<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Lib.
 *
 * @package     local_ned_controller
 * @category    lib
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 *
 * @noinspection PhpUnused
*/

defined('MOODLE_INTERNAL') || die();

define('LOCAL_NED_CONTROLLER_GITHUB_TOKEN', 'b99050d6b5abce53533ea30d27b8f266b998be87');
define('LOCAL_NED_CONTROLLER_GITHUB_USER', 'ned-code');

define('LOCAL_NED_CONTROLLER_REGEX_LIMIT', '/\[\[check_login_limit[ ]*\=[ ]*(\d*)\]\]/m');

use local_ned_controller\ned_notifications as NN;
use local_ned_controller\shared_lib as NED;

require_once(__DIR__.'/local_lib.php');

/**
 * @throws dml_exception
 */
function local_ned_controller_update_plugin_info() {
    global $DB;

    $plugins = $DB->get_records(NED::CTRL);

    foreach ($plugins as $plugin) {
        if (core_plugin_manager::instance()->get_plugin_info(local_ned_controller_get_component_name($plugin))) {
            $installed = 1;
        } else {
            $installed = 0;
        }

        if ($content = local_ned_controller_fetch_version_file_contents(LOCAL_NED_CONTROLLER_GITHUB_USER, $plugin->repository)) {
            $update = new stdClass();
            $update->id = $plugin->id;
            $update->version = $content['plugin_version'];
            $update->installed = $installed;
            $update->timemodified = time();
            $DB->update_record(NED::CTRL, $update);
        }
    }
}

/**
 * @param $plugin
 * @return mixed
 */
function local_ned_controller_get_component_name($plugin) {
    $component = explode('-', $plugin->repository);
    return $component[1];
}

/**
 * @param $plugin
 * @return bool|string
 */
function local_ned_controller_get_plugin_name($plugin) {
    if (($pos = strpos($plugin->repository, '_')) !== false) {
        return substr($plugin->repository, $pos + 1);
    }
    return $plugin->repository;
}

/**
 * @param $plugin
 * @return int|string
 * @throws dml_exception
 */
function local_ned_controller_count_instances($plugin) {
    global $DB;

    $pluginname = local_ned_controller_get_plugin_name($plugin);

    switch ($plugin->plugintype) {
        case 'mod':
            try {
                $count = $DB->count_records_select($pluginname, "course<>0");
            } catch (dml_exception) {
                $count = '-';
            }
            break;
        case 'block':
            $totalcount = $DB->count_records('block_instances', array('blockname' => $pluginname));
            $count = $DB->count_records('block_instances', array('blockname' => $pluginname, 'pagetypepattern' => 'course-view-*'));

            if (!$count) {
                $count = $totalcount;
            }
            break;
        default:
            $count = '-';
    }
    return $count;
}
/**
 * @param $plugin
 * @return int|string
 * @throws dml_exception
 */
function local_ned_controller_count_course_instances($plugin) {
    global $DB;

    $pluginname = local_ned_controller_get_plugin_name($plugin);

    switch ($plugin->plugintype) {
        case 'mod':
            try {
                $sql = "SELECT Count(DISTINCT plg.course) FROM {".$pluginname."} plg WHERE plg.course > 0 ";
                $count = $DB->count_records_sql($sql);
            } catch (dml_exception) {
                $count = '-';
            }
            break;
        case 'block':
            $sql = "SELECT Count(DISTINCT cx.instanceid)
                      FROM {block_instances} bi
                      JOIN {context} cx 
                        ON bi.parentcontextid = cx.id
                     WHERE bi.blockname = ? 
                       AND cx.contextlevel = ?
                       AND bi.pagetypepattern = ?";

            $count = $DB->count_records_sql($sql, array($pluginname, CONTEXT_COURSE, 'course-view-*'));
            break;
        default:
            $count = '-';
    }
    return $count;
}


/**
 * @param $pluginname
 *
 * @return moodle_url|string|null
 */
function local_ned_controller_get_setting_local_urlpath($pluginname){
    global $CFG;
    $setting_path = "/local/$pluginname/settings.php";
    $setting_path_file = "/local/$pluginname/settings_$pluginname.php";
    $path = null;
    if (file_exists($CFG->dirroot.$setting_path_file)) {
        $path = $setting_path_file;
    } elseif(file_exists($CFG->dirroot.$setting_path)) {
        $path = $setting_path.'?section=local&setting='.$pluginname;
    }
    return $path;
}

/**
 * @param \stdClass $plugin
 *
 * @return moodle_url|string
 */
function local_ned_controller_get_setting_url($plugin) {
    global $CFG;
    require_once($CFG->libdir.'/adminlib.php');

    $pluginname = local_ned_controller_get_plugin_name($plugin);
    if ($pluginname == 'custom_menu'){
        $pluginname = 'ned_custom_menu';
    }

    $url = '';
    $setting_name = $plugin->plugintype.'setting'.$pluginname;
    $settings = admin_get_root()->locate($setting_name);

    if ($settings instanceof admin_externalpage) {
        $url = $settings->url;
    } elseif ($settings) {
        $url = new moodle_url("/{$CFG->admin}/settings.php", ['section' => $setting_name]);
    }

    if ($url instanceof moodle_url){
        $url = $url->out(false);
    }
    return $url;
}

/**
 * @param $plugin
 * @return string
 */
function local_ned_controller_get_documentation_url($plugin) {
    $component = local_ned_controller_get_component_name($plugin);
    $urls = [
        NED::CTRL => 'http://ned.ca/plugins/ned-controller/',
        'mod_nedactivitycluster' => 'http://ned.ca/plugins/activity-cluster/',
        NED::NB => 'http://ned.ca/plugins/ned-boost/',
        NED::NC => 'http://ned.ca/plugins/ned-clean/',
        'block_custom_menu' => 'http://ned.ca/plugins/custom-menu/',
        'filter_ned' => 'http://ned.ca/plugins/ned-filter/',
        'format_ned' => 'http://ned.ca/plugins/ned-format/',
        'block_mentor' => 'http://ned.ca/plugins/mentor-manager/',
        NED::SM => 'http://ned.ca/plugins/student-menu/',
        NED::TT => 'http://ned.ca/plugins/teacher-tools/',
    ];
    $url = array_key_exists($component, $urls) ? $urls[$component] : '';
    return $url;
}

/**
 * @param $user
 * @param $repository
 * @return array
 */
function local_ned_controller_fetch_version_file_contents($user, $repository ) {
    $result = array();

    // We have a access token and we can now call the api:
    $api      = new \local_ned_controller\githubapi();
    $response = $api->get_contents_of_repo( $user, $repository, "version.php" );

    $download_url = $api->archive_link($user, $repository);

    $result['download_url'] = $download_url;

    if ( ! empty( $response['content'] ) ) {
        $decoded_text = base64_decode( $response['content'] );

        // escape special characters in the query
        $pattern = preg_quote( "=", '/' );
        // finalise the regular expression, matching the whole line
        $pattern = "/^.*$pattern.*\$/m";
        // search, and store all matching occurences in $matches
        if ( preg_match_all( $pattern, $decoded_text, $matches ) ) {
            foreach ( $matches[0] as $match ) {
                //Search for version number
                if (str_contains($match, '$plugin->version')) {
                    $matched                   = explode( "=", $match );
                    $remove_comments_if_exists = substr( $matched[1], 0, strpos( $matched[1], "//" ) );
                    if ( ! empty( $remove_comments_if_exists ) ) {
                        $result['plugin_version'] = preg_replace( '/[^A-Za-z0-9\-]/', '', $remove_comments_if_exists );
                    } else {
                        $result['plugin_version'] = preg_replace( '/[^A-Za-z0-9\-]/', '', $matched[1] );
                    }
                }

                //Search for release date
                if (str_contains($match, '$plugin->release')) {
                    $matched                   = explode( "=", $match );
                    $remove_comments_if_exists = substr( $matched[1], 0, strpos( $matched[1], "//" ) );
                    if ( ! empty( $remove_comments_if_exists ) ) {
                        $result['plugin_release'] = preg_replace( '/\s+/', '', $remove_comments_if_exists );
                    } else {
                        $result['plugin_release'] = preg_replace( '/\s+/', '', $matched[1] );
                    }

                    $result['plugin_release'] = str_replace( "'", "", $result['plugin_release'] );
                    $result['plugin_release'] = str_replace( ";", "", $result['plugin_release'] );
                }

                //Search for component
                if (str_contains($match, '$plugin->component')) {
                    $matched                   = explode( "=", $match );
                    $remove_comments_if_exists = substr( $matched[1], 0, strpos( $matched[1], "//" ) );
                    if ( ! empty( $remove_comments_if_exists ) ) {
                        $result['component'] = preg_replace( '/\s+/', '', $remove_comments_if_exists );
                    } else {
                        $result['component'] = preg_replace( '/\s+/', '', $matched[1] );
                    }

                    $result['component'] = str_replace( "'", "", $result['component'] );
                    $result['component'] = str_replace( ";", "", $result['component'] );
                }
            }
        }
    } else {
        $result = false;
    }

    return $result;
}

/**
 * Convert the string into JSON format for storage and use.
 * @param string string.
 * @return string JSON or false if cannot convert.
 */
function local_ned_controller_decode($json) {
    $structure = json_decode($json, true);

    if (($structure !== false) && (!empty($structure))) {
        $lines = array();

        foreach ($structure as $category => $categorysettings) {
            $lines[] = $category.', '.implode(', ', $categorysettings);
        }

        return implode(';'.PHP_EOL, $lines);
    } else {
        return '';
    }
}
/**
 * Convert the string into JSON format for storage and use.
 * @param string string.
 *
 * @return mixed true if ok string if error found.
 */
function local_ned_controller_encode($string) {
    $structure = array();

    // Convert string to array.
    $lines = explode(';', $string);
    foreach ($lines as $line) {
        $theline = explode(',', $line);
        // Line must have at least two elements, the category name and the Font Awesome icon.
        if (count($theline) > 1) {
            $theline[0] = ltrim($theline[0]); // Remove newlines on the block name.
            $structure[$theline[0]] = array();
            $structure[$theline[0]][\local_ned_controller\toolbox::$fontawesomekey] = trim($theline[1]);
        }
    }

    $json = json_encode($structure);

    return $json;
}

/**
 * Serves intro attachment files.
 *
 * @param mixed $course course or id of the course
 * @param mixed $cm course module or id of the course module
 * @param context $context
 * @param string $filearea
 * @param array $args
 * @param bool $forcedownload
 * @param array $options additional options affecting the file serving
 *
 * @return bool false if file not found, does not return if found - just send the file
 * @noinspection PhpUnusedParameterInspection
 */
function local_ned_controller_pluginfile($course,
        $cm,
        context $context,
        $filearea,
        $args,
        $forcedownload,
        array $options = array()) {

    $relativepath = join('/', $args);
    $fullpath = join('/', ['', $context->id, NED::CTRL, $filearea, $relativepath]);

    $fs = get_file_storage();
    if (!$file = $fs->get_file_by_hash(sha1($fullpath)) or $file->is_directory()) {
        return false;
    }

    send_stored_file($file, 0, 0, $forcedownload, $options);
    return false;
}

/**
 * Get all rules by criteria
 *
 * @param array $conditions
 * @return array
 */
function local_ned_controller_get_rules($conditions = []) {
    global $DB;
    return $DB->get_records(NED::CTRL.'_rules', $conditions, 'weight');
}

/**
 * Formation of the table for the withdrawal of rules
 *
 * @param $rules
 * @return html_table
 */
function local_ned_controller_table($rules) {
    $rulestable = new html_table();
    $data = [];
    $header = [
        NED::str('name'),
        NED::str('weight'),
        NED::str('status'),
    ];
    foreach ($rules as $rule) {
        $line = new html_table_row();
        $line->cells[] = html_writer::link(new moodle_url('', ['id' => $rule->id]), $rule->name);
        $line->cells[] = $rule->weight;
        $line->cells[] = NED::str($rule->disabled ? 'false' : 'true');
        $data[] = $line;
    }
    $rulestable->head = $header;
    $rulestable->align = array('left', 'left');
    $rulestable->attributes['class'] = 'generaltable';
    $rulestable->data = $data;
    return $rulestable;
}

/**
 * Get data from the active rules
 *
 *
 * @param string $return_first_found_field - optional, name of rule field, if provided, return single first such value from rules
 * @param string|false|\moodle_url $logourl - optional, logo url, if already loaded
 * @param string|false|\moodle_url $compactlogourl - optional, compact logo url, if already loaded
 *
 * @return array|\mixed - [true, $logourl, $compactlogourl, $customcss, $additionalhtmlhead, $additionalhtmltopofbody, $additionalhtmlfooter]
 */
function local_ned_controller_get_active_rules_data($return_first_found_field=false, $logourl='', $compactlogourl='') {

    global $USER;
    static $res = [];
    $userid = $USER->id ?? 0;

    if (!isset($res[$userid])){
        $customcss = '';
        $additionalhtmlhead = '';
        $additionalhtmltopofbody = '';
        $additionalhtmlfooter = '';

        $rules = local_ned_controller_get_rules(['disabled' => 0]);

        // We go through all the rules and look for those that fit our conditions.
        foreach ($rules as $rule) {
            $ruleclass = new \local_ned_controller\rule($rule);
            $ruleid = $ruleclass->is_available();
            if ($ruleid) {
                if ($return_first_found_field){
                    if ($rule->$return_first_found_field ?? false){
                        return $rule->$return_first_found_field;
                    }
                    continue;
                }

                $customcss .= $rule->customcss;
                $additionalhtmlhead .= ($rule->additionalhtmlhead);
                $additionalhtmltopofbody .= ($rule->additionalhtmltopofbody);
                $additionalhtmlfooter .= ($rule->additionalhtmlfooter);

                preg_match_all(LOCAL_NED_CONTROLLER_REGEX_LIMIT, $additionalhtmltopofbody, $matches, PREG_SET_ORDER, 0);
                if (!empty($matches)){
                    if (isset($matches[0][1])){
                        local_ned_controller_apply_concurrent_login_limit($userid, $matches[0][1], session_id());
                        $additionalhtmltopofbody = preg_replace(LOCAL_NED_CONTROLLER_REGEX_LIMIT,'',$additionalhtmltopofbody);
                    }
                }

                $fs = get_file_storage();
                if (empty($logourl)) {
                    $files = $fs->get_area_files(context_system::instance()->id, NED::CTRL, 'logo', $rule->id,
                        "itemid, filepath, filename", false);

                    // There will always be one image, but just in case I call it through foreach so that there is no error if it is more than 2
                    foreach ($files as $file) {
                        $logourl = moodle_url::make_pluginfile_url(
                            $file->get_contextid(),
                            $file->get_component(),
                            $file->get_filearea(),
                            $file->get_itemid(),
                            $file->get_filepath(),
                            $file->get_filename()
                        );
                    }
                }

                if (empty($compactlogourl)) {
                    $files = $fs->get_area_files(context_system::instance()->id, NED::CTRL, 'compactlogo', $rule->id,
                        "itemid, filepath, filename", false);

                    // There will always be one image, but just in case I call it through foreach so that there is no error if it is more than 2
                    foreach ($files as $file) {
                        $compactlogourl = moodle_url::make_pluginfile_url(
                            $file->get_contextid(),
                            $file->get_component(),
                            $file->get_filearea(),
                            $file->get_itemid(),
                            $file->get_filepath(),
                            $file->get_filename()
                        );
                    }
                }
            }
        }

        if ($return_first_found_field){
            return false;
        }

        $res[$userid] = [true, $logourl, $compactlogourl, $customcss, $additionalhtmlhead, $additionalhtmltopofbody, $additionalhtmlfooter];
    }

    return $res[$userid];

}

/**
 * Getting active categories
 *
 * @return array
 */
function local_ned_controller_get_categoty(){
    global $DB;
    return $DB->get_records_menu('course_categories',
                array('visible' => true));
}

/**
 * Triggered as soon as practical on every moodle bootstrap after config has been loaded.
 * The $USER object is available at this point too.
 *
 * @return void
 * @noinspection PhpUnreachableStatementInspection
 */
function local_ned_controller_after_config(){
    // Don't need it really for now
    return;

    /**
     * We don't want to accidentally break ALL site pages,
     *  so we catch everything here with a try-catch.
     */
    try {
        NED::ctrl_after_config();
    } catch (\Exception $e){
        $info = get_exception_info($e);
        $logerrmsg = "[local_ned_controller_after_config()] exception handler: ".$info->message.' '.
            'Debug: '.$info->debuginfo."\n".format_backtrace($info->backtrace, true);
        debugging($logerrmsg, DEBUG_NORMAL, $info->backtrace);
    }
}

/**
 * Give plugin an opportunity touch things before the http headers are sent
 * such as adding additional headers. The return value is ignored.
 */
function local_ned_controller_before_http_headers(){
    /**
     * We don't want to accidentally break ALL site pages,
     *  so we catch everything here with a try-catch.
     */
    try {
        NED::ctrl_before_http_headers();
    } catch (\Exception $e){
        $info = get_exception_info($e);
        $logerrmsg = "[local_ned_controller_before_http_headers()] exception handler: ".$info->message.' '.
            'Debug: '.$info->debuginfo."\n".format_backtrace($info->backtrace, true);
        debugging($logerrmsg, DEBUG_NORMAL, $info->backtrace);
    }
}

/**
 * Give plugins an opportunity to touch the page before JS is finalized.
 */
function local_ned_controller_before_footer(){
    global $CFG, $PAGE;
    if (!empty($CFG->maintenance_enabled) || $PAGE->pagelayout == 'maintenance' || WS_SERVER || AJAX_SCRIPT){
        return;
    }

    if (NN::check_db_tables()){
        $NN = new NN();
        $NN->check_and_show_notifications();
    }

    $DN = new local_ned_controller\deadline_notification();
    $DN->check_and_show_notifications();
}

/**
 * Give plugins an opportunity to inject extra html content. The callback
 * must always return a string containing valid html.
 */
function local_ned_controller_before_standard_top_of_body_html(){
    global $CFG;
    $output = '';

    if (!empty($CFG->messagebar) && isloggedin()) {
        NED::js_call_amd('add_messagebar', 'add', $CFG->messagebar);
    }
    return $output;
}

/**
 * @param $courseid
 *
 * @return bool
 */
function local_ned_controller_has_capability_revokebadge($courseid){
    global $USER;
    $context = context_course::instance($courseid);
    return NED::has_capability('revokebadge', $context, $USER);
}

/**
 * @param $userid
 */
function local_ned_controller_call_revokebadge_js($userid){
    global $COURSE;
    $confirm_text = NED::str('revokebadgeconfirm');
    NED::js_call_amd('revokebadge', 'init', [$userid, $COURSE->id, $confirm_text]);
}

/**
 * @param $userid
 * @param $limit
 * @param null $sid
 * @throws coding_exception
 * @throws dml_exception
 */
function local_ned_controller_apply_concurrent_login_limit($userid, $limit, $sid = null) {
    global $DB;


    if (isguestuser($userid) or empty($userid)) {
        // This applies to real users only!
        return;
    }

    if (empty($limit) or $limit < 0) {
        return;
    }

    $count = $DB->count_records('sessions', array('userid' => $userid));

    if ($count <= $limit) {
        return;
    }

    $i = 0;
    $select = "userid = :userid";
    $params = array('userid' => $userid);
    if ($sid) {
        if ($DB->record_exists('sessions', array('sid' => $sid, 'userid' => $userid))) {
            $select .= " AND sid <> :sid";
            $params['sid'] = $sid;
            $i = 1;
        }
    }

    $sessions = $DB->get_records_select('sessions', $select, $params, 'timecreated DESC', 'id, sid');
    foreach ($sessions as $session) {
        $i++;
        if ($i <= $limit) {
            continue;
        }
        \core\session\manager::kill_session($session->sid);
    }
}

/**
 * Map icons for font-awesome themes.
 */
function local_ned_controller_get_fontawesome_icon_map() {
    return [
        'local_ned_controller:squareo' => 'fa-square-o'
    ];
}

/**
 * Get SCSS to prepend.
 * Return the SCSS to prepend to main SCSS for this theme
 * Note: really works only through the additional themes calls
 * @see theme_config::get_css_content_from_scss() - theme function is called from here
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param theme_config $theme The theme config object.
 *
 * @return string The custom pre SCSS.
 * @noinspection PhpUnusedParameterInspection
 */
function local_ned_controller_get_pre_scss($theme){
    return '';
}

/**
 * NED: load all scss files from the 'scss' dirs in the NED plugins
 * @see \local_ned_controller\shared\C::PLUGIN_DIRS
 * Warning: do NOT load theme files here!
 *
 * Inject additional SCSS.
 * Return the SCSS to append to our main SCSS for this theme
 * Note: really works only through the additional themes calls
 *
 * @see theme_config::get_css_content_from_scss() - theme function is called from here
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param theme_config $theme The theme config object.
 *
 * @return string The custom post SCSS.
 */
function local_ned_controller_get_extra_scss($theme){
    $f_extra_content = function($content, $name){
        $prefix = "\n/** Extra SCSS from $name **/\n";
        $postfix =  "\n/** End extra SCSS from $name **/\n";
        return $prefix.join("\n", $content).$postfix;
    };

    $res_content = [];
    $theme_name = 'theme_'.$theme->name;
    foreach (NED::PLUGIN_DIRS as $plugin => $plugin_dir){
        if (!NED::is_plugin_exists($plugin)) continue;
        if ($theme_name != $plugin && NED::str_starts_with($plugin_dir, '/theme/', false)) continue;

        $dir = NED::path($plugin_dir.'/scss/');
        if (!is_dir($dir)) continue;

        $content = local_ned_controller_load_scss_from_dir($dir);
        if (!empty($content)){
            $res_content[] = $f_extra_content($content, $plugin);
        }
    }

    if (!empty($res_content)){
        return $f_extra_content($res_content, __FUNCTION__);
    }

    return '';
}

/**
 * Load scss files from dir and child dirs
 *
 * @param string $dir - /path/to/dir/with/scss
 * @param array  $content - result array of contents
 *
 * @return array - copy of $content
 */
function local_ned_controller_load_scss_from_dir($dir, &$content=[]){
    if (!is_dir($dir)) return $content;

    if ($dh = opendir($dir)){
        while (($filename = readdir($dh)) !== false){
            if (NED::str_starts_with($filename, ['.', '_'])) continue;

            $filepath = $dir.$filename;
            if (is_dir($filepath)){
                local_ned_controller_load_scss_from_dir($filepath.DIRECTORY_SEPARATOR, $content);
            } else {
                if (!NED::str_ends_with($filename, '.scss')) continue;

                try {
                    $content[] = file_get_contents($dir.$filename);
                } catch (\Exception){
                    continue;
                }
            }
        }
        closedir($dh);
    }

    return $content;
}

/**
 * Add custom CSS.
 * At this moment, core have parsed css image's and font's links, so you should not add them here,
 *  or process them manually
 * @see theme_config::post_process()
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param theme_config $theme The theme config object.
 *
 * @return string The custom CSS.
 * @noinspection PhpUnusedParameterInspection
 */
function local_ned_controller_get_extra_css($theme){
    return '';
}

/**
 * Parses CSS before it is cached.
 * This function can make alterations and replace patterns within the CSS.
 * Please, here - only parse CSS, if you wish to add it: use local_ned_controller_get_extra_css
 * @see local_ned_controller_get_extra_css
 *
 * Note: really works only through the additional themes calls
 * @see theme_config::post_process() - called from here
 *
 * Note: result of this function is saved in theme cache,
 *  so after changes of function work you need purging the caches or enabling theme designer mode
 *
 * @param string $css The CSS
 * @param theme_config $theme The theme config object.
 *
 * @return string The parsed CSS.
 */
function local_ned_controller_process_css($css, $theme){
    $css .= "\n".local_ned_controller_get_extra_css($theme);
    return $css;
}

/**
 * This hook is called from lib/externallib.php and webservice/lib.php
 *
 * @param $externalfunctioninfo
 * @param $params
 *
 * @return bool|null|mixed
 */
function local_ned_controller_override_webservice_execution($externalfunctioninfo, $params){
    $classname = $externalfunctioninfo->classname ?? '';
    $methodname = $externalfunctioninfo->methodname ?? '';
    $result = false;
    $not_false = false;

    if ($classname == 'core_calendar_external' && $methodname == 'get_calendar_event_by_id'){
        // remove description field for modal calendar window
        $result = call_user_func_array([$classname, $methodname], $params);
        if (!empty($result['event'])){
            local_ned_controller\output\core_calendar_renderer::check_event($result['event']);
            $result['event']->description = '';
        }
        $not_false = true;
    }

    if ($not_false && $result === false){
        $result = null;
    }

    return $result;
}

/**
 * Add nodes to myprofile page:
 * - add NGC table
 *
 * @param \core_user\output\myprofile\tree $tree   Tree object
 * @param stdClass                         $user   user object
 * @param bool                             $iscurrentuser
 * @param stdClass                         $course Course object
 *
 * @return bool
 *
 * @noinspection PhpUnusedParameterInspection
 */
function local_ned_controller_myprofile_navigation(\core_user\output\myprofile\tree $tree, $user, $iscurrentuser, $course) {
    if (NED::during_update()) return false;
    if (!NED::role_is_user_default_student($user)) return false;

    $ngc_data = \local_ned_controller\output\ned_grade_controller_render::render_user_profile_data($user, $course);
    if (empty($ngc_data)) return false;

    $after = 'contact';
    if (NED::is_ai_exists()){
        // Create dummy invisible category, to change columns for our main category
        $category_dummy = 'local_ned_controller_dummy';
        $category = new core_user\output\myprofile\category($category_dummy, '', $after, ' hidden display-none ');
        $tree->add_category($category);
        $local_node = new core_user\output\myprofile\node($category_dummy, $category_dummy.'_node', '', null, null, '');
        $tree->add_node($local_node);

        $after = $category_dummy;
    }

    // Add real NGC category
    $category_name = 'local_ned_controller_ned_grade_controller';
    $category_title = NED::str('ngc:profile_table_title');
    $category = new core_user\output\myprofile\category($category_name, $category_title, $after);
    $tree->add_category($category);
    $local_node = new core_user\output\myprofile\node($category_name, $category_name.'_node', '', null, null, $ngc_data);
    $tree->add_node($local_node);

    return true;
}
