<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Filter
 *
 * @package    local_proxy
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_proxy\output;

use local_proxy\helper;
use local_proxy\shared_lib as NED;
use block_ned_teacher_tools\shared_lib as TT;

defined('MOODLE_INTERNAL') || die();
/** @var object $CFG */
require_once($CFG->dirroot . '/mod/assign/locallib.php');

/**
 * Class proxy_overview
 * @package local_proxy\output
 */
class proxy_overview implements \renderable, \templatable {
    /**
     * @var \stdClass
     */
    protected $course;
    /**
     * @var int
     */
    protected $groupid;
    /**
     * @var array
     */
    protected $groups;
    /**
     * @var string
     */
    protected $submissionstatus;
    /**
     * @var string
     */
    protected $gradingstatus;


    /**
     * proxy_overview constructor.
     *
     * @param $course
     * @param $groupid
     * @param $submissionstatus
     * @param $gradingstatus
     */
    public function __construct($course, $groupid, $submissionstatus, $gradingstatus){
        $this->course = $course;
        $this->groups  = NED::get_all_user_course_groups($course->id);
        $this->groupid = 0;
        if ($groupid || is_null($groupid)){
            $this->groupid = NED::get_chosen_group($course->id, $groupid)->id ?? null;
        }
        $this->submissionstatus = $submissionstatus;
        $this->gradingstatus = $gradingstatus;
    }

    /**
     * Export page contents for template
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output){
        $data = new \stdClass();

        // Filter.
        $data->formurl = (new \moodle_url('/local/proxy/overview.php'))->out();
        $data->courseid = $this->course->id;
        $data->hasgroup = !empty($this->groups);
        if (isset($this->groups[$this->groupid])){
            $this->groups[$this->groupid]->selected = true;
        }
        $data->groups = array_values($this->groups);

        $submissionstatuses = ['allclasses', 'submissionmissing', 'submissioncompleted'];
        $data->submissionstatuses = [];
        foreach ($submissionstatuses as $submissionstatus){
            $data->submissionstatuses[] = [
                'id' => $submissionstatus,
                'name' => NED::str($submissionstatus),
                'selected' => $this->submissionstatus === $submissionstatus,
            ];
        }

        $gradingstatuses = ['gradingstatusall', 'gradingstatusnotgraded', 'gradingstatusgraded'];
        $data->gradingstatuses = [];
        foreach ($gradingstatuses as $gradingstatus){
            $data->gradingstatuses[] = [
                'id' => $gradingstatus,
                'name' => NED::str($gradingstatus),
                'selected' => $this->gradingstatus === $gradingstatus
            ];
        }

        // Assignments.
        if (!$assigns = helper::get_proxy_assignments($this->course->id)){
            return $data;
        }

        $classes = [];
        foreach ($assigns as $assign){
            $cm = NED::get_cm_by_params($this->course, NED::ASSIGN, $assign->assignment);
            if (!NED::check_activity_visible_by_cm($cm)) continue;

            foreach ($this->groups as $group){
                if (!empty($this->groupid) && $this->groupid != $group->id) continue;

                $users = NED::get_course_students_by_role($this->course, 0, $group->id, true, false);
                if (empty($users)) continue;

                list($userstotal, $userssubmitted) = helper::get_counted_users_submissions($cm, $users);
                if (!empty($this->submissionstatus) && $this->submissionstatus !== 'allclasses'){
                    if ($this->submissionstatus === 'submissionmissing' && $userssubmitted == $userstotal){
                        continue;
                    } elseif ($this->submissionstatus === 'submissioncompleted' && $userssubmitted == 0){
                        continue;
                    }
                }

                $class = new \stdClass();
                $class->activity = NED::q_cm_link($cm);
                $class->class = $group->name;
                $class->classroomteacher = helper::get_users_by_role(NED::ROLE_CT, $group, true);
                list($window_start_str, $window_end_str, $window_start, $window_end) =
                    NED::dm_get_submission_window($cm, NED::PROXY, null, $group);
                $class->classdeadline = $window_end_str;
                $class->classdeadlinesort = $window_end;

                $time = time();
                if (!$window_start || $window_start > $time){
                    $class->proxystatus = NED::str('pending');
                } elseif ($window_end > $time && $time > $window_start){
                    $class->proxystatus = NED::str('open');
                } elseif ($window_end < $time){
                    $class->proxystatus = NED::str('closed');
                }

                $class->submitted = "$userssubmitted/$userstotal";
                if ($userssubmitted < $userstotal){
                    $class->cellcls = 'bg-red';
                }

                $class->actiontext = TT::str('view');
                $class->actionurl = (new \moodle_url('/local/proxy/submission.php', ['id' => $cm->id, 'groupid' => $group->id]))->out(false);
                $classes[] = $class;
            }
        }

        $data->classes = $classes;

        return $data;
    }
}
