<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * FrontPage notification
 *
 * @package    assignsubmission_ctsubmission
 * @copyright  2022 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace assignsubmission_ctsubmission\output;

use assignsubmission_ctsubmission\shared_lib as NED;
use assignsubmission_ctsubmission\submission;
use assignsubmission_ctsubmission\helper;

defined('MOODLE_INTERNAL') || die();

/** @var object $CFG */
require_once($CFG->dirroot . '/mod/assign/locallib.php');

/**
 * Class ct_alert
 * @package assignsubmission_ctsubmission\output
 */
class ct_alert implements \renderable, \templatable {

    /**
     * Export page contents for template
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output){
        global $USER, $PAGE;

        $data = new \stdClass();

        $courseid = 0;
        if ($PAGE->course->id !== SITEID){
            $courseid = $PAGE->course->id;
        }
        if (!$courses = helper::get_courses([NED::get_full_capability('viewgroupnotifications')], $courseid)){
            return $data;
        }

        $strmisseddeadline = NED::str('misseddeadline');
        $strapproved = NED::str('approved');
        $strwaitingforsubmission = NED::str('waitingforsubmission');
        $strwaitingforapproval = NED::str('waitingforapproval');
        $strdeadlinerequired = NED::str('deadlinerequired');
        $strpending = NED::str('pending');
        $strrevisionrequired = NED::str('revisionrequired');

        $is_admin = is_siteadmin();
        $alerts_any = [$strmisseddeadline, $strrevisionrequired];
        $alert_submit = $strwaitingforsubmission;
        $alert_approve = $strwaitingforapproval;

        $time = time();
        $data->stractionrequired = 0;

        $data->url = NED::url('~/overview.php');
        if ($courseid){
            $data->url->param('courseid', $courseid);
        }

        foreach ($courses as $course){
            if (!$assigns = helper::get_ctsubmission_assignments($course->id)){
                continue;
            }

            $groups = NED::get_all_user_course_groups($course->id);
            foreach ($assigns as $assign){
                $cm = NED::get_cm_by_params($course, NED::ASSIGN, $assign->assignment);
                if (!NED::check_activity_visible_by_cm($cm)) continue;

                if ($is_admin){
                    $hasapprovecap = $hasubmitcap = true;
                } else {
                    $hasapprovecap = NED::has_capability('approve', $cm->context);
                    $hasubmitcap = NED::has_capability('submit', $cm->context);
                    if (!$hasapprovecap && !$hasubmitcap) continue;
                }

                $assign = NED::ned_assign_by_cm($cm);
                /** @var \assign_submission_ctsubmission $assignsubmissionct */
                $assignsubmissionct = $assign->get_submission_plugin_by_type('ctsubmission');

                foreach ($groups as $group){
                    $has_ct = NED::get_users_by_role(NED::ROLE_CT, $cm->course, $cm->id, $group->id, true);
                    if (!$has_ct) continue;

                    $approved = 0;
                    $submissionstatus = '';
                    $submission = $assign->get_group_submission($USER->id, $group->id, true, -1);
                    if ($ctsubmission = submission::get_record(['assignment' => $cm->instance, 'submission' => $submission->id])){
                        $approved = $ctsubmission->get('approved');
                    }

                    list($window_start_str, $window_end_str, $window_start, $window_end) =
                        NED::dm_get_submission_window($cm, NED::CTSUBMISSION, null, $group, true);
                    $count = $assignsubmissionct->count_files($submission->id, ASSIGNSUBMISSION_CTSUBMISSION_FILEAREA);

                    if ($count){
                        if (!$approved){
                            $submissionstatus = $strwaitingforapproval;
                        }
                    } else {
                        if ($window_start){
                            if ($window_start > $time){
                                $submissionstatus = $strpending;
                            } elseif ($window_end >= $time && $time >= $window_start){
                                $submissionstatus = $strwaitingforsubmission;
                            } elseif ($time > $window_end){
                                $submissionstatus = $strmisseddeadline;
                            }
                        } else {
                            $submissionstatus = $strdeadlinerequired;
                        }
                    }

                    if ($approved == ASSIGNSUBMISSION_CTSUBMISSION_APPROVED){
                        $submissionstatus = $strapproved;
                    } elseif ($approved == ASSIGNSUBMISSION_CTSUBMISSION_REVISIONREQUESTED){
                        $submissionstatus = $strrevisionrequired;
                    }

                    if (!$submissionstatus){
                        continue;
                    }

                    $alert = in_array($submissionstatus, $alerts_any);
                    if (!$alert){
                        $alert = ($hasubmitcap && $submissionstatus == $alert_submit) || ($hasapprovecap && $submissionstatus == $alert_approve);
                        if (!$alert) continue;
                    }

                    $data->stractionrequired++;

                    if ($data->stractionrequired){
                        break;
                    }
                }
            }
        }

        return $data;
    }
}
