<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Filter
 *
 * @package    assignsubmission_ctsubmission
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace assignsubmission_ctsubmission\output;

use assignsubmission_ctsubmission\shared_lib as NED;
use block_ned_teacher_tools\shared_lib as TT;

defined('MOODLE_INTERNAL') || die();

/** @var object $CFG */
require_once($CFG->dirroot . '/mod/assign/locallib.php');

/**
 * Class filter
 * @package assignsubmission_ctsubmission\output
 */
class ct_submission implements \renderable, \templatable {
    /**
     * @var \assign $assign
     */
    protected $assign;
    protected $groupid;

    /**
     * assign_summary constructor.
     * @param $assign
     */
    public function __construct($assign, $groupid){
        $this->assign = $assign;
        $this->groupid = $groupid;
    }

    /**
     * Export page contents for template
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output){
        $data = new \stdClass();

        $cm = $this->assign->get_course_module();
        if (!NED::check_activity_visible_by_cm($cm)){
            return $data;
        }

        $this->groupid = $group->id ?? 0;
        $course = $this->assign->get_course();
        $group = NED::get_chosen_group($course->id, $this->groupid);
        $br = \html_writer::empty_tag('br');

        $data->activity = NED::q_cm_link($cm);
        $data->class = $group->name ?? '';
        $data->ct = NED::get_role_users_fullnames(NED::ROLE_CT, $course->id, $this->groupid, 0, true, true);

        // Users.
        $users = NED::get_course_students_by_role($course, 0, $this->groupid, true, false);
        if (empty($users)){
            return $data;
        }

        foreach ($users as $user){
            $user->fullname = NED::q_user_link($user, $course->id);
            list($window_start_str, $window_end_str, $window_start, $window_end) =
                NED::dm_get_submission_window($cm, NED::CTSUBMISSION, $user);

            $dmtimezone = TT::get_user_dm_timezone_ctx($cm->context);
            $user->completiondeadline = $window_start_str;
            $user->submissiondeadline = $window_end_str;

            // Countdown.
            $remainingtime = $window_end - time();
            $user->submissioncountdown = NED::time_counter_to_str_max($remainingtime, 2, true);

            // Submitted
            $submission = $this->assign->get_user_submission($user->id, false);
            if (isset($submission->timemodified) && $submission->status !== ASSIGN_SUBMISSION_STATUS_NEW){
                $user->submitted = userdate($submission->timemodified, TT::str('strtimeformat'), $dmtimezone);
            }

            // Files.
            $out = [];
            if ($submission){
                $fs = get_file_storage();
                $files = $fs->get_area_files($cm->context->id, 'assignsubmission_proxy', 'submission_proxies', $submission->id);
                foreach ($files as $file){
                    if (!$file->is_directory()){
                        $out[] = NED::file_get_link($file, true, true);
                    }
                }
                if ($out){
                    $user->files = implode($br, $out);
                }
            }
            // Grade status.
            $gradestatus = $this->assign->get_grading_status($user->id);
            $user->gradestatus = get_string($gradestatus, 'assign');

            // Action.
            if ($gradestatus == ASSIGN_GRADING_STATUS_GRADED){
                $user->actiontext = "";
            } elseif (!empty($submission->status) && $submission->status === ASSIGN_SUBMISSION_STATUS_SUBMITTED){
                if ($gradestatus === ASSIGN_GRADING_STATUS_GRADED){
                    $user->actiontext = get_string('view');
                } else {
                    $user->actiontext = get_string('edit');
                }
            } else {
                $user->actiontext = get_string('submit');
            }

            $user->actionurl = (new \moodle_url('/local/proxy/submit.php', [
                'id' => $cm->id,
                'groupid' => $this->groupid,
                'userid' => $user->id
            ]))->out(false);
        }
        $data->students = array_values($users);

        return $data;
    }
}
