<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * This file contains the definition for the library class for file submission plugin
 *
 * This class provides all the functionality for the new assign module.
 *
 * @package assignsubmission_ctsubmission
 * @copyright 2021 Michael Gardener <mgardener@cissq.com>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

use \assignsubmission_ctsubmission\shared_lib as NED;

// File areas for file submission assignment.
define('ASSIGNSUBMISSION_CTSUBMISSION_MAXSUMMARYFILES', 5);
define('ASSIGNSUBMISSION_CTSUBMISSION_FILEAREA', 'submission_ctsubmissions');
define('ASSIGNSUBMISSION_CTSUBMISSION_APPROVED', 1);
define('ASSIGNSUBMISSION_CTSUBMISSION_REVISIONREQUESTED', 2);

/**
 * Library class for file submission plugin extending submission plugin base class
 *
 * @package   assignsubmission_ctsubmission
 * @copyright 2021 Michael Gardener <mgardener@cissq.com>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class assign_submission_ctsubmission extends \local_ned_controller\assign_proxy_submission_base {
    /**
     * @return string
     */
    static public function get_table(){
        return 'assignsubmission_ctsubmis';
    }

    /**
     * @return array [config_key => form_data_key]
     */
    static public function get_config_settings(){
        return [
            'maxfilesubmissions' => 'assignsubmission_ctsubmission_maxfiles',
            'maxsubmissionsizebytes' => 'assignsubmission_ctsubmission_maxsizebytes',
            'submissionwindow' => 'assignsubmission_ctsubmission_submissionwindow',
            'ctsubmissionactivity' => 'assignsubmission_ctsubmission_ctsubmissionactivity',
            'filetypeslist' => 'assignsubmission_ctsubmission_filetypes'
        ];
    }

    /**
     * Keys from @see get_config_settings() which influence on the deadline
     *
     * @return string[]
     */
    static public function get_deadline_configs(){
        return ['submissionwindow', 'ctsubmissionactivity'];
    }

    /**
     * @return string
     */
    static public function get_file_area(){
        return ASSIGNSUBMISSION_CTSUBMISSION_FILEAREA;
    }

    /**
     * @return int
     */
    static public function get_maxsummaryfiles(){
        return ASSIGNSUBMISSION_CTSUBMISSION_MAXSUMMARYFILES;
    }

    /**
     * Get the default setting for file submission plugin
     *
     * @param MoodleQuickForm $mform The form to add elements to
     *
     * @return void
     */
    public function get_settings(MoodleQuickForm $mform) {
        global $CFG, $COURSE;

        $this->disable_assign_deadlines($mform);

        if ($this->assignment->has_instance()){
            $this->_set_was_inactive_if_disabled();

            $defaultmaxfilesubmissions = $this->get_config('maxfilesubmissions');
            $defaultmaxsubmissionsizebytes = $this->get_config('maxsubmissionsizebytes');
            $defaultfiletypes = $this->get_config('filetypeslist');
            $defaultsubmissionwindow = $this->get_config('submissionwindow');
            $defaultctsubmissionactivity = $this->get_config('ctsubmissionactivity');
        } else {
            $defaultmaxfilesubmissions = NED::get_config('maxfiles');
            $defaultmaxsubmissionsizebytes = NED::get_config('maxbytes');
            $defaultfiletypes = NED::get_config('filetypes');
            $defaultsubmissionwindow = 0;
            $defaultctsubmissionactivity = '';
        }

        $mform->addElement('duration', 'assignsubmission_ctsubmission_submissionwindow', NED::str('submissionwindow'), ['optional' => true, 'defaultunit' => HOURSECS]);
        $mform->setDefault('assignsubmission_ctsubmission_submissionwindow', $defaultsubmissionwindow);
        $mform->hideIf('assignsubmission_ctsubmission_submissionwindow', 'assignsubmission_ctsubmission_enabled', 'notchecked');


        $mform->addElement('select', 'assignsubmission_ctsubmission_ctsubmissionactivity', NED::str('ctsubmissionactivity'), $this->get_ctsubmission_activities());
        $mform->setDefault('assignsubmission_ctsubmission_ctsubmissionactivity', $defaultctsubmissionactivity);
        $mform->hideIf('assignsubmission_ctsubmission_ctsubmissionactivity', 'assignsubmission_ctsubmission_enabled', 'notchecked');

        $defaultfiletypes = (string)$defaultfiletypes;

        $options = array();
        for ($i = 1; $i <= NED::get_config('maxfiles'); $i++) {
            $options[$i] = $i;
        }

        $name = NED::str('maxfilessubmission');
        $mform->addElement('select', 'assignsubmission_ctsubmission_maxfiles', $name, $options);
        $mform->addHelpButton('assignsubmission_ctsubmission_maxfiles',
                              'maxfilessubmission',
                              'assignsubmission_ctsubmission');
        $mform->setDefault('assignsubmission_ctsubmission_maxfiles', $defaultmaxfilesubmissions);
        $mform->hideIf('assignsubmission_ctsubmission_maxfiles', 'assignsubmission_ctsubmission_enabled', 'notchecked');

        $choices = get_max_upload_sizes($CFG->maxbytes,
                                        $COURSE->maxbytes,
                                        NED::get_config('maxbytes'));

        $name = NED::str('maximumsubmissionsize');
        $mform->addElement('select', 'assignsubmission_ctsubmission_maxsizebytes', $name, $choices);
        $mform->addHelpButton('assignsubmission_ctsubmission_maxsizebytes',
                              'maximumsubmissionsize',
                              'assignsubmission_ctsubmission');
        $mform->setDefault('assignsubmission_ctsubmission_maxsizebytes', $defaultmaxsubmissionsizebytes);
        $mform->hideIf('assignsubmission_ctsubmission_maxsizebytes',
                           'assignsubmission_ctsubmission_enabled',
                           'notchecked');

        $name = NED::str('acceptedfiletypes');
        $mform->addElement('filetypes', 'assignsubmission_ctsubmission_filetypes', $name);
        $mform->addHelpButton('assignsubmission_ctsubmission_filetypes', 'acceptedfiletypes', 'assignsubmission_ctsubmission');
        $mform->setDefault('assignsubmission_ctsubmission_filetypes', $defaultfiletypes);
        $mform->hideIf('assignsubmission_ctsubmission_filetypes', 'assignsubmission_ctsubmission_enabled', 'notchecked');
    }

    /**
     * Save the settings for file submission plugin
     *
     * @param stdClass $formdata
     *
     * @return bool
     */
    public function save_settings(stdClass $formdata) {
        if (empty($formdata->assignsubmission_ctsubmission_filetypes)){
            $formdata->assignsubmission_ctsubmission_filetypes = '';
        }

        return parent::save_settings($formdata);
    }

    /**
     * Save the files and trigger plagiarism plugin, if enabled,
     * to scan the uploaded files via events trigger
     *
     * @param stdClass|null $submission
     * @param stdClass $data
     *
     * @return bool
     */
    public function save($submission, $data){
        global $USER, $DB;

        $fileoptions = $this->_get_file_options();

        $data = file_postupdate_standard_filemanager($data,
                                                     'files',
                                                     $fileoptions,
                                                     $this->assignment->get_context(),
                                                     'assignsubmission_ctsubmission',
                                                     ASSIGNSUBMISSION_CTSUBMISSION_FILEAREA,
                                                    $submission->id);

        $fs = get_file_storage();
        $files = $fs->get_area_files($this->assignment->get_context()->id,
                                     'assignsubmission_ctsubmission',
                                     ASSIGNSUBMISSION_CTSUBMISSION_FILEAREA,
                                     $submission->id,
                                     'id',
                                     false);

        $count = $this->count_files($submission->id, ASSIGNSUBMISSION_CTSUBMISSION_FILEAREA);

        $params = array(
            'context' => context_module::instance($this->assignment->get_course_module()->id),
            'courseid' => $this->assignment->get_course()->id,
            'objectid' => $submission->id,
            'other' => array(
                'content' => '',
                'pathnamehashes' => array_keys($files)
            )
        );
        if (!empty($submission->userid) && ($submission->userid != $USER->id)){
            $params['relateduserid'] = $submission->userid;
        }
        if ($this->assignment->is_blind_marking()){
            $params['anonymous'] = 1;
        }
        $event = \assignsubmission_ctsubmission\event\assessable_uploaded::create($params);
        $event->set_legacy_files($files);
        $event->trigger();

        return true;
    }

    /**
     * @return array
     */
    public function get_ctsubmission_activities(){
        return $this->_get_proxy_filter_activities([NED::ASSIGN, NED::QUIZ]);
    }
}
