<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * This file contains the definition for the library class for file submission plugin
 *
 * This class provides all the functionality for the new assign module.
 *
 * @package assignsubmission_proxy
 * @copyright 2021 Michael Gardener <mgardener@cissq.com>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

use assignsubmission_proxy\shared_lib as NED;

// File areas for file submission assignment.
define('ASSIGNSUBMISSION_PROXY_MAXSUMMARYFILES', 5);
define('ASSIGNSUBMISSION_PROXY_FILEAREA', 'submission_proxies');

/**
 * Library class for file submission plugin extending submission plugin base class
 *
 * @package   assignsubmission_proxy
 * @copyright 2021 Michael Gardener <mgardener@cissq.com>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class assign_submission_proxy extends \local_ned_controller\assign_proxy_submission_base {
    /**
     * @return string
     */
    static public function get_table(){
        return 'assignsubmission_proxy';
    }

    /**
     * @return array [config_key => form_data_key]
     */
    static public function get_config_settings(){
        return [
            'maxfilesubmissions' => 'assignsubmission_proxy_maxfiles',
            'maxsubmissionsizebytes' => 'assignsubmission_proxy_maxsizebytes',
            'submissionwindow' => 'assignsubmission_proxy_submissionwindow',
            'proxyactivity' => 'assignsubmission_proxy_proxyactivity',
            'filetypeslist' => 'assignsubmission_proxy_filetypes'
        ];
    }

    /**
     * Keys from @see get_config_settings() which influence on the deadline
     *
     * @return string[]
     */
    static public function get_deadline_configs(){
        return ['submissionwindow', 'proxyactivity'];
    }

    /**
     * @return string
     */
    static public function get_file_area(){
        return ASSIGNSUBMISSION_PROXY_FILEAREA;
    }

    /**
     * @return int
     */
    static public function get_maxsummaryfiles(){
        return ASSIGNSUBMISSION_PROXY_MAXSUMMARYFILES;
    }

    /**
     * Get the default setting for file submission plugin
     *
     * @param MoodleQuickForm $mform The form to add elements to
     *
     * @return void
     */
    public function get_settings(MoodleQuickForm $mform){
        global $CFG, $COURSE;

        $this->disable_assign_deadlines($mform);

        if ($this->assignment->has_instance()){
            $this->_set_was_inactive_if_disabled();

            $defaultmaxfilesubmissions = $this->get_config('maxfilesubmissions');
            $defaultmaxsubmissionsizebytes = $this->get_config('maxsubmissionsizebytes');
            $defaultfiletypes = $this->get_config('filetypeslist');
            $defaultsubmissionwindow = $this->get_config('submissionwindow');
            $defaultproxyactivity = $this->get_config('proxyactivity');
        } else {
            $defaultmaxfilesubmissions = NED::get_config('maxfiles');
            $defaultmaxsubmissionsizebytes = NED::get_config( 'maxbytes');
            $defaultfiletypes = NED::get_config('filetypes');
            $defaultsubmissionwindow = 0;
            $defaultproxyactivity = '';
        }

        $mform->addElement('duration', 'assignsubmission_proxy_submissionwindow', NED::str('submissionwindow'), ['optional' => true, 'defaultunit' => HOURSECS]);
        $mform->setDefault('assignsubmission_proxy_submissionwindow', $defaultsubmissionwindow);
        $mform->hideIf('assignsubmission_proxy_submissionwindow', 'assignsubmission_proxy_enabled', 'notchecked');


        $mform->addElement('select', 'assignsubmission_proxy_proxyactivity', NED::str('proxyactivity'), $this->get_proxy_activities());
        $mform->setDefault('assignsubmission_proxy_proxyactivity', $defaultproxyactivity);
        $mform->hideIf('assignsubmission_proxy_proxyactivity', 'assignsubmission_proxy_enabled', 'notchecked');

        $defaultfiletypes = (string)$defaultfiletypes;

        $options = array();
        for ($i = 1; $i <= NED::get_config('maxfiles'); $i++){
            $options[$i] = $i;
        }

        $name = NED::str('maxfilessubmission');
        $mform->addElement('select', 'assignsubmission_proxy_maxfiles', $name, $options);
        $mform->addHelpButton('assignsubmission_proxy_maxfiles',
                              'maxfilessubmission',
                              'assignsubmission_proxy');
        $mform->setDefault('assignsubmission_proxy_maxfiles', $defaultmaxfilesubmissions);
        $mform->hideIf('assignsubmission_proxy_maxfiles', 'assignsubmission_proxy_enabled', 'notchecked');

        $choices = get_max_upload_sizes($CFG->maxbytes,
                                        $COURSE->maxbytes,
                                        get_config('assignsubmission_proxy', 'maxbytes'));

        $name = NED::str('maximumsubmissionsize');
        $mform->addElement('select', 'assignsubmission_proxy_maxsizebytes', $name, $choices);
        $mform->addHelpButton('assignsubmission_proxy_maxsizebytes',
                              'maximumsubmissionsize',
                              'assignsubmission_proxy');
        $mform->setDefault('assignsubmission_proxy_maxsizebytes', $defaultmaxsubmissionsizebytes);
        $mform->hideIf('assignsubmission_proxy_maxsizebytes',
                           'assignsubmission_proxy_enabled',
                           'notchecked');

        $name = NED::str('acceptedfiletypes');
        $mform->addElement('filetypes', 'assignsubmission_proxy_filetypes', $name);
        $mform->addHelpButton('assignsubmission_proxy_filetypes', 'acceptedfiletypes', 'assignsubmission_proxy');
        $mform->setDefault('assignsubmission_proxy_filetypes', $defaultfiletypes);
        $mform->hideIf('assignsubmission_proxy_filetypes', 'assignsubmission_proxy_enabled', 'notchecked');
    }

    /**
     * Save the settings for file submission plugin
     *
     * @param stdClass $formdata
     *
     * @return bool
     */
    public function save_settings(stdClass $formdata){
        if (empty($formdata->assignsubmission_proxy_filetypes)){
            $formdata->assignsubmission_proxy_filetypes = '';
        }

        return parent::save_settings($formdata);
    }

    /**
     * Save the files and trigger plagiarism plugin, if enabled,
     * to scan the uploaded files via events trigger
     *
     * @param stdClass $submission
     * @param stdClass $data
     *
     * @return bool
     */
    public function save(stdClass $submission, stdClass $data){
        global $USER, $DB;

        $fileoptions = $this->_get_file_options();

        $data = file_postupdate_standard_filemanager($data,
                                                     'files',
                                                     $fileoptions,
                                                     $this->assignment->get_context(),
                                                     'assignsubmission_proxy',
                                                     ASSIGNSUBMISSION_PROXY_FILEAREA,
                                                     $submission->id);

        $filesubmission = $this->_get_file_submission($submission->id);

        // Plagiarism code event trigger when files are uploaded.

        $fs = get_file_storage();
        $files = $fs->get_area_files($this->assignment->get_context()->id,
                                     'assignsubmission_proxy',
                                     ASSIGNSUBMISSION_PROXY_FILEAREA,
                                     $submission->id,
                                     'id',
                                     false);

        $count = $this->count_files($submission->id, ASSIGNSUBMISSION_PROXY_FILEAREA);

        $params = array(
            'context' => context_module::instance($this->assignment->get_course_module()->id),
            'courseid' => $this->assignment->get_course()->id,
            'objectid' => $submission->id,
            'other' => array(
                'content' => '',
                'pathnamehashes' => array_keys($files)
            )
        );
        if (!empty($submission->userid) && ($submission->userid != $USER->id)){
            $params['relateduserid'] = $submission->userid;
        }
        if ($this->assignment->is_blind_marking()){
            $params['anonymous'] = 1;
        }
        $event = \assignsubmission_proxy\event\assessable_uploaded::create($params);
        $event->set_legacy_files($files);
        $event->trigger();

        $groupname = null;
        $groupid = 0;
        // Get the group name as other fields are not transcribed in the logs and this information is important.
        if (empty($submission->userid) && !empty($submission->groupid)){
            $groupname = $DB->get_field('groups', 'name', array('id' => $submission->groupid), MUST_EXIST);
            $groupid = $submission->groupid;
        } else {
            $params['relateduserid'] = $submission->userid;
        }

        // Unset the objectid and other field from params for use in submission events.
        unset($params['objectid']);
        unset($params['other']);
        $params['other'] = array(
            'submissionid' => $submission->id,
            'submissionattempt' => $submission->attemptnumber,
            'submissionstatus' => $submission->status,
            'filesubmissioncount' => $count,
            'groupid' => $groupid,
            'groupname' => $groupname
        );

        if ($filesubmission){
            $filesubmission->numfiles = $this->count_files($submission->id,
                                                           ASSIGNSUBMISSION_PROXY_FILEAREA);
            $updatestatus = $DB->update_record('assignsubmission_proxy', $filesubmission);
            $params['objectid'] = $filesubmission->id;

            $event = \assignsubmission_proxy\event\submission_updated::create($params);
            $event->set_assign($this->assignment);
            $event->trigger();
            return $updatestatus;
        } else {
            $filesubmission = new stdClass();
            $filesubmission->numfiles = $this->count_files($submission->id,
                                                           ASSIGNSUBMISSION_PROXY_FILEAREA);
            $filesubmission->submission = $submission->id;
            $filesubmission->assignment = $this->assignment->get_instance()->id;
            $filesubmission->id = $DB->insert_record('assignsubmission_proxy', $filesubmission);
            $params['objectid'] = $filesubmission->id;

            $event = \assignsubmission_proxy\event\submission_created::create($params);
            $event->set_assign($this->assignment);
            $event->trigger();
            return $filesubmission->id > 0;
        }
    }

    /**
     * @return array
     */
    public function get_proxy_activities(){
        return $this->_get_proxy_filter_activities(NED::QUIZ);
    }
}
