<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Incident Report Class
 *
 * @package   mod_nedactivitycluster
 * @copyright 2020 Michael Gardener <mgardener@cissq.com>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace mod_nedactivitycluster;


use stdClass;
use moodle_url;
use pix_icon;
use html_writer;
use context_system;
use calendar_event;

defined('MOODLE_INTERNAL') || die();


/**
 * Class helper
 * @package mod_nedactivitycluster
 */
abstract class helper {

    /**
     * @param $surveyid
     * @param string $questionname
     * @return array
     * @throws \dml_exception
     */
    public static function get_survey_data($surveyid, $questionname) {
        global $DB, $USER;

        $sql = "SELECT rb.id, rb.response_id, rb.question_id, rb.response
                      FROM {questionnaire_response_text} rb
                INNER JOIN {questionnaire_response} r ON rb.response_id = r.id
                INNER JOIN {questionnaire_question} q ON rb.question_id = q.id
                INNER JOIN {questionnaire_question_type} qt ON q.type_id = qt.typeid
                     WHERE r.questionnaireid = ? AND r.complete = 'y' AND r.userid = ? AND q.name = ? AND qt.type='Numeric'";

        if ($response = $DB->get_record_sql($sql, [$surveyid, $USER->id, $questionname])) {
            return (float) $response->response;
        }

        return null;
    }

    /**
     * @param $mod
     * @return string
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_completion_status($mod) {
        global $DB, $USER, $OUTPUT;
        if ($mod->completion) {
            if ($modcompletion = $DB->get_record('course_modules_completion', ['coursemoduleid' => $mod->id, 'userid' => $USER->id])) {
                $state = $modcompletion->completionstate;
                $overrideby = $modcompletion->overrideby;
                $date = userdate($modcompletion->timemodified);
            } else {
                $state = COMPLETION_INCOMPLETE;
                $overrideby = 0;
                $date = '';
            }

            // Work out how it corresponds to an icon
            switch($state) {
                case COMPLETION_INCOMPLETE :
                    $completiontype = 'n'.($overrideby ? '-override' : '');
                    break;
                case COMPLETION_COMPLETE :
                    $completiontype = 'y'.($overrideby ? '-override' : '');
                    break;
                case COMPLETION_COMPLETE_PASS :
                    $completiontype = 'pass';
                    break;
                case COMPLETION_COMPLETE_FAIL :
                    $completiontype = 'fail';
                    break;
                case COMPLETION_UNKNOWN:
                default:
                    $completiontype = '';
                    break;
            }

            $completiontrackingstring = $mod->completion == COMPLETION_TRACKING_AUTOMATIC ? 'auto' : 'manual';
            $completionicon = 'completion-' . $completiontrackingstring. '-' . $completiontype;

            if ($overrideby) {
                $overridebyuser = \core_user::get_user($overrideby, '*', MUST_EXIST);
                $describe = get_string('completion-' . $completiontype, 'completion', fullname($overridebyuser));
            } else {
                $describe = get_string('completion-' . $completiontype, 'completion');
            }
            $a = new StdClass;
            $a->state=$describe;
            $a->date=$date;
            $a->user = fullname($USER, has_capability('moodle/site:viewfullnames', $mod->context));
            $a->activity = format_string($mod->name, true, array('context' => $mod->context));
            $fulldescribe = get_string('progress-title','completion',$a);

            return $OUTPUT->pix_icon('i/' . $completionicon, s($fulldescribe));
        }

        return '';
    }

    /**
     * @param $courseactivities
     * @param $mods
     * @return bool|void
     */
    public static function has_questionnaire($courseactivities, $mods) {
        foreach ($mods as $cmid) {
            if (!isset($courseactivities[$cmid])) {
                continue;
            }
            if ($courseactivities[$cmid]->modname == 'questionnaire') {
                return true;
            }
            return false;
        }
    }
}