<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Importer.
 *
 * @package     report_ghs
 * @copyright   2020 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/group/lib.php');

use grade_scale;
use stdClass;
use context_system;
use csv_import_reader;
use block_ned_teacher_tools\deadline_manager;

/**
 * Import framework form.
 *
 * @package     report_ghs
 * @copyright   2020 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class enrollment_importer {

    /** @var string $error The errors message from reading the xml */
    var $error = '';

    /** @var array $flat The flat competencies tree */
    var $flat = array();
    /** @var array $framework The framework info */
    var $framework = array();
    var $mappings = array();
    var $importid = 0;
    var $importer = null;
    var $foundheaders = array();

    public function fail($msg) {
        $this->error = $msg;
        return false;
    }

    public function get_importid() {
        return $this->importid;
    }

    public static function list_required_headers() {
        return array(
            0 => get_string('coursecode', 'report_ghs'),
            1 => get_string('group', 'report_ghs'),
            2 => get_string('groupid', 'report_ghs'),
            3 => get_string('ctname', 'report_ghs'),
            4 => get_string('otname', 'report_ghs'),
            5 => get_string('startdate', 'report_ghs'),
            6 => get_string('enddate', 'report_ghs'),
            7 => get_string('dmrequired', 'report_ghs'),
            8 => get_string('gmessaging', 'report_ghs'),
        );
    }

    public function list_found_headers() {
        return $this->foundheaders;
    }

    private function read_mapping_data($data) {
        if ($data) {
            return array(
                'course' => $data->header0,
                'group' => $data->header1,
                'groupid' => $data->header2,
                'ctname' => $data->header3,
                'otname' => $data->header4,
                'startdate' => $data->header5,
                'enddate' => $data->header6,
                'dmrequired' => $data->header7,
                'gmessaging' => $data->header8,
            );
        } else {
            return array(
                'course' => 0,
                'group' => 1,
                'groupid' => 2,
                'ctname' => 3,
                'otname' => 4,
                'startdate' => 5,
                'enddate' => 6,
                'dmrequired' => 7,
                'gmessaging' => 8,
            );
        }
    }

    private function get_row_data($row, $index) {
        if ($index < 0) {
            return '';
        }
        return isset($row[$index]) ? $row[$index] : '';
    }

    /**
     * Constructor - parses the raw text for sanity.
     */
    public function __construct($text = null, $encoding = null, $delimiter = null, $importid = 0, $mappingdata = null) {
        global $CFG;

        // The idnumber is concatenated with the category names.
        require_once($CFG->libdir . '/csvlib.class.php');

        $type = 'competency_framework';

        if (!$importid) {
            if ($text === null) {
                return;
            }
            $this->importid = csv_import_reader::get_new_iid($type);

            $this->importer = new csv_import_reader($this->importid, $type);

            if (!$this->importer->load_csv_content($text, $encoding, $delimiter)) {
                $this->fail(get_string('invalidimportfile', 'report_ghs'));
                $this->importer->cleanup();
                return;
            }

        } else {
            $this->importid = $importid;

            $this->importer = new csv_import_reader($this->importid, $type);
        }


        if (!$this->importer->init()) {
            $this->fail(get_string('invalidimportfile', 'report_ghs'));
            $this->importer->cleanup();
            return;
        }

        $this->foundheaders = $this->importer->get_columns();

        $domainid = 1;

        $flat = array();
        $csvline = null;

        while ($row = $this->importer->next()) {
            $mapping = $this->read_mapping_data($mappingdata);

            $course = $this->get_row_data($row, $mapping['course']);
            $group = $this->get_row_data($row, $mapping['group']);
            $groupid = $this->get_row_data($row, $mapping['groupid']);
            $ctname = $this->get_row_data($row, $mapping['ctname']);
            $otname = $this->get_row_data($row, $mapping['otname']);
            $startdate = $this->get_row_data($row, $mapping['startdate']);
            $enddate = $this->get_row_data($row, $mapping['enddate']);
            $dmrequired = $this->get_row_data($row, $mapping['dmrequired']);
            $gmessaging = $this->get_row_data($row, $mapping['gmessaging']);

            $csvline = new stdClass();
            $csvline->course = $course;
            $csvline->group = $group;
            $csvline->groupid = $groupid;
            $csvline->ctname = $ctname;
            $csvline->otname = $otname;
            $csvline->startdate = $startdate;
            $csvline->enddate = $enddate;
            $csvline->dmrequired = $dmrequired;
            $csvline->gmessaging = $gmessaging;

            $flat[] = $csvline;
        }

        $this->framework = $flat;

        $this->importer->close();
        if ($this->framework == null) {
            $this->fail(get_string('invalidimportfile', 'report_ghs'));
            return;
        }
    }

    /**
     * @return array of errors from parsing the xml.
     */
    public function get_error() {
        return $this->error;
    }

    /**
     * Do the job.
     */
    public function import() {
        global $DB, $USER;

        $scheduleoptions = deadline_manager::get_schedule_options();

        $records = $this->framework;
        $otrole = $DB->get_record('role', array('shortname' => 'online-teacher'));
        $ctrole = $DB->get_record('role', array('shortname' => 'classroom-teacher'));

        if (!$enrolmanual = enrol_get_plugin('manual')) {
            throw new \coding_exception('Can not instantiate enrol_manual');
        }

        foreach ($records as $record) {
            if (!$course = $DB->get_record('course', array('shortname' => $record->course))) {
                continue;
            }

            if (!$group = $DB->get_record('groups', array('courseid' => $course->id, 'id' => $record->groupid))) {
                continue;
            }

            $coursecontext = \context_course::instance($course->id);

            if ($ctrole && !empty($record->ctname)) {
                if ($ct = $DB->get_record('user', array('firstname' => $record->ctname, 'deleted' => 0))) {
                    // Assign CT
                    if ($instance = $DB->get_record('enrol', array('courseid' => $course->id, 'enrol' => 'manual'))) {
                        $enrolmanual->enrol_user($instance, $ct->id, $ctrole->id);
                    }

                    groups_add_member($group, $ct);
                }
            }

            if ($otrole && !empty($record->otname)) {
                if ($ot = $DB->get_record('user', array('firstname' => $record->otname, 'deleted' => 0))) {
                    // Assign OT
                    if ($instance = $DB->get_record('enrol', array('courseid' => $course->id, 'enrol' => 'manual'))) {
                        $enrolmanual->enrol_user($instance, $ot->id, $otrole->id);
                    }
                    groups_add_member($group, $ot);
                }
            }

            $groupupdate = new stdClass();
            $groupupdate->id = $group->id;
            $gupdate = false;

            if (!empty($record->startdate)) {
                $startdate = strtotime($record->startdate);
                if ($startdate !== false) {
                    // Set start date.
                    $gupdate = true;
                    $groupupdate->startdate = $startdate;

                }
            }

            if (!empty($record->enddate)) {
                $enddate = strtotime($record->enddate);
                if ($enddate !== false) {
                    // Set end date.
                    $gupdate = true;
                    $groupupdate->enddate = $enddate;
                }
            }

            if (!empty($record->dmrequired)) {
                $key = array_search($record->dmrequired, $scheduleoptions);
                if ($key !== false) {
                    // Set DM schedule option.
                    $gupdate = true;
                    $groupupdate->schedule = $key;
                }
            }

            if ($gupdate) {
                $DB->update_record('groups', $groupupdate);
            }

            if (!empty($record->gmessaging)) {
                $record->gmessaging = strtolower($record->gmessaging);
                if ($record->gmessaging == 'yes') {
                    $enablemessaging = 1;
                } else if ($record->gmessaging == 'no') {
                    $enablemessaging = 0;
                }

                // Group conversation messaging.
                if (isset($enablemessaging)) {
                    if (\core_message\api::can_create_group_conversation($USER->id, $coursecontext)) {
                        if ($conversation = \core_message\api::get_conversation_by_area('core_group', 'groups', $group->id, $coursecontext->id)) {
                            if ($enablemessaging && $enablemessaging != $conversation->enabled) {
                                \core_message\api::enable_conversation($conversation->id);
                            }
                            if (!$enablemessaging && $enablemessaging != $conversation->enabled) {
                                \core_message\api::disable_conversation($conversation->id);
                            }
                            \core_message\api::update_conversation_name($conversation->id, $group->name);
                        } else {
                            if (!empty($enablemessaging)) {
                                $conversation = \core_message\api::create_conversation(
                                    \core_message\api::MESSAGE_CONVERSATION_TYPE_GROUP,
                                    [],
                                    $group->name,
                                    \core_message\api::MESSAGE_CONVERSATION_ENABLED,
                                    'core_group',
                                    'groups',
                                    $group->id,
                                    $coursecontext->id
                                );

                                // Add members to conversation if they exists in the group.
                                if ($groupmemberroles = groups_get_members_by_role($group->id, $group->courseid, 'u.id')) {
                                    $users = [];
                                    foreach ($groupmemberroles as $roleid => $roledata) {
                                        foreach ($roledata->users as $member) {
                                            $users[] = $member->id;
                                        }
                                    }
                                    \core_message\api::add_members_to_conversation($users, $conversation->id);
                                }
                            }
                        }
                    }
                }
            }
        }
        $this->importer->cleanup();
        return true;
    }
}
