<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * HELPER
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs;

global $CFG;
if (!class_exists("\\Krizalys\\Onedrive\\Onedrive")) {
    require_once($CFG->dirroot . '/report/ghs/thirdparty/vendor/autoload.php');
}

use block_ned_teacher_tools\activity_status;
use block_ned_teacher_tools\deadline_manager;
use context_system;
use context_course;
use GuzzleHttp\Client as GuzzleHttpClient;
use Krizalys\Onedrive\Client;
use Krizalys\Onedrive\Constant\ConflictBehavior;
use Krizalys\Onedrive\Onedrive;
use local_ned_controller\shared\util;
use Microsoft\Graph\Graph;
use moodle_url;
use pix_icon;
use html_writer;
use stdClass;
use report_ghs\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class helper
 * @package report_ghs
 */
class helper  {
    const TAG_MIDTERM = ['Midterm Point', 'Midterm'];
    const TAG_FINAL = ['Exam', 'Final Exam', 'Final Project'];

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @param bool $export
     * @return false|float|string
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function kica_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'enrolldate':
            case 'startdate':
            case 'enddate':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                }
                break;
            case 'rawgrade':
            case 'kicaavg':
            case 'knowledge':
            case 'inquiry':
            case 'communication':
            case 'application':
                $var = '';
                if (!is_null($data->$column)) {
                    $var = round($data->$column, 2);
                }
                break;
            case 'kica70':
            case 'kica30':
                $var = '';
                if (!empty($data->$column) && $data->$column > 0) {
                    $var = round($data->$column, 2);
                }
                break;
            case 'kicadiff':
                $var = '';
                if (!is_null($data->kica70) && !is_null($data->kica30) && $data->kica30 > 0) {
                    $var = round(($data->kica70 - $data->kica30), 2);
                }
                break;
            case 'kicalink':
            case 'kicalinkshort':
                $action = array(
                    'url' => new moodle_url('/local/kica/grade_user.php',
                        ['courseid' => $data->courseid, 'prevpage' => '/local/kica/grade_user', 'mode' => 0, 'user' => $data->userid]
                    ),
                    'icon' => new pix_icon('t/grades', get_string('grades')),
                    'attributes' => array('class' => 'action-edit', 'role' => 'button', 'target' => '_blank')
                );

                if ($export) {
                    $var = $action['url']->out(false);
                } else {
                    $var = $OUTPUT->action_icon($action['url'], $action['icon'], null, $action['attributes']);
                    if ($data->gradingerror) {
                        $var .= '<i class="fa fa-exclamation-triangle text-danger" aria-hidden="true"></i>';
                    }
                }
                break;
            case 'date':
                $var = '-';
                if ($data->$column) {
                    $var = date("m/d/Y", strtotime($data->$column));
                }
                break;
            case 'active':
                $var = '';
                if ($context = context_course::instance($data->courseid, IGNORE_MISSING)) {
                    $var = 'N';
                    if (is_enrolled($context, $data->userid, '', true)) {
                        $var = 'Y';
                    } else {
                        $data->rowcls = 'not-enrolled';
                    }
                }
                break;
            case 'fcategory':
                $var = self::get_fcatagory($data->categorypath);
                break;
            case 'lcategory':
                $var = $data->category;
                break;
            case 'lastname':
                if (!$export && class_exists('\local_academic_integrity\ai_flag')) {
                    $var = $data->$column . \local_academic_integrity\ai_flag::flag($data->userid);
                } else{
                    $var = $data->$column;
                }
                break;
            default:
                $encoding = mb_detect_encoding($data->$column);
                $var = @iconv($encoding, "Windows-1252", $data->$column);
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @return false|string
     * @throws \dml_exception
     */
    public static function ghs_sla_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'enrolldate':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                }
                break;
            case 'ot':
            case 'ct':
                if (empty($data->$column)) {
                    if ($export) {
                        $var = get_string('missing', 'report_ghs');
                    } else {
                        $pixicon = new pix_icon('i/risk_xss', get_string('missing', 'report_ghs'));
                        $var = $OUTPUT->render($pixicon);
                    }
                } else {
                    $var = $data->$column;
                }
                break;
            case 'basecategory':
                $var = '';
                if ($data->$column > 0) {
                    if ($base = $DB->get_record('course_categories', ['id' => $data->$column])) {
                        $var = $base->name;
                    }
                }
                break;
            case 'submissionstatus':
                $var = '';
                if ($data->$column) {
                    $var = get_string($data->$column, 'report_ghs');
                }
                break;
            case 'course':
                $var = '';
                $shortname = substr($data->$column, 0, 10);
                $courseurl = new moodle_url('/course/view.php',
                    ['id' => $data->courseid]
                );
                if ($export) {
                    $var = $shortname;
                } else {
                    $var = html_writer::link($courseurl, $shortname);
                }
                break;
            case 'slaot':
                $var = '-';
                if ($data->$column > 0) {
                    $var = get_string('yes');
                } else {
                    $var = get_string('no');
                }
                break;
            case 'finalactivity':
                $var = '';
                if ($data->$column == 1) {
                    $var = get_string('final', 'report_ghs');
                } else  if ($data->$column == 2) {
                    $var = get_string('mid', 'report_ghs');
                }
                break;
            case 'gradebooklink':
                $var = '';

                if (empty($data->instancegrade)) {
                    $icon = new pix_icon('i/risk_xss', get_string('grades'));
                } else {
                    $icon = new pix_icon('t/grades', get_string('grades'));
                }
                $gradeitem = $DB->get_record('grade_items', ['itemtype' => 'mod', 'itemmodule' => $data->module, 'iteminstance'=> $data->instance]);

                $groupid = self::get_user_course_groupid($data->userid, $data->courseid);

                $action = array(
                    'url' => new moodle_url('/grade/report/singleview/index.php',
                        ['id' => $data->courseid, 'item' => 'grade', 'itemid' => $gradeitem->id, 'group' => $groupid]
                    ),
                    'icon' => $icon,
                    'attributes' => array('class' => 'action-edit', 'role' => 'button', 'target' => '_blank')
                );

                if ($export) {
                    if (empty($data->instancegrade)) {
                        $var = $action['url']->out(false).'#missing';
                    } else {
                        $var = $action['url']->out(false).'#graded';
                    }
                } else {
                    $var = $OUTPUT->action_icon($action['url'], $action['icon'], null, $action['attributes']);
                }

                break;
            case 'activitygradelink':
                $var = '';
                if (empty($data->timegradedmoodle)) {
                    $icon = new pix_icon('i/risk_xss', get_string('grades'));
                } else {
                    $icon = new pix_icon('t/grades', get_string('grades'));
                }

                $groupid = self::get_user_course_groupid($data->userid, $data->courseid);

                $action = array(
                    'url' => new moodle_url('/mod/'.$data->module.'/view.php',
                        ['id' => $data->cmid, 'action' => 'grading', 'group' => $groupid]
                    ),
                    'icon' => $icon,
                    'attributes' => array('class' => 'action-edit', 'role' => 'button', 'target' => '_blank')
                );

                if ($export) {
                    if (empty($data->timegradedmoodle)) {
                        $var = $action['url']->out(false).'#missing';
                    } else {
                        $var = $action['url']->out(false).'#graded';
                    }
                } else {
                    $var = $OUTPUT->action_icon($action['url'], $action['icon'], null, $action['attributes']);
                }
                break;

            case 'kicagradebooklink':
                if (!$data->kicaenabled) {
                    return 'N/A';
                }

                $cm = NED::get_cm_by_params($data->courseid, $data->module, $data->instance);
                $kicaitem = NED::ki_get_by_cm($cm);
                if (!$kicaitem) {
                    return 'N/A';
                }
                if (empty($data->timegradedkica)) {
                    $icon = new pix_icon('i/risk_xss', get_string('grades'));
                } else {
                    $icon = new pix_icon('t/grades', get_string('grades'));
                }
                $groupid = self::get_user_course_groupid($data->userid, $data->courseid);
                $action = array(
                    'url' => new moodle_url('/local/kica/grade_activity.php', ['courseid' => $data->courseid, 'itemid' => $kicaitem->id, 'group' => $groupid]),
                    'icon' => $icon,
                    'attributes' => array('class' => 'action-edit', 'role' => 'button', 'target' => '_blank')
                );

                if ($export) {
                    if (empty($data->timegradedkica)) {
                        $var = $action['url']->out(false).'#missing';
                    } else {
                        $var = $action['url']->out(false).'#graded';
                    }
                } else {
                    $var = $OUTPUT->action_icon($action['url'], $action['icon'], null, $action['attributes']);
                }
                break;
            case 'duedate':
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                } else {
                    $pixicon = new pix_icon('i/risk_xss', get_string('missing', 'report_ghs'));
                    $var = $OUTPUT->render($pixicon);
                }
                break;
            case 'timesubmission':
            case 'timegraded':
            case 'timegradedmoodle':
            case 'timegradedkica':
                $var = '';
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                }
                break;
            case 'activityname':
                $shortname = substr($data->$column, 0, 10);
                $url = new moodle_url('/mod/'.$data->module.'/view.php', ['id' => $data->cmid]);
                if ($export) {
                    $var = $shortname;
                } else {
                    $var = html_writer::link($url, $shortname, ['target' => '_blank']);
                }
                break;
            case 'firstname':
                if ($data->numofgroups > 1) {
                    $var = '<i class="fa fa-th-large icon-multiple-groups"></i> ' . $data->$column;
                } else {
                    $var = $data->$column;
                }
                break;
            default:
                $var = $data->$column;
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @return false|string
     * @throws \dml_exception
     */
    public static function enrollment_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'coursestart':
            case 'courseend':
            case 'groupstart':
            case 'groupend':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                }
                break;
            case 'basecategory':
                $var = '';
                if ($data->$column > 0) {
                    if ($base = $DB->get_record('course_categories', ['id' => $data->$column])) {
                        $var = $base->name;
                    }
                }
                break;
            case 'firstcourse':
                $sql = "SELECT MIN(ue.timestart) 
                          FROM {course} c
                          JOIN {enrol} en 
                            ON en.courseid = c.id
                          JOIN {user_enrolments} ue
                            ON ue.enrolid = en.id
                         WHERE c.id {$pageparams['insql']} 
                           AND ue.userid = ?
                           AND ue.timestart > 0";
                $var = '-';
                $pageparams['params'][] = $data->userid;
                if ($firstcourse = $DB->get_field_sql($sql,  $pageparams['params'])) {
                    $var = date("m/d/Y", $firstcourse);
                }
                break;
            default:
                $var = $data->$column;
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @return false|string
     * @throws \dml_exception
     */
    public static function enddate_extension_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
            case 'dateapplied':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'groupend':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                }
                break;
            case 'school':
                $var = '-';
                if ($data->$column) {
                    $var = $data->$column;
                }
                break;
            case 'changetype':
                $var = '-';
                if ($data->$column) {
                    $var = get_string($data->$column, 'report_ghs');
                }
                break;
            case 'change':
                $var = date('d M Y', $data->enddateorig) . ' > ' . date('d M Y', $data->enddate);
                break;
            case 'appliedto':
                $name = '';
                $url = null;

                if ($data->changetype == 'classenddate') {
                    if ($group = $DB->get_record('groups', ['id' => $data->groupid])) {
                        $name = $group->name;
                    }
                    $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php',
                        ['courseid' => $data->courseid, 'group' => $data->groupid]
                    );
                } else {
                    if ($user = $DB->get_record('user', ['id' => $data->userid])) {
                        $name = fullname($user);
                        $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php',
                            ['courseid' => $data->courseid, 'group' => $data->groupid, 'user' => $data->userid, 'p' => 'user', 'prevpage' => 'deadline_manager']
                        );
                    }
                }

                if ($export) {
                    if ($name) {
                        $var = $url->out(false);
                    }
                } else {
                    if ($name) {
                        $var = html_writer::link(
                            $url->out(false),
                            $name,
                            ['target' => '_blank']
                        );
                    }
                }
                break;
            case 'submitter':
                if ($export) {
                    $var = $data->$column;
                } else {
                    $var = html_writer::link(
                        (new moodle_url('/user/profile.php', ['id' => $data->submitterid]))->out(),
                        $data->$column,
                        ['target' => '_blank']
                    );
                }
                break;
            default:
                $var = $data->$column;
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @return false|string
     * @throws \dml_exception
     */
    public static function english_proficiency_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'school':
                $var = '-';
                if ($data->$column) {
                    if ($export) {
                        $var = $data->$column;
                    } else {
                        $url = new moodle_url('/local/schoolmanager/view.php', ['schoolid' => $data->schoolid, 'view' => 'students']);
                        $var = html_writer::link($url, $data->$column, ['target' => '_blank']);
                    }
                }
                break;
            case 'student':
                $var = '-';
                $truncatedname = substr($data->$column, 0, 15);
                if ($export) {
                    $var = $data->$column;
                } else {
                    if (!empty($data->userid)) {
                        $flags = NED::get_ai_flag($data->userid);
                        $url = new moodle_url('/user/profile.php', ['id' => $data->userid]);
                        $var = $flags . ' ' . html_writer::link($url, $truncatedname, ['target' => '_blank']);
                    } else {
                        $flags = self::grade_discrepancy_flag(0, $data->id);
                        $var = $flags . ' ' . $truncatedname;
                    }
                }

                break;
            case 'ossltscore':
                $var = $data->$column;
                if (!$export && is_number($data->$column) && $data->$column < 300) {
                    $var = html_writer::span($data->$column, 'color-red');
                }
                break;
            case 'esldo':
            case 'esleo':
            case 'eng3u':
            case 'eng4u':
            case 'olc4o':
                if (!$export) {
                    $inprogress = strpos($data->$column, '(');
                    if ($inprogress !== false && $courses = helper::get_proficiency_courses($data->userid, $column)) {
                        $course = reset($courses);
                        $url = new moodle_url('/blocks/ned_teacher_tools/student_progress.php', ['courseid' => $course->courseid, 'setuser' => $data->userid]);
                        $data->$column = html_writer::link($url, $data->$column, ['target' => '_blank']);
                    }
                }
                $var = $data->$column;
                break;
            default:
                $var = $data->$column;
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @return false|string
     * @throws \dml_exception
     */
    public static function class_deadlines_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'school':
                $var = '-';
                if ($data->$column) {
                    if ($export) {
                        $var = $data->$column;
                    } else {
                        $url = new moodle_url('/local/schoolmanager/view.php', ['schoolid' => $data->cohortid, 'view' => 'students']);
                        $var = html_writer::link($url, $data->$column, ['target' => '_blank']);
                    }
                }
                break;
            case 'course':
                $var = '-';
                if ($data->$column) {
                    if ($export) {
                        $var = $data->$column;
                    } else {
                        $url = new moodle_url('/course/view.php', ['id' => $data->courseid,]);
                        $var = html_writer::link($url, $data->$column, ['target' => '_blank']);
                    }
                }
                break;
            case 'class':
                $var = '-';
                if ($data->groupname) {
                    if ($export) {
                        $var = $data->groupname;
                    } else {
                        $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php', [
                            'courseid' => $data->courseid,
                            'group' => $data->groupid,
                        ]);
                        $var = html_writer::link($url, $data->groupname, ['target' => '_blank']);
                    }
                }
                break;
            case 'activity':
                $var = '-';
                if ($data->$column) {
                    if ($export) {
                        $var = $data->$column;
                    } else {
                        $url = new moodle_url('/mod/'.$data->module.'/view.php', ['id' => $data->cmid]);
                        $var = html_writer::link($url, $data->$column, ['target' => '_blank']);
                    }
                }
                break;
            case 'deadline':
                $var = '-';
                if (!empty($data->$column)) {
                    $var = date('d M Y', $data->$column);
                }
                break;
            case 'students':
                $var = $data->numberofstudents . '/' . $data->totalstudents;
                break;
            case 'deadlinetype':
                $var = '-';
                if (!empty($data->$column)) {
                    $var = get_string($data->$column, 'report_ghs');
                }
                break;
            case 'countdown':
                $var = util::time_counter_to_str_max($data->$column);
                break;
            default:
                $var = $data->$column;
        }

        return $var;
    }
    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @return false|string
     * @throws \dml_exception
     */
    public static function grading_time_monitor_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'grader_':
                if ($export) {
                    $var = $data->$column;
                } else {
                    $url = new moodle_url('/blocks/ned_teacher_tools/grading_tracker.php', ['grader' => $data->graderid]);
                    $var = html_writer::link($url, $data->$column);
                }
                break;
            default:
                if (strpos($column, 'day_') === 0) {
                    $var = round(($data->$column / 60), 1);
                    if ($var >= 7) {
                        $data->{'cls_' .$column} = 'text-red';
                    }
                } else if (strpos($column, 'week_') === 0) {
                    $var = round(($data->$column / 60), 1);
                    if ($var >= 35) {
                        $data->{'cls_' .$column} = 'text-red';
                    }
                } else {
                    $var = $data->$column;
                }
        }

        return $var;
    }

    public static function grade_discrepancy_flag($user_or_id, $epid = 0) {
        global $DB;

        $output = '';
        $userid = NED::get_id($user_or_id);


        if (!has_capability('report/ghs:viewossltengdiscrepancy', context_system::instance())) {
            return $output;
        }

        $data = null;
        if (!empty($userid)) {
            $data = $DB->get_record('report_ghs_english_proficien', ['userid' => $userid]);
        } else if ($epid) {
            $data = $DB->get_record('report_ghs_english_proficien', ['id' => $epid]);
        }

        if (!empty($data)) {
            return self::check_discrepancy($data);
        }

        return $output;
    }

    /**
     * @param $data
     * @return string|void
     */
    public static function check_discrepancy($data) {
        if ($data->ossltstatus !== "Fail") {
            return '';
        }

        if (empty($data->eng3u) || $data->eng3u === 'Not Found') {
            $eng3u = $data->eng3u_grade;
        } else {
            $eng3u = $data->eng3u;
        }

        if (empty($data->eng4u) || $data->eng4u === 'Not Found') {
            $eng4u = $data->eng4u_grade;
        } else {
            $eng4u = $data->eng4u;
        }

        if (($eng3u >= 75 || $eng4u >= 75)) {
            return '<span class="grade-discrepancy-icon"><i class="fa fa-star-half-o"></i></span>';
        }
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @return false|string
     * @throws \dml_exception
     */
    public static function activity_extension_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
            case 'dateapplied':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'groupend':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                }
                break;
            case 'school':
                $var = '-';
                if ($data->$column) {
                    $var = $data->$column;
                }
                break;
            case 'changetype':
                $var = get_string($data->$column, 'report_ghs');
                break;
            case 'activity':
                $var = '-';
                $instance = $DB->get_record($data->modname, ['id' => $data->instance]);
                $url = new moodle_url('/mod/'.$data->modname.'/view.php', ['id' => $data->cmid]);
                if ($export) {
                    $var = $url->out(false);
                } else {
                    $var = html_writer::link(
                        $url->out(false),
                        $instance->name,
                        ['target' => '_blank']
                    );
                }
                break;
            case 'change':
                $var = '';
                if (!empty($data->duedateorig)) {
                    $var .= date('d M Y', $data->duedateorig);
                }
                if (!empty($data->duedateorig) || !empty($data->duedate)) {
                    $var .= ' > ';
                }
                if (!empty($data->duedate)) {
                    $var .= date('d M Y', $data->duedate);
                }
                break;
            case 'appliedto':
                $var = '';
                $name = '';
                $url = null;
                if ($user = $DB->get_record('user', ['id' => $data->userid])) {
                    $name = fullname($user);
                    $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php',
                        ['courseid' => $data->courseid, 'group' => $data->groupid, 'user' => $data->userid, 'p' => 'user', 'prevpage' => 'deadline_manager']
                    );
                }
                if ($name) {
                    if ($export) {
                        $var = $name;
                    } else {
                        $var = html_writer::link(
                            $url->out(false),
                            $name,
                            ['target' => '_blank']
                        );
                    }
                }
                break;
            case 'appliedtourl':
                $var = '';
                $url = null;
                if ($user = $DB->get_record('user', ['id' => $data->userid])) {
                    $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php',
                        ['courseid' => $data->courseid, 'group' => $data->groupid, 'user' => $data->userid, 'p' => 'user', 'prevpage' => 'deadline_manager']
                    );
                }
                if ($url) {
                    $var = $url->out(false);
                }
                break;
            case 'submitter':
                if ($export) {
                    $var = $data->$column;
                } else {
                    $var = html_writer::link(
                        (new moodle_url('/user/profile.php', ['id' => $data->submitterid]))->out(),
                        $data->$column,
                        ['target' => '_blank']
                    );
                }
                break;
            default:
                $var = $data->$column;
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @return false|string
     * @throws \dml_exception
     */
    public static function group_enrollment_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';
        $cellcls = $column.'cls';

        $contextcourse = context_course::instance($data->courseid);

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'startdate':
            case 'enddate':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                }
                break;
            case 'totaldays':
                $var = '-';
                if ($data->startdate > 0 && $data->enddate > 0) {
                    $var = $data->$column;
                    if (!$export && $data->$column > 365) {
                        $data->$cellcls = 'bg-alert';
                        $var .= ' <i class="fa fa-exclamation-triangle" aria-hidden="true"></i>';
                    }
                }
                break;
            case 'gmessaging':
                $var = '-';
                if ($data->$column > 0) {
                    $var = get_string('yes');
                } else {
                    $var = get_string('no');
                }
                break;
            case 'course':
                $courseurl = new moodle_url('/course/view.php', ['id' => $data->courseid]);
                if ($export) {
                    $var = $courseurl->out();
                } else {
                    $var = html_writer::link(
                        $courseurl,
                        $data->$column,
                        ['target' => '_blank']
                    );
                }
                break;
            case 'course_txt':
                $var = $data->course;
                break;
            case 'fcategory':
                $var = self::get_fcatagory($data->categorypath);
                break;
            case 'lcategory':
                $var = $data->category;
                break;
            case 'coursecode':
                $var = $data->$column;

                if ($data->enrolid && has_capability('enrol/manual:enrol', $contextcourse)) {
                    $enrolurl = new moodle_url('/enrol/manual/manage.php', ['enrolid' => $data->enrolid]);
                    if ($export) {
                        $var = $enrolurl->out();
                    } else {
                        $var .= ' ' . html_writer::link($enrolurl,
                                '<i class="fa fa-plus-circle" aria-hidden="true"></i>',
                                ['target' => '_blank']
                            );
                    }
                }
                break;
            case 'coursecode_txt':
                $var = $data->coursecode;
                break;
            case 'groupname':
                $var = $data->$column;

                if (has_capability('moodle/grade:viewall', $contextcourse)) {
                    $progressreporturl = new moodle_url('/blocks/ned_teacher_tools/progress_report.php', [
                        'courseid' => $data->courseid,
                        'tag' => 0,
                        'group' => $data->groupid
                    ]);
                    if ($export) {
                        $var = $progressreporturl->out(false);
                    } else {
                        $var = html_writer::link(
                            $progressreporturl,
                            $data->$column,
                            ['target' => '_blank']
                        );
                    }
                }

                if (!$export && $data->enrolid && has_capability('moodle/course:managegroups', $contextcourse)) {
                    $var .= ' '. html_writer::link(
                            new moodle_url('/group/members.php', ['group' => $data->groupid]),
                            '<i class="fa fa-plus-circle" aria-hidden="true"></i>',
                            ['target' => '_blank']
                        );
                }
                break;
            case 'groupname_txt':
                $var = $data->groupname;
                break;
            case 'dmrequired':
                $var = '-';
                $scheduleoptions = \block_ned_teacher_tools\deadline_manager::get_schedule_options();

                $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php', ['courseid' => $data->courseid, 'p' =>'group', 'group' => $data->groupid]);
                if ($export) {
                    $var = $scheduleoptions[$data->$column];
                } else {
                    $var = html_writer::link($url, $scheduleoptions[$data->$column]);
                }
                break;
            case 'dmstatus':
                $var = '-';
                $scheduleoptions = \block_ned_teacher_tools\deadline_manager::get_schedule_options();

                $dmrequired = $scheduleoptions[$data->dmrequired];

                $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php', ['courseid' => $data->courseid, 'p' =>'group', 'group' => $data->groupid]);

                if ($export || !has_capability('moodle/grade:viewall', $contextcourse)) {
                    $var = ($dmrequired == 'None') ? $dmrequired : $data->$column;;
                } else {
                    $status = ($dmrequired == 'None') ? $dmrequired : $data->$column;
                    if ($status === 'Incomplete') {
                        $data->$cellcls = 'bg-alert';
                        $status .= ' <i class="fa fa-exclamation-triangle" aria-hidden="true"></i>';
                    }
                    $var = html_writer::link($url, $status, ['target' => '_blank']);
                }
                break;
            default:
                $var = $data->$column;
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @param bool $export
     * @return false|float|string
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function missed_grade_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'duedate':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                }
                break;
            case 'kicagradex':
                $var = '';
                if (!is_null($data->$column)) {
                    $var = round($data->$column, 2) . " / " . round($data->{$column.'max'}, 2);
                }
                break;
            case 'activitygrade':
                $var = '';

                if (empty($data->$column)) {
                    $icon = new pix_icon('i/risk_xss', get_string('grades'));
                } else {
                    $icon = new pix_icon('t/grades', get_string('grades'));
                }
                $gradeitem = $DB->get_record('grade_items', ['itemtype' => 'mod', 'itemmodule' => $data->module, 'iteminstance'=> $data->instance]);
                $action = array(
                    'url' => new moodle_url('/grade/report/singleview/index.php',
                        ['id' => $data->courseid, 'item' => 'grade', 'itemid' => $gradeitem->id]
                    ),
                    'icon' => $icon,
                    'attributes' => array('class' => 'action-edit', 'role' => 'button', 'target' => '_blank')
                );

                if ($export) {
                    $var = round($data->$column, 2) . " / " . round($data->{$column.'max'}, 2);
                } else {
                    if (empty($data->$column)) {
                        $var = $OUTPUT->action_icon($action['url'], $action['icon'], null, $action['attributes']);
                    } else {
                        $var = round($data->$column, 2) . " / " . round($data->{$column.'max'}, 2);
                    }
                }

                break;
            case 'kicagrade':
                if (!NED::get_kica_enabled($data->courseid)) {
                    return 'N/A';
                }

                $cm = NED::get_cm_by_params($data->courseid, $data->module, $data->instance);
                $kicaitem = NED::ki_get_by_cm($cm);
                if (!$kicaitem){
                    return 'N/A';
                }
                if (empty($data->timegradedkica)) {
                    $icon = new pix_icon('i/risk_xss', get_string('grades'));
                } else {
                    $icon = new pix_icon('t/grades', get_string('grades'));
                }
                $action = array(
                    'url' => new moodle_url('/local/kica/grade_activity.php', ['courseid' => $data->courseid, 'itemid' => $kicaitem->id]),
                    'icon' => $icon,
                    'attributes' => array('class' => 'action-edit', 'role' => 'button', 'target' => '_blank')
                );

                if ($export) {
                    if (empty($data->timegradedkica)) {
                        $var = $action['url']->out(false).'#missing';
                    } else {
                        $var = $action['url']->out(false).'#graded';
                    }
                } else {
                    $var = $OUTPUT->action_icon($action['url'], $action['icon'], null, $action['attributes']);
                }
                break;
            case 'completed':
            case 'excluded':
            case 'skipped':
                $var = '';
                if (!empty($data->skipped)) {
                    $var = get_string('yes');
                } else {
                    $var = get_string('no');
                }
                break;
            case 'date':
                $var = '-';
                if ($data->$column) {
                    $var = date("m/d/Y", strtotime($data->$column));
                }
                break;
            case 'activityname':
                $shortname = substr($data->$column, 0, 10);
                $url = new moodle_url('/mod/'.$data->module.'/view.php', ['id' => $data->cmid]);
                if ($export) {
                    $var = $shortname;
                } else {
                    $var = html_writer::link($url, $shortname, ['target' => '_blank']);
                }
                break;
            default:
                $var = $data->$column;
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @return false|string
     * @throws \dml_exception
     */
    public static function dm_overview_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'groupname':
            case 'groupnameurl':
                $var = '-';
                $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php',
                    ['courseid' => $data->courseid, 'group' => $data->groupid, 'p' => 'group', 'prevpage' => 'deadline_manager']);
                if ($export) {
                    if ($column == 'groupnameurl') {
                        $var = $url->out(false);
                    } else {
                        $var = $data->$column;
                    }
                } else {
                    $var = html_writer::link(
                        $url->out(false),
                        $data->$column,
                        ['target' => '_blank']
                    );
                }
                break;
            case 'appliedtourl':
                $var = '';
                $url = null;
                if ($user = $DB->get_record('user', ['id' => $data->userid])) {
                    $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php',
                        ['courseid' => $data->courseid, 'group' => $data->groupid, 'user' => $data->userid, 'p' => 'user', 'prevpage' => 'deadline_manager']
                    );
                }
                if ($url) {
                    $var = $url->out(false);
                }
                break;
            default:
                $var = '-';
                if ($data->$column) {
                    $var = $data->$column;
                }
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @param bool $export
     * @return false|float|string
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function grade_changes_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'date':
                $var = '-';
                if ($data->timegrade > 0) {
                    $var = date("m/d/Y g:i A", $data->timegrade);
                }
                break;
            case 'gradebefore':
            case 'gradeafter':
                $var = '';
                if (!is_null($data->$column)) {
                    $var = round($data->$column, 2);
                }
                break;
            case 'fcategory':
                $var = self::get_fcatagory($data->categorypath);
                break;
            case 'lcategory':
                $var = $data->category;
                break;
            case 'timedifference':
                $hours = floor($data->$column / 3600);
                $minutes = floor(($data->$column / 60) % 60);
                $seconds = $data->$column % 60;

                $var = "$hours hr $minutes min $seconds sec";

                break;
            default:
                $encoding = mb_detect_encoding($data->$column);
                $var = @iconv($encoding, "Windows-1252", $data->$column);
        }

        return $var;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @param bool $export
     * @return false|float|string
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function missed_grade_class($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'activitygrade':
                $var = '';
                if (!is_null($data->$column) && $data->$column == 0) {
                    $var = 'bg-orange';
                } else if ($data->duedate < time() && is_null($data->activitygrade)) {
                    $var = 'bg-red';
                }
                break;
            case 'kicagrade':
                $var = '';
                if (!is_null($data->$column) && $data->$column == 0) {
                    $var = 'bg-orange';
                } else if ($data->duedate < time() && is_null($data->kicagrade)) {
                    $var = 'bg-red';
                }
                break;
            case 'skipped':
                $var = '';
                if (!empty($data->skipped)) {
                    $var = 'bg-orange';
                }
                break;
            case 'gcomp':
                $var = '';
                if (!empty($data->gcomp)) {
                    $p = eval('return '.$data->gcomp.';');
                    if ($p > 0 && $p < 100) {
                        $var = 'bg-red';
                    }
                }
                break;
            default:
                $var = '';
        }

        return $var;
    }

    /**
     * @param int $userid User ID
     * @return bool
     * @throws \dml_exception
     */
    public static function is_ot($userid = null) {
        global $DB, $USER;

        if (!$userid) {
            $userid = $USER->id;
        }

        $systemcontext = context_system::instance();

        $sql = "SELECT cm.id
                  FROM {cohort} c
                  JOIN {cohort_members} cm
                    ON c.id = cm.cohortid
                 WHERE c.contextid = ?
                   AND c.idnumber = 'OT'
                   AND cm.userid = ?";

        if ($DB->record_exists_sql($sql, [$systemcontext->id, $userid])) {
            return true;
        }
        return false;
    }

    /**
     * @param $roleid
     * @param $context
     * @param $groupid
     * @return array
     * @throws \dml_exception
     */
    public static function get_role_users($roleid, $context, $groupid = 0) {
        $var = array();
        if ($users = get_role_users($roleid, $context, false, '', null, false, $groupid)) {
            foreach ($users as $user) {
                $var[] = $user;
            }
        }
        return $var;
    }

    /**
     * @param null $userid
     * @param string $capability
     * @return array|bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_enrolled_courses_by_capability($userid=null, $capability = 'mod/assign:grade') {
        global $DB, $USER;

        if (!$userid) {
            $userid = $USER->id;
        }

        if (!$roles = get_roles_with_capability($capability)) {
            return false;
        }

        list($insql, $params) = $DB->get_in_or_equal(array_keys($roles), SQL_PARAMS_NAMED);
        $params['userid'] = $userid;
        $params['ctx'] = CONTEXT_COURSE;
        $params['siteid'] = SITEID;

        $sql = "SELECT DISTINCT c.id, c.category, c.fullname, c.shortname, c.idnumber
                  FROM {role_assignments} ra
                  JOIN {context} cx ON ra.contextid = cx.id
                  JOIN {course} c ON cx.instanceid = c.id
                 WHERE ra.userid = :userid
                   AND cx.contextlevel = :ctx
                   AND ra.roleid {$insql}
                   AND c.id != :siteid
                   AND c.visible = 1";
        return $DB->get_records_sql($sql, $params);
    }

    /**
     * @param $userid
     * @param $courseid
     * @return int
     * @throws \dml_exception
     */
    public static function get_user_course_groupid($userid, $courseid) {
        global $DB;

        $sql = "SELECT gm.id, gm.groupid FROM {groups_members} gm 
                  JOIN {groups} g ON gm.groupid = g.id WHERE gm.userid = ? AND g.courseid = ?";
        $groupid = 0;
        if ($group = $DB->get_record_sql($sql, [$userid, $courseid], IGNORE_MULTIPLE)) {
            $groupid = $group->groupid;
        }

        return $groupid;
    }

    /**
     * @param $userid
     * @return array
     * @throws \dml_exception
     */
    public static function get_user_groups($userid) {
        global $DB, $USER;

        if (!$userid) {
            $userid = $USER->id;
        }
        return $DB->get_records_sql("SELECT *
                                   FROM {groups_members} gm
                                   JOIN {groups} g
                                    ON g.id = gm.groupid
                                  WHERE gm.userid = ?
                                   ORDER BY name ASC", array($userid));
    }

    /**
     * @param null $userid
     * @return array|bool
     */
    public static function get_own_group_users($userid = null, $capability='') {
        global $DB, $USER;

        if (!$userid) {
            $userid = $USER->id;
        }
        if (!$courses = self::get_enrolled_courses_by_capability($userid, $capability)) {
            return false;
        }
        $groups = self::get_user_groups($userid);

        foreach ($groups as $group) {
            if (!isset($courses[$group->courseid])) {
                continue;
            }
            if (!$users = get_enrolled_users(\context_course::instance($group->courseid), '',
                $group->id, 'u.id,u.firstname,u.lastname', null, 0, 0, true)) {
                continue;
            }
            $groups[$group->id]->users = $users;
            $courses[$group->courseid]->groups[$group->id] = $group;
        }
        return $courses;
    }

    /**
     * @param null $userid
     * @return array|bool
     * @throws \dml_exception
     */
    public static function get_own_cohort_users($userid = null) {
        global $DB, $USER;

        if (!$userid) {
            $userid = $USER->id;
        }

        $sql = "SELECT c.* 
                  FROM {cohort} c
                  JOIN {cohort_members} m 
                    ON c.id = m.cohortid 
                 WHERE c.contextid = ? 
                   AND m.userid = ?";

        if (!$cohorts =  $DB->get_records_sql($sql, [(context_system::instance())->id, $userid])) {
            return false;
        }

        foreach ($cohorts as $cohort) {
            $sql = "SELECT u.id, u.firstname, u.lastname FROM {cohort_members}  cm JOIN {user} u ON cm.userid = u.id WHERE cm.cohortid = ?";
            if ($members = $DB->get_records_sql($sql, [$cohort->id])) {
                $cohort->users = $members;
                $cohorts[$cohort->id] = $cohort;
            }
        }
        return $cohorts;
    }

    /**
     * @param $where
     * @param $params
     * @param $report
     * @return bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function report_filter(&$where, &$params, $report, $capability = '') {
        global $DB, $USER;

        $contextsystem = context_system::instance();

        if (is_siteadmin() || has_capability('report/ghs:viewallcourses', $contextsystem)) {
            return true;
        }

        if (has_capability('report/ghs:viewallgroups', $contextsystem)) {
            $viewableusers = \report_ghs\helper::get_own_group_users(null, $capability);
            $courses = [];
            foreach ($viewableusers as $course) {
                $courses[$course->id] = $course->id;
            }
            if ($courses) {
                list($insqlcourse, $paramscourse) = $DB->get_in_or_equal($courses, SQL_PARAMS_NAMED, 'cor');
                $where .= " AND r.courseid {$insqlcourse}";
                $params = array_merge($params, $paramscourse);
            } else {
                $where .= " AND 0=1";
            }
        } else if (has_capability('report/ghs:viewowngroup', $contextsystem)) {
            $groups = [];
            $users = [];
            if ($viewableusers = \report_ghs\helper::get_own_group_users(null, $capability)) {
                foreach ($viewableusers as $course) {
                    if (!isset($course->groups)) {
                        continue;
                    }
                    $courses[$course->id] = $course->id;
                    foreach ($course->groups as $group) {
                        $groups[$group->id] = $group->id;
                        if (isset($group->users)) {
                            foreach ($group->users as $user) {
                                $users[$user->id] = $user->id;
                            }
                        }
                    }
                }
            }
            if ($report != 'ghs_sla') {
                if (!empty($courses) && $groups) {
                    list($insqlcourse, $paramscourse) = $DB->get_in_or_equal($courses, SQL_PARAMS_NAMED, 'cor');
                    $where .= " AND r.courseid {$insqlcourse}";
                    $params = array_merge($params, $paramscourse);

                    list($insqlgrp, $paramsgrp) = $DB->get_in_or_equal($groups, SQL_PARAMS_NAMED, 'gro');
                    $where .= " AND (r.groupid {$insqlgrp}";
                    $params = array_merge($params, $paramsgrp);

                    if (($report == 'ghs_enddate_extension' || $report == 'ghs_activity_extension') && !empty($users)) {
                        list($insqlgrp, $paramsgrp) = $DB->get_in_or_equal($users, SQL_PARAMS_NAMED, 'usr');
                        $where .= " OR r.userid {$insqlgrp})";
                        $params = array_merge($params, $paramsgrp);
                    } else {
                        $where .= ")";
                    }
                } else {
                    $where .= " AND 0=1";
                }
            }
        } else if (has_capability('report/ghs:viewowncohort', $contextsystem)) {
            $viewableusers = \report_ghs\helper::get_own_group_users(null, $capability);
            $courses = [];
            foreach ($viewableusers as $course) {
                $courses[$course->id] = $course->id;
            }
            if ($courses) {
                list($insqlcourse, $paramscourse) = $DB->get_in_or_equal($courses, SQL_PARAMS_NAMED, 'gro');
                $where .= " AND r.courseid {$insqlcourse}";
                $params = array_merge($params, $paramscourse);

                $cohorts = \report_ghs\helper::get_own_cohort_users();
                $users = [];
                foreach ($cohorts as $cohort) {
                    if (isset($cohort->users)) {
                        foreach ($cohort->users as $user) {
                            $users[$user->id] = $user->id;
                        }
                    }
                }

                if ($report != 'ghs_group_enrollment') {
                    if ($users) {
                        list($insquser, $paramsuser) = $DB->get_in_or_equal($users, SQL_PARAMS_NAMED, 'usr');
                        $where .= " AND r.userid {$insquser}";
                        $params = array_merge($params, $paramsuser);
                    } else {
                        $where .= " AND 0=1";
                    }
                }
            } else {
                $where .= " AND 0=1";
            }
        } else {
            if ($enrolledcourses = enrol_get_users_courses($USER->id, true)) {

                $assignedgroups = self::get_user_groups($USER->id);
                $coursegroups = [];
                foreach ($assignedgroups as $group) {
                    $coursegroups[$group->courseid][] = $group;
                }

                $courses = [];
                $groups = [];
                foreach ($enrolledcourses as $course) {
                    $contextcourse = context_course::instance($course->id);

                    if (has_capability('report/ghs:viewallgroups', $contextcourse)) {
                        $courses[$course->id] = $course->id;
                        $cgroups = $DB->get_records('groups', ['courseid' => $course->id]);
                        if (!empty($cgroups)) {
                            foreach ($cgroups as $group) {
                                $groups[$group->id] = $group->id;
                            }
                        }
                    } else if (has_capability('report/ghs:viewowngroup', $contextcourse)) {
                        if (!empty($coursegroups[$course->id])) {
                            $courses[$course->id] = $course->id;
                            foreach ($coursegroups[$course->id] as $group) {
                                $groups[$group->id] = $group->id;
                            }
                        }
                    }
                }
                if ($courses) {
                    list($insqlcourse, $paramscourse) = $DB->get_in_or_equal($courses, SQL_PARAMS_NAMED, 'gro');
                    $where .= " AND r.courseid {$insqlcourse}";
                    $params = array_merge($params, $paramscourse);
                }
                if ($report != 'ghs_sla') {
                    if ($groups) {
                        list($insqlgrp, $paramsgrp) = $DB->get_in_or_equal($groups, SQL_PARAMS_NAMED, 'gro');
                        $where .= " AND r.groupid {$insqlgrp}";
                        $params = array_merge($params, $paramsgrp);
                    }
                }
                if (!$courses && !$groups) {
                    $where .= " AND 0=1";
                }
            } else {
                $where .= " AND 0=1";
            }
        }
    }

    /**
     * @param $tag
     * @param $userid
     * @param $courseid
     * @return string|null
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_rate($tag, $userid, $courseid) {
        global $DB;

        $sql = "SELECT ti.id, ti.tagid, ti.itemid cmid
                  FROM {tag_instance} ti
                  JOIN {tag} t ON ti.tagid = t.id
                  JOIN {course_modules} cm ON ti.itemid = cm.id
                 WHERE t.name = ?
                   AND ti.itemtype = 'course_modules'
                   AND cm.course  = ?
                   AND cm.deletioninprogress = 0";

        if (!$activities = $DB->get_records_sql($sql, [$tag, $courseid])) {
            return null;
        }

        $cmids = [];
        $completed = [];
        foreach ($activities as $activity) {
            $cmids[$activity->cmid] = $activity->cmid;
        }

        foreach ($cmids as $cmid) {
            $cm = NED::get_cm_by_cmid($cmid);
            if (NED::get_kica_enabled($cm->course)) {
                // KICA enabled.
                $grade = NED::kg_get_by_userid_cmid($userid, $cm);
                if (!empty($grade->id) && $grade->is_graded()){
                    $completed[$cmid] = $cmid;
                }
            } else {
                $grade = NED::get_grade_grade($cm, $userid, false);
                if ($grade && !is_null($grade->finalgrade)) {
                    $completed[$cmid] = $cmid;
                }
            }
        }

        return count($completed) . "|" . count($cmids);
    }
    /**
     * Add “Last Activity” column. Show Y or N depending if student has completed activity with “Last Activity” tag.
     *
     * @param $userid
     * @param $courseid
     * @return string|null
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function last_activity_completed($userid, $courseid) {
        global $DB;

        $sql = "SELECT ti.id, ti.tagid, ti.itemid cmid
                  FROM {tag_instance} ti
                  JOIN {tag} t ON ti.tagid = t.id
                  JOIN {course_modules} cm ON ti.itemid = cm.id
                 WHERE t.name = ?
                   AND ti.itemtype = 'course_modules'
                   AND cm.course  = ?
                   AND cm.deletioninprogress = 0";

        if (!$activities = $DB->get_records_sql($sql, ['last activity', $courseid])) {
            return null;
        }

        if (count($activities) > 1) {
            return 'E';
        }

        $cmid = reset($activities)->cmid;

        $completed = [
            activity_status::STATUS_COMPLETED,
            activity_status::STATUS_PASSED,
            activity_status::STATUS_GRADED_KICA,
            activity_status::STATUS_GRADED_KICA_SUMMATIVE,
            activity_status::STATUS_GRADED_KICA_FORMATIVE
        ];

        $cm = NED::get_cm_by_cmid($cmid);
        $activities_info = new activity_status();
        $activities_info->check_activities_list([$cm], $userid);
        $status = $activities_info->check_activity($cm, $userid, false);

        if (in_array($status, $completed)) {
            return 'Y';
        } else {
            return 'N';
        }
    }

    /**
     * Add “Last Activity” column. Show Y or N depending if student has completed activity with “Last Activity” tag.
     *
     * @param $userid
     * @param $courseid
     * @return string|null
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function all_summatives_completed($userid, $courseid) {
        global $DB;

        $sql = "SELECT ti.id, ti.tagid, ti.itemid cmid
                  FROM {tag_instance} ti
                  JOIN {tag} t ON ti.tagid = t.id
                  JOIN {course_modules} cm ON ti.itemid = cm.id
                 WHERE t.name = ?
                   AND ti.itemtype = 'course_modules'
                   AND cm.course  = ?
                   AND cm.deletioninprogress = 0";

        if (!$activities = $DB->get_records_sql($sql, ['summative', $courseid])) {
            return null;
        }

        $completed = [
            activity_status::STATUS_COMPLETED,
            activity_status::STATUS_PASSED,
            activity_status::STATUS_GRADED_KICA,
            activity_status::STATUS_GRADED_KICA_SUMMATIVE,
            activity_status::STATUS_GRADED_KICA_FORMATIVE
        ];

        foreach ($activities as $activity) {
            $cmid = $activity->cmid;

            $cm = NED::get_cm_by_cmid($cmid);
            $activities_info = new activity_status();
            $activities_info->check_activities_list([$cm], $userid);
            $status = $activities_info->check_activity($cm, $userid, false);

            if (!in_array($status, $completed)) {
                return 'N';
            }
        }
        return 'Y';
    }

    /**
     * @param $capability
     * @return bool
     * @throws \coding_exception
     */
    public static function has_capability_in_any_course($capability) {
        $courses = enrol_get_my_courses();
        foreach ($courses as $course) {
            if (has_capability($capability, context_course::instance($course->id))) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return \Krizalys\Onedrive\Client|null
     */
    public static function get_onedrive_client() {
        global $DB;
        try {
            $cfg = get_config('report_ghs');

            $onedrivetoken = new onedrivetoken($cfg->clientid, $cfg->clientsecret, $cfg->onedriveusername, $cfg->onedrivepassword);

            try {
                $token = $onedrivetoken->getAccessToken();
            } catch ( \Exception $e ) {
                mtrace('Invalid access token!');
                $token = null;
            }

            if ($token) {
                $graph = new Graph();
                $graph->setAccessToken($token);

                $httpClient = new GuzzleHttpClient();
                $serviceDefinition = Onedrive::buildServiceDefinition();

                $client = new Client(
                    $cfg->clientid,
                    $graph,
                    $httpClient,
                    $serviceDefinition,
                    []
                );

                return $client;
            }
        } catch ( \Exception $e ) {
            mtrace('No onedrive client!');
            return null;
        }
        return null;
    }

    /**
     * @param Client $client
     * @param $filepath
     */
    public static function upload_file_to_onedrive($client, $filepath, $folder = '') {
        if (empty($folder)) {
            $folder = ONEDRIVE_OLD_DATA;
        }

        $stream = fopen($filepath, 'r'); // Use rb mode for binary.
        try {

            $root = $client->getRoot();
            $proxy = null;

            foreach ($root->getChildren() as $child) {
                if ($child->name === $folder) {
                    $proxy = $child;
                    break;
                }
            }

            if (is_null($proxy)) {
                $proxy = $root;
            }

            $uploadSession = $proxy->startUpload(basename($filepath), $stream, [
                'contentType' => 'text/plain',
                'conflictBehavior' => ConflictBehavior::REPLACE
            ]);
            $childDriveItem = $uploadSession->complete();
        } catch ( \Exception $e ) {
            mtrace('Could not upload file "'.$filepath.'"');
        }
    }

    /**
     * @param $report
     * @param $cfg
     * @throws \coding_exception
     * @throws \moodle_exception
     */
    public static function print_export_task_info($report, $cfg, $return = false) {
        if (is_siteadmin()) {
            $exportcls = '\report_ghs\task\adhoc_' . $report . '_export';

            if (class_exists($exportcls)) {
                if ($task = $exportcls::get_task()) {
                    if ($return) {
                        return html_writer::tag('div', get_string('exportwarning', 'report_ghs', userdate($task->customdata->timecreated)), array('class' => 'alert alert-info'));
                    } else {
                        echo html_writer::tag('div', get_string('exportwarning', 'report_ghs', userdate($task->customdata->timecreated)), array('class' => 'alert alert-info'));
                    }
                } else {
                    $a = new stdClass();
                    $a->lastexport = (isset($cfg->{$report . '_export'})) ? userdate($cfg->{$report . '_export'}) : get_string('never');
                    $a->updateurl = (new moodle_url('/report/ghs/export_report.php', ['report' => $report]))->out();
                    if ($return) {
                        return html_writer::tag('div', get_string('lastexporttewarning', 'report_ghs', $a), array('class' => 'alert alert-info'));
                    } else {
                        echo html_writer::tag('div', get_string('lastexporttewarning', 'report_ghs', $a), array('class' => 'alert alert-info'));
                    }
                }
            }
        }
    }

    /**
     * @param $catid
     * @return array|bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_category_courses_recursive($catid) {
        global $DB;

        $catids = [$catid];

        if (!$category = $DB->get_record('course_categories', ['id' => $catid])) {
            return false;
        }

        if ($subcats = $DB->get_records_select('course_categories', "path LIKE ?", array("$category->path/%"), 'id, visible')) {
            foreach ($subcats as $cat) {
                $catids[] = $cat->id;
            }
        }

        if (!$catids) {
            return false;
        }

        list($insql, $params) = $DB->get_in_or_equal($catids);

        $sql = "SELECT * FROM {course} c WHERE c.category {$insql} AND c.visible = 1";

        return $DB->get_records_sql($sql, $params);
    }

    /**
     * @return array|false
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_report_courses() {
        global $DB;

        $cfg = get_config('report_ghs');

        if (empty($cfg->coursecategories)) {
            return false;
        }

        $courselist = [];
        if ($categoryids = explode(',', $cfg->coursecategories)) {
            foreach ($categoryids as $categoryid) {
                if ($courses = self::get_category_courses_recursive($categoryid)) {
                    $courselist = array_merge($courselist, array_keys($courses));
                }
            }
            return $courselist;
        }

        return false;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @param bool $export
     * @return false|float|string
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function ghs_grade_comparisons_data($data, $column, &$counter, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'classenddate':
                $var = '-';
                if ($data->$column > 0) {
                    $url = new moodle_url('//blocks/ned_teacher_tools/deadline_manager.php',
                        ['courseid' => $data->courseid, 'group' => $data->groupid, 'user' => $data->userid,
                            'prevpage' => '/local/kica/grade_user']);
                    if ($export) {
                        $var = $url->out(false);
                    } else {
                        $var = html_writer::link($url, date("m/d/Y", $data->$column), ['target' => '_blank']);
                    }
                }
                break;
            case 'assigns':
            case 'tests':
                $var = '';
                if (!is_null($data->$column)) {
                    $var = round($data->$column, 2);
                }
                break;
            case 'grade':
                $var = '';
                if (!is_null($data->$column)) {
                    $url = new moodle_url('/local/kica/grade_user.php', ['courseid' => $data->courseid,
                        'group' => $data->groupid, 'mode' => 0, 'user' => $data->userid,
                        'prevpage' => '/local/kica/grade_user']);
                    if ($export) {
                        $var = $url->out(false);
                    } else {
                        $var = html_writer::link($url, round($data->$column, 2), ['target' => '_blank']);
                    }
                }
                break;
            case 'ct':
            case 'gt':
                $uid = $column.'id';
                $var = '-';
                if (!empty($data->$uid)) {
                    $user = \core_user::get_user($data->$uid);
                    $var = fullname($user);
                }
                break;
            case 'atdiff':
                $var = '';
                if (!is_null($data->$column)) {
                    $var = round(abs($data->$column), 2);
                }
                break;
            case 'date':
                $var = '-';
                if ($data->$column) {
                    $var = date("m/d/Y", strtotime($data->$column));
                }
                break;
            default:
                $var = $data->$column;
        }

        return $var;
    }

    public static function get_fcatagory($categorypath) {
        global $DB;

        if ($categorypath) {
            $arr = explode('/', $categorypath);
            if (!empty($arr[1])) {
                return $DB->get_field('course_categories', 'name', ['id' => $arr[1]]);
            }
        }

        return null;
    }

    /**
     * @param $user
     * @return bool
     */
    public static function exclude_user_from_kica_report(&$user) {
        profile_load_data($user);
        if ($user->profile_field_ed_program == 'None') {
            return true;
        }
        return false;
    }

    /**
     * @param $cohort
     * @return bool|void
     */
    public static function exclude_school_from_kica_report($cohort) {
        if ($cohort && $cohort->name) {
            if (substr(trim($cohort->name), 0, 1) === '*') {
                return true;
            }
            return false;
        }
    }

    /**
     * @param $userid
     * @param $coursecode
     * @return array
     * @throws \dml_exception
     */
    public static function get_proficiency_courses($userid, $coursecode) {
        global $DB;

        $sql = "SELECT e.courseid 
                  FROM {user_enrolments} ue
                  JOIN {enrol} e
                    ON ue.enrolid = e.id
                  JOIN {course} c
                    ON e.courseid = c.id
                 WHERE userid = :userid AND c.shortname LIKE :shortname";

        $params = [];
        $params['userid'] = $userid;
        $params['shortname'] = '%'.$DB->sql_like_escape($coursecode).'%';

        return $DB->get_records_sql($sql, $params);
    }
}
