<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use block_ned_teacher_tools\deadline_manager as DM;
use block_ned_teacher_tools\shared_lib as SH;
use report_ghs\helper;
use report_ghs\shared_lib as NED;
use block_ned_teacher_tools\output\menu_bar as MB;
use const block_ned_teacher_tools\TAG_SUMMATIVE;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_dm_overview_report
 * @package report_ghs\task
 */
class adhoc_ghs_dm_overview_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::update_report($cd->userid ?? null);
        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $CFG, $DB, $PAGE;

        $cfg = get_config('report_ghs');

        $DB->execute("TRUNCATE TABLE {report_ghs_dm_overview}");

        $contextsystem = \context_system::instance();

        $courselist = helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);
        $params[] = DM::SCHEDULE_FULL;

        // Classes.
        $sql = "SELECT g.*, TIMESTAMPDIFF(DAY, FROM_UNIXTIME(g.startdate), FROM_UNIXTIME(g.enddate)) numofclassdays
                  FROM {groups} g
	              JOIN {course} c
                    ON g.courseid = c.id
                 WHERE g.courseid {$insql} AND g.schedule = ?
              ORDER BY c.shortname ASC";

        $ctrole = $DB->get_record('role', array('shortname' => 'classroom-teacher'));

        $rs = $DB->get_recordset_sql($sql, $params);

        foreach ($rs as $group) {
            $coursecontext = \context_course::instance($group->courseid);
            $PAGE->set_url(new \moodle_url('/course/view.php', ['id' => $group->courseid]));
            $course = get_course($group->courseid);
            $PAGE->reset_theme_and_output();
            $PAGE->set_course($course);
            $PAGE->initialise_theme_and_output();

            $data = new \stdClass();
            $data->courseid = $group->courseid;
            $data->groupid = $group->id;
            $data->numofclassdays = $group->numofclassdays;

            if ($cohort = DM::get_school_cohort($group)) {
                $data->cohortid = $cohort->id;
            }

            if ($ctusers = helper::get_role_users($ctrole->id, $coursecontext, $group->id)) {
                $data->ctid = (count($ctusers) > 1) ? -1 : $ctusers[0]->id;
            }

            $deadlines = [];

            // Deadline manager.
            $dmcomplete = true;
            $incomplete = false;
            $dmcompletioncheck = false;
            if ($group->schedule == DM::SCHEDULE_FULL) {
                $dmcompletioncheck = true;
            }
            $MB = new MB(MB::PAGE_DM , [], $group->courseid, true);
            if ($MB->is_show_page($MB::PAGE_DM)){
                $deadlinemanagerwarnings = '';
                if ($MB->act) {
                    $deadlinemanager = new DM($group->courseid);
                    if ($deadlinemanager->has_missed_schedule($group->id)){
                        $incomplete = true;
                        $dmcomplete = false;
                    }
                }
            }

            // Check date conflict
            if ($dmcompletioncheck && $dmcomplete) {
                $dm_entity = DM::get_dm_entity($course, $group);
                foreach ($dm_entity->iterate() as $cmid => $dma) {
                    if (!$dma->is_enabled()) continue;
                    if ($dma->has_date_conflict()) {
                        $dmcomplete = false;
                        break;
                    }
                }
            }

            if ($dmcompletioncheck && $dmcomplete) {
                /** @var \report_ghs\task\adhoc_ghs_group_enrollment_report $report */
                $report = '\report_ghs\task\adhoc_ghs_group_enrollment_report';
                if (class_exists($report)) {
                    if ($report::count_dm_conflict($group)) {
                        $dmcomplete = false;
                    }
                }
            }

            if ($incomplete) {
                $data->numofdmdays = get_string('incomplete', 'report_ghs');
            } else {
                $dm_entity = DM::get_dm_entity($group->courseid, $group);
                foreach ($dm_entity->iterate() as $cmid => $dma) {
                    if (!$dma->is_enabled()) continue;

                    $row = $dma->get_row();
                    if (!empty($row->raw_deadline)) {
                        $deadlines[] = $row->raw_deadline;
                    }
                }

                if ($deadlines) {
                    $data->numofdmdays = round((abs(max($deadlines) - min($deadlines)) / DAYSECS), 0);
                }
            }

            if ($dmcompletioncheck) {
                $data->dmcomplete = ($dmcomplete) ? get_string('yes') : get_string('no');
            } else {
                $data->dmcomplete = '';
            }

            $data->numofsummatives = count(NED::cmids_get_by_tags([TAG_SUMMATIVE], [], $course));

            $data->timecreated = time();

            $DB->insert_record('report_ghs_dm_overview', $data);

            NED::purge_course_depended_caches();
        }

        $rs->close();

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsdmoverview', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }
        set_config('dmoverviewlastupdate', time(), 'report_ghs');
        return true;
    }
}