<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for Grade changes report
 *
 * @package     report_ghs
 * @copyright   2023 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use local_kica as kica;
use local_kica\helper as kica_helper;
use report_ghs\helper;
use report_ghs\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');

/**
 * Class adhoc_ghs_grades_report
 * @package report_ghs\task
 */
class adhoc_ghs_grade_changes_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::update_report($cd->userid ?? null);
        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $CFG, $DB, $USER;

        $cfg = get_config('report_ghs');

        $USER = get_admin();

        $DB->execute("TRUNCATE TABLE {report_ghs_grade_changes}");

        $contextsystem = \context_system::instance();

        $courselist = helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);

        $time = time() - YEARSECS;
        $params[] = $time;

        $sql = "SELECT DISTINCT gi.courseid, gh.itemid, gh.userid	
                  FROM {grade_grades_history} gh
	              JOIN {grade_items} gi ON gh.itemid = gi.id
                 WHERE gi.courseid {$insql} AND gh.timemodified > ? AND gi.itemtype = 'mod'
              ORDER BY gi.courseid ASC, gh.itemid ASC, gh.userid ASC";


        $sqlgh = "SELECT DISTINCT gh.* FROM {grade_grades_history} gh
              	    JOIN {grade_items} gi ON gh.itemid = gi.id WHERE gh.itemid = ? AND gh.userid = ? AND gh.timemodified > ?
                ORDER BY gh.timemodified DESC";

        $sqlcohort = "SELECT c.* FROM {cohort} c JOIN {cohort_members} m ON c.id = m.cohortid WHERE c.contextid = ? AND m.userid = ? AND c.name NOT LIKE '*%'";

        $sqlgroup = "SELECT g.* FROM {groups} g JOIN {groups_members} m ON g.id = m.groupid WHERE g.courseid = ? AND m.userid = ?";

        $rs = $DB->get_recordset_sql($sql, $params);

        foreach ($rs as $ghdata) {
            $grades = $DB->get_records_sql($sqlgh, [$ghdata->itemid, $ghdata->userid, $time]);
            if (count($grades) == 1) {
                continue;
            }

            $grades = array_values($grades);

            if (!$gradeitem = $DB->get_record('grade_items', ['id' => $ghdata->itemid, 'itemtype' => 'mod'])) {
                continue;
            }
            if (!$cm = get_coursemodule_from_instance($gradeitem->itemmodule, $gradeitem->iteminstance, $gradeitem->courseid)) {
                continue;
            }
            if (!$tags = array_values(\core_tag_tag::get_item_tags_array('core', 'course_modules', $cm->id))) {
                continue;
            }
            if (!in_array('Summative', $tags)) {
                continue;
            }

            foreach ($grades as $index => $grade) {
                if ($index == 0) {
                    continue;
                }

                $previndex = $index - 1;

                if ($grades[$previndex]->rawgrade != $grade->rawgrade) {
                    $grades[$previndex]->gradebefore = $grade->rawgrade;
                    $grades[$previndex]->timedifference = ($grades[$previndex]->timemodified - $grade->timemodified);
                }
            }

            foreach ($grades as $grade) {
                if (isset($grade->timedifference) && $grade->timedifference < DAYSECS && !is_null($grade->gradebefore)) {
                    $data = new \stdClass();
                    $data->userid = $grade->userid;
                    $data->courseid = $ghdata->courseid;
                    $data->module = $gradeitem->itemmodule;
                    $data->instance = $gradeitem->iteminstance;
                    $data->activityname = $gradeitem->itemname;

                    $data->cmid = $cm->id;
                    if (!$tags = array_values(\core_tag_tag::get_item_tags_array('core', 'course_modules', $cm->id))) {
                        continue;
                    }
                    if (!in_array('Summative', $tags)) {
                        continue;
                    }

                    if ($cohort = $DB->get_record_sql($sqlcohort, [$contextsystem->id, $grade->userid], IGNORE_MULTIPLE)) {
                        $data->cohortid = $cohort->id;
                    }

                    if ($groups = $DB->get_records_sql($sqlgroup, [$ghdata->courseid, $grade->userid])) {
                        $data->groupid = (reset($groups))->id;
                    }

                    $data->graderid = $grade->usermodified;
                    $data->gradeafter = $grade->rawgrade;
                    $data->gradebefore = $grade->gradebefore;
                    $data->timegrade = $grade->timemodified;
                    $data->timedifference = $grade->timedifference;
                    $data->timecreated = time();

                    $DB->insert_record('report_ghs_grade_changes', $data);
                }
            }
        }

        $rs->close();

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsgradechanges', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }
        set_config('ghsgradechangeslastupdate', time(), 'report_ghs');
        return true;
    }
}