<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use local_kica as kica;
use local_kica\helper as kica_helper;
use report_ghs\helper;
use report_ghs\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_grades_report
 * @package report_ghs\task
 */
class adhoc_ghs_grades_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::update_report($cd->userid ?? null);
        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $CFG, $DB, $USER;

        adhoc_ghs_activity_report::update_report();

        $cfg = get_config('report_ghs');

        $USER = get_admin();

        $DB->execute("TRUNCATE TABLE {report_ghs_grades}");

        $contextsystem = \context_system::instance();

        $courselist = helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);

        // Courses.
        $sql = "SELECT c.* 
                  FROM {local_kica} k 
                  JOIN {course} c 
                    ON k.courseid = c.id
                 WHERE k.enabled = 1
                   AND c.id {$insql}
              ORDER BY c.shortname ASC";

        $sqlcourseavg = "SELECT gg.id, gg.finalgrade
             FROM {grade_items} gi
             JOIN {grade_grades} AS gg
               ON gi.id = gg.itemid
            WHERE gi.itemtype = 'course' AND gi.courseid = ? AND gg.userid = ?";

        $sqlcohort = "SELECT c.* FROM {cohort} c JOIN {cohort_members} m ON c.id = m.cohortid WHERE c.contextid = ? AND m.userid = ? AND c.name NOT LIKE '*%'";

        $sqlgroup = "SELECT g.* FROM {groups} g JOIN {groups_members} m ON g.id = m.groupid WHERE g.courseid = ? AND m.userid = ?";

        $sqlenrol = "SELECT MIN(ue.timestart) FROM {user_enrolments} ue JOIN {enrol} e ON ue.enrolid = e.id WHERE ue.timestart > 0 AND e.courseid = ? AND ue.userid = ?";

        $rs = $DB->get_recordset_sql($sql, $params);

        foreach ($rs as $course) {
            $kica = NED::get_kica($course->id);
            $calc_natual = NED::is_kica_calculation_natural($kica);
            if (!$kica){
                continue;
            }

            $coursecontext = \context_course::instance($course->id);

            if (!$users = get_enrolled_users($coursecontext, 'mod/assign:submit', 0, 'u.*', 'u.id', 0, 0, false)) {
                continue;
            }

            $graders = [];
            $kicagroups = NED::get_kica_groups($course->id, true);
            NED::kg_get_grades_by_course($course->id, array_keys($users), false, true);

            foreach ($users as $index => $user) {
                $data = new \stdClass();
                $data->courseid = $course->id;
                $data->userid = $user->id;
                $data->gradingerror = 0;

                // Grading error check.
                foreach ($kicagroups as $kicagroupvalue) {
                    $kica_items = NED::ki_get_all_by_course($course->id, $kicagroupvalue);
                    foreach ($kica_items as $kicaitem) {
                        $kicagrade = kica\kica_grade::get_by_userid_itemid($user->id, $kicaitem->id);
                        if (empty($kicagrade->id) || $kicagrade->flag) {
                            continue;
                        }

                        $needsupdate = kica_helper::is_gradingform_needupdate($kicaitem, $user->id) || $kicagrade->grade_mismatch();
                        if ($needsupdate && $kicagrade->is_gradable() && !$kicagrade->is_graded()){
                            $data->gradingerror++;
                        }
                    }
                }

                if ($cohort = $DB->get_record_sql($sqlcohort, [$contextsystem->id, $user->id], IGNORE_MULTIPLE)) {
                    $data->chortid = $cohort->id;
                }

                if (helper::exclude_user_from_kica_report($user)) {
                    continue;
                }

                if ($groups = $DB->get_records_sql($sqlgroup, [$course->id, $user->id])) {
                    $data->groupid = (reset($groups))->id;

                    if (!isset($graders[$data->groupid])) {
                        if ($users = get_enrolled_users($coursecontext, 'mod/assign:grade', $data->groupid)) {
                            $graders[$data->groupid] = (reset($users))->id;
                            $data->graderid = $graders[$data->groupid];
                        }
                    } else {
                        $data->graderid = $graders[$data->groupid];
                    }
                }

                if ($enrolldate = $DB->get_field_sql($sqlenrol, [$course->id, $user->id])) {
                    $data->enrolldate =  $enrolldate;
                }

                $kica_average_data = NED::kg_get_kica_average($course->id, $user->id, null, null, null);
                $kicagrades = NED::kg_get_course_average($course->id, $user->id, null, false, $kica_average_data);
                $data->kicaavg = $kicagrades['finalgrade'] ?? null;

                foreach (NED::KICA_KEYS as $key) {
                    $data->{$key} = $kicagrades[$key] ?? null;
                    if (!$calc_natual) {
                        $max = $kica->$key ?? 0;
                        $data->{$key} = $max ? ($data->{$key}/$max * 100) : 0;
                    }
                }
                if (!is_null($data->kicaavg)) {
                    $gr70 = reset($kicagroups);
                    $kica_total = [];
                    foreach ($kicagroups as $kicagroup){
                        $g_key = ($gr70 == $kicagroup) ? '70' : '30';
                        $total = 0;
                        $maxtotal = 0;
                        foreach (NED::KICA_KEYS as $kica_key){
                            $val = $kica_average_data[$kicagroup][$kica_key] ?? null;
                            if ($max = $kica_average_data['max'][$kicagroup][$kica_key] ?? null) {
                                $maxtotal += $max;
                            }
                            if (is_null($val)) continue;
                            $total += $val;

                        }

                        $kica_total[$g_key] = ($maxtotal) ? ($total / $maxtotal * 100) : null;
                    }

                    foreach ($kica_total as $g_key => $total){
                        $kica_key = 'kica'.$g_key;
                        $data->$kica_key = round($total, 5);
                    }
                }

                if ($finalgrade = $DB->get_record_sql($sqlcourseavg, [$course->id, $user->id])) {
                    $coursefinal = $finalgrade->finalgrade;
                    if (!is_null($coursefinal)) {
                        $data->coursegrade = round($coursefinal, 5);
                    }
                }

                $data->frate = helper::get_rate('formative', $user->id, $course->id);
                $data->srate = helper::get_rate('summative', $user->id, $course->id);
                $data->lastactivity = helper::all_summatives_completed($user->id, $course->id);

                $data->timecreated = time();

                $DB->insert_record('report_ghs_grades', $data);
            }

            NED::purge_course_depended_caches();
        }

        $rs->close();

        set_config('kicalastupdate', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsgrades', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }
}