<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use local_kica as kica;
use local_kica\helper as kica_helper;
use report_ghs\helper;
use report_ghs\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_grades_report
 * @package report_ghs\task
 */
class adhoc_ghs_grading_time_monitor_report extends base_ghs_adhoc_task {
    const START_DATE = '2023-09-04';
    const END_DATE = '2024-09-01';

    /**
     * @return bool
     */
    public function execute() {
        $cd = $this->get_custom_data();
        self::update_report($cd->userid ?? null);
        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $CFG, $DB, $USER;

        $cfg = get_config('report_ghs');

        $USER = get_admin();

        $DB->execute("TRUNCATE TABLE {report_ghs_grading_time_moni}");

        $contextsystem = \context_system::instance();

        $field = $DB->get_record('customfield_field', ['shortname' => 'regt'], '*', MUST_EXIST);

        $sql = "SELECT gt.graderid, gt.courseid, gt.cmid, SUM(COALESCE(d.value, 0)) totaltime
                  FROM {block_ned_teacher_tools_grtr} gt
                  JOIN {course} c ON c.id = gt.courseid
                  JOIN {course_modules} cm ON cm.id = gt.cmid
                  JOIN {modules} m ON cm.module = m.id 
	         LEFT JOIN {customfield_data} AS d ON gt.cmid = d.instanceid AND d.fieldid = ?
                 WHERE gt.timegrade >= ?
                   AND gt.timegrade < ? 
                   AND gt.hidden = 0 
                   AND gt.graderid <> gt.userid
              GROUP BY gt.graderid, gt.courseid, gt.cmid";

        list($days, $today) = self::get_school_days();

        foreach ($days as $day) {
            $params = [$field->id, $day['timestart'], $day['timestart']+DAYSECS];
            $rs = $DB->get_recordset_sql($sql, $params);
            foreach ($rs as $grader) {
                $data = new \stdClass();
                $data->graderid = $grader->graderid;
                $data->courseid = $grader->courseid;
                $data->cmid = $grader->cmid;
                $data->week = $day['week'];
                $data->timestart = $day['timestart'];
                $data->totaltime = $grader->totaltime;
                $DB->insert_record('report_ghs_grading_time_moni', $data);
            }

            $rs->close();
        }

        // Estimated time.
        $sql = "SELECT gt.graderid, gt.courseid, gt.cmid, SUM(COALESCE(d.value, 0)) estimatedtime
                  FROM {block_ned_teacher_tools_grtr} gt
                  JOIN {course} c ON c.id = gt.courseid
                  JOIN {course_modules} cm ON cm.id = gt.cmid
                  JOIN {modules} m ON cm.module = m.id 
	         LEFT JOIN {customfield_data} AS d ON gt.cmid = d.instanceid AND d.fieldid = ?
                 WHERE gt.timeend >= ?
                   AND gt.timeend < ? 
                   AND gt.timegrade = 0 
                   AND gt.hidden = 0 
                   AND gt.graderid <> gt.userid
              GROUP BY gt.graderid, gt.courseid, gt.cmid";

        foreach ($days as $day) {
            if ($today > $day['timestart']) {
                continue;
            }
            $params = [$field->id, $day['timestart'], $day['timestart']+DAYSECS];
            $rs = $DB->get_recordset_sql($sql, $params);
            foreach ($rs as $grader) {
                if ($gtm = $DB->get_record('report_ghs_grading_time_moni', [
                    'graderid' => $grader->graderid,
                    'courseid' => $grader->courseid,
                    'cmid' => $grader->cmid,
                    'timestart' => $day['timestart'],
                ])) {
                    $data = new \stdClass();
                    $data->id = $gtm->id;
                    $data->estimatedtime = $grader->estimatedtime;
                    $DB->update_record('report_ghs_grading_time_moni', $data);
                } else {
                    $data = new \stdClass();
                    $data->graderid = $grader->graderid;
                    $data->courseid = $grader->courseid;
                    $data->cmid = $grader->cmid;
                    $data->week = $day['week'];
                    $data->timestart = $day['timestart'];
                    $data->estimatedtime = $grader->estimatedtime;
                    $DB->insert_record('report_ghs_grading_time_moni', $data);
                }
            }

            $rs->close();
        }

        set_config('gradingtimemonitorlastupdate', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsgradingtimemonitor', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }

    /**
     * @return array
     */
    public static function get_school_days() {
        global $USER;

        $USER = get_admin();

        $days = [];

        $starttime = usergetmidnight(strtotime(self::START_DATE));
        $endtime = usergetmidnight(strtotime(self::END_DATE));
        $today = usergetmidnight(time());

        $week = 0;
        $day = 0;
        for ($i = $starttime; $i <= $endtime; $i += DAYSECS) {
            $day++;
            if (($day % 7) == 1) {
                $week++;
            }
            $days[$i] = ['day' => $day, 'week' => $week, 'timestart' => $i];
        }
        return [$days, $today];
    }
}