<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for GRoup Enrollment Report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use block_ned_teacher_tools\deadline_manager;
use report_ghs\helper;
use PHPExcel;
use report_ghs\onedrivetoken;
use stdClass;
use Krizalys\Onedrive\Onedrive;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_group_enrollment_report
 * @package report_ghs\task
 */
class adhoc_ghs_group_enrollment_export extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::export_report($cd->userid);

        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @return string
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function export_report($userid = null, $client = null) {
        global $DB, $CFG;

        raise_memory_limit(MEMORY_EXTRA);

        if (!$client) {
            $client = helper::get_onedrive_client();
        }

        $columnsexport = array(
            'rowcount',
            'course_txt',
            'coursecode',
            'subject',
            'grade',
            'department',
            'moe_code',
            'moe_name',
            'fcategory',
            'lcategory',
            'school',
            'groupname_txt',
            'gmessaging',
            'activestudents',
            'suspendedstudents',
            'ctname',
            'otname',
            'startdate',
            'enddate',
            'totaldays',
            'dmrequired',
            'dmstatus',
            'dmdateconflicts',
            'enrolldateconflicts',
        );

        $params = null;
        $pageparams = [];

        $sql = "SELECT r.id,
               c.fullname course,
               c.shortname coursecode,
                coh.name school,
               g.name groupname,
               g.idnumber groupidnumber,
               r.activestudents,
               r.suspendedstudents,
               if(r.ctid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.ctid)) ctname,
               if(r.otid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.otid)) otname,
               g.startdate,
               g.enddate,
               (CEIL((g.enddate - g.startdate) / 86400)) totaldays,
               g.schedule dmrequired,
               r.dmdateconflicts,
               r.dmstatus,
               r.enrolldateconflicts,
               r.courseid,
               r.groupid,
               cc.name category,
               cc.path  categorypath,
               cc.parent basecategory,
               (SELECT COUNT(1) FROM {message_conversations} conv WHERE conv.type = '2' AND conv.itemid = g.id AND conv.enabled = '1') gmessaging,
               r.subject,
               r.grade,
               r.department,
               r.moe_code,
               r.moe_name
          FROM {report_ghs_group_enrollment} r
          JOIN {course} c
            ON r.courseid = c.id
          JOIN {course_categories} cc 
            ON c.category = cc.id                
          JOIN {groups} g
            ON r.groupid = g.id
LEFT OUTER JOIN {cohort} coh ON r.cohortid = coh.id            
         WHERE 0 = 0";

        // Delete firs rowcount column.
        array_shift($columnsexport);

        $headers = $columnsexport;

        foreach ($headers as $ckey => $column) {
            $headers[$ckey] = get_string($column, 'report_ghs');
        }

        $tempdir = $CFG->tempdir . '/phpexcel';
        if (!file_exists($tempdir)) {
            mkdir($tempdir, 0777, true);
        }

        $reportname = 'ghs_group_enrollment';
        $filename = $tempdir . '/' . $reportname . '_' . date('Y-m-d') . '.csv';

        $outputcsv = fopen($filename, 'w+');

        // Output the column headings.
        fputcsv($outputcsv, $headers);

        $rs = $DB->get_recordset_sql($sql, $params);
        foreach ($rs as $tablerow) {
            $row = array();
            foreach ($columnsexport as $column) {
                $row[] = \report_ghs\helper::group_enrollment_data($tablerow, $column, $counter, $pageparams, true);
            }
            fputcsv($outputcsv, $row);
        }
        $rs->close();

        if (!empty($outputcsv)) {
            fclose($outputcsv);
        }

        helper::upload_file_to_onedrive($client, $filename, ONEDRIVE_OLD_DATA);

        $newfile = $tempdir . '/' . $reportname . '.csv';
        if (copy($filename, $newfile)) {
            helper::upload_file_to_onedrive($client, $newfile, ONEDRIVE_NEW_DATA);
        }

        set_config('ghs_group_enrollment_export', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsgroupenrollment', 'report_ghs') . ' ' . get_string('export', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }

}