<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

defined('MOODLE_INTERNAL') || die();

use core_tag_tag;
use report_ghs\helper;
use report_ghs\shared_lib as NED;

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_sla_report
 * @package report_ghs\task
 */
class adhoc_ghs_sla_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::update_report($cd->userid ?? null);

        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $DB;

        $DB->execute("TRUNCATE TABLE {report_ghs_sla}");

        $contextsystem = \context_system::instance();

        // $courses81 = \core_course_category::get(81, MUST_EXIST, true)->get_courses(['recursive' => 1]);
        // $courses82 = \core_course_category::get(82, MUST_EXIST, true)->get_courses(['recursive' => 1]);

        $courselist = helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);

        // Courses.
        $sql = "SELECT c.*
                 FROM {course} c
                WHERE c.id {$insql}
             ORDER BY c.shortname ASC";

        $sqlsubmassign = "SELECT s.timemodified FROM {assign_submission} s WHERE s.assignment = ? AND s.userid = ? AND s.status = 'submitted' AND s.latest = 1";
        $sqlsubmquiz = "SELECT qa.timefinish FROM {quiz_attempts} qa WHERE qa.quiz = ? AND qa.userid = ? AND qa.state = 'finished' ORDER BY qa.timefinish DESC LIMIT 1";
        $sqlsubm = ['assign' => $sqlsubmassign, 'quiz' => $sqlsubmquiz];

        $sqlactivitygrade = "SELECT gg.id, gg.finalgrade, gg.timemodified, gi.gradetype, gi.grademax, gg.excluded
                               FROM {grade_items} gi
                               JOIN {grade_grades} AS gg ON gi.id = gg.itemid
                              WHERE gi.itemtype = 'mod' AND gi.itemmodule = ? AND gi.iteminstance = ? AND gg.userid = ?";

        $sqlgroup = "SELECT g.* FROM {groups} g JOIN {groups_members} m ON g.id = m.groupid WHERE g.courseid = ? AND m.userid = ?";

        $sqlcompletion = "SELECT cmc.* FROM {course_modules_completion} cmc INNER JOIN {course_modules} cm ON cmc.coursemoduleid = cm.id
                           WHERE cmc.coursemoduleid = ? AND cmc.userid = ? AND cm.deletioninprogress = 0";

        $sqlcohort = "SELECT c.* FROM {cohort} c JOIN {cohort_members} m ON c.id = m.cohortid WHERE c.contextid = ? AND m.userid = ?";

        $rs = $DB->get_recordset_sql($sql, $params);

        $stractive = get_string('active', 'report_ghs');
        $strsuspendedcourse = get_string('suspendedcourse', 'report_ghs');
        $strsuspendedsite = get_string('suspendedsite', 'report_ghs');
        $strcompleted = get_string('completed', 'report_ghs');
        $otrole = $DB->get_record('role', array('shortname' => 'online-teacher'));
        $ctrole = $DB->get_record('role', array('shortname' => 'classroom-teacher'));
        $DM = NED::get_DM();

        foreach ($rs as $course) {
            $coursecontext = \context_course::instance($course->id);
            $graders = [];
            if (!$users = get_enrolled_users($coursecontext, 'mod/assign:submit', 0, 'u.*', 'u.id', 0, 0, false)) {
                continue;
            }

            $courseid = $course->id;
            $kica = NED::get_kica_enabled($courseid);
            if ($kica){
                NED::ki_get_all_by_course($courseid, null, $users, true);
                NED::kg_get_grades_by_course($courseid, array_keys($users), false, true);
            }

            foreach ($users as $user) {
                $group = null;
                if ($groups = $DB->get_records_sql($sqlgroup, [$course->id, $user->id])) {
                    $group = reset($groups);
                }

                $cms = NED::get_important_activities($course->id, $user);
                $cohort = $DB->get_record_sql($sqlcohort, [$contextsystem->id, $user->id], IGNORE_MULTIPLE);

                foreach ($cms as $cm) {
                    if ($cm->modname == 'assign' || $cm->modname == 'quiz') {
                        $var_sql = $sqlsubm[$cm->modname] ?? null;

                        // Deadline.
                        if (!$DM || empty($var_sql) || !$DM::is_cm_enabled($cm)) {
                            continue;
                        }

                        $module = $DM::get_dmm_by_cm($cm);
                        if (!$module){
                            continue;
                        }

                        if ($deadline = $module->get_user_effective_access($user->id)) {
                            if ($deadline < 0) {
                                $deadline = null;
                            }
                        }

                        $submissiontime = $DB->get_field_sql($var_sql, [$cm->instance, $user->id]);

                        $finalgrade = $DB->get_record_sql($sqlactivitygrade, [$cm->modname, $cm->instance, $user->id]);

                        $data = new \stdClass();
                        $data->userid = $user->id;
                        $data->courseid = $course->id;
                        if ($cohort) {
                            $data->chortid =  $cohort->id;
                        }
                        if ($tags = array_values(core_tag_tag::get_item_tags_array('core', 'course_modules', $cm->id))) {
                            if (!in_array('SLA-OT', $tags)) {
                                continue;
                            } else {
                                $data->slaot = 1;
                            }
                            if (array_intersect(helper::TAG_FINAL, $tags)) {
                                $data->finalactivity = 1;
                            } else if (array_intersect(helper::TAG_MIDTERM, $tags)) {
                                $data->finalactivity = 2;
                            }

                            if (!in_array('Summative', $tags) && !in_array('Formative', $tags)) {
                                continue;
                            }
                            $_tags = [];
                            if (in_array('Formative', $tags)) {
                                $_tags[] = 'Formative';
                            }
                            if (in_array('Summative', $tags)) {
                                $_tags[] = 'Summative';;
                            }
                            $data->activitytag = implode(', ', $_tags);
                        } else {
                            continue;
                        }
                        $data->cmid = $cm->id;
                        $data->module = $cm->modname;
                        $data->instance = $cm->instance;
                        if (method_exists($module, 'get_mod_grade')) {
                            if ($instancegrade = $module->get_mod_grade($user->id)) {
                                $data->instancegrade = $instancegrade;
                            }
                        }

                        $submission = $module->get_user_submission($user->id);
                        $data->submissionstatus = self::get_submission_status($cm->modname, $submission);

                        $data->activityname = $cm->name;
                        if ($deadline) {
                            $data->duedate = $deadline;
                        } else {
                            $data->duedate = 0;
                        }

                        // Do not show records if due date is in the future and there is no submission.
                        if ($data->submissionstatus == 'notsubmitted' && $data->duedate > time()) {
                            continue;
                        }

                        if ($submissiontime) {
                            $data->timesubmission = $submissiontime;
                        } else {
                            $data->timesubmission = 0;
                        }

                        if (isset($finalgrade->id)) {
                            $data->gradeid = $finalgrade->id;
                        }

                        if (isset($finalgrade->timemodified)) {
                            $data->timegradedmoodle = $finalgrade->timemodified;
                        } else {
                            $data->timegradedmoodle = 0;
                        }

                        if ((!empty($submissiontime) && ($submissiontime > 0)) || (!empty($deadline) && ($deadline > 0))) {
                            if ($deadline > time()) {
                                $data->numberofdaysmoodle = 0;
                            } else if (!empty($data->timegradedmoodle)) {
                                $data->numberofdaysmoodle = floor((($data->timegradedmoodle - max($submissiontime, $deadline)) / DAYSECS));
                            } else {
                                $data->numberofdaysmoodle = floor(((time() - max($submissiontime, $deadline)) / DAYSECS));
                            }
                        }

                        $data->kicaenabled = 0;
                        $data->timegradedkica = 0;
                        if ($kica){
                            $kicaitem = NED::ki_get_by_cm($cm, true);
                            if ($kicaitem){
                                $data->kicaenabled = 1;
                                // KICA.
                                $kica_grade = NED::kg_get_by_userid_cmid($user->id, $cm);
                                if (!empty($kica_grade->id) && isset($kica_grade->timemodified)) {
                                    $data->timegradedkica = $kica_grade->timemodified;
                                }
                            }
                        }

                        if (!$data->duedate && !$data->timesubmission && !$data->timegradedmoodle && !$data->timegradedkica) {
                            continue;
                        }

                        if ((!empty($submissiontime) && ($submissiontime > 0)) || (!empty($deadline) && ($deadline > 0))) {
                            if ($deadline > time()) {
                                $data->timegradedkica = 0;
                            } else if (!empty($data->timegradedkica)) {
                                $data->numberofdayskica = floor((($data->timegradedkica - max($submissiontime, $deadline)) / DAYSECS));
                            } else {
                                $data->numberofdayskica = floor(((time() - max($submissiontime, $deadline)) / DAYSECS));
                            }
                        }

                        $data->timecreated = time();

                        $data->userstatus = $stractive;
                        $completion = $DB->get_record_sql($sqlcompletion, [$cm->id, $user->id]);
                        if ($user->suspended) {
                            continue;
                        } else if (enrol_get_enrolment_end($course->id, $user->id) === false) {
                            continue;
                        } else if (!empty($completion) && $completion->completionstate > 1) {
                            $data->userstatus = $strcompleted;
                        }

                        if ($group) {
                            if (!isset($graders[$group->id])) {
                                if ($users = get_enrolled_users($coursecontext, 'mod/assign:grade', $group->id)) {
                                    $graders[$group->id] = (reset($users))->id;
                                    $data->graderid = $graders[$group->id];
                                }
                            } else {
                                $data->graderid = $graders[$group->id];
                            }
                        }

                        if ($ctusers = helper::get_role_users($ctrole->id, $coursecontext, ((isset($group)) ? $group->id : 0) )) {
                            $data->ctid = (count($ctusers) > 1) ? -1 : $ctusers[0]->id;
                        }

                        $DB->insert_record('report_ghs_sla', $data);

                    }
                }
            }

            NED::purge_course_depended_caches();
        }

        $rs->close();

        set_config('activitylastupdate', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsactivity', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }

    /**
     * @param $mod
     * @param $submission
     * @return string|null
     */
    public static function get_submission_status($mod, $submission) {
        if (empty($submission)) {
            return 'notsubmitted';
        }

        if ($mod == 'assign') {
            if ($submission->status == ASSIGN_SUBMISSION_STATUS_SUBMITTED || $submission->status == ASSIGN_SUBMISSION_STATUS_REOPENED) {
                return 'submitted';
            } else if ($submission->status == ASSIGN_SUBMISSION_STATUS_NEW) {
                return 'notsubmitted';
            } else if ($submission->status == ASSIGN_SUBMISSION_STATUS_DRAFT) {
                return 'draft';
            }
        } else if ($mod == 'quiz') {
            return 'submitted';
        }
        return null;
    }
}