<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * RADIX
 *
 * @package     report_ghs
 * @copyright   2020 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(__FILE__, 3) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use local_schoolmanager\school_handler as SC;
use block_ned_teacher_tools\shared_lib as SH;
use \local_academic_integrity\shared_lib as NED;
use local_schoolmanager\output\reports_header;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use report_ghs\helper;

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 20, PARAM_INT);
$sort      = optional_param('sort', 'dateapplied', PARAM_ALPHANUM);
$dir       = optional_param('dir', 'DESC', PARAM_ALPHA);
// Action.
$action    = optional_param('action', false, PARAM_ALPHA);
$search    = optional_param('search', '', PARAM_TEXT);

// Filters.
$schoolid = optional_param('schoolid', 0, PARAM_INT);
$filterclass = optional_param('filterclass', 0, PARAM_INT);
$filtersubmitter = optional_param('filtersubmitter', 0, PARAM_INT);
$filtercourse = optional_param('filtercourse', 0, PARAM_INT);
$filterstudent = optional_param('filterstudent', 0, PARAM_INT);

$cfg = get_config('report_ghs');

require_login(null, false);
$contextsystem = context_system::instance();

// Permission.
if (!has_capability('report/ghs:viewactivityextensions', $contextsystem)
    && !\report_ghs\helper::has_capability_in_any_course('report/ghs:viewactivityextensions')) {
    throw new required_capability_exception($contextsystem, 'report/ghs:viewactivityextensions', 'nopermissions', '');
}

$thispageurl = new moodle_url('/report/ghs/ghs_activity_extension.php');
$fullpageurl = new moodle_url('/report/ghs/ghs_activity_extension.php', [
    'page' => $page,
    'perpage' => $perpage,
    'sort' => $sort,
    'dir' => $dir,
    'action' => $action,
    'schoolid' => $schoolid,
    'filterclass' => $filterclass,
    'filtersubmitter' => $filtersubmitter,
    'filtercourse' => $filtercourse,
    'filterstudent' => $filterstudent,
]);
$report = basename($thispageurl->out(), '.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('schoolmanager');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$name = get_string('ghsactivityextension', 'report_ghs');
$title = get_string('ghsactivityextension', 'report_ghs');
$heading = $SITE->fullname;

// Breadcrumb.
if (has_capability('report/ghs:viewactivityextensions', $contextsystem)) {
    admin_externalpage_setup('reportghsactivityextension', null, null, '', ['pagelayout' => 'schoolmanager']);
}

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$datacolumns = array(
    'id' => 'r.id',
    'school' => 'coh.name',
    'cohortid' => 'r.cohortid',
    'course' => 'c.shortname',
    'courseid' => 'c.id',
    'class' => 'g.name',
    'groupid' => 'g.id',
    'userid' => 'r.userid',
    'overridenby' => 'r.overridenby',
    'submitter' => "CONCAT(su.firstname, ' ',  su.lastname)",
    'appliedto' => 'g.name',
    'change' => 'r.duedate',
    'dateapplied' => 'r.timecreated',
    'reasonprovided' => 'r.reason',
    'activity' => "(CASE WHEN r.modname = 'assign'  THEN (SELECT a.name FROM {assign} a WHERE a.id = r.instance)                
                   WHEN r.modname  = 'quiz'  THEN (SELECT q.name FROM {quiz} q WHERE q.id = r.instance)                   
                   ELSE '' END)"
);

$params = [];

// Filter.
$where = '';

if (!has_capability('report/ghs:viewactivityextensionallschools', $contextsystem)) {
    $sh = new SC();
    if (!$schools = $sh->get_schools()) {
        $where .= " AND 0=1";
    } else {
        list($insql, $aparams) = $DB->get_in_or_equal(array_keys($schools), SQL_PARAMS_NAMED);
        $where .= " AND " . $datacolumns['cohortid'] . " {$insql}";
        $params = array_merge($params, $aparams);
    }
} else if ($schoolid) {
    $where .= " AND ".$datacolumns['cohortid']." = :school";
    $params['school'] = $schoolid;
}

if ($filterclass) {
    $where .= " AND ".$datacolumns['groupid']." = :class";
    $params['class'] = $filterclass;
}
if ($filtersubmitter) {
    $where .= " AND ".$datacolumns['overridenby']." = :submitter";
    $params['submitter'] = $filtersubmitter;
}
if ($filtercourse) {
    $where .= " AND ".$datacolumns['courseid']." = :course";
    $params['course'] = $filtercourse;
}
if ($filterstudent) {
    $where .= " AND ".$datacolumns['userid']." = :student";
    $params['student'] = $filterstudent;
}

// Sort.
$order = '';
if ($sort) {
    $order = " ORDER BY $datacolumns[$sort] $dir";
}

$courselist = helper::get_report_courses();

list($insql, $cparams) = $DB->get_in_or_equal($courselist);

$pageparams = array();
$pageparams['insql'] = $insql;
$pageparams['params'] = $cparams;

// Filter by capabilies.
\report_ghs\helper::report_filter($where, $params, $report, 'report/ghs:viewactivityextensions');

// Count records for paging.
$countsql = "SELECT COUNT(1) 
          FROM {block_ned_teacher_tools_exte} r
LEFT OUTER JOIN {course} c ON r.courseid = c.id
    INNER JOIN {groups} g ON r.groupid = g.id
    INNER JOIN {user} su ON r.overridenby = su.id
LEFT OUTER JOIN {cohort} coh ON r.cohortid = coh.id
         WHERE 0=0
$where";

$totalcount = $DB->count_records_sql($countsql, $params);

// Table columns.
$columns = array(
    'rowcount',
    'school',
    'course',
    'class',
    'submitter',
    'activity',
    'appliedto',
    'change',
    'dateapplied',
    'reasonprovided'
);
$exportcpilumns = array(
    'rowcount',
    'school',
    'course',
    'class',
    'submitter',
    'activity',
    'appliedto',
    'appliedtourl',
    'change',
    'dateapplied',
    'reasonprovided'
);

$sql = "SELECT r.id,
               r.userid,
               r.groupid,
               r.courseid,
               r.cmid,
               r.modname,
               r.instance,
               r.duedateorig,
               r.duedate,
               r.reason reasonprovided,
               r.timecreated dateapplied,
               c.shortname course,
               g.name class, 
               r.overridenby submitterid,
               CONCAT(su.firstname, ' ',  su.lastname) AS submitter,
               g.name appliedto,
               coh.name school,
               CASE
                   WHEN r.modname = 'assign'  THEN (SELECT a.name FROM {assign} a WHERE a.id = r.instance)                
                   WHEN r.modname  = 'quiz'  THEN (SELECT q.name FROM {quiz} q WHERE q.id = r.instance)                   
                   ELSE ''
                END AS activity
          FROM {block_ned_teacher_tools_exte} r
LEFT OUTER JOIN {course} c ON r.courseid = c.id
    INNER JOIN {groups} g ON r.groupid = g.id
    INNER JOIN {user} su ON r.overridenby = su.id
LEFT OUTER JOIN {cohort} coh ON r.cohortid = coh.id
         WHERE 0=0
               $where              
               $order";

if ($action == 'excel') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);

    $table = new stdClass();
    $table->head = $exportcpilumns;

    // Delete first rowcount column.
    $itemid = array_shift($table->head);
    // Delete last action column.
    //array_pop($table->head);

    $counter = 0;
    $filename = $report.'_'.(date('Y-m-d'));
    $downloadfilename = clean_filename($filename);

    $workbook = new Spreadsheet();
    $myxls = $workbook->setActiveSheetIndex(0);

    $numberofcolumns = count($table->head);

    // Header row.
    foreach ($table->head as $key => $heading) {
        $cell = Coordinate::stringFromColumnIndex($key + 1) . '1'; // A1 cell address.
        $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br(get_string($heading, 'report_ghs'))))));
    }

    // Data rows.
    $rs = $DB->get_recordset_sql($sql, $params);
    $rownum = 2;
    foreach ($rs as $tablerow) {
        $row = array();
        $columnum = 1;
        foreach ($table->head as $column) {
            $data = \report_ghs\helper::activity_extension_data($tablerow, $column, $counter, $pageparams, true);
            $cell = Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.
            if (preg_match("/^[fh]tt?p:\/\//", $data)) {
                $myxls->setCellValue($cell, 'Link');
                $myxls->getCell($cell)->getHyperlink()->setUrl($data);
            } else {
                $myxls->setCellValue($cell, $data);
            }
            $columnum++;
        }
        $rownum++;
    }
    $rs->close();


    // Freeze header.
    $myxls->freezePane('A2');

    // Filter.
    $myxls->setAutoFilter(
        $myxls->calculateWorksheetDimension()
    );

    // Auto column width calculation.
    foreach (range('A', $myxls->getHighestDataColumn()) as $col) {
        $myxls->getColumnDimension($col)->setAutoSize(true);
    }

    // Header format.
    $styleArray = array(
        'font' => array(
            'bold' => true,
        ),
        'fill' => array(
            'fillType' => Fill::FILL_SOLID,
            'color' => array(
                'argb' => 'FFFFF000',
            ),
        ),
    );
    $myxls->getStyle('A1:'.Coordinate::stringFromColumnIndex($numberofcolumns).'1')->applyFromArray($styleArray);

    // Rename worksheet
    $myxls->setTitle('export');

    // Set active sheet index to the first sheet, so Excel opens this as the first sheet
    $workbook->setActiveSheetIndex(0);

    $objWriter = new Xlsx($workbook);

    if (ob_get_length()) {
        ob_end_clean();
    }
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
    header('Cache-Control: max-age=0');

    ob_end_clean();
    $objWriter->save('php://output');
    exit;
} else if ($action == 'csv') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);
    $table = new stdClass();

    // Delete firs rowcount column.
    array_shift($columns);
    // Delete last action column.
    //array_pop($columns);

    $headers = $columns;

    foreach ($headers as $ckey => $column) {
        $headers[$ckey] = get_string($column, 'report_ghs');
    }

    if (ob_get_length()) {
        ob_end_clean();
    }
    // Output headers so that the file is downloaded rather than displayed.
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=ghs_enrollment_'.(date('Y-m-d')).'.csv');

    // Create a file pointer connected to the output stream.
    $outputcsv = fopen('php://output', 'w');

    // Output the column headings.
    fputcsv($outputcsv, $headers);

    $counter = 0;

    $rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        foreach ($columns as $column) {
            $row[] = \report_ghs\helper::activity_extension_data($tablerow, $column, $counter, $pageparams, true);
        }
        fputcsv($outputcsv, $row);
    }
    $rs->close();
    exit;
} else {

    foreach ($columns as $column) {
        $string[$column] = get_string($column, 'report_ghs');
        if ($sort != $column) {
            $columnicon = "";
            $columndir = "ASC";
        } else {
            $columndir = $dir == "ASC" ? "DESC" : "ASC";
            $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
            $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
        }
        if (($column == 'rowcount') || ($column == 'action')) {
            $$column = $string[$column];
        } else {
            $sorturl = clone $fullpageurl;
            $sorturl->param('perpage', $perpage);
            $sorturl->param('sort', $column);
            $sorturl->param('dir', $columndir);
            $sorturl->param('search', $search);

            $$column = html_writer::link($sorturl->out(false), $string[$column]).' '.$columnicon;
        }
    }

    $table = new html_table();

    $table->head = array();
    $table->wrap = array();
    $table->attributes = ['class' => 'nedtable fullwidth'];

    foreach ($columns as $column) {
        $table->head[$column] = $$column;
        $table->wrap[$column] = '';
    }

    // Override cell wrap.
    $table->wrap['rowcount'] = 'nowrap';
    $table->wrap['action'] = 'nowrap';

    $tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

    $counter = ($page * $perpage);

    foreach ($tablerows as $tablerow) {
        $row = new html_table_row();

        foreach ($columns as $column) {
            $varname = 'cell'.$column;
            $$varname = new html_table_cell(\report_ghs\helper::activity_extension_data($tablerow, $column, $counter, $pageparams));
        }

        $row->cells = array();
        foreach ($columns as $column) {
            $varname = 'cell' . $column;
            $row->cells[$column] = $$varname;
        }
        $table->data[] = $row;

    }

    echo $OUTPUT->header();
    echo html_writer::start_div('page-content-wrapper');
    echo $OUTPUT->render(new reports_header());
    echo html_writer::tag('h1', $title, array('class' => 'page-title'));

    // Filter form.
    $searchformurl = new moodle_url('/report/ghs/ghs_activity_extension.php');

    // Schools
    $handler = new SC();
    $schoolfilter = $handler->get_control_form($schoolid, $fullpageurl, false, true);

    // Classes
    $sql = "SELECT DISTINCT r.groupid, g.name
              FROM {block_ned_teacher_tools_exte} r
              JOIN {groups} g ON r.groupid = g.id
          ORDER BY g.name ASC";
    $classoptions =  ['0' => get_string('all')] + $DB->get_records_sql_menu($sql, $params);

    // Submitters
    $sql = "SELECT DISTINCT r.overridenby, CONCAT(su.firstname, ' ',  su.lastname) name
              FROM {block_ned_teacher_tools_exte} r
              JOIN {user} su ON r.overridenby = su.id
          ORDER BY su.lastname ASC";
    $submitteroptions = ['0' => get_string('all')] + $DB->get_records_sql_menu($sql, $params);

    // Courses
    $sql = "SELECT DISTINCT r.courseid, c.shortname name
              FROM {block_ned_teacher_tools_exte} r
              JOIN {course} c ON r.courseid = c.id
          ORDER BY c.shortname ASC";
    $courseoptions = ['0' => get_string('all')] + $DB->get_records_sql_menu($sql, $params);

    // Students
    $sql = "SELECT DISTINCT r.userid, CONCAT(u.firstname, ' ',  u.lastname) name
              FROM {block_ned_teacher_tools_exte} r
              JOIN {user} u ON r.userid = u.id
          ORDER BY u.lastname ASC";
    $studentoptions = ['0' => get_string('all')] + $DB->get_records_sql_menu($sql, $params);


    $searchform =
        // First row.
        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').

        // Class.
        NED::single_select($fullpageurl, 'filterclass', $classoptions, $filterclass, get_string('class', 'report_ghs'), null, null, ['class' => 'mb-2 mr-sm-2']).

        // Submitter.
        NED::single_select($fullpageurl, 'filtersubmitter', $submitteroptions, $filtersubmitter, get_string('submitter', 'report_ghs'), null, null, ['class' => 'mb-2 mr-sm-2']).

        html_writer::end_div().
        html_writer::end_div().

        // Second row.
        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').

        // School.
        html_writer::start_div('p-2').
        $schoolfilter.
        html_writer::end_div().

        // Course.
        SH::single_autocomplete($fullpageurl, 'filtercourse', $courseoptions, $filtercourse, get_string('course', 'report_ghs'), 'mb-2 mr-sm-2').

        // Student.
        SH::single_autocomplete($fullpageurl, 'filterstudent', $studentoptions, $filterstudent, get_string('student', 'report_ghs'), 'mb-2 mr-sm-2').

        html_writer::end_div().
        html_writer::end_div();

    echo html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));

    $pagingurl = clone $fullpageurl;
    $pagingurl->remove_params('page');
    $pagingurl->remove_params('action');

    $pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

    // Export link.
    $exporexcelurl = $thispageurl;
    $exporexcelurl->remove_all_params();
    $exporexcelurl->param('action', 'excel');
    $exportexceliconurl = $OUTPUT->image_url('f/spreadsheet');
    $exportexcelicon = html_writer::img($exportexceliconurl, '', array('width' => '16', 'height' => '16'));
    $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
    $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

    $exportexturl = $thispageurl;
    $exportexturl->remove_all_params();
    $exportexturl->param('action', 'csv');
    $exporttexticonurl = $OUTPUT->image_url('f/text');
    $exporttexticon = html_writer::img($exporttexticonurl, '', array('width' => '16', 'height' => '16'));
    $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
    $exporttextlink = html_writer::link($exportexturl, 'CSV');

    $exportbuttons =  html_writer::div(
        get_string('export', 'report_ghs').' :&nbsp;&nbsp;'.
        $exportexceliconlink.'&nbsp;'.$exportexcellink.'&nbsp;|&nbsp; '
        .$exporttexticonlink.'&nbsp;'.$exporttextlink, 'export-link-wrapper',
        array('style' => 'text-align:center;')
    );

    if ($outputpagingbar = $OUTPUT->render($pagingbar)) {
        echo $outputpagingbar;
    } else {
        echo html_writer::tag('div', '', ['class' => 'dummy-pagination']);
    }
    echo html_writer::div($exportbuttons, 'export-link-wrapper-top');
    echo html_writer::table($table);
    echo $outputpagingbar;
    echo $exportbuttons;

    echo html_writer::end_div(); // Main wrapper.
    echo $OUTPUT->footer();
}