<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * RADIX
 *
 * @package     report_ghs
 * @copyright   2020 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use local_schoolmanager\school_handler as SH;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use report_ghs\helper;
use local_schoolmanager\output\reports_header;
use report_ghs\shared_lib as NED;

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 200, PARAM_INT);
$sort      = optional_param('sort', 'school', PARAM_ALPHANUM);
$dir       = optional_param('dir', 'DESC', PARAM_ALPHA);
// Action.
$action    = optional_param('action', false, PARAM_ALPHA);
$search    = optional_param('search', '', PARAM_TEXT);

// Filters.
$schoolid  = optional_param('schoolid', 0, PARAM_INT);
$filter = optional_param('filter', 14, PARAM_INT);
$designation = optional_param('designation', 0, PARAM_INT);
$completion = optional_param('completion', 0, PARAM_INT);
$optfilter = optional_param('opt', 0, PARAM_INT);
$ossltyearfilter = optional_param('ossltyear', '2023-24', PARAM_RAW);
$statusfilter = optional_param('status', 0, PARAM_RAW);

$cfg = get_config('report_ghs');

require_login(null, false);
$contextsystem = context_system::instance();

// Permission.
if (!has_any_capability(['report/ghs:viewenglishproficiencyallschools', 'report/ghs:viewenglishproficiencyownschool'], $contextsystem)) {
    throw new required_capability_exception($contextsystem, 'report/ghs:viewenglishproficiencyallschools', 'nopermissions', '');
}

$thispageurl = new moodle_url('/report/ghs/ghs_english_proficiency.php');
$fullpageurl = new moodle_url('/report/ghs/ghs_english_proficiency.php', [
    'page' => $page,
    'perpage' => $perpage,
    'sort' => $sort,
    'dir' => $dir,
    'action' => $action,
    'schoolid' => $schoolid,
    'filter' => $filter,
    'designation' => $designation,
    'completion' => $completion,
    'opt' => $optfilter,
    'ossltyear' => $ossltyearfilter,
    'status' => $statusfilter,
]);
$report = basename($thispageurl->out(), '.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('schoolmanager');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$name = get_string('ghsenglishproficiency', 'report_ghs');
$title = get_string('ghsenglishproficiency', 'report_ghs');
$heading = $SITE->fullname;

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$stractive = get_string('active', 'report_ghs');
$strinactive = get_string('inactive', 'report_ghs');

$datacolumns = array(
    'id' => 'r.id',
    'schoolid' => 's.id',
    'school' => 's.name',
    'student' => "(CASE
                    WHEN u.id > 0 THEN CONCAT(u.firstname, ' ', u.lastname)            
                    ELSE CONCAT(d.firstname, ' ', d.lastname)  
               END)",
    'status' => "(CASE
                    WHEN u.id > 0 AND u.deleted = 0 AND u.suspended = 0 THEN '".$stractive."'            
                    ELSE '".$strinactive."'  
               END)",
    'program' => 'd.program',
    'firstlanguage' => 'd.firstlanguage',
    'opt' => 'd.opt',
    'ossltstatus' => 'd.ossltstatus',
    'ossltscore' => 'd.ossltscore',
    'ossltyear' => 'd.ossltyear',
    'esl' => 'd.esl',
    'gradyear' => "d.gradyear",
    'graduation' => "d.graduation",
    'grade' => "d.grade",
    'esldo' => "(CASE
                    WHEN d.esldo_grade > 0 AND (d.esldo = '' OR d.esldo IS NULL OR d.esldo = 'Not Found') THEN d.esldo_grade            
                    ELSE d.esldo
               END)",
    'esleo' => "(CASE
                    WHEN d.esleo_grade > 0 AND (d.esleo = '' OR d.esleo IS NULL OR d.esleo = 'Not Found') THEN d.esleo_grade            
                    ELSE d.esleo
               END)",
    'eng3u' => "(CASE
                    WHEN d.eng3u_grade > 0 AND (d.eng3u = '' OR d.eng3u IS NULL OR d.eng3u = 'Not Found') THEN d.eng3u_grade            
                    ELSE d.eng3u
               END)",
    'eng4u' => "(CASE
                    WHEN d.eng4u_grade > 0 AND (d.eng4u = '' OR d.eng4u IS NULL OR d.eng4u = 'Not Found') THEN d.eng4u_grade            
                    ELSE d.eng4u
               END)",
    'olc4o' => "(CASE
                    WHEN d.olc4o_grade > 0 AND (d.olc4o = '' OR d.olc4o IS NULL OR d.olc4o = 'Not Found') THEN d.olc4o_grade            
                    ELSE d.olc4o
               END)",
);

$filteroptions = [
    1 => get_string('completedosslteng3u', 'report_ghs'),
    2 => get_string('completedosslteng4u', 'report_ghs'),
    3 => get_string('passedosslt', 'report_ghs'),
    4 => get_string('failedosslt', 'report_ghs'),
    5 => get_string('failedosslteng3uover', 'report_ghs'),
    6 => get_string('failedosslteng4uover', 'report_ghs'),
    7 => get_string('failedosslteng3ueng4uover', 'report_ghs'),
    8 => get_string('completedeng3u', 'report_ghs'),
    9 => get_string('completedeng4u', 'report_ghs'),
    10 => get_string('eng3ugradeunder75', 'report_ghs'),
    11 => get_string('eng3ugradeover75', 'report_ghs'),
    12 => get_string('eng4ugradeunder75', 'report_ghs'),
    13 => get_string('eng4ugradeover75', 'report_ghs'),
    14 => get_string('completedosslt', 'report_ghs'),
    15 => get_string('didnotcompleteosslt', 'report_ghs'),
];

$designationoptions = [
    1 => get_string('esl', 'report_ghs'),
    2 => get_string('nonesl', 'report_ghs'),
];

$coursecompletionoptions = [
    1 => get_string('completed', 'report_ghs'),
    2 => get_string('inprogress', 'report_ghs'),
    3 => get_string('both', 'report_ghs'),
];

$optoptions = [
    1 => get_string('c2', 'report_ghs'),
    2 => get_string('c1', 'report_ghs'),
    3 => get_string('b2', 'report_ghs'),
    4 => get_string('b1', 'report_ghs'),
    5 => get_string('a2', 'report_ghs'),
    6 => get_string('a1', 'report_ghs'),
    7 => get_string('notreported', 'report_ghs'),
    8 => get_string('notrequired', 'report_ghs'),
];

$ossltyearoptions = [
    '2022-23' => '2022-23',
    '2023-24' => '2023-24',
];

$statusoptions = [
    $stractive => $stractive,
    $strinactive => $strinactive,
];

$params = [];

// Filter.
$where = '';

if (!has_capability('report/ghs:viewenglishproficiencyallschools', $contextsystem)) {
    $sh = new SH();
    if (!$schools = $sh->get_schools()) {
        $where .= " AND 0=1";
    } else {
        list($insql, $aparams) = $DB->get_in_or_equal(array_keys($schools), SQL_PARAMS_NAMED);
        $where .= " AND " . $datacolumns['schoolid'] . " {$insql}";
        $params = array_merge($params, $aparams);
    }
} else if ($schoolid) {
    $where .= " AND ".$datacolumns['schoolid']." = :schoolid";
    $params['schoolid'] = $schoolid ?? 0;
}

if ($filter && array_key_exists($filter, $filteroptions)) {
    switch ($filter) {
        case 1:
            $where .= " AND ".$datacolumns['ossltstatus']." IN ('Pass', 'Fail') AND ".$datacolumns['eng3u']." IS NOT NULL";
            break;
        case 2:
            $where .= " AND ".$datacolumns['ossltstatus']." IN ('Pass', 'Fail') AND ".$datacolumns['eng4u']." IS NOT NULL";
            break;
        case 3:
            $where .= " AND ".$datacolumns['ossltstatus']." = 'Pass'";
            break;
        case 4:
            $where .= " AND ".$datacolumns['ossltstatus']." = 'Fail'";
            break;
        case 5:
            $where .= " AND ".$datacolumns['ossltstatus']." = 'Fail' AND (d.eng3u >= 75 OR d.eng3u_grade >= 75)";
            break;
        case 6:
            $where .= " AND ".$datacolumns['ossltstatus']." = 'Fail' AND (d.eng4u >= 75 OR d.eng4u_grade >= 75)";
            break;
        case 7:
            $where .= " AND ".$datacolumns['ossltstatus']." = 'Fail' AND ((d.eng3u >= 75 OR d.eng3u_grade >= 75) OR (d.eng4u >= 75 OR d.eng4u_grade >= 75))";
            break;
        case 8:
            $where .= " AND d.eng3u >= 0 AND d.eng3u <> 'Not Found' AND d.eng3u <> ''";
            break;
        case 9:
            $where .= " AND d.eng4u >= 0 AND d.eng4u <> 'Not Found' AND d.eng4u <> ''";
            break;
        case 10:
            $where .= " AND ((d.eng3u < 75 AND d.eng3u <> 'Not Found' AND d.eng3u <> '') OR (d.eng3u_grade < 75 AND d.eng3u_grade <> ''))";
            break;
        case 11:
            $where .= " AND ((d.eng3u >= 75 AND d.eng3u <> 'Not Found' AND d.eng3u <> '') OR (d.eng3u_grade >= 75 AND d.eng3u_grade <> ''))";
            break;
        case 12:
            $where .= " AND ((d.eng4u < 75 AND d.eng4u <> 'Not Found' AND d.eng4u <> '') OR (d.eng4u_grade < 75 AND d.eng4u_grade <> ''))";
            break;
        case 13:
            $where .= " AND ((d.eng4u >= 75 AND d.eng4u <> 'Not Found' AND d.eng4u <> '') OR (d.eng4u_grade >= 75 AND d.eng4u_grade <> ''))";
            break;
        case 14:
            $where .= " AND ".$datacolumns['ossltstatus']." IN ('Pass', 'Fail')";
            break;
        case 15:
            $where .= " AND (".$datacolumns['ossltstatus']." NOT IN ('Pass', 'Fail')";
            $where .= " OR ".$datacolumns['ossltstatus']." IS NULL)";
            break;
    }
}

if ($optfilter && array_key_exists($optfilter, $optoptions)) {
    $where .= " AND ".$datacolumns['opt']." = '" . $optoptions[$optfilter] . "'";
}
if ($ossltyearfilter && array_key_exists($ossltyearfilter, $ossltyearoptions)) {
    $where .= " AND ".$datacolumns['ossltyear']." = '" . $ossltyearoptions[$ossltyearfilter] . "'";
}
if ($statusfilter && array_key_exists($statusfilter, $statusoptions)) {
    $where .= " AND ".$datacolumns['status']." = '" . $statusoptions[$statusfilter] . "'";
}

if ($designation && array_key_exists($designation, $designationoptions)) {
    switch ($designation) {
        case 1:
            $where .= " AND (".$datacolumns['firstlanguage']." <> '' AND " . $datacolumns['firstlanguage']." <> 'English')";
            break;
        case 2:
            $where .= " AND ".$datacolumns['firstlanguage']." = 'English'";
            break;
    }
}

if ($completion && array_key_exists($completion, $coursecompletionoptions)) {
    switch ($completion) {
        case 1: // Completed
            $where .= " AND ((CASE WHEN (d.esldo = '' OR d.esldo IS NULL OR d.esldo = 'Not Found') AND d.esldo_grade > 0 THEN 0 WHEN (d.esldo = '' OR d.esldo IS NULL) THEN 1 WHEN d.esldo = 'Not Found' THEN 2 ELSE d.esldo END) *
	                        (CASE WHEN (d.esleo = '' OR d.esleo IS NULL OR d.esleo = 'Not Found') AND d.esleo_grade > 0  THEN 0 WHEN (d.esleo = '' OR d.esleo IS NULL) THEN 1 WHEN d.esleo = 'Not Found' THEN 2 ELSE d.esleo END) *
                            (CASE WHEN (d.eng3u = '' OR d.eng3u IS NULL OR d.eng3u = 'Not Found') AND d.eng3u_grade > 0  THEN 0 WHEN (d.eng3u = '' OR d.eng3u IS NULL) THEN 1 WHEN d.eng3u = 'Not Found' THEN 2 ELSE d.eng3u END) *
                            (CASE WHEN (d.eng4u = '' OR d.eng4u IS NULL OR d.eng4u = 'Not Found') AND d.eng4u_grade > 0  THEN 0 WHEN (d.eng4u = '' OR d.eng4u IS NULL) THEN 1 WHEN d.eng4u = 'Not Found' THEN 2 ELSE d.eng4u END) *
                            (CASE WHEN (d.olc4o = '' OR d.olc4o IS NULL OR d.olc4o = 'Not Found') AND d.olc4o_grade > 0  THEN 0 WHEN (d.olc4o = '' OR d.olc4o IS NULL) THEN 1 WHEN d.olc4o = 'Not Found' THEN 2 ELSE d.olc4o END)) > 1";
            break;
        case 2:
            $where .= " AND ((d.esldo_grade > 0 AND (d.esldo = '' OR d.esldo IS NULL OR d.esldo = 'Not Found'))
                             OR (d.esleo_grade > 0 AND (d.esleo = '' OR d.esleo IS NULL OR d.esleo = 'Not Found'))
                             OR (d.eng3u_grade > 0 AND (d.eng3u = '' OR d.eng3u IS NULL OR d.eng3u = 'Not Found'))
                             OR (d.eng4u_grade > 0 AND (d.eng4u = '' OR d.eng4u IS NULL OR d.eng4u = 'Not Found'))
                             OR (d.olc4o_grade > 0 AND (d.olc4o = '' OR d.olc4o IS NULL OR d.olc4o = 'Not Found')))";
            break;
        case 3:
            $where .= " AND (((CASE WHEN (d.esldo = '' OR d.esldo IS NULL OR d.esldo = 'Not Found') AND d.esldo_grade > 0 THEN 0 WHEN (d.esldo = '' OR d.esldo IS NULL) THEN 1 WHEN d.esldo = 'Not Found' THEN 2 ELSE d.esldo END) *
	                        (CASE WHEN (d.esleo = '' OR d.esleo IS NULL OR d.esleo = 'Not Found') AND d.esleo_grade > 0  THEN 0 WHEN (d.esleo = '' OR d.esleo IS NULL) THEN 1 WHEN d.esleo = 'Not Found' THEN 2 ELSE d.esleo END) *
                            (CASE WHEN (d.eng3u = '' OR d.eng3u IS NULL OR d.eng3u = 'Not Found') AND d.eng3u_grade > 0  THEN 0 WHEN (d.eng3u = '' OR d.eng3u IS NULL) THEN 1 WHEN d.eng3u = 'Not Found' THEN 2 ELSE d.eng3u END) *
                            (CASE WHEN (d.eng4u = '' OR d.eng4u IS NULL OR d.eng4u = 'Not Found') AND d.eng4u_grade > 0  THEN 0 WHEN (d.eng4u = '' OR d.eng4u IS NULL) THEN 1 WHEN d.eng4u = 'Not Found' THEN 2 ELSE d.eng4u END) *
                            (CASE WHEN (d.olc4o = '' OR d.olc4o IS NULL OR d.olc4o = 'Not Found') AND d.olc4o_grade > 0  THEN 0 WHEN (d.olc4o = '' OR d.olc4o IS NULL) THEN 1 WHEN d.olc4o = 'Not Found' THEN 2 ELSE d.olc4o END)) > 1";
            $where .= " OR ((d.esldo_grade > 0 AND (d.esldo = '' OR d.esldo IS NULL OR d.esldo = 'Not Found'))
                             OR (d.esleo_grade > 0 AND (d.esleo = '' OR d.esleo IS NULL OR d.esleo = 'Not Found'))
                             OR (d.eng3u_grade > 0 AND (d.eng3u = '' OR d.eng3u IS NULL OR d.eng3u = 'Not Found'))
                             OR (d.eng4u_grade > 0 AND (d.eng4u = '' OR d.eng4u IS NULL OR d.eng4u = 'Not Found'))
                             OR (d.olc4o_grade > 0 AND (d.olc4o = '' OR d.olc4o IS NULL OR d.olc4o = 'Not Found'))))";

            break;
    }
}

// Sort.
$order = '';
if ($sort) {
    $order = " ORDER BY $datacolumns[$sort] IS NULL, $datacolumns[$sort] = '', $datacolumns[$sort] $dir";
}

$pageparams = array();

// Count records for paging.
$countsql = "SELECT COUNT(1) 
          FROM {report_ghs_english_proficien} AS d 
     LEFT JOIN {user} AS u ON d.userid = u.id
     LEFT JOIN {local_schoolmanager_school} AS s ON d.school = s.code
         WHERE d.defaultrole = 'Student'
$where";

$totalcount = $DB->count_records_sql($countsql, $params);

// Table columns.
$columns = array(
    'rowcount',
    'school',
    'student',
    'status',
    'grade',
    'program',
    'graduation',
    'opt',
    'ossltstatus',
    'ossltscore',
    'ossltyear',
    'esldo',
    'esleo',
    'eng3u',
    'eng4u',
    'olc4o',
);
$columns_export = array(
    'rowcount',
    'school',
    'student',
    'oen',
    'status',
    'grade',
    'program',
    'graduation',
    'opt',
    'ossltstatus',
    'ossltscore',
    'ossltyear',
    'esl',
    'esldo',
    'esleo',
    'eng3u',
    'eng4u',
    'olc4o',
);

$sql = "SELECT d.id, u.id userid, u.deleted, u.suspended,
               CASE
                    WHEN u.id > 0 AND u.deleted = 0 AND u.suspended = 0 THEN '".$stractive."'            
                    ELSE '".$strinactive."'  
               END AS status,
               CASE
                    WHEN u.id > 0 THEN CONCAT(u.firstname, ' ', u.lastname)            
                    ELSE CONCAT(d.firstname, ' ', d.lastname)  
               END AS student,
               s.id schoolid,
               s.code school,
               d.program,
               d.oen,
               d.opt,
               d.ossltstatus,
               d.ossltscore,
               d.ossltyear,
               CASE
                    WHEN d.esldo_grade > 0 AND (d.esldo = '' OR d.esldo IS NULL OR d.esldo = 'Not Found') THEN CONCAT('(', ROUND(d.esldo_grade), ')')            
                    ELSE d.esldo
               END AS esldo,
               d.esldo_grade,
               
               CASE
                    WHEN d.esleo_grade > 0 AND (d.esleo = '' OR d.esleo IS NULL OR d.esleo = 'Not Found') THEN CONCAT('(', ROUND(d.esleo_grade), ')')            
                    ELSE d.esleo
               END AS esleo,
               d.esleo_grade,
               
               CASE
                    WHEN d.eng3u_grade > 0 AND (d.eng3u = '' OR d.eng3u IS NULL OR d.eng3u = 'Not Found') THEN CONCAT('(', ROUND(d.eng3u_grade), ')')            
                    ELSE d.eng3u
               END AS eng3u,
               d.eng3u _eng3u,
               d.eng3u_grade,
               
               CASE
                    WHEN d.eng4u_grade > 0 AND (d.eng4u = '' OR d.eng4u IS NULL OR d.eng4u = 'Not Found') THEN CONCAT('(', ROUND(d.eng4u_grade), ')')            
                    ELSE d.eng4u
               END AS eng4u,
               d.eng4u _eng4u,
               d.eng4u_grade,
               
               CASE
                    WHEN d.olc4o_grade > 0 AND (d.olc4o = '' OR d.olc4o IS NULL OR d.olc4o = 'Not Found') THEN CONCAT('(', ROUND(d.olc4o_grade), ')')            
                    ELSE d.olc4o
               END AS olc4o,
               d.olc4o_grade,
               
               d.gradyear,
               d.graduation,
               d.grade,
               d.esl
          FROM {report_ghs_english_proficien} AS d 
     LEFT JOIN {user} AS u ON d.userid = u.id
     LEFT JOIN {local_schoolmanager_school} AS s ON d.school = s.code
         WHERE d.defaultrole = 'Student'
               $where              
               $order";

if ($action == 'excel') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);

    $table = new stdClass();
    $table->head = $columns_export;

    // Delete first rowcount column.
    $itemid = array_shift($table->head);
    // Delete last action column.
    //array_pop($table->head);

    $counter = 0;
    $filename = $report.'_'.(date('Y-m-d'));
    $downloadfilename = clean_filename($filename);

    $workbook = new Spreadsheet();
    $myxls = $workbook->setActiveSheetIndex(0);

    $numberofcolumns = count($table->head);

    // Header row.
    foreach ($table->head as $key => $heading) {
        $cell = Coordinate::stringFromColumnIndex($key + 1) . '1'; // A1 cell address.
        $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br(get_string($heading, 'report_ghs'))))));
    }

    // Data rows.
    $rs = $DB->get_recordset_sql($sql, $params);
    $rownum = 2;
    foreach ($rs as $tablerow) {
        $row = array();
        $columnum = 1;
        foreach ($table->head as $column) {
            $data = \report_ghs\helper::english_proficiency_data($tablerow, $column, $counter, $pageparams, true);
            $cell = Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.
            if (preg_match("/^[fh]tt?p:\/\//", $data)) {
                $myxls->setCellValue($cell, 'Link');
                $myxls->getCell($cell)->getHyperlink()->setUrl($data);
            } else {
                $myxls->setCellValue($cell, $data);
            }
            $columnum++;
        }
        $rownum++;
    }
    $rs->close();


    // Freeze header.
    $myxls->freezePane('A2');

    // Filter.
    $myxls->setAutoFilter(
        $myxls->calculateWorksheetDimension()
    );

    // Auto column width calculation.
    foreach (range('A', $myxls->getHighestDataColumn()) as $col) {
        $myxls->getColumnDimension($col)->setAutoSize(true);
    }

    // Header format.
    $styleArray = array(
        'font' => array(
            'bold' => true,
        ),
        'fill' => array(
            'fillType' => Fill::FILL_SOLID,
            'color' => array(
                'argb' => 'FFFFF000',
            ),
        ),
    );
    $myxls->getStyle('A1:'.Coordinate::stringFromColumnIndex($numberofcolumns).'1')->applyFromArray($styleArray);

    // Rename worksheet
    $myxls->setTitle('export');

    // Set active sheet index to the first sheet, so Excel opens this as the first sheet
    $workbook->setActiveSheetIndex(0);

    $objWriter = new Xlsx($workbook);

    if (ob_get_length()) {
        ob_end_clean();
    }
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
    header('Cache-Control: max-age=0');

    ob_end_clean();
    $objWriter->save('php://output');
    exit;
} else if ($action == 'csv') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);
    $table = new stdClass();

    $headers = $columns_export;

    foreach ($headers as $ckey => $column) {
        $headers[$ckey] = get_string($column, 'report_ghs');
    }

    if (ob_get_length()) {
        ob_end_clean();
    }
    // Output headers so that the file is downloaded rather than displayed.
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=ghs_enrollment_'.(date('Y-m-d')).'.csv');

    // Create a file pointer connected to the output stream.
    $outputcsv = fopen('php://output', 'w');

    // Output the column headings.
    fputcsv($outputcsv, $headers);

    $counter = 0;

    $rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        foreach ($columns_export as $column) {
            $row[] = \report_ghs\helper::english_proficiency_data($tablerow, $column, $counter, $pageparams, true);
        }
        fputcsv($outputcsv, $row);
    }
    $rs->close();
    exit;
} else {

    foreach ($columns as $column) {
        $string[$column] = get_string($column, 'report_ghs');
        if ($sort != $column) {
            $columnicon = "";
            $columndir = "ASC";
        } else {
            $columndir = $dir == "ASC" ? "DESC" : "ASC";
            $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
            $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
        }
        if (($column == 'rowcount') || ($column == 'action') || ($column == 'gradebooklink') || ($column == 'kicagradebooklink')) {
            $$column = $string[$column];
        } else {
            $sorturl = clone $fullpageurl;
            $sorturl->param('perpage', $perpage);
            $sorturl->param('sort', $column);
            $sorturl->param('dir', $columndir);
            $sorturl->param('search', $search);

            $$column = html_writer::link($sorturl->out(false), $string[$column]).' '.$columnicon;
        }
    }

    $table = new html_table();

    $table->head = array();
    $table->wrap = array();
    $table->attributes = ['class' => 'nedtable table'];

    foreach ($columns as $column) {
        $table->head[$column] = $$column;
        $table->wrap[$column] = '';
    }

    // Override cell wrap.
    $table->wrap['rowcount'] = 'nowrap';
    $table->wrap['action'] = 'nowrap';
    $table->wrap['student'] = 'nowrap';

    $tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

    $counter = ($page * $perpage);

    foreach ($tablerows as $tablerow) {
        $row = new html_table_row();

        foreach ($columns as $column) {
            $varname = 'cell'.$column;
            $$varname = new html_table_cell(\report_ghs\helper::english_proficiency_data($tablerow, $column, $counter, $pageparams));
        }
        if ($tablerow->status == $strinactive) {
            $row->attributes['class'] = 'greyed';
        }
        $row->cells = array();
        foreach ($columns as $column) {
            $varname = 'cell' . $column;
            $row->cells[$column] = $$varname;
        }
        $table->data[] = $row;

    }

    echo $OUTPUT->header();
    echo html_writer::start_div('page-content-wrapper');
    echo $OUTPUT->render(new reports_header());
    echo html_writer::tag('h1', $title, array('class' => 'page-title'));

    $handler = new \local_schoolmanager\school_handler();
    $schoolfilter =  $handler->get_control_form($schoolid, $fullpageurl, false, true);

    // Filter form.
    $searchformurl = new moodle_url('/report/ghs/ghs_english_proficiency.php');

    if (is_siteadmin()) {
        $languagefilterhtml =  html_writer::start_div('form-group').
        html_writer::label(get_string('language', 'report_ghs'), 'designation', true, ['class' => '']).
        html_writer::select(
            $designationoptions, 'designation', $designation, array('' => get_string('all', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).
        html_writer::end_div();
    }

    $searchform = html_writer::tag('form',
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sesskey',
            'value' => sesskey(),
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'perpage',
            'value' => $perpage,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sort',
            'value' => $sort,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'dir',
            'value' => $dir,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'schoolid',
            'value' => $schoolid,
        )).

        // First row.
        html_writer::start_div('form-inline').

        html_writer::start_div('form-group').
        html_writer::label(get_string('opt', 'report_ghs'), 'filter', true, ['class' => '']).
        html_writer::select(
            $optoptions, 'opt', $optfilter, array('' => get_string('all', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).
        html_writer::end_div().

        html_writer::start_div('form-group').
        html_writer::label(get_string('students', 'report_ghs'), 'filter', true, ['class' => '']).
        html_writer::select(
            $filteroptions, 'filter', $filter, array('' => get_string('allstudents', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).
        html_writer::end_div().

        $languagefilterhtml.

        html_writer::start_div('form-group').
        html_writer::label(get_string('coursestatus', 'report_ghs'), 'completion', true, ['class' => '']).
        html_writer::select(
            $coursecompletionoptions, 'completion', $completion, array('' => get_string('all', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).
        html_writer::end_div().

        html_writer::start_div('form-group').
        html_writer::label(get_string('ossltyear', 'report_ghs'), 'ossltyear', true, ['class' => '']).
        html_writer::select(
            $ossltyearoptions, 'ossltyear', $ossltyearfilter, array('' => get_string('all', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).
        html_writer::end_div().

        html_writer::start_div('form-group').
        html_writer::label(get_string('status', 'report_ghs'), 'status', true, ['class' => '']).
        html_writer::select(
            $statusoptions, 'status', $statusfilter, array('' => get_string('all', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).
        html_writer::end_div().

        html_writer::empty_tag('input', array(
            'type' => 'submit',
            'value' => get_string('filter', 'report_ghs'),
            'class' => 'btn btn-primary form-control mb-2 mr-sm-2',
        )).

        html_writer::link($searchformurl,
            get_string('reset', 'report_ghs'), array('class' => 'btn btn-secondary form-control mb-2 mr-sm-2 filter-reset')
        ).

        html_writer::end_div(),
        array(
            'action' => $searchformurl->out(),
            'method' => 'post',
            'class' => '_form-inline mb-2 mt-2',
            'autocomplete' => 'off'
        )
    );

    echo html_writer::start_div('d-flex flex-row');
    echo html_writer::start_div('p-2');
    echo $schoolfilter;
    echo html_writer::end_div();
    echo html_writer::start_div('d-flex align-items-end');
    echo html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));
    echo html_writer::end_div();
    echo html_writer::end_div();


    $pagingurl = clone $fullpageurl;
    $pagingurl->remove_params('page');
    $pagingurl->remove_params('action');

    $pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

    // Export link.
    $exporexcelurl = $thispageurl;
    $exporexcelurl->remove_all_params();
    $exporexcelurl->param('schoolid', $schoolid);
    $exporexcelurl->param('filter', $filter);
    $exporexcelurl->param('designation', $designation);
    $exporexcelurl->param('completion', $completion);
    $exporexcelurl->param('opt', $optfilter);
    $exporexcelurl->param('action', 'excel');
    $exportexceliconurl = $OUTPUT->image_url('f/spreadsheet');
    $exportexcelicon = html_writer::img($exportexceliconurl, '', array('width' => '16', 'height' => '16'));
    $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
    $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

    $exportexturl = $exporexcelurl;
    $exportexturl->param('action', 'csv');
    $exporttexticonurl = $OUTPUT->image_url('f/text');
    $exporttexticon = html_writer::img($exporttexticonurl, '', array('width' => '16', 'height' => '16'));
    $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
    $exporttextlink = html_writer::link($exportexturl, 'CSV');

    $importfile = '';
    if (is_siteadmin()) {
        $importfile = html_writer::link(
            new moodle_url('/report/ghs/ghs_english_proficiency_import.php'),
            get_string('importfile', 'report_ghs'),
            ['class' => 'badge badge-info ml-2']
        );
    }

    $exportbuttons =  html_writer::div(
        get_string('export', 'report_ghs').' :&nbsp;&nbsp;'.
        $exportexceliconlink.'&nbsp;'.$exportexcellink.'&nbsp;|&nbsp; ' .
        $exporttexticonlink.'&nbsp;'.$exporttextlink.
        $importfile,
        'export-link-wrapper',
        array('style' => 'text-align:center;')
    );

    if ($outputpagingbar = $OUTPUT->render($pagingbar)) {
        echo $outputpagingbar;
    } else {
        echo html_writer::tag('div', '', ['class' => 'dummy-pagination']);
    }
    echo html_writer::div($exportbuttons, 'export-link-wrapper-top');

    echo html_writer::start_div('row');
    echo html_writer::start_div('col-md-9');
    echo html_writer::start_div('table-responsive');
    echo $OUTPUT->render_from_template('report_ghs/osslt_performance_requirement', []);
    echo html_writer::table($table);
    echo html_writer::end_div();
    echo html_writer::end_div();
    echo html_writer::start_div('col-md-3 summary');
    $summary = new \report_ghs\output\ghs_english_proficiency_summary($totalcount, $sql, $params);
    echo $OUTPUT->render($summary);
    echo html_writer::end_div();
    echo html_writer::end_div();

    echo $outputpagingbar;
    echo $exportbuttons;
    echo html_writer::end_div(); // Main wrapper.

    echo $OUTPUT->footer();
}