<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * RADIX
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 20, PARAM_INT);
$sort      = optional_param('sort', 'course', PARAM_ALPHANUM);
$dir       = optional_param('dir', 'ASC', PARAM_ALPHA);
// Action.
$action    = optional_param('action', false, PARAM_ALPHA);
$search    = optional_param('search', '', PARAM_TEXT);

$cfg = get_config('report_ghs');

require_login(null, false);
$contextsystem = context_system::instance();

// Role
$isot = report_ghs\helper::is_ot($USER->id);

// Permission.
if (!$isot) {
    if (!has_capability('report/ghs:viewgrades', $contextsystem) && !\report_ghs\helper::has_capability_in_any_course('report/ghs:viewgrades')) {
        throw new required_capability_exception($context, $capability, 'nopermissions', '');
    }
}

$thispageurl = new moodle_url('/report/ghs/ghs_grades.php');
$report = basename($thispageurl->out(), '.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('report');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$name = get_string('ghsgrades', 'report_ghs');
$title = get_string('ghsgrades', 'report_ghs');
$heading = $SITE->fullname;

// Breadcrumb.
if (has_capability('report/ghs:viewgrades', $contextsystem)) {
    admin_externalpage_setup('reportghsgrades', null, null);
}

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$datacolumns = array(
    'id' => 'r.id',
    'firstname' => 'u.firstname',
    'lastname' => 'u.lastname',
    'school' => 'coh.idnumber',
    'course' => 'cor.shortname',
    'category' => 'cat.name',
    'groupname' => 'g.name',
    'graderid' => 'r.graderid',
    'grader' => 'u2.firstname',
    'enrolldate' => 'r.enrolldate',
    'startdate' => 'g.startdate',
    'enddate' => 'g.enddate',
    'rawgrade' => 'r.coursegrade',
    'kicaavg' => 'r.kicaavg',
    'knowledge' => 'r.knowledge',
    'inquiry' => 'r.inquiry',
    'communication' => 'r.communication',
    'application' => 'r.application',
    'kica70' => 'r.kica70',
    'kica30' => 'r.kica30',
    'kicadiff' => 'r.kica30',
    'frate' => 'r.frate',
    'srate' => 'r.srate',
    'gradingerror' => 'r.gradingerror',
);

$params = [];

// Filter.
$where = '';
if ($isot) {
    $where .= " AND ".$datacolumns['graderid']." = :graderid";
    $params['graderid'] = $USER->id;
}

// Sort.
$order = '';
if ($action == 'excel') {
    $sort = 'kicaavg';
}
if ($sort) {
    if ($sort == 'kicalink') {
        $order = " ORDER BY {$datacolumns['gradingerror']} $dir";
    } else {
        $order = " ORDER BY $datacolumns[$sort] $dir";
    }
}

$pageparams = array();

// Filter by capabilies.
\report_ghs\helper::report_filter($where, $params, $report);

// Count records for paging.
$countsql = "SELECT COUNT(1)
               FROM {report_ghs_grades} r
         INNER JOIN {user} u ON r.userid = u.id
    LEFT OUTER JOIN {cohort} coh ON r.chortid = coh.id
    LEFT OUTER JOIN {course} cor ON r.courseid = cor.id
    LEFT OUTER JOIN {course_categories} cat ON cor.category = cat.id
    LEFT OUTER JOIN {groups} g ON r.groupid = g.id 
    LEFT OUTER JOIN {user} u2 ON r.graderid = u2.id
              WHERE 0 = 0 $where";
$totalcount = $DB->count_records_sql($countsql, $params);

// Table columns.
$columns = array(
    'rowcount',
    'firstname',
    'lastname',
    'school',
    'course',
    'category',
    'groupname',
    'grader',
    'startdate',
    'enddate',
    'active',
    'rawgrade',
    'kicaavg',
    'kica70',
    'kica30',
    'kicadiff',
    'frate',
    'srate',
    'kicalinkshort',
);
$columnsexport = array(
    'rowcount',
    'firstname',
    'lastname',
    'school',
    'course',
    'category',
    'groupname',
    'grader',
    'startdate',
    'enddate',
    'active',
    'rawgrade',
    'kicaavg',
    'knowledge',
    'inquiry',
    'communication',
    'application',
    'kica70',
    'kica30',
    'kicadiff',
    'frate',
    'srate',
    'kicalinkshort',
);

$sql = "SELECT  r.*, r.coursegrade rawgrade,
                u.firstname,
                u.lastname,
                coh.idnumber  school,
                cor.shortname  course,
                cat.name  category,
                g.name  groupname,
                g.startdate,
                g.enddate,
                u2.firstname grader
           FROM {report_ghs_grades} r
     INNER JOIN {user} u ON r.userid = u.id
LEFT OUTER JOIN {cohort} coh ON r.chortid = coh.id
LEFT OUTER JOIN {course} cor ON r.courseid = cor.id
LEFT OUTER JOIN {course_categories} cat ON cor.category = cat.id
LEFT OUTER JOIN {groups} g ON r.groupid = g.id
LEFT OUTER JOIN {user} u2 ON r.graderid = u2.id
          WHERE 0 = 0
                $where
                $order";

if ($action == 'excel') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);

    $table = new stdClass();
    $table->head = $columnsexport;

    // Delete first rowcount column.
    $itemid = array_shift($table->head);
    // Delete last action column.
    //array_pop($table->head);

    $counter = 0;
    $filename = $report.'_'.(date('Y-m-d'));
    $downloadfilename = clean_filename($filename);

    $gradecolumns = array('coursegrade', 'kicaavg', 'kica70', 'kica30');

    $workbook = new Spreadsheet();
    $myxls = $workbook->setActiveSheetIndex(0);

    $numberofcolumns = count($table->head);

    // Header row.
    foreach ($table->head as $key => $heading) {
        $cell = Coordinate::stringFromColumnIndex($key + 1) . '1'; // A1 cell address.
        $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br(get_string($heading, 'report_ghs'))))));
    }

    // Data rows.
    $rs = $DB->get_recordset_sql($sql, $params);
    $rownum = 2;
    foreach ($rs as $tablerow) {
        $row = array();
        $columnum = 1;
        $col = [];
        foreach ($table->head as $column) {
            $data = \report_ghs\helper::kica_data($tablerow, $column, $counter, $pageparams, true);
            $col[$column] = ['index' => $columnum, 'value' => $data];

            $lowgrade  = false;
            if (in_array($column, $gradecolumns) && is_numeric($data) && $data < 50)  {
                $lowgrade  = true;
            }

            $cell = Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.
            if (preg_match("/^[fh]tt?p:\/\//", $data)) {
                $myxls->setCellValue($cell, 'Link');
                $myxls->getCell($cell)->getHyperlink()->setUrl($data);
            } else {
                $myxls->setCellValue($cell, $data);
            }

            if ($lowgrade) {
                $myxls->getStyle($cell)->getFill()->setFillType(Fill::FILL_SOLID);
                $myxls->getStyle($cell)->getFill()->getStartColor()->setARGB('FFFCC7CE');
                $myxls->getStyle($cell)->getFont()->getColor()->setARGB('FF9C0006');
            }

            $columnum++;
        }

        if (is_numeric($col['coursegrade']['value']) && is_numeric($col['kica70']['value'])) {
            if ($col['coursegrade']['value'] > 50 && $col['kica70']['value'] < 50) {
                $cell = Coordinate::stringFromColumnIndex($col['kica70']['index']) . $rownum;
                $myxls->getStyle($cell)->getFill()->setFillType(Fill::FILL_SOLID);
                $myxls->getStyle($cell)->getFill()->getStartColor()->setARGB('FFC0514D');
                $myxls->getStyle($cell)->getFont()->getColor()->setARGB('FFFFFFFF');
            }
        }

        $rownum++;
    }
    $rs->close();

    // Freeze header.
    $myxls->freezePane('A2');

    // Filter.
    $myxls->setAutoFilter(
        $myxls->calculateWorksheetDimension()
    );

    // Auto column width calculation.
    foreach (range('A', $myxls->getHighestDataColumn()) as $col) {
        $myxls->getColumnDimension($col)->setAutoSize(true);
    }

    // Header format.
    $styleArray = array(
        'font' => array(
            'bold' => true,
        ),
        'fill' => array(
            'fillType' => Fill::FILL_SOLID,
            'color' => array(
                'argb' => 'FFFFF000',
            ),
        ),
    );
    $myxls->getStyle('A1:'.Coordinate::stringFromColumnIndex($numberofcolumns).'1')->applyFromArray($styleArray);

    // Rename worksheet
    $myxls->setTitle('export');

    // Set active sheet index to the first sheet, so Excel opens this as the first sheet
    $workbook->setActiveSheetIndex(0);

    $objWriter = new Xlsx($workbook);

    if (ob_get_length()) {
        ob_end_clean();
    }
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
    header('Cache-Control: max-age=0');

    ob_end_clean();
    $objWriter->save('php://output');
    exit;
} else if ($action == 'csv') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);
    $table = new stdClass();

    // Delete firs rowcount column.
    array_shift($columnsexport);
    // Delete last action column.
    //array_pop($columnsexport);

    $headers = $columnsexport;

    foreach ($headers as $ckey => $column) {
        $headers[$ckey] = get_string($column, 'report_ghs');
    }

    if (ob_get_length()) {
        ob_end_clean();
    }
    // Output headers so that the file is downloaded rather than displayed.
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=ghs_grades_'.(date('Y-m-d')).'.csv');

    // Create a file pointer connected to the output stream.
    $outputcsv = fopen('php://output', 'w');

    // Output the column headings.
    fputcsv($outputcsv, $headers);

    $counter = 0;

    $rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        foreach ($columnsexport as $column) {
            $row[] = \report_ghs\helper::kica_data($tablerow, $column, $counter, $pageparams, true);
        }
        fputcsv($outputcsv, $row);
    }
    $rs->close();
    exit;
} else {

    foreach ($columns as $column) {
        $string[$column] = get_string($column, 'report_ghs');
        if ($sort != $column) {
            $columnicon = "";
            $columndir = "ASC";
        } else {
            $columndir = $dir == "ASC" ? "DESC" : "ASC";
            $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
            $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
        }
        if (($column == 'rowcount') || ($column == 'action') || ($column == 'kicadiff') || ($column == 'active')) {
            $$column = $string[$column];
        } else {
            $sorturl = $thispageurl;
            $sorturl->param('perpage', $perpage);
            $sorturl->param('sort', $column);
            $sorturl->param('dir', $columndir);
            $sorturl->param('search', $search);

            $$column = html_writer::link($sorturl->out(false), $string[$column]).' '.$columnicon;
        }
    }

    $table = new html_table();

    $table->head = array();
    $table->wrap = array();
    $table->attributes = ['class' => 'nedtable fullwidth'];

    foreach ($columns as $column) {
        $table->head[$column] = $$column;
        $table->wrap[$column] = '';
    }

    // Override cell wrap.
    $table->wrap['action'] = 'nowrap';

    $tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

    $counter = ($page * $perpage);

    foreach ($tablerows as $tablerow) {
        $row = new html_table_row();

        foreach ($columns as $column) {
            $varname = 'cell'.$column;
            $$varname = new html_table_cell(\report_ghs\helper::kica_data($tablerow, $column, $counter, $pageparams));
        }

        if (!empty($tablerow->rowcls)) {
            $row->attributes = ['class' => $tablerow->rowcls];
        }

        $row->cells = array();
        foreach ($columns as $column) {
            $varname = 'cell' . $column;
            $row->cells[$column] = $$varname;
        }
        $table->data[] = $row;

    }

    echo $OUTPUT->header();
    echo html_writer::start_div('page-content-wrapper', array('id' => 'page-content'));
    echo html_writer::tag('h1', $title, array('class' => 'page-title'));

    $reportcls = '\report_ghs\task\adhoc_'.$report.'_report';

    if (class_exists($reportcls)) {
        if ($task = $reportcls::get_task()) {
            echo html_writer::tag('div', get_string('taskwarning', 'report_ghs', userdate($task->customdata->timecreated)), array('class' => 'alert alert-info'));
        } else {
            $a = new stdClass();
            $a->lastupdate = userdate($cfg->kicalastupdate);
            $a->updateurl = (new moodle_url('/report/ghs/update_report.php', ['report' => $report]))->out();
            echo html_writer::tag('div', get_string('lastupdatewarning', 'report_ghs', $a), array('class' => 'alert alert-info'));
        }
    }

    \report_ghs\helper::print_export_task_info($report, $cfg);

    $pagingurl = new moodle_url('/report/ghs/ghs_grades.php',
        array(
            'perpage' => $perpage,
            'sort' => $sort,
            'dir' => $dir,
            'search' => $search
        )
    );

    $pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

    // Export link.
    $exporexcelurl = $thispageurl;
    $exporexcelurl->remove_all_params();
    $exporexcelurl->param('action', 'excel');
    $exportexceliconurl = $OUTPUT->image_url('f/spreadsheet');
    $exportexcelicon = html_writer::img($exportexceliconurl, '', array('width' => '16', 'height' => '16'));
    $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
    $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

    $exportexturl = $thispageurl;
    $exportexturl->remove_all_params();
    $exportexturl->param('action', 'csv');
    $exporttexticonurl = $OUTPUT->image_url('f/text');
    $exporttexticon = html_writer::img($exporttexticonurl, '', array('width' => '16', 'height' => '16'));
    $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
    $exporttextlink = html_writer::link($exportexturl, 'CSV');

    $exportbuttons =  html_writer::div(
        get_string('export', 'report_ghs').' :&nbsp;&nbsp;'.
        $exportexceliconlink.'&nbsp;'.$exportexcellink.'&nbsp;|&nbsp; '
        .$exporttexticonlink.'&nbsp;'.$exporttextlink, 'export-link-wrapper',
        array('style' => 'text-align:center;')
    );

    if ($outputpagingbar = $OUTPUT->render($pagingbar)) {
        echo $outputpagingbar;
    } else {
        echo html_writer::tag('div', '', ['class' => 'dummy-pagination']);
    }
    echo html_writer::div($exportbuttons, 'export-link-wrapper-top');
    echo html_writer::table($table);
    echo $outputpagingbar;
    echo $exportbuttons;

    echo html_writer::end_div(); // Main wrapper.
    echo $OUTPUT->footer();
}