<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * RADIX
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use report_ghs\task\adhoc_ghs_grading_time_monitor_report;

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 40, PARAM_INT);
$sort      = optional_param('sort', 'grader_', PARAM_TEXT);
$dir       = optional_param('dir', 'ASC', PARAM_ALPHA);
// Action.
$action    = optional_param('action', false, PARAM_ALPHA);
$search    = optional_param('search', '', PARAM_TEXT);

// Filters.
$view = optional_param('view', 1, PARAM_INT);
$subject = optional_param('subject', 0, PARAM_INT);
$courseid = optional_param('courseid', 0, PARAM_INT);
$tagid = optional_param('tagid', 0, PARAM_INT);
$period = optional_param('period', 0, PARAM_INT);
$gtstatus = optional_param('gtstatus', 'FT', PARAM_TEXT);

$cfg = get_config('report_ghs');

require_login(null, false);
$contextsystem = context_system::instance();

// Permission.
require_capability('report/ghs:viewgradingtimemonitor', $contextsystem);

if ($view == 1) {
    $periodoptions = [
        '1' => get_string('twoweeks', 'report_ghs'),
        '2' => get_string('fourweeks', 'report_ghs'),
    ];
} else {
    $periodoptions = [
        '3' => get_string('lastmonth', 'report_ghs'),
        '4' => get_string('lastthreemonth', 'report_ghs'),
        '5' => get_string('entiremonth', 'report_ghs'),
    ];
}

if (!array_key_exists($period, $periodoptions)) {
    $keys = array_keys($periodoptions);
    $period = reset($keys);
}

$thispageurl = new moodle_url('/report/ghs/ghs_grading_time_monitor.php');
$fullpageurl = new moodle_url('/report/ghs/ghs_grading_time_monitor.php', [
    'page' => $page,
    'perpage' => $perpage,
    'sort' => $sort,
    'dir' => $dir,
    'action' => $action,
    'view' => $view,
    'subject' => $subject,
    'courseid' => $courseid,
    'tagid' => $tagid,
    'period' => $period,
    'gtstatus' => $gtstatus,
]);
$report = basename($thispageurl->out(), '.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('report');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$name = get_string('ghsgradingtimemonitor', 'report_ghs');
$title = get_string('ghsgradingtimemonitor', 'report_ghs');
$heading = $SITE->fullname;

// Breadcrumb.
if (has_capability('report/ghs:viewgrades', $contextsystem)) {
    admin_externalpage_setup('reportghsgradingtimemonitor', null, null);
}

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$datacolumns = array(
    'id' => 'r.id',
    'grader_' => "CONCAT(u.firstname, '', u.lastname)",
);

$params = [];

$pageparams = array();

// Table columns.
$columns = array(
    'grader_',
);
$weekheader['grader_'] = 1;

$headerstr = [];
$sqlpivot = [];

list($shooldays, $today) = adhoc_ghs_grading_time_monitor_report::get_school_days();

if ($view == 1) {
    if ($period == 1) {
        $starttime = usergetmidnight(strtotime('previous monday') - WEEKSECS);
        $endtime = usergetmidnight(strtotime('next monday') - 12 * HOURSECS);
    } else if ($period == 2) {
        $starttime = usergetmidnight(strtotime('previous monday') - 3 * WEEKSECS);
        $endtime = usergetmidnight(strtotime('next monday') - 12 * HOURSECS);
    }
    $counter = 0;
    $highlightted = count($columns);
    for ($i = $starttime; $i <= $endtime; $i += DAYSECS) {
        $counter++;
        if ($i == $today) {
            $highlightted += $counter;
        }
        $label = date('M j', $i);
        $key = 'day_' . $counter;
        if (isset($weekheader[$shooldays[$i]['week']])) {
            $weekheader[$shooldays[$i]['week']] += 1;
        } else {
            $weekheader[$shooldays[$i]['week']] = 1;
        }
        $columns[] = $key;
        $headerstr[$key] = $label;
        if ($today >=  $i) {
            $sqlpivot[] = "SUM(CASE WHEN r.timestart = $i THEN r.totaltime ELSE 0 END) AS '$key'";
        } else {
            $sqlpivot[] = "SUM(CASE WHEN r.timestart = $i THEN r.estimatedtime ELSE 0 END) AS '$key'";
        }
        $datacolumns[$key] = $key;
    }
} else {
    if ($period == 3) {
        $starttime = usergetmidnight(strtotime('previous monday') - 4 * WEEKSECS);
        $endtime = usergetmidnight(strtotime('next monday') + WEEKSECS - 12 * HOURSECS);
    } else if ($period == 4) {
        $starttime = usergetmidnight(strtotime('previous monday') - 13 * WEEKSECS);
        $endtime = usergetmidnight(strtotime('next monday') + WEEKSECS - 12 * HOURSECS);
    } else {
        $starttime = usergetmidnight(strtotime('2023-09-04'));
        $endtime = usergetmidnight(strtotime('2024-09-01'));
    }

    $week = 0;
    $day = 0;
    for ($i = $starttime; $i <= $endtime; $i += DAYSECS) {
        $day++;
        if (($day % 7) == 1) {
            $week = $shooldays[$i]['week'];
            $label = "W".$week;
            $key = 'week_' . $week;

            if (isset($weekheader[date('n', $i)])) {
                $weekheader[date('n', $i)]['colspan'] += 1;
                $weekheader[date('n', $i)]['enddate'] = $i;
            } else {
                $weekheader[date('n', $i)]['stardate'] = $i;
                $weekheader[date('n', $i)]['enddate'] = $i;
                $weekheader[date('n', $i)]['colspan'] = 1;
            }

            $columns[] = $key;
            $headerstr[$key] = $label;
            $sqlpivot[] = "SUM(CASE WHEN r.week = $week THEN r.totaltime ELSE 0 END) AS '$key'";
            $datacolumns[$key] = $key;
        }
    }
}
// Header strings.
$headerstring  = [];
foreach ($columns as $column) {
    if (strpos($column, 'day_') === 0 || strpos($column, 'week_') === 0) {
        $headerstring[$column] = $headerstr[$column];
    } else {
        $headerstring[$column] = get_string($column, 'report_ghs');
    }
}

$sqlpivot = implode(',', $sqlpivot);

// Filter.
$where = '';
$inwhere = '';

$subjectfield = $DB->get_record('customfield_field', ['shortname' => 'subject'], '*', MUST_EXIST);

if ($subject) {
    $sql = 'SELECT c.id, c.shortname
              FROM {customfield_data} cd
              JOIN {course} c ON cd.instanceid = c.id
             WHERE cd.fieldid = ? 
               AND cd.value = ?
          ORDER BY c.shortname';

    $courses = $DB->get_records_sql_menu($sql, [$subjectfield->id, $subject]);

    if (!$courseid) {
        list($insql, $params) = $DB->get_in_or_equal(array_keys($courses));
    }
} else {
    $sql = "SELECT DISTINCT c.id, c.shortname
              FROM {report_ghs_grading_time_moni} tm
              JOIN {course} c ON tm.courseid = c.id
          ORDER BY c.shortname";
    $courses = $DB->get_records_sql_menu($sql);
}

if ($courseid) {
    list($insql, $params) = $DB->get_in_or_equal($courseid);
    $inwhere .= " AND r.courseid = ?";
    $params[] = $courseid;
}

if ($tagid) {
    $inwhere .= " AND r.cmid IN (SELECT ti.itemid FROM {tag_instance} ti WHERE ti.component = 'core' 
    AND ti.itemtype = 'course_modules' AND ti.tagid = ?)";
    $params[] = $tagid;
}
if ($gtstatus) {
    $inwhere .= " AND r.graderid IN (SELECT d.userid FROM {user_info_data} d JOIN {user_info_field} f 
    ON d.fieldid = f.id WHERE f.shortname = 'gt_status' AND d.data = ?)";
    $params[] = $gtstatus;
}

// Sort.
$order = '';
if ($sort) {
    $order = " ORDER BY $datacolumns[$sort] $dir";
}

// Count records for paging.
$countsql = "SELECT COUNT(1)
            FROM (SELECT r.graderid, $sqlpivot     
                    FROM {report_ghs_grading_time_moni} r
                    WHERE 0 = 0
                    $inwhere
                    GROUP BY r.graderid) rep
              JOIN {user} u 
                ON rep.graderid = u.id
             WHERE 0 = 0
                   $where";
$totalcount = $DB->count_records_sql($countsql, $params);


$sql = "SELECT CONCAT(u.firstname, '', u.lastname) grader_,
               rep.* 
          FROM (SELECT r.graderid, $sqlpivot     
                FROM {report_ghs_grading_time_moni} r
                WHERE 0 = 0
                $inwhere
                GROUP BY r.graderid) rep
          JOIN {user} u 
            ON rep.graderid = u.id
         WHERE 0 = 0
               $where
               $order";

if ($action == 'excel') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);

    $table = new stdClass();
    $table->head = $columns;

    $counter = 0;
    $filename = $report.'_'.(date('Y-m-d'));
    $downloadfilename = clean_filename($filename);

    $workbook = new Spreadsheet();
    $myxls = $workbook->setActiveSheetIndex(0);

    $numberofcolumns = count($columns);


    // Header row.
    $c = 0;
    foreach ($table->head  as $column) {
        $c++;
        $cell = Coordinate::stringFromColumnIndex($c) . '1'; // A1 cell address.
        $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br($headerstring[$column])))));
    }

    // Data rows.
    $rs = $DB->get_recordset_sql($sql, $params);
    $rownum = 2;
    foreach ($rs as $tablerow) {
        $row = array();
        $columnum = 1;
        $col = [];
        foreach ($table->head as $column) {
            $data = \report_ghs\helper::grading_time_monitor_data($tablerow, $column, $counter, $pageparams, true);
            $col[$column] = ['index' => $columnum, 'value' => $data];

            $cell = Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.
            if (preg_match("/^[fh]tt?p:\/\//", $data)) {
                $myxls->setCellValue($cell, 'Link');
                $myxls->getCell($cell)->getHyperlink()->setUrl($data);
            } else {
                $myxls->setCellValue($cell, $data);
            }

            $columnum++;
        }

        $rownum++;
    }
    $rs->close();

    // Freeze header.
    $myxls->freezePane('A2');

    // Filter.
    $myxls->setAutoFilter(
        $myxls->calculateWorksheetDimension()
    );

    // Auto column width calculation.
    foreach (range('A', $myxls->getHighestDataColumn()) as $col) {
        $myxls->getColumnDimension($col)->setAutoSize(true);
    }

    // Header format.
    $styleArray = array(
        'font' => array(
            'bold' => true,
        ),
        'fill' => array(
            'fillType' => Fill::FILL_SOLID,
            'color' => array(
                'argb' => 'FFFFF000',
            ),
        ),
    );
    $myxls->getStyle('A1:'.Coordinate::stringFromColumnIndex($numberofcolumns).'1')->applyFromArray($styleArray);

    // Rename worksheet
    $myxls->setTitle('export');

    // Set active sheet index to the first sheet, so Excel opens this as the first sheet
    $workbook->setActiveSheetIndex(0);

    $objWriter = new Xlsx($workbook);

    if (ob_get_length()) {
        ob_end_clean();
    }
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
    header('Cache-Control: max-age=0');

    ob_end_clean();
    $objWriter->save('php://output');
    exit;
} else

    if ($action == 'csv') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);
    $table = new stdClass();

    if (ob_get_length()) {
        ob_end_clean();
    }
    // Output headers so that the file is downloaded rather than displayed.
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=ghs_grades_'.(date('Y-m-d')).'.csv');

    // Create a file pointer connected to the output stream.
    $outputcsv = fopen('php://output', 'w');

    // Output the column headings.
    fputcsv($outputcsv, $headerstring);

    $counter = 0;

    $rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        foreach ($columns as $column) {
            $row[] = \report_ghs\helper::grading_time_monitor_data($tablerow, $column, $counter, $pageparams, true);
        }
        fputcsv($outputcsv, $row);
    }
    $rs->close();
    exit;
} else {

    foreach ($columns as $column) {
        if ($sort != $column) {
            $columnicon = "";
            $columndir = "ASC";
        } else {
            $columndir = $dir == "ASC" ? "DESC" : "ASC";
            $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
            $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
        }
        if (($column == 'rowcount') || ($column == 'action') || ($column == 'kicadiff') || ($column == 'active')) {
            $$column = $headerstring[$column];
        } else {
            $sorturl = $thispageurl;
            $sorturl->param('perpage', $perpage);
            $sorturl->param('sort', $column);
            $sorturl->param('dir', $columndir);
            $sorturl->param('search', $search);

            $$column = html_writer::link($sorturl->out(false), $headerstring[$column]).' '.$columnicon;
        }
    }

    $table = html_writer::start_tag('table', array('class' => 'nedtable fullwidth'));

    $table .= html_writer::start_tag('tr');
    foreach ($weekheader as $wheader => $colspan) {
        if (!is_array($colspan)) {
            if (!is_numeric($wheader)) {
                $table .= html_writer::tag('td', '', ['class' => 'text-center bg-grey']);
            } else {
                $table .= html_writer::tag('td', get_string('week') . ' ' . $wheader, ['class' => 'text-center bg-grey', 'colspan' => $colspan]);
            }
        } else {
            if (!is_numeric($wheader)) {
                $table .= html_writer::tag('td', '', ['class' => 'text-center bg-grey']);
            } else {
                $theader = date('M j', $colspan['stardate']) . '-' . date('M j', $colspan['enddate'] + 6 * DAYSECS);
                $table .= html_writer::tag('td', $theader, ['class' => 'text-center bg-grey', 'colspan' => $colspan['colspan']]);
            }
        }
    }
    $table .= html_writer::end_tag('tr');


    $table .= html_writer::start_tag('tr');
    $c = 0;
    foreach ($columns as $column) {
        $c++;
        $cls = '';
        if (isset($highlightted) && $highlightted == $c) {
            $cls = 'bg-yellow';
        }
        $table .= html_writer::tag('td', $$column, ['class' => 'text-center bg-grey ' . $cls, 'nowrap' => 'nowrap']);
    }
    $table .= html_writer::end_tag('tr');


    $tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

    $counter = ($page * $perpage);

    foreach ($tablerows as $tablerow) {
        $row = new html_table_row();
        $table .= html_writer::start_tag('tr');
        $c = 0;
        foreach ($columns as $column) {
            $c++;
            $cls = [];
            $attributes = [];

            if (isset($highlightted) && $highlightted == $c) {
                $cls[] = 'bg-yellow';
            }

            $celldata = \report_ghs\helper::grading_time_monitor_data($tablerow, $column, $counter, $pageparams);

            if ($column !== 'grader_') {
                $cls[] = 'text-center';
            }
            if (isset($tablerow->{'cls_' .$column})) {
                $cls[] = $tablerow->{'cls_' .$column};
            }
            $attributes['class'] = implode(' ', $cls);
            $attributes['nowrap'] = 'nowrap';
            $table .= html_writer::tag('td', $celldata, $attributes);
        }
        $table .= html_writer::end_tag('tr');
    }

    $table .= html_writer::end_tag('table');

    echo $OUTPUT->header();
    echo html_writer::start_div('page-content-wrapper', array('id' => 'page-content'));
    echo html_writer::tag('h1', $title, array('class' => 'page-title'));

    $reportcls = '\report_ghs\task\adhoc_'.$report.'_report';

    if (class_exists($reportcls)) {
        if ($task = $reportcls::get_task()) {
            echo html_writer::tag('div', get_string('taskwarning', 'report_ghs', userdate($task->customdata->timecreated)), array('class' => 'alert alert-info'));
        } else {
            $a = new stdClass();
            $a->lastupdate = userdate($cfg->gradingtimemonitorlastupdate ?? 0);
            $a->updateurl = (new moodle_url('/report/ghs/update_report.php', ['report' => $report]))->out();
            echo html_writer::tag('div', get_string('lastupdatewarning', 'report_ghs', $a), array('class' => 'alert alert-info'));
        }
    }

    // Filter form.
    $searchformurl = clone $thispageurl;

    $viewoptions = [
        '1' => get_string('days', 'report_ghs'),
        '2' => get_string('weeks', 'report_ghs'),
    ];

    $gtstatusoptions = [
        'FT' => 'FT',
        'PT' => 'PT',
    ];

    $fc = new \customfield_select\field_controller($subjectfield->id);
    $subjectoptions = $fc->get_options();

    $sql = "SELECT DISTINCT t.id, t.rawname
            FROM {tag_instance} ti
            JOIN {tag} t ON ti.tagid = t.id
            WHERE ti.component = 'core'
            AND ti.itemtype = 'course_modules'
            GROUP BY t.rawname";
    $tagoptions = $DB->get_records_sql_menu($sql);

    $searchform = html_writer::tag('form',
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sesskey',
            'value' => sesskey(),
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'perpage',
            'value' => $perpage,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sort',
            'value' => $sort,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'dir',
            'value' => $dir,
        )).

        // First row.
        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').

        html_writer::label(get_string('periodview', 'report_ghs'), 'view', true, ['class' => '']).
        html_writer::select(
            $viewoptions, 'view', $view, false, ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('subject', 'report_ghs'), 'subject', true, ['class' => '']).
        html_writer::select(
            $subjectoptions, 'subject', $subject, array(0 => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('course', 'report_ghs'), 'courseid', true, ['class' => '']).
        html_writer::select(
            $courses, 'courseid', $courseid, array(0 => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('activitytags', 'report_ghs'), 'tagid', true, ['class' => '']).
        html_writer::select(
            $tagoptions, 'tagid', $tagid, array(0 => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('period', 'report_ghs'), 'period', true, ['class' => '']).
        html_writer::select(
            $periodoptions, 'period', $period, null, ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('gtstatus', 'report_ghs'), 'gtstatus', true, ['class' => '']).
        html_writer::select(
            $gtstatusoptions, 'gtstatus', $gtstatus, null, ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::end_div().
        html_writer::end_div().

        // Second row.
        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').


        html_writer::empty_tag('input', array(
            'type' => 'submit',
            'value' => get_string('filter', 'report_ghs'),
            'class' => 'btn btn-primary form-control mb-2 mr-sm-2',
        )).

        html_writer::link($searchformurl,
            get_string('reset', 'report_ghs'), array('class' => 'btn btn-secondary form-control mb-2 mr-sm-2 filter-reset')
        ).

        html_writer::end_div().
        html_writer::end_div(),
        array(
            'action' => $searchformurl->out(),
            'method' => 'post',
            'class' => '_form-inline mb-2 mt-2',
            'autocomplete' => 'off'
        )
    );

    echo html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));

    $pagingurl = new moodle_url('/report/ghs/ghs_grading_time_monitor.php',
        array(
            'perpage' => $perpage,
            'sort' => $sort,
            'dir' => $dir,
            'search' => $search
        )
    );

    $pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

    // Export link.
    $exporexcelurl = $thispageurl;
    $exporexcelurl->remove_all_params();
    $exporexcelurl->param('action', 'excel');
    $exportexceliconurl = $OUTPUT->image_url('f/spreadsheet');
    $exportexcelicon = html_writer::img($exportexceliconurl, '', array('width' => '16', 'height' => '16'));
    $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
    $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

    $exportexturl = $thispageurl;
    $exportexturl->remove_all_params();
    $exportexturl->param('action', 'csv');
    $exporttexticonurl = $OUTPUT->image_url('f/text');
    $exporttexticon = html_writer::img($exporttexticonurl, '', array('width' => '16', 'height' => '16'));
    $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
    $exporttextlink = html_writer::link($exportexturl, 'CSV');

    $exportbuttons =  html_writer::div(
        get_string('export', 'report_ghs').' :&nbsp;&nbsp;'.
        $exportexceliconlink.'&nbsp;'.$exportexcellink.'&nbsp;|&nbsp; '
        .$exporttexticonlink.'&nbsp;'.$exporttextlink, 'export-link-wrapper',
        array('style' => 'text-align:center;')
    );

    if ($outputpagingbar = $OUTPUT->render($pagingbar)) {
        echo $outputpagingbar;
    } else {
        echo html_writer::tag('div', '', ['class' => 'dummy-pagination']);
    }
    echo html_writer::div($exportbuttons, 'export-link-wrapper-top');
    echo html_writer::start_div('table-responsive');
    echo $table;
    echo html_writer::end_div();
    echo $outputpagingbar;
    echo $exportbuttons;

    echo html_writer::end_div(); // Main wrapper.
    echo $OUTPUT->footer();
}