<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * RADIX
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 200, PARAM_INT);
$sort      = optional_param('sort', 'numberofdaysmoodle', PARAM_ALPHANUM);
$dir       = optional_param('dir', 'DESC', PARAM_ALPHA);
// Action.
$action    = optional_param('action', false, PARAM_ALPHA);

// Filters.
$filterbasecategory = optional_param('filterbasecategory', '', PARAM_TEXT);
$filtercategory = optional_param('filtercategory', '', PARAM_INT);
$filteractivitytag = optional_param('filteractivitytag', '', PARAM_TEXT);
$filteruserstatus = optional_param('filteruserstatus', 'Active', PARAM_TEXT);
$filterschool = optional_param('filterschool', 0, PARAM_INT);
$filtergrader = optional_param('filtergrader', 0, PARAM_INT);
$filterct = optional_param('filterct', 0, PARAM_INT);
$filtergraded = optional_param('filtergraded', 0, PARAM_INT);
$filtercoursetype = optional_param('filtercoursetype', -1, PARAM_INT);
$filternumberofdays = optional_param('filternumberofdays', 1, PARAM_INT);

$cfg = get_config('report_ghs');

require_login(null, false);
$contextsystem = context_system::instance();

// Role
$isot = report_ghs\helper::is_ot($USER->id);

// Permission.
if (!$isot) {
    if (!has_capability('report/ghs:viewsla', $contextsystem) && !\report_ghs\helper::has_capability_in_any_course('report/ghs:viewsla')) {
        throw new required_capability_exception($context, $capability, 'nopermissions', '');
    }
}

$thispageurl = new moodle_url('/report/ghs/ghs_sla.php');
$fullpageurl = new moodle_url('/report/ghs/ghs_sla.php', [
    'page' => $page,
    'perpage' => $perpage,
    'sort' => $sort,
    'dir' => $dir,
    'action' => $action,
    'filteractivitytag' => $filteractivitytag,
    'filterbasecategory' => $filterbasecategory,
    'filtercategory' => $filtercategory,
    'filteruserstatus' => $filteruserstatus,
    'filterschool' => $filterschool,
    'filtergrader' => $filtergrader,
    'filterct' => $filterct,
    'filtergraded' => $filtergraded,
    'filtercoursetype' => $filtercoursetype,
    'filternumberofdays' => $filternumberofdays,
]);
$report = basename($thispageurl->out(), '.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('report');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$name = get_string('ghsactivity', 'report_ghs');
$title = get_string('ghsactivity', 'report_ghs');
$heading = $SITE->fullname;

// Breadcrumb.
if (has_capability('report/ghs:viewsla', $contextsystem)) {
    admin_externalpage_setup('reportghsactivity', null, null);
}

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$datacolumns = array(
    'id' => 'a.id',
    'firstname' => 'u.firstname',
    'lastname' => 'u.lastname',
    'course' => 'c.shortname',
    'category' => 'cc.name',
    'categoryid' => 'cc.id',
    'basecategory' => 'cc.parent',
    'activityname' => 'r.activityname',
    'graderid' => 'r.graderid',
    'ot' => 'u2.firstname',
    'ct' => "if(r.ctid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.ctid))",
    'ctid' => "r.ctid",
    'numberofdaysmoodle' => 'r.numberofdaysmoodle',
    'numberofdayskica' => 'r.numberofdayskica',
    'userstatus' => 'r.userstatus',
    'activitytag' => 'r.activitytag',
    'duedate' => 'r.duedate',
    'excluded' => 'gg.excluded',
    'timesubmission' => 'r.timesubmission',
    'timegradedmoodle' => 'r.timegradedmoodle',
    'timegradedkica' => 'r.timegradedkica',
    'kicaenabled' => 'r.kicaenabled',
    'school' => 'coh.name',
    'chortid' => 'r.chortid',
    'slaot' => 'r.finalactivity',
    'finalactivity' => 'r.finalactivity',
    'gradebooklink' => 'r.timegradedmoodle',
    'instancegrade' => 'r.instancegrade',
    'submissionstatus' => 'r.submissionstatus',
    'coursetype' => "(SELECT Count(0) FROM {local_kica} k WHERE k.courseid = c.id AND k.enabled = 1)",
);

$params = [];

// Filter.
$where = '';
if ($filterbasecategory) {
    $where .= " AND ".$datacolumns['basecategory']." = :basecategory";
    $params['basecategory'] = $filterbasecategory;
}

if ($filtercategory) {
    $where .= " AND ".$datacolumns['categoryid']." = :category";
    $params['category'] = $filtercategory;
}

if ($filteractivitytag) {
    $where .= " AND ".$datacolumns['activitytag']." = :activitytag";
    $params['activitytag'] = $filteractivitytag;
}

if ($filteruserstatus) {
    $where .= " AND ".$datacolumns['userstatus']." = :userstatus";
    $params['userstatus'] = $filteruserstatus;
}

if ($filterschool) {
    $where .= " AND ".$datacolumns['chortid']." = :school";
    $params['school'] = $filterschool;
}

if ($filtergrader) {
    $where .= " AND ".$datacolumns['graderid']." = :grader";
    $params['grader'] = $filtergrader;
}

if ($filterct) {
    $where .= " AND ".$datacolumns['ctid']." = :ct";
    $params['ct'] = $filterct;
}

if ($filtergraded > -1) { // kicaenabled
    if ($filtergraded == 2) { // Excluded
        $where .= " AND " . $datacolumns['excluded'] . " > 0";
    } else if ($filtergraded == 1) { // Graded
        $where .= " AND " . $datacolumns['timegradedmoodle'] . " > 0";
        $where .= " AND " . $datacolumns['excluded'] . " = 0";
    } else {
        $where .= " AND " . $datacolumns['excluded'] . " = 0";
        $where .= " AND (" . $datacolumns['timegradedmoodle'] . " = 0";
        $where .= " OR (" . $datacolumns['timegradedkica'] . " = 0";
        $where .= " AND " . $datacolumns['kicaenabled'] . " = 1))";
    }
}

if ($filtercoursetype > -1) {
    if ($filtercoursetype) {
        $where .= " AND " . $datacolumns['coursetype'] . " > 0";
    } else {
        $where .= " AND " . $datacolumns['coursetype'] . " = 0";
    }
}
if ($filternumberofdays) {
    $where .= " AND (".$datacolumns['numberofdaysmoodle']." >= :numberofdaysmoodle".
        " OR ".$datacolumns['numberofdayskica']." >= :numberofdayskica)";
    $params['numberofdaysmoodle'] = $filternumberofdays;
    $params['numberofdayskica'] = $filternumberofdays;
}

$otfilter = '';
if ($isot) {
    $otfilter .= " AND ".$datacolumns['graderid']." = :graderid";
    $params['graderid'] = $USER->id;
}

// Sort.
$order = '';
if ($sort) {
    if ($sort == 'gradebooklink') {
        $order = " ORDER BY CASE WHEN $datacolumns[$sort] > 0 then 1 else 0 END $dir, ".$datacolumns["numberofdaysmoodle"]." DESC";
    } else if ($sort == 'activitygradelink') {
        $order = " ORDER BY CASE WHEN ".$datacolumns["instancegrade"]." > 0 then 1 else 0 END $dir, ".$datacolumns["numberofdaysmoodle"]." DESC";
    } else if ($sort == 'kicagradebooklink') {
        $order = " ORDER BY CASE 
        WHEN ".$datacolumns["timegradedkica"]." > 0 THEN 1 
        WHEN (".$datacolumns["kicaenabled"]." = 1 AND ".$datacolumns["timegradedkica"]." = 0) THEN 0 
        ELSE -1 END $dir, ".$datacolumns["numberofdayskica"]." DESC";
    } else {
        $order = " ORDER BY $datacolumns[$sort] $dir";
    }
}

$pageparams = array();

// Filter by capabilies.
\report_ghs\helper::report_filter($where, $params, $report);

// Count records for paging.
$countsql = "SELECT COUNT(1) 
           FROM {report_ghs_sla} r
     INNER JOIN {user} u ON r.userid = u.id
     INNER JOIN {course} c ON r.courseid = c.id
     INNER JOIN {course_categories} cc ON c.category = cc.id
     INNER JOIN {grade_grades} gg ON r.gradeid = gg.id
LEFT OUTER JOIN {cohort} coh ON r.chortid = coh.id
LEFT OUTER JOIN {user} u2 ON r.graderid = u2.id
WHERE 0 = 0 $otfilter $where";
$totalcount = $DB->count_records_sql($countsql, $params);

// Table columns.
$columns = array(
    'rowcount',
    'firstname',
    'lastname',
    'school',
    'course',
    'activityname',
    'ot',
    'ct',
    'duedate',
    'timesubmission',
    'timegradedmoodle',
    'timegradedkica',
    'numberofdaysmoodle',
    'numberofdayskica',
    'slaot',
    'finalactivity',
    'submissionstatus',
    'gradebooklink',
    'activitygradelink',
    'kicagradebooklink',
);

$columnsexport = array(
    'rowcount',
    'firstname',
    'lastname',
    'school',
    'userstatus',
    'course',
    'category',
    'basecategory',
    'activityname',
    'activitytag',
    'ot',
    'ct',
    'duedate',
    'timesubmission',
    'timegradedmoodle',
    'timegradedkica',
    'numberofdaysmoodle',
    'numberofdayskica',
    'slaot',
    'finalactivity',
    'submissionstatus',
    'gradebooklink',
    'activitygradelink',
    'kicagradebooklink',
);

$sql =  "SELECT r.*, u.firstname, u.lastname, c.shortname course, cc.name category, cc.parent basecategory, coh.name school, u2.firstname ot, 
                if(r.ctid = -1, '*multiple', (SELECT u3.firstname FROM {user} u3 WHERE u3.id = r.ctid)) ct,
                (SELECT COUNT(1) FROM {groups_members} gm JOIN {groups} g ON gm.groupid = g.id WHERE gm.userid = r.userid AND g.courseid = r.courseid) numofgroups
           FROM {report_ghs_sla} r
     INNER JOIN {user} u ON r.userid = u.id
     INNER JOIN {course} c ON r.courseid = c.id
     INNER JOIN {course_categories} cc ON c.category = cc.id
     INNER JOIN {grade_grades} gg ON r.gradeid = gg.id
LEFT OUTER JOIN {cohort} coh ON r.chortid = coh.id
LEFT OUTER JOIN {user} u2 ON r.graderid = u2.id
         WHERE 0 = 0
               $otfilter
               $where
               $order";

$sqlqroupcheck = "SELECT r.*,        
            (SELECT COUNT(1) FROM {groups_members} gm JOIN {groups} g ON gm.groupid = g.id WHERE gm.userid = r.userid AND g.courseid = r.courseid) numofgroups
           FROM {report_ghs_sla} r
     INNER JOIN {user} u ON r.userid = u.id
     INNER JOIN {course} c ON r.courseid = c.id
     INNER JOIN {course_categories} cc ON c.category = cc.id
     INNER JOIN {grade_grades} gg ON r.gradeid = gg.id
LEFT OUTER JOIN {cohort} coh ON r.chortid = coh.id
LEFT OUTER JOIN {user} u2 ON r.graderid = u2.id
         WHERE 0 = 0
               $otfilter
               $where
       HAVING numofgroups > 1";

if ($action == 'excel') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);

    $table = new stdClass();
    $table->head = $columnsexport;

    // Delete first rowcount column.
    $itemid = array_shift($table->head);
    // Delete last action column.
    //array_pop($table->head);

    $counter = 0;
    $filename = $report.'_'.(date('Y-m-d'));
    $downloadfilename = clean_filename($filename);

    $workbook = new Spreadsheet();
    $myxls = $workbook->setActiveSheetIndex(0);

    $numberofcolumns = count($table->head);

    // Header row.
    foreach ($table->head as $key => $heading) {
        $cell = Coordinate::stringFromColumnIndex($key + 1) . '1'; // A1 cell address.
        $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br(get_string($heading, 'report_ghs'))))));
    }

    // Data rows.
    $rs = $DB->get_recordset_sql($sql, $params);
    $rownum = 2;
    foreach ($rs as $tablerow) {
        $row = array();
        $columnum = 1;
        foreach ($table->head as $column) {
            $data = \report_ghs\helper::ghs_sla_data($tablerow, $column, $counter, $pageparams, true);
            $cell = Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.
            if (preg_match("/^[fh]tt?p:\/\//", $data)) {
                $myxls->setCellValue($cell, 'Link');
                $myxls->getCell($cell)->getHyperlink()->setUrl($data);
            } else {
                $myxls->setCellValue($cell, $data);
            }
            $columnum++;
        }
        $rownum++;
    }
    $rs->close();


    // Freeze header.
    $myxls->freezePane('A2');

    // Filter.
    $myxls->setAutoFilter(
        $myxls->calculateWorksheetDimension()
    );

    // Auto column width calculation.
    foreach (range('A', $myxls->getHighestDataColumn()) as $col) {
        $myxls->getColumnDimension($col)->setAutoSize(true);
    }

    // Header format.
    $styleArray = array(
        'font' => array(
            'bold' => true,
        ),
        'fill' => array(
            'fillType' => Fill::FILL_SOLID,
            'color' => array(
                'argb' => 'FFFFF000',
            ),
        ),
    );
    $myxls->getStyle('A1:'.Coordinate::stringFromColumnIndex($numberofcolumns).'1')->applyFromArray($styleArray);

    // Rename worksheet
    $myxls->setTitle('export');

    // Set active sheet index to the first sheet, so Excel opens this as the first sheet
    $workbook->setActiveSheetIndex(0);

    $objWriter = new Xlsx($workbook);

    if (ob_get_length()) {
        ob_end_clean();
    }
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
    header('Cache-Control: max-age=0');

    ob_end_clean();
    $objWriter->save('php://output');
    exit;
} else if ($action == 'csv') {
    ob_start();
    set_time_limit(300);
    raise_memory_limit(MEMORY_EXTRA);
    $table = new stdClass();

    // Delete firs rowcount column.
    array_shift($columnsexport);
    // Delete last action column.
    //array_pop($columnsexport);

    $headers = $columnsexport;

    foreach ($headers as $ckey => $column) {
        $headers[$ckey] = get_string($column, 'report_ghs');
    }

    if (ob_get_length()) {
        ob_end_clean();
    }
    // Output headers so that the file is downloaded rather than displayed.
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=ghs_sla_'.(date('Y-m-d')).'.csv');

    // Create a file pointer connected to the output stream.
    $outputcsv = fopen('php://output', 'w');

    // Output the column headings.
    fputcsv($outputcsv, $headers);

    $counter = 0;

    $rs = $DB->get_recordset_sql($sql, $params);
    foreach ($rs as $tablerow) {
        $row = array();
        foreach ($columnsexport as $column) {
            $row[] = \report_ghs\helper::ghs_sla_data($tablerow, $column, $counter, $pageparams, true);
        }
        fputcsv($outputcsv, $row);
    }
    $rs->close();
    exit;
} else {

    foreach ($columns as $column) {
        $string[$column] = get_string($column, 'report_ghs');
        if ($sort != $column) {
            $columnicon = "";
            $columndir = "ASC";
        } else {
            $columndir = $dir == "ASC" ? "DESC" : "ASC";
            $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
            $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
        }
        if (($column == 'rowcount') || ($column == 'action')) {
            $$column = $string[$column];
        } else {
            $sorturl = clone $fullpageurl;
            $sorturl->remove_params('page');
            $sorturl->param('sort', $column);
            $sorturl->param('dir', $columndir);
            $$column = html_writer::link($sorturl->out(false), $string[$column]).' '.$columnicon;
        }
    }

    $table = new html_table();

    $table->attributes = ['class' => 'generaltable table-sm'];
    $table->head = array();
    $table->wrap = array();
    $table->attributes = ['class' => 'nedtable fullwidth'];

    foreach ($columns as $column) {
        $table->head[$column] = $$column;
        $table->wrap[$column] = '';
    }

    // Override cell wrap.
    $table->wrap['action'] = 'nowrap';
    $table->wrap['gradebooklink'] = 'nowrap';

    $tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

    $counter = ($page * $perpage);

    foreach ($tablerows as $tablerow) {
        $row = new html_table_row();
        if ($tablerow->numofgroups > 1) {
            $row->attributes = ['class' => 'multiple-groups'];
        }

        foreach ($columns as $column) {
            $varname = 'cell'.$column;
            $$varname = new html_table_cell(\report_ghs\helper::ghs_sla_data($tablerow, $column, $counter, $pageparams));
        }

        $row->cells = array();
        foreach ($columns as $column) {
            $varname = 'cell' . $column;
            $row->cells[$column] = $$varname;
        }
        $table->data[] = $row;

    }

    echo $OUTPUT->header();
    echo html_writer::start_div('page-content-wrapper', array('id' => 'page-content'));
    echo html_writer::tag('h1', $title, array('class' => 'page-title'));

    $reportcls = '\report_ghs\task\adhoc_'.$report.'_report';

    if ($DB->record_exists_sql($sqlqroupcheck, $params)) {
        echo html_writer::tag('div', get_string('multiplegroupswarning', 'report_ghs'), array('class' => 'alert alert-danger'));
    }

    if (class_exists($reportcls)) {
        if ($task = $reportcls::get_task()) {
            echo html_writer::tag('div', get_string('taskwarning', 'report_ghs', userdate($task->customdata->timecreated)), array('class' => 'alert alert-info'));
        } else {
            $a = new stdClass();
            $a->lastupdate = userdate($cfg->activitylastupdate);
            $a->updateurl = (new moodle_url('/report/ghs/update_report.php', ['report' => $report]))->out();
            echo html_writer::tag('div', get_string('lastupdatewarning', 'report_ghs', $a), array('class' => 'alert alert-info'));
        }
    }

    \report_ghs\helper::print_export_task_info($report, $cfg);

    // Filter form.
    $searchformurl = new moodle_url('/report/ghs/ghs_sla.php');

    $basecategoryoptions = [
        '81' => '2019-20',
        '82' => '2018-19',
    ];

    $sql = "SELECT DISTINCT cc.id, cc.name
              FROM {report_ghs_sla} r
              JOIN {course} c ON r.courseid = c.id
              JOIN {course_categories} cc ON c.category = cc.id
             WHERE 0 = 0
                   $otfilter
          ORDER BY cc.name";
    $categoryoptions = $DB->get_records_sql_menu($sql, $params);

    $sql= "SELECT DISTINCT r.activitytag id, r.activitytag FROM {report_ghs_sla} r
            WHERE 0 = 0 $otfilter";
    $activitytagoptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT DISTINCT r.userstatus id, r.userstatus FROM {report_ghs_sla} r
             WHERE 0 = 0 $otfilter";
    $userstatusoptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT DISTINCT r.chortid, c.name
              FROM {report_ghs_sla} r
              JOIN {cohort} c ON r.chortid = c.id
             WHERE 0 = 0 $otfilter
          ORDER BY c.name ASC";

    $schooloptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT r.graderid, u.firstname
              FROM {report_ghs_sla} r
              JOIN {user} u ON r.graderid = u.id
             WHERE 0 = 0
                   $otfilter
          GROUP BY r.graderid
          ORDER BY u.firstname ASC";
    $graderoptions = $DB->get_records_sql_menu($sql, $params);

    $sql = "SELECT r.ctid, u.firstname
              FROM {report_ghs_sla} r
              JOIN {user} u ON r.ctid = u.id
             WHERE 0 = 0
                   $otfilter
          GROUP BY r.ctid
          ORDER BY u.firstname ASC";
    $ctoptions = $DB->get_records_sql_menu($sql, $params);

    $gradedoptions = ['1' => get_string('graded', 'report_ghs'), '0' => get_string('notgraded', 'report_ghs'), '2' => get_string('excluded', 'report_ghs')];

    $coursetypeoptions = [
        '1' => get_string('withkica', 'report_ghs'),
        '0' => get_string('withoutkica', 'report_ghs')
    ];

    $numberofdaysoptions = [
        '4' => '4+',
        '6' => '6+',
        '10' => '10+',
    ];

    $searchform = html_writer::tag('form',
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sesskey',
            'value' => sesskey(),
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'perpage',
            'value' => $perpage,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'sort',
            'value' => $sort,
        )).
        html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => 'dir',
            'value' => $dir,
        )).

        // First row.
        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').

        html_writer::label(get_string('coursetype', 'report_ghs'), 'filtercoursetype', true, ['class' => '_sr-only']).
        html_writer::select(
            $coursetypeoptions, 'filtercoursetype', $filtercoursetype, array('-1' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('basecategory', 'report_ghs'), 'filterbasecategory', true, ['class' => '_sr-only']).
        html_writer::select(
            $basecategoryoptions, 'filterbasecategory', $filterbasecategory, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('category', 'report_ghs'), 'filtercategory', true, ['class' => 'ml-3']).
        html_writer::select(
            $categoryoptions, 'filtercategory', $filtercategory, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('activitytag', 'report_ghs'), 'filteractivitytag', true, ['class' => 'ml-3']).
        html_writer::select(
            $activitytagoptions, 'filteractivitytag', $filteractivitytag, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('userstatus', 'report_ghs'), 'filteruserstatus', true, ['class' => 'ml-3']).
        html_writer::select(
            $userstatusoptions, 'filteruserstatus', $filteruserstatus, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::end_div().
        html_writer::end_div().

        // Second row.
        html_writer::start_div('form-inline').
        html_writer::start_div('form-group').

        html_writer::label(get_string('school', 'report_ghs'), 'filterschool', true, ['class' => '']).
        html_writer::select(
            $schooloptions, 'filterschool', $filterschool, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('ot', 'report_ghs'), 'filtergrader', true, ['class' => 'ml-3']).
        html_writer::select(
            $graderoptions, 'filtergrader', $filtergrader, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('ct', 'report_ghs'), 'filterct', true, ['class' => 'ml-3']).
        html_writer::select(
            $ctoptions, 'filterct', $filterct, array('' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('gradestatus', 'report_ghs'), 'filtergraded', true, ['class' => 'ml-3']).
        html_writer::select(
            $gradedoptions, 'filtergraded', $filtergraded, array('-1' => get_string('showall', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::label(get_string('numberofdays', 'report_ghs'), 'filternumberofday', true, ['class' => 'ml-3']).
        html_writer::select(
            $numberofdaysoptions, 'filternumberofdays', $filternumberofdays, array('1' => get_string('all', 'report_ghs')), ['class' => 'form-control mb-2 mr-sm-2']
        ).

        html_writer::empty_tag('input', array(
            'type' => 'submit',
            'value' => get_string('filter', 'report_ghs'),
            'class' => 'btn btn-primary form-control mb-2 mr-sm-2',
        )).

        html_writer::link($searchformurl,
            get_string('reset', 'report_ghs'), array('class' => 'btn btn-secondary form-control mb-2 mr-sm-2 filter-reset')
        ).

        html_writer::end_div().
        html_writer::end_div(),
        array(
            'action' => $searchformurl->out(),
            'method' => 'post',
            'class' => '_form-inline mb-2 mt-2',
            'autocomplete' => 'off'
        )
    );

    echo html_writer::div($searchform, 'search-form-wrapper', array('id' => 'search-form'));

    $pagingurl = clone $fullpageurl;
    $pagingurl->remove_params('page');
    $pagingurl->remove_params('action');

    $pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

    // Export link.
    $exporexcelurl = clone $fullpageurl;
    $exporexcelurl->param('action', 'excel');
    $exportexceliconurl = $OUTPUT->image_url('f/spreadsheet');
    $exportexcelicon = html_writer::img($exportexceliconurl, '', array('width' => '16', 'height' => '16'));
    $exportexceliconlink = html_writer::link($exporexcelurl, $exportexcelicon);
    $exportexcellink = html_writer::link($exporexcelurl, 'XLS');

    $exportexturl = clone $fullpageurl;
    $exportexturl->param('action', 'csv');
    $exporttexticonurl = $OUTPUT->image_url('f/text');
    $exporttexticon = html_writer::img($exporttexticonurl, '', array('width' => '16', 'height' => '16'));
    $exporttexticonlink = html_writer::link($exportexturl, $exporttexticon);
    $exporttextlink = html_writer::link($exportexturl, 'CSV');

    $exportbuttons =  html_writer::div(
        get_string('export', 'report_ghs').' :&nbsp;&nbsp;'.
        $exportexceliconlink.'&nbsp;'.$exportexcellink.'&nbsp;|&nbsp; '
        .$exporttexticonlink.'&nbsp;'.$exporttextlink, 'export-link-wrapper',
        array('style' => 'text-align:center;')
    );

    $outputpagingbar = $OUTPUT->render($pagingbar);

    echo html_writer::div(
        html_writer::div(
            (($outputpagingbar) ? $outputpagingbar :  html_writer::tag('div', '', ['class' => 'dummy-pagination'])) .
            html_writer::div(
                get_string('displayingrecords', 'report_ghs', number_format($totalcount)),
                'paging-total-number'
            ),
            'paging-inline-wrapper'
        ),
        'paging-wrapper'
    );
    echo html_writer::div($exportbuttons, 'export-link-wrapper-top');
    echo html_writer::table($table);
    echo $outputpagingbar;
    echo $exportbuttons;

    echo html_writer::end_div(); // Main wrapper.
    echo $OUTPUT->footer();
}