<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    theme_ned_boost
 * @subpackage NED Boost
 * @copyright  NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @developer  G J Barnard - {@link http://about.me/gjbarnard} and
 *                           {@link http://moodle.org/user/profile.php?id=442195}
 */

namespace theme_ned_boost\output;
use local_schoolmanager\school;
use local_schoolmanager\shared_lib as SML;
use theme_ned_boost\shared_lib as NED;
use local_ned_controller\shared_lib as CTRL;

defined('MOODLE_INTERNAL') || die;
/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/theme/ned_boost/classes/output/block_myprofile_renderer.php');
require_once($CFG->dirroot . '/theme/ned_boost/classes/output/core_user/myprofile_renderer.php');


/**
 * Class core_renderer
 *
 * @package theme_ned_boost\output
 *
 * @property-read \xhtml_container_stack $opencontainers - The xhtml_container_stack to use.
 * @property-read \moodle_page $page - The Moodle page the renderer has been created to assist with.
 * @property-read \string $target - The requested rendering target.
 *
 * @property-read string $contenttype - Used to pass information from {@link core_renderer::doctype()}
 *      to {@link core_renderer::standard_head_html()}.
 * @property-read string $metarefreshtag - Used by {@link core_renderer::redirect_message()}
 *      method to communicate with {@link core_renderer::header()}.
 * @property-read string $unique_end_html_token - Unique token for the closing HTML
 * @property-read string $unique_performance_info_token - Unique token for performance information
 * @property-read string $unique_main_content_token - Unique token for the main content.
 * @property-read \custom_menu_item $language - The language menu if created
 */
class core_renderer extends \theme_boost_union\output\core_renderer {
    use \local_ned_controller\output\core_renderer;

    /**
     * The standard tags that should be included in the <head> tag
     * including a meta description for the front page
     *
     * @return string HTML fragment.
     */
    public function standard_head_html() {
        global $USER, $COURSE, $PAGE;

        $output = parent::standard_head_html();
        if ($this->customcss){
            $output .=  "<style>{$this->customcss}</style>";
        }
        if (!empty($this->additionalhtmlhead)) {
            $output .= $this->additionalhtmlhead;
        }

        if (!isguestuser() && isloggedin()) {
            $user = \core_user::get_user($USER->id);
            profile_load_data($user);
            $output .= " <script type='text/javascript'>
                window.JR_USER_PROFILE = {
                    fullname: '".fullname($user)."',
                    email: '".$user->email."',
                    city: '".$user->city."',
                    country: '".$user->country."',
                    coursename: '".$COURSE->fullname."',
                    coursecode: '".$COURSE->shortname."',
                    partnerschool: '".(NED::get_user_school_name())."',
                    defaultrole: '".($user->profile_field_default_role ?? "")."',
                    pageurl: '".$PAGE->url->out()."',
                };
            </script>";
        }

        return $output;
    }

    /**
     * The standard tags (typically skip links) that should be output just inside
     * the start of the <body> tag. Designed to be called in theme layout.php files.
     *
     * @return string HTML fragment.
     */
    public function standard_top_of_body_html() {
        $output = parent::standard_top_of_body_html();

        if ($this->page->pagelayout !== 'embedded' && !empty($this->additionalhtmltopofbody)) {
            $output .=  $this->additionalhtmltopofbody;
        }

        return $output;
    }

    public function get_extra_buttons() {
        global $PAGE, $SESSION;

        $contextsystem = \context_system::instance();

        // Home
        $url = new \moodle_url('/');
        $extrabuttons[] = \html_writer::link($url, get_string('home'), ['class' => 'btn btn-secondary mr-1']);

        // Classes.
        if (NED::is_plugin_exists(NED::REPORT_GHS)) {
            if (has_capability('report/ghs:viewgroupenrollment', $contextsystem) || \report_ghs\helper::has_capability_in_any_course('report/ghs:viewgroupenrollment')) {
                $extracls = '';
                if (strpos($PAGE->url->get_path(), '/report/ghs/ghs_group_enrollment.php') !== false) {
                    $extracls = ' active';
                }
                $url = new \moodle_url('/report/ghs/ghs_group_enrollment.php');
                $extrabuttons[] = \html_writer::link($url, 'Classes', ['class' => 'btn btn-secondary mr-1'.$extracls]);
            }

            if (NED::is_ai_exists()){
                // Reports.
                if (has_any_capability([
                    'theme/ned_boost:seealldashboards',
                    'theme/ned_boost:seeowncohortdashboards',
                    'theme/ned_boost:seeowngroupsdashboards'], $contextsystem)) {
                    $extracls = '';
                    $path = $PAGE->url->get_path();
                    if (strpos($path, '/local/academic_integrity/infractions.php') !== false
                        || strpos($path, '/report/ghs/ghs_activity_extension.php') !== false
                        || strpos($path, '/report/ghs/ghs_enddate_extension.php') !== false
                        || strpos($path, '/report/ghs/ghs_class_deadlines.php') !== false
                    ) {
                        $extracls = ' active';
                    }
                    $url = new \moodle_url('/local/academic_integrity/infractions.php');
                    $extrabuttons[] = \html_writer::link($url, get_string('reports'), ['class' => 'btn btn-secondary mr-1'.$extracls]);
                }
            }
        }

        // My School.
        if (NED::is_school_manager_exists()) {
            if (has_any_capability([
                'local/schoolmanager:viewownschooldashboard',
                'local/schoolmanager:viewallschooldashboards'], $contextsystem)) {
                $extracls = '';
                if (strpos($PAGE->url->get_path(), '/local/schoolmanager/view.php') !== false) {
                    $extracls = ' active';
                }
                $url = new \moodle_url('/local/schoolmanager/view.php');
                if (has_capability('local/schoolmanager:viewallschooldashboards', $contextsystem)) {
                    $title = get_string('schools', 'theme_ned_boost');
                } else {
                    $title = get_string('myschool', 'theme_ned_boost');
                }
                $extrabuttons[] = \html_writer::link($url, $title, ['class' => 'btn btn-secondary mr-1'.$extracls]);
            }
        }

        // Test & Exams
        $enabletem = false;
        if (!is_siteadmin() && NED::is_school_manager_exists() && $schoolid = SML::get_user_school(null, true)) {
            if ($school = new school($schoolid)) {
                $enabletem = $school->enabletem;
            }
        }

        if (is_siteadmin() || ($enabletem && \local_tem\helper::can_view_sessions())) {
            $extracls = '';
            if (strpos($PAGE->url->get_path(), '/local/tem/sessions.php') !== false) {
                $extracls .= ' active';
            }

            $url = new \moodle_url('/local/tem/sessions.php');
            $title = get_string('testsexams', 'local_tem');
            if (isset($SESSION->temwarning) && !empty($SESSION->temwarning)) {
                $extracls .= ' blinking';
                $title .= ' <span class="red-text"><i class="fa fa-exclamation-triangle" aria-hidden="true"></i></span>';
            }
            $extrabuttons[] = \html_writer::link($url, $title, ['class' => 'btn btn-secondary mr-1 btn-tem'.$extracls]);
        }

        // My Dashboard/Student Dashboards.
        if (has_any_capability([
            'theme/ned_boost:seeowndashboard',
            'theme/ned_boost:seeowngroupsdashboards',
            'theme/ned_boost:seeowncohortdashboards',
            'theme/ned_boost:seealldashboards',
        ], $contextsystem)) {
            $url = new \moodle_url('/my/');
            if (has_any_capability([
                'theme/ned_boost:seeowngroupsdashboards',
                'theme/ned_boost:seeowncohortdashboards',
                'theme/ned_boost:seealldashboards',
            ], $contextsystem)) {
                $title = get_string('studentdashboards', 'theme_ned_boost');
            } else {
                $title = get_string('mydashboard', 'theme_ned_boost');
            }
            $extrabuttons[] = \html_writer::link($url, $title, ['class' => 'btn btn-secondary mr-1']);
        }

        return implode(' ', $extrabuttons);
    }

    /**
     * Wrapper for header elements.
     *
     * @return string HTML to display the main header.
     */
    public function full_header() {
        global $PAGE, $CFG;
        if ($CFG->branch >= 35) {
            $acourse = (($this->page->pagelayout == 'course') || ($this->page->pagelayout == 'incourse'));
            $dasboardfrontpage = (($this->page->pagelayout == 'dashboard') || ($this->page->pagelayout == 'frontpage'));

            $extrabuttons = $this->get_extra_buttons();

            $pageheaderclass = 'row';
            $headerinfo = null;
            $quicklinks = array();
            if ($acourse) {
                $pageheaderclass .= ' acourse';

                $nctoolbox = \local_ned_controller\toolbox::get_instance();
                $categoryicons = $nctoolbox->get_categoryicons();
                if (!empty($categoryicons)) {
                    $category = \core_course_category::get($this->get_current_category(), IGNORE_MISSING);
                    if ((!empty($category)) && (array_key_exists($category->name, $categoryicons))) {
                        $toolbox = \theme_ned_boost\toolbox::get_instance();
                        $headerinfo = array('heading' => $toolbox->getfontawesomemarkup(
                                $categoryicons[$category->name][\local_ned_controller\toolbox::$fontawesomekey]) . ' ' . $this->page->heading
                        );
                    }
                }

                $manualenrolmentquicklink = $nctoolbox->get_manualenrollment_quicklink($this->page->context, $this->page->course->id);
                if ($manualenrolmentquicklink) {
                    $quicklinks[] = array(
                        'url' => $manualenrolmentquicklink,
                        'icon' => $this->pix_icon('t/enrolusers', CTRL::str('quicklinksmanualenrollment'))
                    );
                }
                $courseparticipantsquicklink = $nctoolbox->get_courseparticipants_quicklink($this->page->context, $this->page->course->id);
                if ($courseparticipantsquicklink) {
                    $quicklinks[] = array(
                        'url' => $courseparticipantsquicklink,
                        'icon' => $this->pix_icon('t/groups', CTRL::str('quicklinkscourseparticpants'))
                    );
                }
            }

            $shownavbar = (($this->page->pagelayout == 'frontpage') || empty($this->page->layout_options['nonavbar'])) &&
                (!(($this->page->pagelayout == 'course') || ($this->page->pagelayout == 'incourse')));

            $header = new \stdClass();
            $header->settingsmenu = $this->context_header_settings_menu();
            $header->contextheader = $this->context_header($headerinfo);
            $header->hasnavbar = empty($PAGE->layout_options['nonavbar']);
            $header->navbar = $this->navbar();
            $header->courseheader = $this->course_header();
            $header->pageheaderclass = $pageheaderclass;
            $header->acourse = $acourse;
            $header->dasboardfrontpage = $dasboardfrontpage;
            $header->pbr = $this->image_url('pbr', NED::$PLUGIN_NAME);
            $header->editingbutton = $this->editing_button();
            if ($quicklinks) {
                $header->quicklinks = $quicklinks;
            }
            $header->pageheadingbutton = $extrabuttons . $this->page_heading_button();
            $header->shownavbar = $shownavbar;

            return NED::render_from_template('header', $header);
        } else {
            $acourse = (($this->page->pagelayout == 'course') || ($this->page->pagelayout == 'incourse'));
            $pageheaderclass = 'row';
            $headerinfo = null;
            if ($acourse) {
                $pageheaderclass .= ' acourse';

                $nctoolbox = \local_ned_controller\toolbox::get_instance();
                $categoryicons = $nctoolbox->get_categoryicons();
                if (!empty($categoryicons)) {
                    $category = \core_course_category::get($this->get_current_category());
                    if ((!empty($category)) && (array_key_exists($category->name, $categoryicons))) {
                        $toolbox = \theme_ned_boost\toolbox::get_instance();
                        $headerinfo = array('heading' => $toolbox->getfontawesomemarkup(
                                $categoryicons[$category->name][\local_ned_controller\toolbox::$fontawesomekey]).' '.$this->page->heading
                        );
                    }
                }
            }
            $html = \html_writer::start_tag('header', array('id' => 'page-header', 'class' => $pageheaderclass));
            $html .= \html_writer::start_div('col-xs-12 p-a-1');
            $html .= \html_writer::start_div('card');
            $html .= \html_writer::start_div('card-block');

            if (!$acourse) {
                $html .= \html_writer::start_div('pull-xs-left');
            } else {
                $html .= \html_writer::start_div('nedcoursename');
            }
            $html .= $this->context_header($headerinfo);
            $html .= \html_writer::end_div();

            $html .= \html_writer::start_div('pull-xs-right acoursemenus');
            $contextheadersettingsmenu = $this->context_header_settings_menu();
            if (!empty($contextheadersettingsmenu)) {
                $html .= \html_writer::div($contextheadersettingsmenu, 'context-header-settings-menu');
            }
            if ($acourse) {
                $html .= \html_writer::div($this->editing_button(), 'editing-button pull-right');

                $quicklinks = array();

                $manualenrolmentquicklink = $nctoolbox->get_manualenrollment_quicklink($this->page->context, $this->page->course->id);
                if ($manualenrolmentquicklink) {
                    $icon = $this->pix_icon('t/enrolusers', CTRL::str('quicklinksmanualenrollment'));
                    $quicklinks[] = \html_writer::link($manualenrolmentquicklink, $icon);
                }

                $courseparticipantsquicklink = $nctoolbox->get_courseparticipants_quicklink($this->page->context, $this->page->course->id);
                if ($courseparticipantsquicklink) {
                    $icon = $this->pix_icon('t/groups', CTRL::str('quicklinkscourseparticpants'));
                    $quicklinks[] = \html_writer::link($courseparticipantsquicklink, $icon);
                }

                if (!empty($quicklinks)) {
                    $html .= \html_writer::div(\html_writer::span('', 'ned-divider'), 'pull-right');
                    foreach ($quicklinks as $quicklink) {
                        $html .= \html_writer::div($quicklink, 'quicklink');
                    }
                }
                $html .= \html_writer::div(\html_writer::span('', 'ned-holder'), 'pull-right');
            }
            $html .= \html_writer::end_div();

            $pageheadingbutton = $this->page_heading_button();
            if ((empty($this->page->layout_options['nonavbar'])) &&
                (!(($this->page->pagelayout == 'course') || ($this->page->pagelayout == 'incourse')))) {
                $html .= \html_writer::start_div('clearfix w-100 pull-xs-left', array('id' => 'page-navbar'));
                $html .= \html_writer::tag('div', $this->navbar(), array('class' => 'breadcrumb-nav'));
                $html .= \html_writer::div($pageheadingbutton, 'breadcrumb-button pull-xs-right');
                $html .= \html_writer::end_div();
            } else if ($pageheadingbutton) {
                $html .= \html_writer::div($pageheadingbutton, 'breadcrumb-button nonavbar pull-xs-right');
            }
            $html .= \html_writer::tag('div', $this->course_header(), array('id' => 'course-header'));
            $html .= \html_writer::end_div();
            $html .= \html_writer::end_div();
            $html .= \html_writer::end_div();
            $html .= \html_writer::end_tag('header');
            return $html;
        }
    }

    /**
     * @return int|mixed
     */
    protected function get_current_category() {
        $catid = 0;

        if (is_array($this->page->categories)) {
            $catids = array_keys($this->page->categories);
            $catid = reset($catids);
        } else if (!empty($this->page->course->category)) {
            $catid = $this->page->course->category;
        }

        return $catid;
    }

    /**
     * Whether we should display the logo in the navbar.
     *
     * We will when there are no main logos, and we have compact logo.
     *
     * @return bool
     */
    public function should_display_navbar_logo() {
        $logo = $this->get_compact_logo_url();
        return !empty($logo) && !$this->should_display_main_logo();
    }

    /**
     * Whether we should display the main logo.
     *
     * @param int $headinglevel The heading level we want to check against.
     * @return bool
     */
    public function should_display_main_logo($headinglevel=1) {
        return $headinglevel == 1 && $this->page->pagelayout !== 'mypublic' && !empty($this->get_compact_logo_url());
    }

    /**
     * Override to inject the logo.
     *
     * @param array $headerinfo The header info.
     * @param int $headinglevel What level the 'h' tag will be.
     * @return string HTML for the header bar.
     */
    public function context_header($headerinfo = null, $headinglevel = 1): string {
        global $SITE;
        if ($this->should_display_main_logo($headinglevel)) {
            $logourl = $this->page->theme->setting_file_url('logo', 'logo');
            $sitename = format_string($SITE->fullname, true, array('context' => \context_course::instance(SITEID)));
            return \html_writer::div(\html_writer::empty_tag('img', [
                'src' => $logourl, 'alt' => $sitename, 'class' => 'img-responsive']), 'logo');
        }

        return \core_renderer::context_header($headerinfo, $headinglevel); // Bypass Boost's method.
    }

    /**
     * Renders the login form.
     *
     * @param \core_auth\output\login $form The renderable.
     * @return string
     */
    public function render_login(\core_auth\output\login $form) {
        global $CFG, $SITE;

        $context = $form->export_for_template($this);

        // Override because rendering is not supported in template yet.
        if ($CFG->rememberusername == 0) {
            $context->cookieshelpiconformatted = $this->help_icon('cookiesenabledonlysession');
        } else {
            $context->cookieshelpiconformatted = $this->help_icon('cookiesenabled');
        }
        $context->errorformatted = $this->error_text($context->error);
        $url = $this->page->theme->setting_file_url('loginlogo', 'loginlogo');
        $context->logourl = $url;
        $context->sitename = format_string($SITE->fullname, true,
            ['context' => \context_course::instance(SITEID), "escape" => false]);
        $context->showlocallogin = true;

        return $this->render_from_template('core/loginform', $context);
    }

    /**
     * @return string
ё     */
    public function editing_button() {
        $html = '';
        if ($this->page->user_allowed_editing()) {
            $pagetype = $this->page->pagetype;
            if (strpos($pagetype, 'admin-setting') !== false) {
                $pagetype = 'admin-setting'; // Deal with all setting page types.
            } else if ((strpos($pagetype, 'mod-') !== false) &&
                ((strpos($pagetype, 'edit') !== false) ||
                 (strpos($pagetype, 'view') !== false) ||
                 (strpos($pagetype, '-mod') !== false))) {
                $pagetype = 'mod-edit-view'; // Deal with all mod edit / view / -mod page types.
            } else if (strpos($pagetype, 'mod-data-field') !== false) {
                $pagetype = 'mod-data-field'; // Deal with all mod data field page types.
            } else if (strpos($pagetype, 'mod-lesson') !== false) {
                $pagetype = 'mod-lesson'; // Deal with all mod lesson page types.
            }
            switch ($pagetype) {
                case 'site-index':
                case 'calendar-view':  // Slightly faulty as even the navigation link goes back to the frontpage.  TODO: MDL.
                    $url = new \moodle_url('/course/view.php');
                    $url->param('id', 1);
                    if ($this->page->user_is_editing()) {
                        $url->param('edit', 'off');
                    } else {
                        $url->param('edit', 'on');
                    }
                break;
                case 'admin-index':
                case 'admin-setting':
                    $url = $this->page->url;
                    if ($this->page->user_is_editing()) {
                        $url->param('adminedit', 0);
                    } else {
                        $url->param('adminedit', 1);
                    }
                break;
                case 'course-index':
                case 'course-management':
                case 'course-search':
                case 'mod-resource-mod':
                case 'tag-search':
                    $url = new \moodle_url('/tag/search.php');
                    if ($this->page->user_is_editing()) {
                        $url->param('edit', 'off');
                    } else {
                        $url->param('edit', 'on');
                    }
                break;
                case 'mod-data-field':
                case 'mod-edit-view':
                case 'mod-forum-discuss':
                case 'mod-forum-index':
                case 'mod-forum-search':
                case 'mod-forum-subscribers':
                case 'mod-lesson':
                case 'mod-quiz-index':
                case 'mod-scorm-player':
                    $url = new \moodle_url('/course/view.php');
                    $url->param('id', $this->page->course->id);
                    $url->param('return', $this->page->url->out_as_local_url(false));
                    if ($this->page->user_is_editing()) {
                        $url->param('edit', 'off');
                    } else {
                        $url->param('edit', 'on');
                    }
                break;
                case 'my-index':
                case 'user-profile':
                    // TODO: Not sure how to get 'id' param and if it is really needed.
                    $url = $this->page->url;
                    // Umm! Both /user/profile.php and /user/profilesys.php have the same page type but different parameters!
                    if ($this->page->user_is_editing()) {
                        $url->param('adminedit', 0);
                        $url->param('edit', 0);
                    } else {
                        $url->param('adminedit', 1);
                        $url->param('edit', 1);
                    }
                break;
                default:
                    $url = $this->page->url;
                    if ($this->page->user_is_editing()) {
                        $url->param('edit', 'off');
                    } else {
                        $url->param('edit', 'on');
                    }
                break;
            }

            $url->param('sesskey', sesskey());
            if ($this->page->user_is_editing()) {
                $editstring = get_string('turneditingoff');
                $colourclass = 'edit-on';
            } else {
                $editstring = get_string('turneditingon');
                $colourclass = 'edit-off';
            }
            $toolbox = \theme_ned_boost\toolbox::get_instance();
            $edit = $toolbox->getfontawesomemarkup('fa fa-pencil-square-o', array('fa-fw'));
            $html = \html_writer::link($url, $edit, array('title' => $editstring, 'class' => $colourclass));
        }

        return $html;
    }

    /**
     * Get the compact logo URL.
     *
     * @param int $maxwidth
     * @param int $maxheight
     *
     * @return string
     */
    public function get_compact_logo_url($maxwidth = 100, $maxheight = 100) {
        if ($this->compactlogourl == null) {
            $this->compactlogourl = parent::get_compact_logo_url($maxwidth, $maxheight);
        }

        return $this->compactlogourl;
    }

    /**
     * Return the site's logo URL, if any.
     *
     * @param int $maxwidth The maximum width, or null when the maximum width does not matter.
     * @param int $maxheight The maximum height, or null when the maximum height does not matter.
     * @return \moodle_url|false
     */
    public function get_logo_url($maxwidth = null, $maxheight = 200) {
        if ($this->logourl == null) {
            $this->logourl = parent::get_logo_url($maxwidth, $maxheight);
        }

        return $this->logourl;
    }

    /**
     * @return array
     */
    protected function get_dynamicbase() {
        global $SITE, $OUTPUT;
        $regionmainsettingsmenu = $this->region_main_settings_menu();
        $primary = new navigation\primary($this->page);
        $renderer = $this->page->get_renderer('core');
        $primarymenu = $primary->export_for_template($renderer);
        $secondarynavigation = false;
        $overflow = '';

        $extraclasses = [];
        // Add a special class selector to improve the Smart menus SCSS selectors.
        if (isset($primarymenu['includesmartmenu']) && $primarymenu['includesmartmenu'] == true) {
            $extraclasses[] = 'theme-boost-union-smartmenu';
        }
        if (isset($primarymenu['bottombar']) && !empty($primarymenu['includesmartmenu'])) {
            $extraclasses[] = 'theme-boost-union-bottombar';
        }

        $bodyattributes = $this->body_attributes($extraclasses);

        if ($this->page->has_secondary_navigation()) {
            $tablistnav = $this->page->has_tablist_secondary_navigation();
            $moremenu = new \core\navigation\output\more_menu($this->page->secondarynav, 'nav-tabs', true, $tablistnav);
            $secondarynavigation = $moremenu->export_for_template($OUTPUT);
            $overflowdata = $this->page->secondarynav->get_overflow_menu_data();
            if (!is_null($overflowdata)) {
                $overflow = $overflowdata->export_for_template($OUTPUT);
            }
        }

        return [
            'sitename' => format_string($SITE->shortname, true, ['context' => \context_course::instance(\SITEID), "escape" => false]),
            'output' => $this,
            'bodyattributes' => $bodyattributes,
            'regionmainsettingsmenu' => $regionmainsettingsmenu,
            'hasregionmainsettingsmenu' => !empty($regionmainsettingsmenu),
            'primarymoremenu' => $primarymenu['moremenu'],
            'secondarymoremenu' => $secondarynavigation ?: false,
            'mobileprimarynav' => $primarymenu['mobileprimarynav'],
            'usermenu' => $primarymenu['user'],
            'langmenu' => $primarymenu['lang'],
            'overflow' => $overflow,
        ];
    }

    /**
     * @param $templatecontext
     * @param $position
     */
    protected function determine_dynamic_block_positions(&$templatecontext, $position) {
        $blockslefthtml = $this->blocks('side-pre');
        $hasleftblocks = strpos($blockslefthtml, 'data-block=') !== false;
        $blocksrighthtml = $this->blocks('side-post');
        $hasrightblocks = strpos($blocksrighthtml, 'data-block=') !== false;
        $hasblocks = ($hasleftblocks || $hasrightblocks);

        switch ($position) { // 1 is 'Both' - default for this layout.
            case 2: // Right.
                $blocksrighthtml = $blockslefthtml.$blocksrighthtml;
                $blockslefthtml = '';
                $hasrightblocks = $hasblocks;
                $hasleftblocks = false;
                break;
            case 3: // Left.
                $blockslefthtml = $blockslefthtml.$blocksrighthtml;
                $blocksrighthtml = '';
                $hasleftblocks = $hasblocks;
                $hasrightblocks = false;
                break;
        }
        $bothblocks = ($hasleftblocks && $hasrightblocks);

        $templatecontext['hasblocks'] = $hasblocks;
        $templatecontext['bothblocks'] = $bothblocks;
        $templatecontext['sidepreblocks'] = $blockslefthtml;
        $templatecontext['hasleftblocks'] = $hasleftblocks;
        $templatecontext['sidepostblocks'] = $blocksrighthtml;
        $templatecontext['hasrightblocks'] = $hasrightblocks;
    }

    /**
     * @return array
     */
    public function get_course_context() {
        global $CFG;

        require_once($CFG->libdir.'/behat/lib.php');

        $templatecontext = $this->get_dynamicbase();
        $templatecontext['homeicon'] = true;

        if (empty($this->page->layout_options['nonavbar'])) {
            $navbaritems = $this->page->navbar->get_items();
            $navbaritems = array_slice($navbaritems, 2);
            $navbarcontext = ['get_items' => $navbaritems];
            $templatecontext['headernavbar'] = \html_writer::div($this->render_from_template('core/navbar', $navbarcontext), 'coursenavbar');
        }
        // Remove courselevelblockpositions setting
        $position = 2;  // Right.
        $this->determine_dynamic_block_positions($templatecontext, $position);

        if ($CFG->branch >= 34) {
            $templatecontext['hasactivity_navigation'] = true;
        }

        return $templatecontext;
    }

    /**
     * @return array
     */
    public function get_frontdashboard_context() {
        global $CFG;
        require_once($CFG->libdir.'/behat/lib.php');

        return $this->get_dynamicbase();
    }

    /**
     * @return array
     */
    public function get_layout2_context() {
        global $CFG;
        require_once($CFG->libdir.'/behat/lib.php');

        $templatecontext = $this->get_dynamicbase();

        $blockshtml = $this->blocks('side-pre');
        $hasblocks = strpos($blockshtml, 'data-block=') !== false;
        $templatecontext['sidepreblocks'] = $blockshtml;
        $templatecontext['hasblocks'] = $hasblocks;

        return $templatecontext;
    }

    /**
     * @return array
     */
    public function get_columns2_context() {
        return $this->get_layout2_context();
    }

    public function get_schoolmanager_context() {
        global $SITE, $PAGE, $OUTPUT;

        $blockshtml = $this->blocks('side-pre');
        $hasblocks = strpos($blockshtml, 'data-block=') !== false;
        $buildregionmainsettings = !$PAGE->include_region_main_settings_in_header_actions();
        // If the settings menu will be included in the header then don't add it here.
        $regionmainsettingsmenu = $buildregionmainsettings ? $this->region_main_settings_menu() : false;
        $templatecontext = $this->get_dynamicbase();

        $templatecontext['sitename'] = format_string($SITE->shortname, true, ['context' => \context_course::instance(SITEID), "escape" => false]);
        $templatecontext['output'] = $OUTPUT;
        $templatecontext['sidepreblocks'] = $blockshtml;
        $templatecontext['hasblocks'] = $hasblocks;
        $templatecontext['regionmainsettingsmenu'] = $regionmainsettingsmenu;

        $handler = new \local_schoolmanager\school_handler();
        $templatecontext['headermenu'] = $handler->get_control_form(optional_param('schoolid', 0, PARAM_INT));

        return $templatecontext;
    }

    /**
     * Prints a nice side block with an optional header.
     *
     * @param \block_contents $bc HTML for the content
     * @param string $region the region the block is appearing in.
     * @return string the HTML to be output.
     */
    public function block(\block_contents $bc, $region) {
        $bc = clone($bc); // Avoid messing up the object passed in.
        // if we show 'myoverview' of frontpage, we should not render it in block region
        if (theme_ned_boost_should_add_showmyoverview_block() && $region == 'side-post' && $bc->attributes['data-block'] == 'myoverview'){
            return '';
        }

        if (empty($bc->blockinstanceid) || !strip_tags($bc->title)) {
            $bc->collapsible = \block_contents::NOT_HIDEABLE;
        }

        if(($bc->attributes['data-block'] ?? '') == 'badges'){
            $bc->title = NED::str('badges');
        }
        $title = $bc->title;
        $showheader = true;

        if (empty($title)) {
            if (!$this->page->user_is_editing()) {
                $showheader = false;
            }
        } else {
            $toolbox = \theme_ned_boost\toolbox::get_instance();
            $customiseindividualblocks = $toolbox->get_customiseindividualblocks($this->page->theme);
            if (!empty($customiseindividualblocks)) {
                foreach ($customiseindividualblocks as $blockname => $blocksettings) {
                    if ($blockname == $bc->attributes['data-block']) {
                        if (!empty($blocksettings[\theme_ned_boost\toolbox::$fontawesomekey])) {
                            $title = $toolbox->getfontawesomemarkup($blocksettings[\theme_ned_boost\toolbox::$fontawesomekey]).$title;
                        }
                        break;
                    }
                }
            }
        }

        $id = !empty($bc->attributes['id']) ? $bc->attributes['id'] : uniqid('block-');
        $context = new \stdClass();
        $context->skipid = $bc->skipid;
        $context->blockinstanceid = $bc->blockinstanceid;
        $context->dockable = $bc->dockable;
        $context->id = $id;
        $context->hidden = $bc->collapsible == \block_contents::HIDDEN;
        $context->skiptitle = strip_tags($bc->title);
        $context->showskiplink = !empty($context->skiptitle);
        $context->arialabel = $bc->arialabel;
        $context->ariarole = !empty($bc->attributes['role']) ? $bc->attributes['role'] : 'complementary';
        $context->type = $bc->attributes['data-block'];
        $context->title = $title;
        $context->content = $bc->content;
        $context->annotation = $bc->annotation;
        $context->footer = $bc->footer;
        $context->hascontrols = !empty($bc->controls);
        if ($context->hascontrols) {
            $context->controls = $this->block_controls($bc->controls, $id);
        }
        $context->showheader = $showheader;

        return $this->render_from_template('core/block', $context);
    }
}
