<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    theme_ned_classic
 * @subpackage NED Classic
 * @copyright  2024 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace theme_ned_classic\output;
use theme_ned_classic\shared_lib as NED;
use block_ned_teacher_tools\output\menu_bar as MB;

defined('MOODLE_INTERNAL') || die;

//require_once(__DIR__ . '/../../renderers.php');
//require_once(__DIR__ . '/mod_assign_renderer.php');
//require_once(__DIR__ . '/core_user/myprofile_renderer.php');

/**
 * Class core_renderer
 *
 * @package theme_ned_classic\output
 * @property-read \xhtml_container_stack $opencontainers - The xhtml_container_stack to use.
 * @property-read \moodle_page $page - The Moodle page the renderer has been created to assist with.
 * @property-read \string $target - The requested rendering target.
 *
 * @property-read string $contenttype - Used to pass information from {@link core_renderer::doctype()}
 *      to {@link core_renderer::standard_head_html()}.
 * @property-read string $metarefreshtag - Used by {@link core_renderer::redirect_message()}
 *      method to communicate with {@link core_renderer::header()}.
 * @property-read string $unique_end_html_token - Unique token for the closing HTML
 * @property-read string $unique_performance_info_token - Unique token for performance information
 * @property-read string $unique_main_content_token - Unique token for the main content.
 * @property-read \custom_menu_item $language - The language menu if created
 */
class core_renderer extends \theme_classic\output\core_renderer {
    use \local_ned_controller\output\core_renderer;

    const PREF_TT_SHOW_MB = 'block_ned_teacher_tools_showttbar';
    const PARAM_TT_SHOW_MB = self::PREF_TT_SHOW_MB;

    /**
     * Determines if we should render the logo.
     *
     * @param int $headinglevel What level the 'h' tag will be.
     *
     * @return bool Should the logo be rendered.
     */
    protected function _should_render_logo($headinglevel = 1) {
        /* Only render the logo if we're on the front page or login page
           and the theme has a logo. */
        $logo = $this->get_logo_url();
        if ($headinglevel == 1 && !empty($logo)) {
            if ($this->page->pagelayout == 'frontpage' || $this->page->pagelayout == 'login') {
                return (bool)$logo;
            }
        }

        return null;
    }

    /**
     * Returns a reference to the site home.
     *
     * It can be either a link or a span.
     *
     * @param bool $returnlink
     *
     * @return string
     */
    protected function _get_home_ref($returnlink = true) {
        global $SITE;

        $sitename = format_string($SITE->shortname, true, ['context' => \context_course::instance(SITEID)]);

        if ($returnlink) {
            return \html_writer::link(new \moodle_url('/'), $sitename, ['class' => 'brand', 'title' => get_string('home')]);
        }

        return \html_writer::tag('span', $sitename, ['class' => 'brand']);
    }

    /**
     * Get notifications for course header pages
     *
     * @return string
     */
    protected function _get_header_notifications(){
        $notifications = '';
        $course = $this->page->course;
        if ($course->id == SITEID || $this->page->pagetype !== 'course-view-ned') return $notifications;

        if (class_exists('local_proxy\\output\\renderer')){
            $renderer = $this->page->get_renderer('local_proxy');
            $notifications .= $renderer->render(new \local_proxy\output\proxy_notification($course));
        }
        if (class_exists('assignsubmission_ctsubmission\\output\\renderer')){
            $ctrenderer = $this->page->get_renderer('assignsubmission_ctsubmission');
            $notifications .= $ctrenderer->render(new \assignsubmission_ctsubmission\output\ct_notification($course));
        }
        if (has_capability('assignsubmission/ctsubmission:viewsubmissionalertcourse', NED::ctx(), null, false)
            && class_exists('assignsubmission_ctsubmission\\output\\renderer')){
            $ctrenderer = $ctrenderer ?? $this->page->get_renderer('assignsubmission_ctsubmission');
            if ($msg = $ctrenderer->render(new \assignsubmission_ctsubmission\output\ct_alert())) {
                $notifications .= NED::div($msg, 'alert alert-danger row-fluid', [
                    'style' => 'overflow: hidden;',
                ]);
            }
        }

        return $notifications;
    }

    /**
     * Return both MB course elements: $menutoggle and $menubar
     *
     * @param \moodle_page $page
     *
     * @return string[] - [$menutoggle, $menubar] - HTML elements, $menutoggle (show/hide MB) and rendered menu_bar
     */
    protected function _get_tt_menutoggle_menubar($page){
        static $res = null;

        if (is_null($res)){
            $res = ['', ''];
            if (!NED::is_tt_exists()) return $res;

            $dm_capabilities = [
                'block/ned_teacher_tools:view_all_groups',
                'block/ned_teacher_tools:view_own_groups',
                'block/ned_teacher_tools:deadlineviewonly',
            ];
            $pagetypes = [
                'course-view-ned',
                'blocks-ned_teacher_tools-deadline_manager',
                'blocks-ned_teacher_tools-progress_report',
                'blocks-ned_teacher_tools-student_progress',
                'blocks-ned_teacher_tools-marking_manager',
                'local-kica-grade_user',
                'blocks-ned_teacher_tools-raps',
            ];

            $coursecontext = NED::ctx($page->course->id);
            $canviewmenu = has_any_capability($dm_capabilities, $coursecontext);

            $menutoggle = '';
            $showttbar = 0;
            if ($canviewmenu && in_array($page->pagetype, $pagetypes)) {
                $pref = get_user_preferences(static::PREF_TT_SHOW_MB, 0);
                $showttbar = optional_param(static::PARAM_TT_SHOW_MB, $pref, PARAM_INT);
                if ($showttbar != $pref) {
                    set_user_preference(static::PREF_TT_SHOW_MB, $showttbar);
                }

                $menutoggle .= NED::single_checkbox2($page->url, $showttbar, static::PARAM_TT_SHOW_MB,
                    NED::fa('fa-bookmark color-green'), NED::fa('fa-bookmark-o'), 'ttmenutoggler ml-1 fa-lg',
                    NED::str('showttmb'), NED::str('hidettmb')
                );
            }

            $menubar = '';
            if ($canviewmenu && $showttbar) {
                $page_path = explode('.', $page->url->get_path());
                $page_file = reset($page_path);
                if ($page_file == MB::PAGE_COURSE) {
                    $MB = new MB(MB::PAGE_COURSE, ['id' => $page->course->id]);
                } elseif ($page_file == MB::PAGE_PARTICIPANTS) {
                    $MB = new MB(MB::PAGE_PARTICIPANTS, ['contextid' => $coursecontext->id]);
                }
                if (isset($MB)) {
                    $renderer = $page->get_renderer(NED::TT);
                    $menubar = $renderer->render($MB);
                }
            }

            $res = [$menutoggle, $menubar];
        }

        return $res;
    }

    /**
     * Returns the navigation bar home reference.
     *
     * The small logo is only rendered on pages where the logo is not displayed.
     *
     * @param bool $returnlink Whether to wrap the icon and the site name in links or not
     *
     * @return string The site name, the small logo or both depending on the theme settings.
     */
    public function navbar_home($returnlink = true) {
        debugging('navbar_home HERE!');
        $imageurl = $this->get_compact_logo_url(null, 35);
        if ($this->_should_render_logo() || empty($imageurl)) {
            // If there is no small logo we always show the site name.
            return $this->_get_home_ref($returnlink);
        }
        $image = \html_writer::img($imageurl,  NED::str('sitelogo'), ['class' => 'small-logo']);

        // Disable home link on the quiz pages
        if (NED::str_has($this->page->url->get_path(), '/mod/quiz/')){
            $returnlink = false;
        }

        if ($returnlink) {
            $logocontainer = \html_writer::link(new \moodle_url('/'), $image,
                ['class' => 'small-logo-container', 'title' => get_string('home')]);
        } else {
            $logocontainer = \html_writer::tag('span', $image, ['class' => 'small-logo-container']);
        }

        // Sitename setting defaults to true.
        if (!isset($this->page->theme->settings->sitename) || !empty($this->page->theme->settings->sitename)) {
            return $logocontainer . $this->_get_home_ref($returnlink);
        }

        return $logocontainer;
    }

    /** == From Classic == */

    /**
     * Internal implementation of user image rendering.
     * Implemented here to add academic integrity flag.
     *
     * @param \user_picture $userpicture
     * @return string
     */
    protected function render_user_picture(\user_picture $userpicture) {
        $user_picture = parent::render_user_picture($userpicture);
        if (empty($user_picture)) return $user_picture;

        $user = $userpicture->user;
        if (empty($userpicture->courseid)) {
            $courseid = $this->page->course->id;
        } else {
            $courseid = $userpicture->courseid;
        }

        return NED::div(NED::span(
            NED::get_profile_with_menu_flag($user->id, $courseid, $user_picture, $userpicture->includefullname)
        ), 'd-inline-block');
    }

    /**
     * Renders the header bar.
     * NED: + add user time to header actions (right column of page-course header)
     *
     * @param \context_header $contextheader Header bar object.
     *
     * @return string HTML for the header bar.
     */
    protected function render_context_header(\context_header $contextheader) {
        $res = parent::render_context_header($contextheader);
        if (empty($res)) return $res;

        $usertime = NED::div(userdate(time(), get_string('strftimedatetime', 'langconfig')),
            'user-time-display fa-lg');
        $this->page->add_header_action($usertime);

        return NED::div($res);
    }

    /**
     * Either returns the parent version of the header bar, or a version with the logo replacing the header.
     *
     * @since Moodle 2.9
     *
     * @param array $headerinfo   An array of header information, dependant on what type of header is being displayed. The following
     *                            array example is user specific.
     *                            heading => Override the page heading.
     *                            user => User object.
     *                            usercontext => user context.
     * @param int   $headinglevel What level the 'h' tag will be.
     *
     * @return string HTML for the header bar.
     */
    public function context_header($headerinfo = null, $headinglevel = 1) {

        $logo = $this->_should_render_logo($headinglevel);
        if ($logo != null) {
            return NED::div(NED::img($logo,'img-responsive', null, [], NED::str('logo')), 'logo');
        }

        $html = parent::context_header($headerinfo, $headinglevel);
        // check if displayed user picture (with NED context menu)
        if (!empty($headerinfo['user']) || NED::page()->context->contextlevel == CONTEXT_USER){
            $html = NED::div($html, 'ned-user-context-header');
        }

        return $html;
    }

    /**
     * Return the theme logo URL, else the site's logo URL, if any.
     *
     * Note that maximum sizes are not applied to the theme logo.
     *
     * @param int $maxwidth  The maximum width, or null when the maximum width does not matter.
     * @param int $maxheight The maximum height, or null when the maximum height does not matter.
     *
     * @return \moodle_url|false
     */
    public function get_logo_url($maxwidth = null, $maxheight = 100) {
        global $CFG;

        if (!empty($this->logourl)) {
            return $this->logourl;
        }

        if (!empty($this->page->theme->settings->logo)) {
            $url = $this->page->theme->setting_file_url('logo', 'logo');
            // Get a URL suitable for \moodle_url.
            $relativebaseurl = preg_replace('|^https?://|i', '//', $CFG->wwwroot);
            $url = str_replace($relativebaseurl, '', $url);
            return new \moodle_url($url);
        }

        return parent::get_logo_url($maxwidth, $maxheight);
    }

    /**
     * Return the theme's compact logo URL, else the site's compact logo URL, if any.
     *
     * Note that maximum sizes are not applied to the theme logo.
     *
     * @param int $maxwidth  The maximum width, or null when the maximum width does not matter.
     * @param int $maxheight The maximum height, or null when the maximum height does not matter.
     *
     * @return \moodle_url|false
     */
    public function get_compact_logo_url($maxwidth = 100, $maxheight = 100) {
        global $CFG;

        if (!empty($this->compactlogourl)) {
            return $this->compactlogourl;
        }

        if (!empty($this->page->theme->settings->smalllogo)) {
            $url = $this->page->theme->setting_file_url('smalllogo', 'smalllogo');
            // Get a URL suitable for \moodle_url.
            $relativebaseurl = preg_replace('|^https?://|i', '//', $CFG->wwwroot);
            $url = str_replace($relativebaseurl, '', $url);
            return new \moodle_url($url);
        }

        return parent::get_compact_logo_url($maxwidth, $maxheight);
    }

    /**
     * The standard tags (meta tags, links to stylesheets and JavaScript, etc.)
     * that should be included in the <head> tag. Designed to be called in theme
     * layout.php files.
     *
     * @return string HTML fragment.
     */
    public function standard_head_html() {
        $output = parent::standard_head_html();

        if ($this->customcss){
            $output .=  "<style>{$this->customcss}</style>";
        }
        if (!empty($this->additionalhtmlhead)) {
            $output .= $this->additionalhtmlhead;
        }

        if ($additionalhtmlhead = NED::get_config('additionalhtmlhead')) {
            $output .= "\n".$additionalhtmlhead;
        }
        return $output;
    }

    /**
     * The standard tags (typically skip links) that should be output just inside
     * the start of the <body> tag. Designed to be called in theme layout.php files.
     *
     * @return string HTML fragment.
     */
    public function standard_top_of_body_html() {
        $output = parent::standard_top_of_body_html();

        if ($this->page->pagelayout !== 'embedded' && !empty($this->additionalhtmltopofbody)) {
            $output .=  $this->additionalhtmltopofbody;
        }

        return $output;
    }

    /**
     * Gets the HTML for the page heading button.
     *
     * @return string HTML.
     */
    public function page_heading_button() {
        $res = parent::page_heading_button();
        [$menutoggle,] = $this->_get_tt_menutoggle_menubar($this->page);
        $res .= $menutoggle;
        return $res;
    }

    /**
     * Wrapper for header elements.
     *
     * @return string HTML to display the main header.
     */
    public function full_header() {
        $orig_header = parent::full_header();
        $notifications = $this->_get_header_notifications();
        [, $menubar] = $this->_get_tt_menutoggle_menubar($this->page);

        return $orig_header . $notifications . $menubar;
    }
}
