<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Activity other completion condition.
 *
 * @package     availability_ctsubmission
 * @copyright   2022 Michael Gardener <mgardener@cissq.com>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace availability_ctsubmission;

use assign;
use assignsubmission_ctsubmission\output\ct_overview;
use assignsubmission_ctsubmission\submission;
use context_module;
use core_availability\info_module;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/mod/assign/locallib.php');

/**
 * availability_ctsubmission condition.
 *
 * @package     availability_ctsubmission
 * @copyright   2022 Michael Gardener <mgardener@cissq.com>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or laterr
 */
class condition extends \core_availability\condition {

    /**
     * @var bool
     */
    protected $allow;

    /** @var int ID of module that this depends on */
    protected $cmid;


    /**
     * @param $structure
     */
    public function __construct($structure) {
        if (isset($structure->cm) && is_number($structure->cm)) {
            $this->cmid = (int)$structure->cm;
        } else {
            throw new \coding_exception('Missing or invalid ->cm for completion condition');
        }
        $this->allow = true;
    }

    /**
     * Saves tree data back to a structure object.
     *
     * @return \stdClass Structure object (ready to be made into JSON format)
     */
    public function save() {
        return (object)array('type' => 'ctsubmission');
    }

    /**
     * @param $not
     * @param \core_availability\info $info
     * @param $grabthelot
     * @param $userid
     * @return bool
     */
    public function is_available($not, \core_availability\info $info, $grabthelot, $userid) {
        global $USER;

        $allow = true;
        $course = $info->get_course();

        if (!has_capability('availability/ctsubmission:restrict', $info->get_context(), null, false)) {
            return $this->return_function($allow, $not);
        }

        ct_overview::set_course_groups($course);

        if (empty($course->assigns)) {
            return $this->return_function(false, $not);
        }

        $cm = get_coursemodule_from_id('assign', $this->cmid);
        $context = context_module::instance($cm->id);

        $assignments = [];
        foreach ($course->assigns as $assigndata) {
            $assignments[] = $assigndata->assignment;
        }
        if (!in_array($cm->instance, $assignments)) {
            // If the cmid cannot be found, always return false regardless
            // of the condition or $not state. (Will be displayed in the
            // information message.)
            $allow = false;
        } else {
            if ($usergroups = groups_get_user_groups($course->id)) {
                $assign = new assign($context, $cm, $course);
                foreach ($usergroups as $groupingid => $ggroups) {
                    foreach ($ggroups as $groupid) {
                        $assignsubmissionct = $assign->get_submission_plugin_by_type('ctsubmission');
                        if ($submission = $assign->get_group_submission($USER->id, $groupid, false, -1)) {
                            $count = $assignsubmissionct->count_files($submission->id, ASSIGNSUBMISSION_CTSUBMISSION_FILEAREA);
                            if (!$count) {
                                return $this->return_function(false, $not);
                            }
                        } else {
                            return $this->return_function(false, $not);
                        }
                    }
                }
            } else {
                $allow = false;
            }
        }

        return $this->return_function($allow, $not);
    }

    /**
     * @param $allow
     * @param $not
     * @return bool|mixed
     */
    public function return_function($allow, $not) {
        if ($not) {
            $allow = !$allow;
        }
        return $allow;
    }

    /**
     * Obtains a string describing this restriction (whether or not
     * it actually applies). Used to obtain information that is displayed to
     * students if the activity is not available to them, and for staff to see
     * what conditions are.
     *
     * The $full parameter can be used to distinguish between 'staff' cases
     * (when displaying all information about the activity) and 'student' cases
     * (when displaying only conditions they don't meet).
     *
     * If implementations require a course or modinfo, they should use
     * the get methods in $info.
     *
     * The special string <AVAILABILITY_CMNAME_123/> can be returned, where
     * 123 is any number. It will be replaced with the correctly-formatted
     * name for that activity.
     *
     * @param bool $full Set true if this is the 'full information' view
     * @param bool $not Set true if we are inverting the condition
     * @param \core_availability\info $info Item we're checking
     * @return string Information string (for admin) about all restrictions on
     *   this item
     */
    public function get_description($full, $not, \core_availability\info $info) {
        $cm = get_coursemodule_from_id('assign', $this->cmid);
        $a = new \stdClass();
        $a->name = $cm->name ?? '';
        $a->url = (new \moodle_url('/mod/assign/view.php', ['id' => $this->cmid]))->out();
        return $this->get_either_description($not, false, $a);
    }

    /**
     * Shows the description using the different lang strings for the standalone
     * version or the full one.
     *
     * @param bool $not        True if NOT is in force
     * @param bool $standalone True to use standalone lang strings
     * @param string $activityname Activity name
     *
     * @return string
     */
    protected function get_either_description($not, $standalone, $activityname) {
        return get_string('eitherdescription', 'availability_ctsubmission', $activityname);
    }

    /**
     * @return string
     */
    protected function get_debug_string() {
        return $this->allow ? 'YES' : 'NO';
    }
}
