<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * DM - Missing Due Date
 *
 * @package availability_dmmissingduedate
 * @copyright 2020 NED
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace availability_dmmissingduedate;

defined('MOODLE_INTERNAL') || die();

use availability_dmmissingduedate\shared_lib as NED;

/**
 * dmmissingduedate condition.
 *
 * @package availability_dmmissingduedate
 * @copyright 2020 Michael Gardener <mgardener@cissq.com>
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class condition extends \core_availability\condition {

    /**
     * Saves tree data back to a structure object.
     *
     * @return \stdClass Structure object (ready to be made into JSON format)
     */
    public function save(){
        return (object)array('type' => 'dmmissingduedate');
    }

    /**
     * Returns a JSON object which corresponds to a condition of this type.
     *
     * Intended for unit testing, as normally the JSON values are constructed
     * by JavaScript code.
     *
     * @return \stdClass Object representing condition
     */
    public static function get_json(){
        return (object)array('type' => 'dmmissingduedate');
    }

    /**
     * Determines whether a particular item is currently available
     * according to this availability condition.
     *
     * If implementations require a course or modinfo, they should use
     * the get methods in $info.
     *
     * The $not option is potentially confusing. This option always indicates
     * the 'real' value of NOT. For example, a condition inside a 'NOT AND'
     * group will get this called with $not = true, but if you put another
     * 'NOT OR' group inside the first group, then a condition inside that will
     * be called with $not = false. We need to use the real values, rather than
     * the more natural use of the current value at this point inside the tree,
     * so that the information displayed to users makes sense.
     *
     * @param bool                    $not        Set true if we are inverting the condition
     * @param \core_availability\info $info       Item we're checking
     * @param bool                    $grabthelot Performance hint: if true, caches information
     *                                            required for all course-modules, to make the front page and similar
     *                                            pages work more quickly (works only for current user)
     * @param int                     $userid     User ID to check availability for
     *
     * @return bool True if available
     */
    public function is_available($not, \core_availability\info $info, $grabthelot, $userid){
        /**
         * Allow user access to the activity,  only if user has deadline in this activity
         */
        $course = $info->get_course();
        $context = $info->get_context();
        $allow = true;

        do {
            if ($context->contextlevel != CONTEXT_MODULE){
                break;
            }

            $cmid = $context->instanceid;
            if (!NED::dm_get_deadline_by_cm($cmid, $userid, $course)){
                $allow = false;
            }
        } while(false);

        return $not ? !$allow : $allow;
    }

    /**
     * Obtains a string describing this restriction (whether or not
     * it actually applies). Used to obtain information that is displayed to
     * students if the activity is not available to them, and for staff to see
     * what conditions are.
     *
     * The $full parameter can be used to distinguish between 'staff' cases
     * (when displaying all information about the activity) and 'student' cases
     * (when displaying only conditions they don't meet).
     *
     * If implementations require a course or modinfo, they should use
     * the get methods in $info. They should not use any other functions that
     * might rely on modinfo, such as format_string.
     *
     * To work around this limitation, use the functions:
     *
     * description_cm_name()
     * description_format_string()
     * description_callback()
     *
     * These return special markers which will be added to the string and processed
     * later after modinfo is complete.
     *
     * @param bool                    $full Set true if this is the 'full information' view
     * @param bool                    $not  Set true if we are inverting the condition
     * @param \core_availability\info $info Item we're checking
     *
     * @return string Information string (for admin) about all restrictions on
     *   this item
     */
    public function get_description($full, $not, \core_availability\info $info){
        return NED::str('eitherdescription');
    }

    /**
     * Obtains a representation of the options of this condition as a string,
     * for debugging.
     *
     * @return string Text representation of parameters
     */
    protected function get_debug_string(){
        return gmdate('Y-m-d H:i:s');
    }

    /**
     * Obtains a string describing this restriction, used when there is only
     * a single restriction to display. (I.e. this provides a 'short form'
     * rather than showing in a list.)
     *
     * Default behaviour sticks the prefix text, normally displayed above
     * the list, in front of the standard get_description call.
     *
     * If implementations require a course or modinfo, they should use
     * the get methods in $info. They should not use any other functions that
     * might rely on modinfo, such as format_string.
     *
     * To work around this limitation, use the functions:
     *
     * description_cm_name()
     * description_format_string()
     * description_callback()
     *
     * These return special markers which will be added to the string and processed
     * later after modinfo is complete.
     *
     * @param bool                    $full Set true if this is the 'full information' view
     * @param bool                    $not  Set true if we are inverting the condition
     * @param \core_availability\info $info Item we're checking
     *
     * @return string Information string (for admin) about all restrictions on
     *   this item
     */
    public function get_standalone_description($full, $not, \core_availability\info $info){
        return $this->get_description($full, $not, $info);
    }
}
