/**
 * Add a create new group modal to the page.
 *
 * @module     block_ned_teacher_tools/classextension
 * @class      ClassExtension
 * @package    block_ned_teacher_tools
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
define(['jquery', 'core/str', 'core/modal_factory', 'core/modal_events', 'core/fragment', 'core/ajax', 'core/yui', 'core/notification',],
    function($, Str, ModalFactory, ModalEvents, Fragment, Ajax, Y, Notification) {

        /**
         * Constructor
         *
         * @param {int} contextid
         * @param {int} groupid
         *
         * Each call to init gets it's own instance of this class.
         */
        var ClassExtension = function(contextid, groupid) {
            this.contextid = contextid;
            this.groupid = groupid;
            this.init();
        };

        /**
         * @var {Modal} modal
         * @private
         */
        ClassExtension.prototype.modal = null;

        /**
         * @var {int} contextid
         * @private
         */
        ClassExtension.prototype.contextid = -1;

        /**
         * @var {int} groupid
         * @private
         */
        ClassExtension.prototype.groupid = -1;

        /**
         * Initialise the class.
         *
         * @param {String} selector used to find triggers for the new group modal.
         * @private
         * @return {Promise}
         */
        ClassExtension.prototype.init = function() {
            return Str.get_strings([{
                    key: 'extension',
                    component: 'block_ned_teacher_tools'
                }
            ]).then(function(strings) {
                // Create the modal.
                return ModalFactory.create({
                    type: ModalFactory.types.SAVE_CANCEL,
                    title: strings[0],
                    body: this.getBody()
                });
            }.bind(this)).then(function(modal) {
                // Keep a reference to the modal.
                this.modal = modal;

                // Forms are big, we want a big modal.
                this.modal.setLarge();

                // We want to reset the form every time it is opened.
                this.modal.getRoot().on(ModalEvents.hidden, function() {
                    this.modal.setBody(this.getBody());
                }.bind(this));

                // We want to hide the submit buttons every time it is opened.
                this.modal.getRoot().on(ModalEvents.shown, function() {
                    this.modal.getRoot().append('<style>[data-fieldtype=submit] { display: none ! important; }</style>');
                }.bind(this));

                // We catch the modal save event, and use it to submit the form inside the modal.
                // Triggering a form submission will give JS validation scripts a chance to check for errors.
                this.modal.getRoot().on(ModalEvents.save, this.submitForm.bind(this));
                // We also catch the form submit event and use it to submit the form with ajax.
                this.modal.getRoot().on('submit', 'form', this.submitFormAjax.bind(this));

                modal.show();
            }.bind(this));
        };

        /**
         * @method getBody
         * @private
         * @return {Promise}
         */
        ClassExtension.prototype.getBody = function(formdata) {
            if (typeof formdata === "undefined") {
                formdata = {};
            }
            // Get the content of the modal.
            var params = {
                jsonformdata: JSON.stringify(formdata),
                groupid: this.groupid,
            };
            return Fragment.loadFragment('block_ned_teacher_tools', 'class_extension_form', this.contextid, params);
        };

        /**
         * @method handleFormSubmissionResponse
         * @private
         * @return {Promise}
         */
        ClassExtension.prototype.handleFormSubmissionResponse = function(data) {
            this.modal.hide();
            // We could trigger an event instead.
            // Yuk.
            Y.use('moodle-core-formchangechecker', function() {
                M.core_formchangechecker.reset_form_dirty_state();
            });
            document.location.reload();
        };

        /**
         * @method handleFormSubmissionFailure
         * @private
         * @return {Promise}
         */
        ClassExtension.prototype.handleFormSubmissionFailure = function(data) {
            // Oh noes! Epic fail :(
            // Ah wait - this is normal. We need to re-display the form with errors!
            this.modal.setBody(this.getBody(data));
        };

        /**
         * Private method
         *
         * @method submitFormAjax
         * @private
         * @param {Event} e Form submission event.
         */
        ClassExtension.prototype.submitFormAjax = function(e) {
            // We don't want to do a real form submission.
            e.preventDefault();

            var changeEvent = document.createEvent('HTMLEvents');
            changeEvent.initEvent('change', true, true);

            // Prompt all inputs to run their validation functions.
            // Normally this would happen when the form is submitted, but
            // since we aren't submitting the form normally we need to run client side
            // validation.
            this.modal.getRoot().find(':input').each(function(index, element) {
                element.dispatchEvent(changeEvent);
            });

            // Now the change events have run, see if there are any "invalid" form fields.
            var invalid = $.merge(
                this.modal.getRoot().find('[aria-invalid="true"]'),
                this.modal.getRoot().find('.error')
            );

            // If we found invalid fields, focus on the first one and do not submit via ajax.
            if (invalid.length) {
                invalid.first().focus();
                return;
            }

            // Convert all the form elements values to a serialised string.
            var formData = this.modal.getRoot().find('form').serialize();

            // Now we can continue...
            Ajax.call([{
                methodname: 'block_ned_teacher_tools_submit_class_extension_form',
                args: {
                    contextid: this.contextid,
                    jsonformdata: JSON.stringify(formData)
                },
                // done: this.handleFormSubmissionResponse.bind(this, formData),
                fail: this.handleFormSubmissionFailure.bind(this, formData)
            }])[0].done(function(data) {
                if (data.status) {
                    this.modal.hide();
                    // We could trigger an event instead.
                    Y.use('moodle-core-formchangechecker', function() {
                        M.core_formchangechecker.reset_form_dirty_state();
                    });
                    document.location.reload();
                } else {
                    this.modal.hide();
                    Notification.addNotification({
                        message: data.warnings[0].message,
                        type: 'error'
                    });
                }
            }.bind(this));
        };

        /**
         * This triggers a form submission, so that any mform elements can do final tricks before the form submission is processed.
         *
         * @method submitForm
         * @param {Event} e Form submission event.
         * @private
         */
        ClassExtension.prototype.submitForm = function(e) {
            e.preventDefault();
            this.modal.getRoot().find('form').submit();
        };

        return /** @alias module:block_ned_teacher_tools/ClassExtension */ {
            // Public variables and functions.
            /**
             * Attach event listeners to initialise this module.
             *
             * @method init
             * @param {int} contextid The course context id.
             * @param {int} groupid The user id.
             * @return {Promise}
             */
            init: function(contextid, groupid) {
                return new ClassExtension(contextid, groupid);
            }
        };
    });