/**
 * Create new modal with form
 *
 * @module     block_ned_teacher_tools/modal_form
 * @class      ModalForm
 * @package    block_ned_teacher_tools
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
// @noinspection TypeScriptValidateTypes
define(['jquery', 'core/str', 'core/modal_factory', 'core/modal_events', 'core/fragment'],
    function($, Str, ModalFactory, ModalEvents, Fragment) {
        let texts = {};

        function str(key){
            if (texts[key] === undefined){
                return key;
            }
            return texts[key];
        }

        /**
         * Constructor
         *
         * @param {integer} contextid The course context id.
         * @param {object} data - other useful data
         *
         * Each call to init gets it's own instance of this class.
         */
        let ModalForm;
        ModalForm = function (contextid, data) {
            this.contextid = contextid;
            this.data = data || {};
            this.type = data['type'] || 'default';
            if (typeof this.type === 'string'){
                this.type = this.type.toUpperCase();
                this.modal_type = ModalFactory.types[this.type] || ModalFactory.types.DEFAULT;
            }

            // noinspection JSIgnoredPromiseFromCall
            this.init();
        };

        /**
         * @var {Modal} modal
         * @private
         */
        ModalForm.prototype.modal = null;

        /**
         * @var {string} last_body
         * @private
         */
        ModalForm.prototype.last_body = null;

        /**
         * @var {string} | {object} last_response
         * @private
         */
        ModalForm.prototype.last_response = null;

        /**
         * @var {number}|{integer} contextid
         * @private
         */
        ModalForm.prototype.contextid = -1;

        /**
         * @var {string} type
         * @private
         */
        ModalForm.prototype.type = '';

        /**
         * @var {string} modal_type
         * @private
         */
        ModalForm.prototype.modal_type = ModalFactory.types.DEFAULT;

        /**
         * @var {object} data
         * @private
         */
        ModalForm.prototype.data = {};


        /**
         * @var {boolean} confirm
         * @private
         */
        ModalForm.prototype.confirm = false;

        /**
         * Initialise the class.
         *
         * @private
         * @return {Promise}
         */
        ModalForm.prototype.init = function() {
            let strings = [];
            let strings_comp = [];
            this.TryCall('callback_init');

            if (this.modal_type === ModalFactory.types.SAVE_CANCEL){
                strings.push('save');
            }

            strings.forEach(el => strings_comp.push({key: el, component: 'block_ned_teacher_tools'}));
            return Str.get_strings(strings_comp).then(function(string_result) {
                strings.forEach((el,i) => (texts[el] = string_result[i]));
                return this.get_modal();
            }.bind(this));
        };

        ModalForm.prototype.TryCall = function(callback_name){
            let fun = this.data[callback_name];
            if (typeof fun === 'function'){
                fun.bind(this)();
            }
        };

        /**
         * @method get_modal
         * @private
         * @return {Promise}
         */
        ModalForm.prototype.get_modal = function() {
            return this.getBody().then(function(body) {
                if (body === null || !this.last_body){
                    if (this.modal){
                        this.modal.destroy.bind(this.modal)();
                    }
                    return null;
                }

                let $submit = $(this.last_body).find('[type="submit"]');
                let modal_type = this.modal_type;
                if (!$submit.length && modal_type === ModalFactory.types.DEFAULT){
                    modal_type = ModalFactory.types.ALERT;
                }

                if (this.modal){
                    if (modal_type !== this.modal_type){
                        this.modal.destroy.bind(this.modal)();
                    } else {
                        return this.modal.setBody(body);
                    }
                }

                return ModalFactory.create({
                    type: modal_type,
                    title: this.data['title'] || '',
                    body: this.last_body,
                });
            }.bind(this)).then(function(modal) {
                if (!modal){
                    return;
                }
                // Keep a reference to the modal.
                this.modal = modal;
                let root = this.modal.getRoot();

                // Forms are big, we want a big modal.
                this.modal.setLarge();
                modal.getModal().addClass('modal-form ' + (this.data['add_class'] || ''));

                let save_button = false;
                if (this.modal_type === ModalFactory.types.SAVE_CANCEL){
                    save_button = str('save');
                }

                if (save_button){
                    this.modal['setSaveButtonText'](save_button);
                }

                root.on(ModalEvents.shown, function() {
                    this.TryCall('callback_show');
                    let $input = $.merge(
                        this.modal.getModal().find('form :input:visible'),
                        this.modal.getModal().find('.btn-primary:visible')
                    );

                    if ($input.length){
                        $input.first().focus();
                    }
                }.bind(this));

                // We want to reset the form every time it is opened.
                root.on(ModalEvents.hidden, function(e){
                    e.preventDefault();
                    this.TryCall('callback_hide');
                    if (!this.data['prevent_hide']){
                        this.modal.destroy.bind(this.modal)();
                    }
                }.bind(this));
                root.on('click', '[name="cancel"]', this.oncancel.bind(this));

                // We catch the modal save event, and use it to submit the form inside the modal.
                // Triggering a form submission will give JS validation scripts a chance to check for errors.
                root.on(ModalEvents.save, this.onsave.bind(this));
                // We also catch the form submit event and use it to submit the form with ajax.
                root.on('submit', 'form', this.onsave.bind(this));

                this.TryCall('callback_create');
                this.modal.show();
            }.bind(this));
        };

        /**
         * @method onsave
         * @param {Event} e Form submission event.
         * @private
         */
        ModalForm.prototype.onsave = function(e){
            e.preventDefault();
            this.confirm = true;
            this.TryCall('callback_save');
            if (!this.data['prevent_save']){
                return this.get_modal();
            }
            //document.location.reload();
        };

        /**
         * @method oncancel
         * @param {Event} e Form submission event.
         * @private
         */
        ModalForm.prototype.oncancel = function(e){
            e.preventDefault();
            this.TryCall('callback_cancel');
            if (!this.data['prevent_cancel']){
                this.modal.destroy.bind(this.modal)();
            }
        };

        /**
         * @method getBody
         * @private
         * @return {Promise}
         */
        ModalForm.prototype.getBody = function(ignore_callback=false){
            return this._getBody(ignore_callback).then(function(body){
                this.last_response = body;
                this.last_body = body;
                if (typeof body === "string"){
                    try {
                        this.last_response = JSON.parse(body);
                        this.last_body = this.last_response['body'] || null;
                    } catch (e) {
                        this.last_response = body;
                        this.last_body = body;
                    }
                }

                this.TryCall('callback_body');
                if (this.last_body === 'OK'){
                    this.TryCall('callback_body_ok');
                    if (this.data['finish_if_body_ok']){
                        return null;
                    }
                }

                return body;
            }.bind(this));
        };

        /**
         * @method _getBody
         * @private
         * @return {Promise}
         */
        ModalForm.prototype._getBody = function(ignore_callback=false) {
            let fun = this.data['getBody'];
            if (!ignore_callback && typeof fun === 'function'){
                let possiblePromise = fun.bind(this)();
                if (possiblePromise instanceof Promise){
                    return possiblePromise;
                } else {
                    return new Promise((resolve) => {resolve(possiblePromise)});
                }
            }

            let formdata = this.getForm();
            if (formdata == null){
                return null;
            }

            let params = {
                jsonformdata: JSON.stringify(formdata),
                data: JSON.stringify(this.data),
                confirm: this.confirm,
                classname: this.data['classname'] || '',
                methodname: this.data['methodname'] || '',
            };
            this.confirm = false;
            return Fragment.loadFragment('block_ned_teacher_tools', 'modal_form', this.contextid, params);
        };

        /**
         * @method getForm
         * @private
         * @return {string} | {object} | {null}
         */
        ModalForm.prototype.getForm = function(ignore_callback=false){
            if (!this.modal){
                return {};
            }

            let fun = this.data['getForm'];
            if (ignore_callback && typeof fun === 'function'){
                return fun.bind(this)();
            }

            let form = this.modal.getRoot().find('form');
            if (!form.length){
                return {};
            }

            let changeEvent = document.createEvent('HTMLEvents');
            changeEvent.initEvent('change', true, true);

            // Prompt all inputs to run their validation functions.
            // Normally this would happen when the form is submitted, but
            // since we aren't submitting the form normally we need to run client side
            // validation.
            this.modal.getRoot().find(':input').each(function(index, element) {
                element.dispatchEvent(changeEvent);
            });

            // Now the change events have run, see if there are any "invalid" form fields.
            let invalid = $.merge(
                this.modal.getRoot().find('[aria-invalid="true"]'),
                this.modal.getRoot().find('.error')
            );

            // If we found invalid fields, focus on the first one and do not submit via ajax.
            if (invalid.length) {
                invalid.first().focus();
                return null;
            }

            // Convert all the form elements values to a serialised string.
            return form.serialize();
        };

        return /** @alias module:block_ned_teacher_tools/ModalForm */ {
            // Public variables and functions.

            /**
             * Attach event listeners to initialise this module.
             *
             * @method init
             *
             * @param {integer} contextid The course context id.
             * @param {object} data - other useful data:
             *      title: string,
             *      type: string - form type
             *      add_class: string,
             *      callback_body: function - when we get body text
             *      callback_body_ok: function - when we body text is "OK"
             *      finish_if_body_ok: boolean - hide modal, if answer is "OK"
             *      callback_hide: function,
             *      prevent_hide: boolean,
             *      callback_save: function
             *      prevent_save: boolean,
             *      callback_cancel: function,
             *      prevent_cancel: boolean,
             *      getBody: function, return Promise | function
             *      getForm: function, return string | null
             *
             *
             * @return {ModalForm}
             */
            init: function(contextid, data) {
                return new ModalForm(contextid, data);
            }
        };
    }
);
