<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Deadline Manager
 *
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools;

use block_ned_teacher_tools\shared_lib as SH;
use local_ned_controller\tt_config_manager as CM;
use block_ned_teacher_tools\output\menu_bar as MB;
use block_ned_teacher_tools\mod\deadline_manager_quiz;

require_once(__DIR__.'/../lib.php');
require_once($CFG->dirroot.'/cohort/lib.php');
require_once($CFG->dirroot.'/user/lib.php');

defined('MOODLE_INTERNAL') || die();

/**
 * Class deadline_manager
 * @package block_ned_teacher_tools
 */
class deadline_manager {
    use \local_ned_controller\shared\global_util;

    const TABLE_EXTENSION = 'block_ned_teacher_tools_exte';

    /**
     *
     */
    const PAGE_OVERVIEW = 'overview';
    /**
     *
     */
    const PAGE_GROUP = 'group';
    /**
     *
     */
    const PAGE_GROUP_EDIT = 'group_edit';
    /**
     *
     */
    const PAGE_USER = 'user';

    /**
     *
     */
    const PAGE_USER_OVERVIEW = 'user_overview';
    /**
     *
     */
    const PAGE_USER_EDIT = 'user_edit';
    /**
     *
     */
    const SCHEDULE_NONE = 0;
    /**
     *
     */
    const SCHEDULE_OPTIONAL = 1;
    /**
     *
     */
    const SCHEDULE_FULL = 2;

    /**
     *  Enabled DM modules
     */
    const ENABLED_MODULES = [
        SH::ASSIGN => true,
        SH::QUIZ   => true,
    ];

    const SOURCE_EXTENSION = 'extension';
    const SOURCE_USER = 'user';
    const SOURCE_GROUP = 'group';
    const SOURCE_ACTIVITY = 'activity';
    const SOURCE_NONE = 'none';
    /**
     * All deadline sources, order by priority
     * @var array
     */
    const SOURCES = [
        self::SOURCE_EXTENSION,
        self::SOURCE_USER,
        self::SOURCE_GROUP,
        self::SOURCE_ACTIVITY,
        self::SOURCE_NONE,
    ];
    /**
     * Sources, which are editable in DM, order by priority
     * @var array
     */
    const DM_SOURCES = [
        self::SOURCE_EXTENSION,
        self::SOURCE_USER,
        self::SOURCE_GROUP,
    ];

    /**
     * @var array static store
     */
    static protected $_global_data = [];

    /**
     * @var array List of deadline_managers for cache
     */
    static protected $_deadline_managers = [];

    /**
     * @var  \block_ned_teacher_tools\mod\deadline_manager_mod[] List of deadline_manager_mods for cache
     */
    static protected $_deadline_manager_mods = [];

    /**
     * @var object Course
     */
    public $course;

    /**
     * @var array List of cmids that are enabled with block setting
     */
    protected $activities;

    /**
     * @var array List of cmids that are enabled with block setting
     */
    protected $extensions;

    /**
     * deadline_manager constructor.
     *
     * @param int|string $courseid
     *
     * @throws \moodle_exception
     */
    public function __construct($courseid){
        $this->course = SH::get_course($courseid);
        if (!$this->course){
            SH::print_error("There is no course with '$courseid'!");
        }
        $this->set_enabled_activities();
        $this->set_enabled_extensions();

        static::$_deadline_managers[$courseid] = $this;
    }

    /**
     * @param int|string $courseid
     *
     * @return static
     */
    public static function get_dm_by_courseid($courseid){
        if (!isset(static::$_deadline_managers[$courseid])){
            return new static($courseid);
        }
        return static::$_deadline_managers[$courseid];
    }

    /**
     * @param \cm_info|int|string $cm_or_id
     * @param object|numeric      $courseorid - Optional course object (or its id) if loaded, improves optimization if $cm_or_id is represented as ID
     *
     * @return \block_ned_teacher_tools\mod\deadline_manager_mod|false
     */
    public static function get_dmm_by_cm($cm_or_id, $courseorid=null){
        $cmid = SH::get_id($cm_or_id);
        if (!$cmid){
            return false;
        }

        if (!isset(static::$_deadline_manager_mods[$cmid])){
            $cm = SH::get_cm_by_cmorid($cm_or_id, $courseorid);

            /** @var \block_ned_teacher_tools\mod\deadline_manager_mod|string $deadline_classname */
            $deadline_classname = '\block_ned_teacher_tools\mod\deadline_manager_'.($cm->modname ?? '');
            if (class_exists($deadline_classname)){
                static::$_deadline_manager_mods[$cmid] = $deadline_classname::get_dmm_by_cm($cm, $courseorid);
            } else {
                static::$_deadline_manager_mods[$cmid] = false;
            }
        }

        return static::$_deadline_manager_mods[$cmid];
    }

    /**
     * deadline_manager_entity constructor.
     * Normally you should provide at least group or user
     *
     * @param object|numeric $course_or_id
     * @param object|numeric $group_or_id (optional)
     * @param object|numeric $user_or_id  (optional)
     *
     * @return \block_ned_teacher_tools\deadline_manager_entity
     */
    public static function get_dm_entity($course_or_id, $group_or_id=null, $user_or_id=null){
        $keys = SH::get_ids($course_or_id, $group_or_id, $user_or_id);
        $dme = static::g_get(__FUNCTION__, $keys);
        if (is_null($dme)){
            $dme = new \block_ned_teacher_tools\deadline_manager_entity($course_or_id, $group_or_id, $user_or_id);
            static::g_set(__FUNCTION__, $keys, $dme);
        }

        return $dme;
    }

    /**
     * @param \cm_info|int|string $cm_or_id - it's better to use cm_info, if you already loaded it
     * @param object|numeric      $course_or_id
     *
     * @return bool
     */
    public static function is_cm_enabled($cm_or_id, $course_or_id=null){
        $cmid = SH::get_id($cm_or_id);
        $e = static::g_get(__FUNCTION__, [$cmid]);
        if (is_null($e)){
            $e = false;
            if (static::is_cm_enabled_by_type($cm_or_id, $course_or_id)){
                $courseid = SH::get_id($course_or_id) ?: SH::get_courseid_by_cmorid($cm_or_id);
                $activities = static::get_enabled_activities_by_courseid($courseid);

                if (!empty($activities)){
                    $e = in_array($cmid, $activities);
                }
            }
            static::g_set(__FUNCTION__, [$cmid], $e);
        }

        return $e;
    }

    /**
     * Checked, can this course module use extension
     * NOTE: Here we check none capabilities, only cm settings
     *
     * @param \cm_info|numeric $cm_or_id     - it's better to use cm_info, if you already loaded it
     * @param numeric|object   $course_or_id - (optional)
     *
     * @return bool
     */
    public static function can_cm_use_extension($cm_or_id, $course_or_id=null){
        [$cmid, $courseid] = SH::get_ids($cm_or_id, $course_or_id);
        $res = static::g_get(__FUNCTION__, [$cmid]);
        if (is_null($res)){
            $res = false;

            do {
                $courseid = $courseid ?: SH::get_courseid_by_cmorid($cm_or_id);
                if (!static::is_cm_enabled($cm_or_id, $courseid)){
                    break;
                }

                /** @see \block_ned_teacher_tools\mod\deadline_manager_mod::is_proxy() */
                if (SH::dm_is_proxy_activity($cm_or_id)){
                    break;
                }

                $extensions = static::get_enabled_extensions_by_courseid($courseid);
                if (empty($extensions) || !in_array($cmid, $extensions)){
                    break;
                }

                $res = true;
            } while(false);

            static::g_set(__FUNCTION__, [$cmid], $res);
        }

        return $res;
    }

    /**
     * Return, can grader add extension in some activity & userid
     * Function for both get_extension_course_data and get_extension_data
     * @see get_activity_extension_data()
     *
     * @param \cm_info|int|string      $cm_or_id        - it's better to use cm_info, if you already loaded it
     * @param object|int|string        $student_or_id
     * @param object|numeric|null      $grader_or_id    (optional) Uses current $USER by default
     * @param \context_course|\context $context         (optional) Uses $course context by default
     * @param int|null                 $deadline        (optional) current deadline, if already loaded
     * @param bool                     $check_submitted (optional) if false, ignore submitted user activity or not
     * @param bool                     $load_by_course  (optional) if true, load all data by courseid, otherwise by courseid and userid
     * @param object|numeric           $courseorid      (optional) course object (or its id) if loaded,
     *                                                  improves optimization if $cm_or_id is represented as ID
     *
     * @return bool
     */
    public static function can_grader_add_extension_user_in_cm($cm_or_id, $student_or_id, $grader_or_id=null, $context=null,
        $deadline=null, $check_submitted=true, $load_by_course=false, $courseorid=null){
        [, , $can_add_extension] =
            static::get_activity_extension_data($cm_or_id, $student_or_id, $grader_or_id, $context,
                $deadline, $check_submitted, $load_by_course, $courseorid);
        return $can_add_extension;
    }


    /**
     * @param $roleshortname
     * @param $context
     * @param $groupid
     *
     * @return array
     */
    public static function get_role_users($roleshortname, $context, $groupid){
        global $DB;

        if ($role = $DB->get_record('role', ['shortname' => $roleshortname])){
            if ($roleusers = get_role_users($role->id, $context, false, '', null, false, $groupid)){
                return $roleusers;
            }
        }
        return [];
    }

    /**
     * @param      $roleshortname
     * @param      $context
     * @param      $groupid
     * @param bool $linkable
     * @param null $courseid
     *
     * @return array
     */
    public static function get_role_users_fullnames($roleshortname, $context, $groupid, $linkable=false, $courseid=null){
        $userfullnames = [];
        if ($users = self::get_role_users($roleshortname, $context, $groupid)){
            foreach ($users as $user){
                if ($linkable){
                    $params = ['id' => $user->id];
                    if ($courseid){
                        $params['course'] = $courseid;
                    }
                    $url = new \moodle_url('/user/view.php', $params);
                    $userfullnames[] = \html_writer::link($url, fullname($user));
                } else {
                    $userfullnames[] = fullname($user);
                }
            }
            return $userfullnames;
        }
        return [];
    }

    /**
     * @param $groupid
     * @param $deadlines
     *
     * @return int|mixed
     */
    public static function get_group_first_deadline($groupid, $deadlines){
        $first = 0;
        $counter = 0;
        $groupoverride = null;
        foreach ($deadlines as $cmid => $deadline){
            if (isset($deadline['groupoverrides'])){
                foreach ($deadline['groupoverrides'] as $item){
                    if ($item->groupid == $groupid && !is_null($item->deadline)){
                        $groupoverride = $item->deadline;
                        break;
                    }
                }
            }
            $deadlinedate = (!is_null($groupoverride)) ? $groupoverride : $deadline['mod'];
            if ($deadlinedate){
                if ($counter == 0){
                    $first = $deadlinedate;
                }
                if ($deadlinedate < $first){
                    $first = $deadlinedate;
                }
                $counter++;
            }
        }
        return $first;


    }

    /**
     * @param      $userid
     * @param      $groupid
     * @param      $deadlines
     * @param bool $min = true
     *
     * @return int|mixed
     */
    public static function get_user_first_deadline($userid, $groupid, $deadlines, $min=true){
        $first = null;
        foreach ($deadlines as $cmid => $deadline){
            $deadlinedate = null;

            if (isset($deadline['useroverrides'])){
                foreach ($deadline['useroverrides'] as $item){
                    if ($item->userid == $userid){
                        $deadlinedate = $item->deadline;
                        break;
                    }
                }
            }

            if (is_null($deadlinedate) && $groupid && isset($deadline['groupoverrides'])){
                foreach ($deadline['groupoverrides'] as $item){
                    if ($item->groupid == $groupid){
                        $deadlinedate = $item->deadline;
                        break;
                    }
                }
            }

            if (is_null($deadlinedate)){
                $deadlinedate = $deadline['mod'];
            }

            if (is_null($first) || ($min && ($deadlinedate < $first)) || (!$min && ($deadlinedate > $first))){
                $first = $deadlinedate;
            }
        }
        return $first;
    }

    /**
     * @param $groupid
     * @param $deadlines
     *
     * @return int|mixed
     */
    public static function get_group_last_deadline($groupid, $deadlines){
        $last = 0;
        $groupoverride = null;
        foreach ($deadlines as $cmid => $deadline){
            if (isset($deadline['groupoverrides'])){
                foreach ($deadline['groupoverrides'] as $item){
                    if ($item->groupid == $groupid && !is_null($item->deadline)){
                        $groupoverride = $item->deadline;
                        break;
                    }
                }
            }
            $deadlinedate = (!is_null($groupoverride)) ? $groupoverride : $deadline['mod'];
            if ($deadlinedate){
                if ($deadlinedate > $last){
                    $last = $deadlinedate;
                }
            }
        }

        return $last;
    }

    /**
     * @param $groupid
     *
     * @return array|null
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public function group_users_summary($groupid){
        global $MB;
        $numberofmissedusers = 0;
        $extensionsgranted = 0;
        $numofgrades = 0;
        $numofzeros = 0;

        $users = SH::get_course_students_by_role($this->course, 0, $groupid, $MB->show_inactive ?? false, false);
        if (!$users){
            return null;
        }

        $ousers = $users;
        $user_extensions = [];
        $cms = SH::get_important_activities_for_users($this->course, $users);

        foreach ($cms as $cm){
            $dmm_module = static::get_dmm_by_cm($cm, $this->course);
            if (!$dmm_module || !$dmm_module->is_enabled()) continue;

            // Number of graded
            $numofgrades += $dmm_module->get_number_of_grades(array_keys($users));

            // Number of zeros.
            $numofzeros += $dmm_module->get_number_of_zeros(array_keys($users));

            foreach ($users as $index => $user){
                // Missed users.
                $timesubmit = $dmm_module->get_user_submission_time($user->id);
                $deadline = $dmm_module->get_user_effective_access($user->id);

                if (empty($timesubmit) || $timesubmit > $deadline){
                    // On-time users.
                    unset($ousers[$user->id]);

                    if ($deadline && $deadline < time()){
                        // Missed users.
                        $numberofmissedusers++;
                    }
                }

                // Extensions.
                if (!isset($user_extensions[$user->id])){
                    $user_extensions[$user->id] = static::get_number_of_extensions_in_course($user, $this->course, true);
                    $extensionsgranted += $user_extensions[$user->id];
                }
            }
        }
        // missedusers, ontimeusers, $extensionsgranted
        return [$numberofmissedusers, count($ousers), $extensionsgranted, $numofgrades, $numofzeros];
    }

    /**
     * @param $userid
     *
     * @return array|null
     */
    public function user_summary($userid){
        $numberofmissedactivities = 0;
        $numberofontimeactivities = 0;
        $numofgrades = 0;
        $numofzeros = 0;

        if (!$user = SH::get_user($userid)){
            return null;
        }

        // Fetching all modules in the course.
        $cms = SH::get_important_activities_for_users($this->course, $user);

        foreach ($cms as $cm){
            $dmm_module = static::get_dmm_by_cm($cm, $this->course);
            if (!$dmm_module || !$dmm_module->is_enabled()) continue;

            // Number of graded
            $numofgrades += $dmm_module->get_number_of_grades([$user->id]);

            // Number of zeros.
            $numofzeros += $dmm_module->get_number_of_zeros([$user->id]);

            // Missed activities.
            $timesubmit = $dmm_module->get_user_submission_time($user->id);
            $deadline = $dmm_module->get_user_effective_access($user->id);

            if (empty($timesubmit) || $timesubmit > $deadline){
                if ($deadline && $deadline < time()){
                    $numberofmissedactivities++;
                }
            } else {
                $numberofontimeactivities++;
            }
        }

        // Extensions.
        $numberofextendedactivities = static::get_number_of_extensions_in_course($user, $this->course);

        // Missed, On-Time, Extended.
        return [$numberofmissedactivities, $numberofontimeactivities, $numberofextendedactivities, $numofgrades, $numofzeros];
    }

    /**
     * @param $groupid
     *
     * @return bool
     */
    public static function is_overridden_group($groupid){
        $hasoverrides = false;

        $plugins = \core_plugin_manager::instance()->get_installed_plugins('mod');
        foreach ($plugins as $plugin => $version){
            /** @var \block_ned_teacher_tools\mod\deadline_manager_mod|string $classname */
            $classname = '\block_ned_teacher_tools\mod\deadline_manager_'.$plugin;
            if (class_exists($classname)){
                $hasoverrides = $hasoverrides || $classname::group_has_any_overrides($groupid);
            }
        }

        return $hasoverrides;
    }

    /**
     * @param $course_or_id
     *
     * @return array|string[]
     */
    public static function get_enabled_activities_by_courseid($course_or_id){
        $courseid = SH::get_id($course_or_id);
        $activities = static::g_get(__FUNCTION__, [$courseid]);
        if (is_null($activities)){
            $activities = CM::get_enabled_dm_activities($courseid);
            static::g_set(__FUNCTION__, [$courseid], $activities);
        }

        return $activities;
    }

    /**
     * @param $course_or_id
     *
     * @return array|string[]
     */
    public static function get_enabled_extensions_by_courseid($course_or_id){
        $courseid = SH::get_id($course_or_id);
        $extensions = static::g_get(__FUNCTION__, [$courseid]);
        if (is_null($extensions)){
            $extensions = CM::get_enabled_extension_activities($courseid);
            static::g_set(__FUNCTION__, [$courseid], $extensions);
        }

        return $extensions;
    }

    /**
     * init param
     */
    protected function set_enabled_activities(){
        $this->activities = static::get_enabled_activities_by_courseid($this->course);
    }

    /**
     * init param
     */
    protected function set_enabled_extensions(){
        $this->extensions = static::get_enabled_extensions_by_courseid($this->course);
    }

    /**
     * @return array|null
     */
    public function get_enabled_activities(){
        return $this->activities;
    }

    /**
     * @return array|null
     */
    public function get_enabled_extensions(){
        return $this->extensions;
    }

    /**
     * @param $cmid
     *
     * @return bool
     */
    public function is_enabled_activity($cmid){
        return static::is_cm_enabled($cmid, $this->course);
    }

    /**
     * @param $cmid
     *
     * @return bool
     */
    public function is_enabled_extension($cmid){
        return static::can_cm_use_extension($cmid, $this->course);
    }

    /**
     * @return array
     * @throws \coding_exception
     */
    public static function get_schedule_options(){
        return [
            self::SCHEDULE_NONE     => get_string('none', 'block_ned_teacher_tools'),
            self::SCHEDULE_OPTIONAL => get_string('optional', 'block_ned_teacher_tools'),
            self::SCHEDULE_FULL     => get_string('full', 'block_ned_teacher_tools'),
        ];
    }


    /**
     * Return group id for the missed schedule or false if there are no such group
     *
     * @param int|object $group_or_id - if false check all groups
     *
     * @return numeric|bool
     */
    public function has_missed_schedule($group_or_id=null){
        if (empty($this->activities)){
            return false;
        }

        $groupid = $group_or_id ? SH::get_id($group_or_id) : 0;
        if ($groupid == SH::GROUP_NONE) return false;

        [$show_inactive, $groups] =
            MB::get_showinactive_groups_group_groupid_allstudents_students_user($this->course, null, 0, 0, true);
        if (empty($groups)) return false;

        if ($groupid){
            if (empty($groups[$groupid])) return false;

            $groups = [$groupid => $groups[$groupid]];
        }

        foreach ($groups as $group){
            if (!$show_inactive && empty($group->users)) continue;
            if ($group->schedule != static::SCHEDULE_FULL) continue;

            foreach ($this->activities as $cmid){
                $dmm = static::get_dmm_by_cm($cmid, $this->course);
                if (!$dmm) continue;

                if (!$dmm->get_group_override_date($group->id)){
                    return $group->id;
                }
            }
        }

        return false;
    }

    /**
     * @param $context
     *
     * @return bool
     * @throws \coding_exception
     */
    public static function deadline_viewonly($context){
        if (has_any_capability(['block/ned_teacher_tools:view_all_groups', 'block/ned_teacher_tools:view_own_groups'], $context)){
            return false;
        } else if (has_capability('block/ned_teacher_tools:deadlineviewonly', $context)){
            return true;
        }
        return true;
    }

    /**
     * @param numeric|object $course_or_id
     * @param numeric|object $user_or_id
     * @param numeric|object $cm_or_id
     * @param bool           $load_by_course - if true, load all data by courseid, otherwise by courseid and userid
     *
     * @return array|object|null
     */
    protected static function _load_extensions($course_or_id=null, $user_or_id=null, $cm_or_id=null, $load_by_course=false){
        global $DB;
        if (!$user_or_id || !($course_or_id || $cm_or_id)){
            debugging('You can not call _load_extensions functions without userid and courseid (or cmid)');
            return $cm_or_id ? null : [];
        }

        [$courseid, $userid, $cmid] = SH::get_ids($course_or_id, $user_or_id, $cm_or_id);
        if (!$courseid){
            $courseid = SH::get_courseid_by_cmorid($cm_or_id);
        }

        $keys = [$courseid, $userid];
        if ($cmid){
            $keys[] = $cmid;
        }

        $res = static::g_get(__FUNCTION__, $keys);
        if (is_null($res)){
            $records = [];
            if ($load_by_course){
                if (!static::g_get(__FUNCTION__, [$courseid, 0, 0])){
                    // load all course data
                    $records = $DB->get_records(static::TABLE_EXTENSION, ['courseid' => $courseid]);
                    static::g_set(__FUNCTION__, [$courseid, 0, 0], true);
                }
            } else {
                if (!static::g_get(__FUNCTION__, [$courseid, $userid, 0])){
                    // load all course-user data
                    $records = $DB->get_records(static::TABLE_EXTENSION, ['courseid' => $courseid, 'userid' => $userid]);
                    static::g_set(__FUNCTION__, [$courseid, $userid, 0], true);
                }
            }

            if (!empty($records)){
                foreach ($records as $record){
                    static::g_set(__FUNCTION__, [$record->courseid, $record->userid, $record->cmid], $record);
                }

                $res = static::g_get(__FUNCTION__, $keys);
            }
        }

        if ($res){
            if (is_array($res)){
                unset($res[0]);
            }
            return $res;
        } else {
            return $cm_or_id ? null : [];
        }
    }

    /**
     * Clear static extension cache
     *
     * @param numeric|object $course_or_id
     * @param numeric|object $user_or_id (optional)
     * @param numeric|object $cm_or_id   (optional) you can provide course module to load course
     *
     * @return bool
     */
    protected static function _clear_extension_cache($course_or_id=null, $user_or_id=null, $cm_or_id=null){
        if (!$course_or_id && !$cm_or_id){
            return false;
        }

        /** @see \block_ned_teacher_tools\deadline_manager::_load_extensions() */
        $f = '_load_extensions';
        [$courseid, $userid] = SH::get_ids($course_or_id, $user_or_id);
        $course_loaded = false;
        if (!$courseid){
            $courseid = SH::get_courseid_by_cmorid($cm_or_id);
        }

        if ($userid){
            $course_loaded = static::g_get($f, [$courseid, 0, 0]);
        }
        if ($course_loaded || !$userid){
            return static::g_remove($f, [$courseid, 0, 0]);
        }

        return static::g_remove($f, [$courseid, $userid]);
    }

    /**
     * @param numeric|object $user_or_id
     * @param numeric|object $cm_or_id
     * @param numeric|object $course_or_id
     * @param bool           $load_by_course - if true, load all data by courseid, otherwise by courseid and userid
     *
     * @return int Number of extensions in the course module by user
     */
    public static function get_number_of_extensions($user_or_id, $cm_or_id, $course_or_id=null, $load_by_course=false){
        $record = static::_load_extensions($course_or_id, $user_or_id, $cm_or_id, $load_by_course);
        return $record->number ?? 0;
    }

    /**
     * Get user extension by activity
     *
     * @param numeric|object          $user_or_id
     * @param \cm_info|numeric|object $cm_or_id
     * @param numeric|object          $course_or_id
     * @param bool                    $load_by_course - if true, load all data by courseid, otherwise by courseid and userid
     *
     * @return \block_ned_teacher_tools\support\dm_extensions_record|object|false
     * @noinspection PhpReturnDocTypeMismatchInspection
     */
    public static function get_extension($user_or_id, $cm_or_id, $course_or_id=null, $load_by_course=false){
        return static::_load_extensions($course_or_id, $user_or_id, $cm_or_id, $load_by_course) ?: false;
    }

    /**
     * Remove user extension for some activity
     * This function doesn't check any capabilities or existing extensions
     *
     * @param numeric|object          $user_or_id
     * @param \cm_info|numeric|object $cm_or_id
     * @param numeric|object          $course_or_id
     */
    protected static function _delete_extension_by_params($user_or_id=null, $cm_or_id=null, $course_or_id=null){
        [$userid, $cmid, $courseid] = SH::get_ids($user_or_id, $cm_or_id, $course_or_id);
        $params = SH::sql_filter_params(['userid' => $userid, 'cmid' => $cmid, 'courseid' => $courseid]);
        if (empty($params)) return;

        SH::db()->delete_records(static::TABLE_EXTENSION, $params);
        static::_clear_extension_cache($course_or_id, $user_or_id, $cm_or_id);
    }

    /**
     * @param string                  $capability
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     * @param bool                    $doanything        - if false, ignores effect of admin role assignment
     *
     * @return bool
     */
    protected static function _has_capability($capability, $course_or_context=null, $user_or_id=null, $doanything=true){
        $ctx = SH::course2ctx($course_or_context);
        $userid = SH::get_id($user_or_id);
        $res = static::g_get(__FUNCTION__, [$capability, $ctx->id, $userid]);
        if (is_null($res)){
            $res = SH::has_capability($capability, $ctx, $userid ?: null, $doanything);
            static::g_set(__FUNCTION__, [$capability, $ctx->id, $userid], $res);
        }

        return $res;
    }

    /**
     * Remove user extension for some activity
     * This function doesn't check any capabilities or existing extensions
     *
     * @param numeric|object          $user_or_id
     * @param \cm_info|numeric|object $cm_or_id
     * @param numeric|object          $course_or_id - optional, if already loaded or for more safety
     */
    public static function delete_extension($user_or_id, $cm_or_id, $course_or_id=null){
        if (!$user_or_id || !$cm_or_id) return;

        static::_delete_extension_by_params($user_or_id, $cm_or_id, $course_or_id);
    }

    /**
     * Remove all extensions by some activity
     * This function doesn't check any capabilities or existing extensions
     *
     * @param \cm_info|numeric|object $cm_or_id
     * @param numeric|object          $course_or_id - optional, if already loaded or for more safety
     */
    public static function delete_all_extensions_by_activity($cm_or_id, $course_or_id=null){
        if (!$cm_or_id) return;

        static::_delete_extension_by_params(null, $cm_or_id, $course_or_id);
    }

    /**
     * Remove all user extension from the course
     * This function doesn't check any capabilities or existing extensions
     *
     * @param numeric|object $user_or_id
     * @param numeric|object $course_or_id
     */
    public static function delete_all_user_extensions_by_course($user_or_id, $course_or_id){
        if (!$user_or_id || !$course_or_id) return;

        static::_delete_extension_by_params($user_or_id, null, $course_or_id);
    }

    /**
     * Set new or update old extension
     *
     * @param object|numeric          $user_or_id   - student id to get extension
     * @param \cm_info|object|numeric $cm_or_id
     * @param null|numeric            $duedate      - new duedate
     * @param bool                    $set_overrule - set overrule flag, if null - don't change it
     * @param string                  $reason       - reason, shouldn't be empty
     * @param null|numeric            $duedate_orig - old duedate
     * @param int                     $cohortid     - optional extra information for the extension
     * @param object|numeric          $courseorid   - optional course object (or its id) if loaded,
     *                                              improves optimization if $cm_or_id is represented as ID
     *
     * @return object|false|null - return new extension object, false if error, or null if can't do operation
     */
    public static function set_extension($user_or_id, $cm_or_id, $duedate=null, $set_overrule=null, $reason='', $duedate_orig=null,
            $cohortid=0, $courseorid=null){
        $cm = SH::get_cm_by_cmorid($cm_or_id, $courseorid);
        $userid = SH::get_userid_or_global($user_or_id);
        $graderid = SH::get_userid_or_global();
        $context = $cm->context;

        $dmm_module = static::get_dmm_by_cm($cm, $courseorid);
        if (!$dmm_module || !$dmm_module->is_enabled()){
            return null;
        }

        if (!static::can_grader_add_extension_user_in_cm($cm, $userid, $graderid, $context, null, true, false, $courseorid)){
            return null;
        }

        if ($duedate){
            $dmm_module->set_user_override($userid, $duedate);
        } else {
            $dmm_module->delete_user_override($userid);
        }

        if (empty($reason)){
            return false;
        }

        $extension = static::get_extension($userid, $cm->id, $cm->course) ?: new \stdClass();
        $extension->courseid = $cm->course;
        $extension->cmid = $cm->id;
        $extension->modname = $cm->modname;
        $extension->instance = $cm->instance;
        $extension->overridenby = $graderid;
        $extension->userid = $userid;
        $extension->duedate = $duedate;
        $extension->overrule = $set_overrule ?? ($extension->overrule ?? false);
        $extension->reason = $reason;
        $extension->timecreated = time();

        $extension->cohortid = $cohortid;
        $extension->duedateorig = $duedate_orig;

        $groupids = SH::get_user_groupids($cm->course, $userid);
        $extension->groupid = reset($groupids) ?? 0;

        $extension->number = ($extension->number ?? 0) + 1;
        $res = static::_update_extension_record($extension);
        if (!$res){
            return false;
        }

        static::_clear_extension_cache($cm->course, $user_or_id, $cm);
        return $extension;
    }

    /**
     * Just save/update record in the DB, there are none checks here
     *
     * @param object $extension data to update
     *
     * @return bool
     */
    protected static function _update_extension_record($extension){
        if (!empty($extension->id)){
            return SH::db()->update_record(static::TABLE_EXTENSION, $extension);
        } else {
            return SH::db()->insert_record(static::TABLE_EXTENSION, $extension);
        }
    }

    /**
     * @param object|numeric $user_or_id     User ID
     * @param object|numeric $course_or_id   Course module id
     * @param bool           $load_by_course - if true, load all data by courseid, otherwise by courseid and userid
     * @param int            $lastdays - count only for some last days (num of days)
     *
     * @return int Number of records
     */
    public static function get_number_of_extensions_in_course($user_or_id, $course_or_id, $load_by_course = false, $lastdays = 0){
        $records = static::_load_extensions($course_or_id, $user_or_id, null, $load_by_course);
        if ($records) {
            if ($lastdays) {
                foreach ($records as $index => $record) {
                    if ((time() - $record->timecreated) > $lastdays * DAYSECS) {
                        unset($records[$index]);
                    }
                }
            }
            return count($records);
        }
        return 0;
    }

    /**
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     *
     * @return bool
     */
    public static function can_edit_group_deadline($course_or_context=null, $user_or_id=null){
        return static::_has_capability('manage_group_deadlines', $course_or_context, $user_or_id);
    }

    /**
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     *
     * @return bool
     */
    public static function can_edit_user_deadline($course_or_context=null, $user_or_id=null){
        return static::_has_capability('manage_user_deadlines', $course_or_context, $user_or_id);
    }

    /**
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     *
     * @return bool
     */
    public static function can_extend_group_user_enddate($course_or_context=null, $user_or_id=null){
        $ctx = SH::course2ctx($course_or_context);
        $config = SH::get_tt_block_config($ctx->instanceid);
        return !empty($config->classstudentenddateextension) &&
            static::_has_capability('extendclassstudentenddate', $ctx, $user_or_id);
    }

    /**
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     *
     * @return bool
     */
    public static function can_override_restrictions($course_or_context=null, $user_or_id=null){
        return static::_has_capability('overriderestrictions', $course_or_context, $user_or_id);
    }

    /**
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     *
     * @return bool
     */
    public static function can_view_only($course_or_context=null, $user_or_id=null){
        return static::_has_capability('deadlineviewonly', $course_or_context, $user_or_id, false);
    }

    /**
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     *
     * @return bool
     */
    public static function can_view_own_groups($course_or_context=null, $user_or_id=null){
        return static::_has_capability('view_own_groups', $course_or_context, $user_or_id);
    }

    /**
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     *
     * @return bool
     */
    public static function can_add_extension($course_or_context=null, $user_or_id=null){
        $cap_default = 'add_extension';
        $cap_sa = 'add_extension_sa';
        $check_other_capability = SH::dm_should_use_other_extension_capability($user_or_id);
        if (is_null($check_other_capability)){
            // indeterminacy, check any capability
            return static::_has_capability($cap_default, $course_or_context, $user_or_id) ||
                static::_has_capability($cap_sa, $course_or_context, $user_or_id);
        } elseif ($check_other_capability){
            return static::_has_capability($cap_sa, $course_or_context, $user_or_id);
        } else {
            return static::_has_capability($cap_default, $course_or_context, $user_or_id);
        }
    }

    /**
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     *
     * @return bool
     */
    public static function can_view_extension($course_or_context=null, $user_or_id=null){
        return static::_has_capability('view_extension_detail', $course_or_context, $user_or_id);
    }

    /**
     * @param \context|object|numeric $course_or_context - course, or its id, or context
     * @param object|numeric          $user_or_id
     *
     * @return bool
     */
    public static function can_ignore_dm_limits($course_or_context=null, $user_or_id=null){
        return static::_has_capability('ignore_max_extension_limit', $course_or_context, $user_or_id);
    }

    /**
     * @param object $group
     * @param int    $roleid
     * @param int    $duration
     *
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function sync_group_enrollments($group, $roleid, $duration){
        global $DB;

        $context = \context_course::instance($group->courseid);

        // We are looking for all users with this role assigned in this context or higher.
        [$relatedctxsql, $relatedctxparams] = $DB->get_in_or_equal($context->get_parent_context_ids(true), SQL_PARAMS_NAMED, 'relatedctx');

        $sql = "SELECT DISTINCT u.id AS userid
                  FROM {groups_members} gm
                  JOIN {user} u ON u.id = gm.userid
             LEFT JOIN {role_assignments} ra ON (ra.userid = u.id AND ra.contextid $relatedctxsql)
             LEFT JOIN {role} r ON r.id = ra.roleid
                 WHERE gm.groupid = :groupid
                   AND r.id = :roleid";

        $params = array_merge($relatedctxparams, ['groupid' => $group->id, 'roleid' => $roleid]);

        if ($members = $DB->get_records_sql($sql, $params)){
            [$sqluser, $paramuser] = $DB->get_in_or_equal(array_keys($members), SQL_PARAMS_NAMED, 'usr');

            if ($enrols = $DB->get_records('enrol', ['courseid' => $group->courseid])){
                [$sqlenrol, $paramenrol] = $DB->get_in_or_equal(array_keys($enrols), SQL_PARAMS_NAMED, 'enr');

                $enddate = $group->enddate + $duration * DAYSECS;
                $params = $paramuser + $paramenrol;
                $params['timeend'] = $enddate;

                if ($enddate > time()){
                    $sql = "UPDATE {user_enrolments} SET timeend = :timeend, status=0 WHERE userid {$sqluser} AND enrolid {$sqlenrol}";
                } else {
                    $sql = "UPDATE {user_enrolments} SET timeend = :timeend, status=1 WHERE userid {$sqluser} AND enrolid {$sqlenrol}";
                }

                $DB->execute($sql, $params);
            }
        }
    }

    /**
     * @param numeric $groupid
     * @param numeric $userid
     *
     * @return int
     */
    public function get_midterm_date($groupid=null, $userid=null){
        if (empty($groupid) && empty($userid)) return false;
        if (!$enabledactivites = $this->get_enabled_activities()) return false;

        $midterms = SH::cmids_get_midterm($this->course->id);
        $enable_midterms = array_intersect($midterms, $enabledactivites);
        $cmid = reset($enable_midterms);
        if (!$cmid) return false;

        return static::get_dmm_by_cm($cmid, $this->course)->get_override($groupid, $userid, true);
    }

    /**
     * @param $grouporid
     */
    public static function set_dm_timezone($grouporid){
        global $USER, $DB;

        $user = $DB->get_record('user', ['id' => $USER->id]);

        if ($cohort = self::get_school_cohort($grouporid, $user->id)){
            if (isset($cohort->assigned)){
                if (!$cohort->timezone){
                    $cohort->timezone = SH::NED_TIMEZONE;
                }
                if ($cohort->timezone != $user->timezone){
                    $USER->timezone = $cohort->timezone;
                    $user->timezone = $cohort->timezone;
                    user_update_user($user, false, false);
                }
            }
        }
    }

    /**
     * @param      $grouporid
     * @param null $userid
     *
     * @return mixed|void
     * @throws \dml_exception
     */
    public static function get_school_cohort($grouporid, $userid = null){
        global $USER, $DB;

        if (!$userid){
            $userid = $USER->id;
        }

        if (is_object($grouporid)){
            $group = $grouporid;
        } else {
            $group = $DB->get_record('groups', ['id' => $grouporid]);
        }

        $schoolcohort = null;

        if ($group){
            if ($cohorts = cohort_get_user_cohorts($userid)){
                foreach ($cohorts as $cohort){
                    if (strtoupper(substr($group->name, 0, 4)) == strtoupper(substr($cohort->name, 0, 4))){
                        $schoolcohort = $cohort;
                        $schoolcohort->assigned = true;
                        break;
                    }
                }
            }

            // Search.
            if (!$schoolcohort){
                $search = strtoupper(substr($group->name, 0, 4));
                $filter = $DB->sql_like('co.name', ':search', false, false);
                $params['search'] = $DB->sql_like_escape($search).'%';

                $sql = "SELECT * FROM {cohort} co  WHERE {$filter}";

                $schoolcohort = $DB->get_record_sql($sql, $params, IGNORE_MULTIPLE);
            }
        }

        return $schoolcohort;
    }

    /**
     * @param $grouporid
     * @param $cohort
     *
     * @return bool
     */
    public static function has_different_timezone_users_in_school($grouporid, $cohort){
        global $DB;

        if (is_object($grouporid)){
            $group = $grouporid;
        } else {
            $group = $DB->get_record('groups', ['id' => $grouporid]);
        }

        $sql = "SELECT cm.id, cm.userid, u.firstname, u.lastname,u.timezone
                  FROM {cohort_members} cm
                  JOIN {user} u ON cm.userid = u.id
                 WHERE cm.cohortid = ?
                   AND cm.userid IN (SELECT gm.userid
                                       FROM {groups_members} gm
                                      WHERE gm.groupid = ?)
                   AND u.timezone != ?";

        return $DB->record_exists_sql($sql, [$cohort->id, $group->id, $cohort->timezone]);
    }

    /**
     * Return some course data relative to extensions and current context:
     *  max extension per student, max extension per activity, can add extension, can ignore max extension limit
     *
     * @param object|numeric           $course_or_id
     * @param \context_course|\context $context      (optional) Uses $course context by default
     * @param object|numeric|null      $grader_or_id (optional) Uses current $USER by default
     *
     * @return array($maxextensionperstudent, $maxextensionperactivity, $canaddextension, $canignoredmlimits)
     */
    public static function get_extension_course_data($course_or_id, $context=null, $grader_or_id=null){
        $courseid = SH::get_id($course_or_id);
        $graderid = SH::get_userid_or_global($grader_or_id);

        $data = static::g_get(__FUNCTION__, [$courseid, $graderid]);
        if (is_null($data)){
            $ctx = $context ?? \context_course::instance($courseid);
            $blocksettings = SH::get_tt_block_config($courseid);
            $maxextensionperstudent = $blocksettings->maxextensionperstudent ?? 0;
            $maxextensionperactivity = $blocksettings->maxextensionperactivity ?? 0;
            $allowextentionsafterccompletion = $blocksettings->allowextentionsafterccompletion ?? 0;
            if (!$allowextentionsafterccompletion
                && !static::_has_capability('ignore_extensions_after_ccompletion_limit', $ctx, $graderid)
                && SH::is_course_final_evaluation_completed($courseid)){
                $canaddextension = false;
            } else {
                $canaddextension = static::can_add_extension($ctx, $graderid);
            }
            $canignoredmlimits = static::can_ignore_dm_limits($ctx, $graderid);

            $data = [$maxextensionperstudent, $maxextensionperactivity, $canaddextension, $canignoredmlimits];
            static::g_set(__FUNCTION__, [$courseid, $graderid], $data);
        }

        return $data;
    }

    /**
     * Return full extension data by some activity & userid
     * Function for both get_extension_course_data and get_extension_data
     * @see get_extension_course_data()
     * @see \block_ned_teacher_tools\mod\deadline_manager_mod::get_extension_data()
     *
     * @param \cm_info|int|string      $cm_or_id        - it's better to use cm_info, if you already loaded it
     * @param object|int|string        $student_or_id
     * @param object|numeric|null      $grader_or_id    (optional) Uses current $USER by default
     * @param \context_course|\context $context         (optional) Uses $course context by default
     * @param int|null                 $deadline        (optional) current deadline, if already loaded
     * @param bool                     $check_submitted (optional) if false, ignore submitted user activity or not
     * @param bool                     $load_by_course  (optional) if true, load all data by courseid, otherwise by courseid and userid
     * @param object|numeric           $courseorid      (optional) course object (or its id) if loaded,
     *                                                  improves optimization if $cm_or_id is represented as ID
     *
     * @return array($use_extension, $numberofextensions, $can_add_extension, $showextensionicon)
     */
    public static function get_activity_extension_data($cm_or_id, $student_or_id, $grader_or_id=null, $context=null,
        $deadline=null, $check_submitted=true, $load_by_course=false, $courseorid=null){

        [$cmid, $studentid, $graderid] = SH::get_ids($cm_or_id, $student_or_id, $grader_or_id);
        $keys = [$cmid, $studentid, $graderid, (int)$check_submitted];
        $res = static::g_get(__FUNCTION__, $keys);

        if (is_null($res)){
            $use_extension = $numberofextensions = $can_add_extension = $showextensionicon = null;
            $cm = SH::get_cm_by_cmorid($cm_or_id, $courseorid);
            $dm_module = static::get_dmm_by_cm($cm, $courseorid);
            if ($dm_module && $dm_module->can_use_extension()){
                $use_extension = true;
                [$maxextensionperstudent, $maxextensionperactivity, $canaddextension, $canignoredmlimits] =
                    static::get_extension_course_data($cm->course, $context, $grader_or_id);
                [$numberofextensions, $can_add_extension, $showextensionicon] =
                    $dm_module->get_extension_data($student_or_id,
                        $maxextensionperstudent, $maxextensionperactivity, $canaddextension, $canignoredmlimits,
                        $deadline, $check_submitted, $load_by_course);
            }

            $res = [$use_extension, $numberofextensions, $can_add_extension, $showextensionicon];
            static::g_set(__FUNCTION__, $keys, $res);
        }

        return $res; // array($use_extension, $numberofextensions, $can_add_extension, $showextensionicon);
    }

    /**
     * @param \core\event\base | \core\event\course_module_created $event
     */
    public static function course_module_created_event($event){
        $cmid = $event->contextinstanceid;
        if (SH::check_event_loop('DM: course module created event '.$cmid)){
            return;
        }

        // update (create) proxy deadlines, if it's proxy activity
        SH::dm_proxy_updated($cmid);
    }

    /**
     * @param \core\event\base | \core\event\course_module_updated $event
     */
    public static function course_module_updated_event($event){
        $cmid = $event->contextinstanceid;
        if (SH::check_event_loop('DM: course module updated event '.$cmid)){
            return;
        }

        // update proxy default deadlines, if it's proxy point activity
        SH::dm_point_update_proxy($cmid);

        // update time limit for quiz overrides
        if ($cm_quiz = SH::get_cm_by_cmid($cmid, $event->courseid, null, SH::QUIZ)){
            $dmq = deadline_manager_quiz::get_dmm_by_cm($cm_quiz, $event->courseid);
            if ($dmq->is_enabled()){
                $dmq->set_timelimit(null);
            }
        }
    }

    /**
     * @param object $cm
     */
    public static function course_module_pre_deleted_event($cm){
        $cmid = $cm->id;
        if (SH::check_event_loop('DM: course module pre deleted event '.$cmid)){
            return;
        }

        // delete proxy deadlines, if it's proxy point activity
        SH::dm_point_delete_proxy_deadlines($cmid, true);
    }

    /**
     * Return or render DM report object
     *
     * For the $class_data
     * @see output\deadline_manager_report::__construct()
     *
     * @param string $pagetype   - type of the page and report
     * @param array  $class_data - arguments for class constructor
     * @param bool   $render     - if true, return rendered data
     *
     * @return output\deadline_manager_report|string|null - return string if $render is true, null - if such pagetype wasn't found
     */
    public static function get_dm_report_by_pagetype($pagetype='overview', $class_data=[], $render=false){
        $classname = "\\".SH::$PLUGIN_NAME."\output\deadline_manager_report_".$pagetype;
        if (class_exists($classname)){
            /**
             * @var $report output\deadline_manager_report
             */
            $report = new $classname(...$class_data);

            if ($render){
                return SH::get_renderer()->render($report);
            }

            return $report;
        }

        return $render ? '' : null;
    }

    /**
     * Translate source
     *
     * @param string $source - one of the {@see SOURCES}
     * @param bool   $sub    - substr answer to one letter
     *
     * @return string
     */
    public static function format_source($source, $sub=false){
        $source = SH::str('dm_source_'.$source);
        return $sub ? substr($source, 0, 1) : $source;
    }

    /**
     * Format deadline to user-view format
     *
     * @param int        $deadline
     * @param int|string $dm_timezone
     *
     * @return string
     */
    public static function format_deadline($deadline, $dm_timezone=99){
        if (empty($deadline)){
            return static::format_none_deadline();
        } else {
            return userdate($deadline, SH::str('strtimeformat'), $dm_timezone);
        }
    }

    /**
     * Return empty deadline string
     *
     * @return string
     */
    public static function format_none_deadline(){
        return SH::str('dm_none_deadline');
    }

    /**
     * Get rendered proxy icon
     *
     * @return string
     */
    public static function icon_proxy(){
        return SH::render_from_template('dm_icon_proxy');
    }

    /**
     * Get rendered overruled icon
     *
     * @return string
     */
    public static function icon_overruled(){
        return SH::render_from_template('dm_icon_overruled');
    }

    /**
     * Get rendered expired icon
     *
     * @return string
     */
    public static function icon_expired(){
        return SH::render_from_template('dm_icon_expired');
    }

    /**
     * Function check module by its mod type
     *
     * @param \cm_info|object|numeric $cm_or_id
     * @param object|numeric          $course_or_id (optional) course to load course-module (if it was provided as cmid) more quickly
     *
     * @return bool true if module type is enabled {@see deadline_manager::ENABLED_MODULES}
     */
    public static function is_cm_enabled_by_type($cm_or_id, $course_or_id=null){
        $cmid = SH::get_id($cm_or_id);
        $result = static::g_get(__FUNCTION__, $cmid);
        if (is_null($result)){
            $cm = SH::get_cm_by_cmorid($cm_or_id, $course_or_id);
            $result = $cm && static::is_modtype_enabled($cm->modname);
            static::g_set(__FUNCTION__, $cmid, $result);
        }

        return $result;
    }

    /**
     * @param string $modtype
     *
     * @return bool true if current modtype is enabled
     */
    public static function is_modtype_enabled($modtype){
        return static::ENABLED_MODULES[$modtype] ?? false;
    }

    /**
     * Function create adhoc task which removing extension records from {@see deadline_manager::TABLE_EXTENSION}
     *
     * @param array $task_data          associative array with {@see \block_ned_teacher_tools\task\adhoc_delete_extensions} task parameters
     * @param bool  $ignore_other_tasks - skip tasks after current or not. Needs for hooks pre_course_delete and pre_user_delete
     */
    public static function create_delete_extension_task($task_data, $ignore_other_tasks = false){
        static $ignore_tasks = false;
        if (empty($task_data) || $ignore_tasks) return;

        $ignore_tasks = $ignore_other_tasks;
        $task = new \block_ned_teacher_tools\task\adhoc_delete_extensions();
        $task->set_custom_data($task_data);
        \local_ned_controller\task\base_adhoc_task::just_add_new_job(0, $task, false);
    }
}
