<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Form activity_settings_changer
 *
 * @package    block_ned_teacher_tools
 * @subpackage NED
 * @copyright  2020 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools\form;
use \block_ned_teacher_tools\shared_lib as SH;

defined('MOODLE_INTERNAL') || die();

use block_ned_teacher_tools as NED;

/** @var \stdClass $CFG */
require_once($CFG->libdir.'/formslib.php');
require_once($CFG->dirroot . '/course/modlib.php');
require_once($CFG->libdir.'/gradelib.php');

/**
 * Form update_scripts
 *
 * @package    block_ned_teacher_tools
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class activity_settings_changer extends \moodleform {

    const MOD_ASSIGN = 'assign';
    const MOD_QUIZ = 'quiz';
    const MOD_FORUM = 'forum';
    const MODS = [self::MOD_ASSIGN, self::MOD_QUIZ, self::MOD_FORUM];

    const COMPLETION_TRACKING_NOT_CHANGE = -1;
    const COMPLETION_KEYS = [
        'completion', 'completionview', 'completionusegrade', 'completiongradeitemnumber', 'completionexpected', 'completionsubmit',
        'completionpass', 'completionpassgroup', 'completionpostsenabled', 'completionposts', 'completionpostsgroup',
        'completiondiscussionsenabled', 'completionattemptsexhausted', 'completiondiscussions', 'completiondiscussionsgroup',
        'completionrepliesenabled', 'completionreplies', 'completionrepliesgroup',
    ];
    const TAG_KEYS = ['tags'];

    const DEPENDENCE_KEYS = [
        'completiondiscussionsenabled' => 'completiondiscussions',
        'completionrepliesenabled' => 'completionreplies',
        'completionpostsenabled' => 'completionposts',
    ];

    protected $_data = false;
    protected $_courses = null;
    protected $_cms = null;


    public function definition() {
        $data  = $this->_customdata;
        $mform = $this->_form;
        $mform->_attributes['class'] = str_replace('\\', ' ', __CLASS__) . ' ned-form';
        if ($data['class'] ?? false){
            $mform->_attributes['class'] .= ' '.$data['class'];
        }

        if (isset($data['hiddens'])){
            foreach ($data['hiddens'] as $key => $item){
                $mform->addElement('hidden', $key, $item);
                $mform->setType($key, PARAM_TEXT);
            }
        }

        $mform->addElement('tags', 'have_tags', NED\str('havealltags'), ['itemtype' => 'course_modules', 'component' => 'core']);
        $mform->addElement('tags', 'havent_tags', NED\str('nonetags'), ['itemtype' => 'course_modules', 'component' => 'core']);

        $categorieslist = \core_course_category::make_categories_list();
        $def_cats = ['0' => get_string('all')];
        $mform->addElement('select', 'coursecategory', NED\str('locatedincoursecategory'), $def_cats + $categorieslist);

        $mod_group = [];
        foreach (static::MODS as $modname){
            $mod_group[] = $mform->createElement('advcheckbox', $modname, get_string('modulename', $modname));
        }
        $mform->addGroup($mod_group, 'mod_group', NED\str('applythesesettings'),
            \html_writer::div('', 'br'), false);

        foreach (static::MODS as $modname){
            $this->add_static($modname.'_header', get_string('modulename', $modname), 2);
            $this->add_activity_completion($modname);
            $this->add_tags($modname);
        }

        $this->add_action_buttons(true, NED\str('applysettings'));
    }

    /**
     * This method is called after definition(), data submission and set_data().
     * All form setup that is dependent on form values should go in here.
     */
    function definition_after_data() {
        parent::definition_after_data();
        $mform =& $this->_form;
        $mods = array_combine(static::MODS, static::MODS);

        foreach ($mform->_elements as $element) {
            $ename = $element->getName();
            $prefix = explode('_', $ename, 2)[0] ?? '';
            if (isset($mods[$prefix]) && $ename != $prefix){
                $mform->hideIf($ename, $prefix, 'eq', 0);
            }
        }
    }

    /**
     * @return \local_ned_controller\stdClass2|object|null
     */
    public function get_data(){
        if ($this->_data === false){
            $data = parent::get_data();

            if ($data){
                $mod_data = array_fill_keys(static::MODS, []);
                foreach ($data as $key => $item){
                    $expl_key = explode('_', $key, 2);
                    $mod = $expl_key[0] ?? '';
                    $mod_key = $expl_key[1] ?? '';
                    if ($mod && $mod_key && isset($mod_data[$mod])){
                        $mod_data[$mod][$mod_key] = $item;
                    }
                }

                $data->data = [];
                foreach ($mod_data as $mod => $mod_datum){
                    foreach (static::DEPENDENCE_KEYS as $dependence => $key){
                        if (isset($mod_datum[$dependence]) && empty($mod_datum[$dependence])){
                            $mod_datum[$key] = 0;
                        }
                    }
                    $data->data[$mod] = SH::stdClass2($mod_datum);
                }
                $data = SH::stdClass2($data);
            }
            $this->_data = $data;
        }

        return $this->_data;
    }

    /* NEW FUNCTIONS */

    /**
     * Method hideIf doesn't work with standard 'static' element, so we insert in a group
     *
     * @param     $name
     * @param     $text
     * @param int $h
     */
    public function add_static($name, $text, $h=0){
        $group = [];
        if ($h){
            $text = \html_writer::tag('h'.$h, $text);
        }
        $group[] =& $this->_form->createElement('static', $name.'_text', '', $text);
        $this->_form->addGroup($group, $name, '', '', false);
    }

    /**
     * Activity completion
     * see course/moodleform_mod.php \core_grades\component_gradeitems\moodleform_mod->standard_coursemodule_elements()
     *
     * @param $modname
     */
    public function add_activity_completion($modname){
        $mform = $this->_form;
        $n = function($name) use ($modname){
              return $modname.'_'.$name;
        };
        $str = function($identifier, $a=null){
            return get_string($identifier, 'completion', $a);
        };

        $this->add_static($n('activitycompletionheader'), $str('activitycompletion'), 4);

        $trackingdefault = static::COMPLETION_TRACKING_NOT_CHANGE;
        $mform->addElement('select', $n('completion'), $str('completion'), [
                static::COMPLETION_TRACKING_NOT_CHANGE => NED\str('completion_not_change'),
                COMPLETION_TRACKING_NONE => $str('completion_none'),
                COMPLETION_TRACKING_MANUAL => $str('completion_manual')
            ]);
        $mform->setDefault($n('completion'), $trackingdefault);
        $mform->addHelpButton($n('completion'), 'completion', 'completion');

        $this->add_static($n('completed_warning'), \html_writer::tag('i', NED\str('completedwarningtext')));
        $mform->hideIf($n('completed_warning'), $n('completion'), 'eq', static::COMPLETION_TRACKING_NOT_CHANGE);

        // Automatic completion once you view it
        $gotcompletionoptions = false;
        if (plugin_supports('mod', $modname, FEATURE_COMPLETION_TRACKS_VIEWS, false)) {
            $mform->addElement('advcheckbox', $n('completionview'), $str('completionview'),
                $str('completionview_desc'));
            $mform->hideIf($n('completionview'), $n('completion'), 'ne', COMPLETION_TRACKING_AUTOMATIC);
            // Check by default if automatic completion tracking is set.
            if ($trackingdefault == COMPLETION_TRACKING_AUTOMATIC) {
                $mform->setDefault($n('completionview'), 1);
            }
            $gotcompletionoptions = true;
        }

        if (plugin_supports('mod', $modname, FEATURE_GRADE_HAS_GRADE, false)) {
            // This activity supports grading.
            $gotcompletionoptions = true;

            $component = 'mod_'.$modname;
            $itemnames = \core_grades\component_gradeitems::get_itemname_mapping_for_component($component);

            if (count($itemnames) === 1) {
                // Only one gradeitem in this activity.
                // We use the completionusegrade field here.
                $mform->addElement('advcheckbox', $n('completionusegrade'),
                    $str('completionusegrade'), $str('completionusegrade_desc')
                );
                $mform->hideIf($n('completionusegrade'), $n('completion'), 'ne', COMPLETION_TRACKING_AUTOMATIC);
                $mform->addHelpButton($n('completionusegrade'), 'completionusegrade', 'completion');
            } elseif (count($itemnames) > 1) {
                // There are multiple grade items in this activity.
                // Show them all.
                $options = ['' => $str('activitygradenotrequired')];
                foreach ($itemnames as $itemnumber => $itemname) {
                    $options[$itemnumber] = get_string("grade_{$itemname}_name", $component);
                }

                $mform->addElement(
                    'select',
                    $n('completiongradeitemnumber'),
                    $str('completionusegrade'),
                    $options
                );
                $mform->hideIf($n('completiongradeitemnumber'), $n('completion'), 'ne', COMPLETION_TRACKING_AUTOMATIC);
            }
        }

        // Automatic completion according to module-specific rules
        $customcompletionelements = $this->add_completion_rules($modname);
        foreach ($customcompletionelements as $element) {
            $mform->hideIf($element, $n('completion'), 'ne', COMPLETION_TRACKING_AUTOMATIC);
        }

        $gotcompletionoptions = $gotcompletionoptions || count($customcompletionelements)>0;

        // Automatic option only appears if possible
        if ($gotcompletionoptions) {
            $mform->getElement($n('completion'))->addOption($str('completion_automatic'), COMPLETION_TRACKING_AUTOMATIC);
        }

        // Completion expected at particular date? (For progress tracking)
        $mform->addElement('date_time_selector', $n('completionexpected'), $str('completionexpected'), ['optional' => true]);
        $mform->addHelpButton($n('completionexpected'), 'completionexpected', 'completion');
        $mform->hideIf($n('completionexpected'), $n('completion'), 'eq', COMPLETION_TRACKING_NONE);
        $mform->hideIf($n('completionexpected'), $n('completion'), 'eq', static::COMPLETION_TRACKING_NOT_CHANGE);
    }

    /**
     * Tags
     * see course/moodleform_mod.php \core_grades\component_gradeitems\moodleform_mod->standard_coursemodule_elements()
     *
     * @param $modname
     */
    public function add_tags($modname){
        $mform = $this->_form;
        $n = function($name) use ($modname){
            return $modname . '_' . $name;
        };
        $str = function($identifier, $a = null){
            return get_string($identifier, 'tag', $a);
        };

        if (\core_tag_tag::is_enabled('core', 'course_modules')) {
            $this->add_static($n('tagshdr'), $str('tags'), 4);
            $mform->addElement('advcheckbox', $n('set_tags'), NED\str('changetags'), NED\str('changetags_desc'));
            $mform->addElement('tags', $n('tags'), get_string('tags'),
                array('itemtype' => 'course_modules', 'component' => 'core'));
            $mform->hideIf($n('tags'), $n('set_tags'), 'eq', 0);
        }
    }

    /**
     * Render & return form as html
     *
     * @return string
     */
    public function draw() {
        //finalize the form definition if not yet done
        if (!$this->_definition_finalized) {
            $this->_definition_finalized = true;
            $this->definition_after_data();
        }

        return $this->_form->toHtml();
    }

    /**
     * Add error text to the form
     *
     * @param $text
     */
    public function error($text){
        $this->_form->addElement('html', \html_writer::div(
            \html_writer::span(get_string('error').': ', 'error-label').
            \html_writer::span($text, 'error-text'),
            'error'));
    }

    /**
     * Do standard action for this form
     *
     * @param \stdClass $data
     * @param bool $print
     *
     * @return bool
     */
    public function do_action($data=null, $print=true){
        $p = function($text, $h=0) use ($print){
            if (!$print){
                return;
            }
            if(!NO_OUTPUT_BUFFERING){
                ob_flush();
                flush();
            }
            if ($h){
                echo \html_writer::tag('h'.$h, $text);
            } else {
                echo \html_writer::div($text);
            }
        };

        $data = $data ?? $this->get_data();
        $cms = $this->get_cms($data);
        if (empty($cms)){
            return false;
        }

        $courses = $this->get_data_courses();
        $course_count = count($courses);
        $cms_count = count($cms);
        $p(NED\str('coursecmforupdate', ['courses' => $course_count, 'cms' => $cms_count]));
        $p('<br>');
        $progressbar = new \progress_bar('activity_settings_changer_'.time(), 500, true);
        $a = array(
            'count' => $cms_count,
            'done'  => 0,
        );

        foreach ($cms as $cm){
            $this->update_cm($cm, $data, $courses[$cm->course] ?? null);
            $a['done']++;
            $a['name'] = $cm->name;
            $progressbar->update($a['done'], $a['count'], NED\str('updateactivity', $a));
        }

        return true;
    }

    /**
     * Update single course module
     *
     * @param      $cm
     * @param      $formdata
     * @param null $course
     */
    public function update_cm($cm, $formdata, $course=null){
        global $COURSE;
        $mod_data = $formdata->data[$cm->modname] ?? null;
        if (empty($mod_data)){
            return;
        }

        $course = $course ?? $this->get_data_courses($cm->course);
        $_course = null;
        if ($course){
            // cheat next create form and update functions, which use global $COURSE
            $_course = $COURSE;
            $COURSE = $course;
        }

        list($cm, $context, $module, $data, $cw) = get_moduleinfo_data($cm, $course);
        $mformclassname = '\\'.NED\PLUGIN_NAME.'\\form\\mod_'.$module->name.'_mod_form';
        /** @var mod_assign_mod_form | mod_quiz_mod_form | mod_assign_mod_form | \moodleform_mod $mform */
        $mform = new $mformclassname($data, $cw->section, $cm, $course);
        $mform->set_data($data);
        $upd_data = $mform->exportValues();
        $upd_data->completionunlocked = 1; // need to activate some checks in data_postprocessing
        $mform->data_postprocessing($upd_data);

        if (isset($data->type) && !isset($upd_data->type)){
            $upd_data->type = $data->type;
        }
        $upd_data->cmidnumber = $cm->id;

        if ($mod_data->completion != static::COMPLETION_TRACKING_NOT_CHANGE){
            $upd_data->completionunlocked = 1;
            foreach (static::COMPLETION_KEYS as $key){
                if (isset($mod_data->$key)){
                    $skip = false;
                    switch($key){
                        case 'completiongradeitemnumber':
                            // see lib/completionlib.php: 668
                            if (!is_null($mod_data->$key)){
                                $item = \grade_item::fetch([
                                    'courseid' => $cm->course, 'itemtype' => 'mod',
                                    'itemmodule' => $cm->modname, 'iteminstance' => $cm->instance,
                                    'itemnumber' => $mod_data->completiongradeitemnumber
                                ]);
                                if (!$item){
                                    $upd_data->$key = null;
                                    $skip = true;
                                    break;
                                }
                            }
                            break;
                    }
                    if ($skip){
                        continue;
                    }
                    $upd_data->$key = $mod_data->$key;
                } elseif(isset($upd_data->$key)){
                    unset($upd_data->$key);
                }
            }
        } else {
            $upd_data->completionunlocked = 0;
        }

        if ($mod_data->set_tags){
            foreach (static::TAG_KEYS as $key){
                if (isset($mod_data->$key)){
                    $upd_data->$key = $mod_data->$key;
                } elseif(isset($upd_data->$key)){
                    unset($upd_data->$key);
                }
            }
        }

        // if it will be an error with one cm, it shouldn't break all others
        try {
            update_moduleinfo($cm, $upd_data, $course);
        } catch (\Exception $ex){
            // detect active db transactions, rollback and log as error
            abort_all_db_transactions();
            $info = get_exception_info($ex);
            $message = NED\str('cmcourse_error',
                ['cmname' => $cm->name, 'cmid' => $cm->id, 'coursename' => $COURSE->fullname, 'courseid' => $COURSE->id]) .
                "\n".$info->message;
            echo NED\fatal_error($message, $info->moreinfourl, null, $info->backtrace, $info->debuginfo, true);
        }

        if ($_course){
            // return real $COURSE
            $COURSE = $_course;
        }
    }

    /**
     * @param null $data
     *
     * @return array
     */
    public function get_cms($data=null){
        global $DB, $CFG;
        if (is_null($this->_courses)){
            $data = $data ?? $this->get_data();
            $where = [];
            $params = [];
            $join = [];
            $tagfield = empty($CFG->keeptagnamecase) ? 'name' : 'rawname';

            $mods = [];
            $select_name = [];
            foreach (static::MODS as $mod){
                if(!$data->$mod){
                    continue;
                }
                $mods[] = $mod;
                $l = $mod[0];
                $join[] = "
                LEFT JOIN {{$mod}} $l
                    ON m.name = '$mod'
                    AND $l.id = cm.instance
                ";
                $select_name[] = "$l.name";
            }
            if(empty($mods)){
                return [];
            }
            $select_name = join(", ", $select_name);
            list($mods_sql, $mods_params) = $DB->get_in_or_equal($mods, SQL_PARAMS_NAMED);
            $params = array_merge($params, $mods_params);
            $where[] = 'm.name ' . $mods_sql;

            if($data->coursecategory ?? false){
                $course_ids = array_keys($this->get_data_courses());
                list($course_sql, $course_params) = $DB->get_in_or_equal($course_ids, SQL_PARAMS_NAMED);
                $params = array_merge($params, $course_params);
                $where[] = 'cm.course ' . $course_sql;
            }

            if($data->have_tags){
                list($have_tags_sql, $have_tags_params) = $DB->get_in_or_equal($data->have_tags, SQL_PARAMS_NAMED);
                $params = array_merge($params, $have_tags_params);
                $params['tiitemtype_1'] = 'course_modules';
                $params['ticomponent_1'] = 'core';
                $params['have_need_tags_count'] = count($data->have_tags);
                $join[] = "
                    LEFT JOIN (
                        SELECT tg.id, ti.itemid AS itemid, COUNT(tg.id) AS tag_count 
                        FROM {course_modules} cm
                        LEFT JOIN {tag_instance} ti
                            ON ti.itemid = cm.id
                            AND ti.itemtype = :tiitemtype_1
                            AND ti.component = :ticomponent_1
                        LEFT JOIN {tag} tg 
                            ON tg.id = ti.tagid
                        WHERE tg.$tagfield $have_tags_sql
                        GROUP BY ti.itemid
                    ) AS have_need_tags
                    ON have_need_tags.itemid = cm.id
                ";
                $where[] = 'have_need_tags.tag_count = :have_need_tags_count';
            }

            if($data->havent_tags){
                list($havent_tags_sql, $havent_tags_params) = $DB->get_in_or_equal($data->havent_tags, SQL_PARAMS_NAMED);
                $params = array_merge($params, $havent_tags_params);
                $params['tiitemtype_2'] = 'course_modules';
                $params['ticomponent_2'] = 'core';
                $join[] = "
                    LEFT JOIN (
                        SELECT tg.id, ti.itemid AS itemid 
                        FROM {tag} tg 
                        LEFT JOIN {tag_instance} ti
                            ON ti.tagid = tg.id
                            AND ti.itemtype = :tiitemtype_2
                            AND ti.component = :ticomponent_2
                        LEFT JOIN {course_modules} cm
                            ON cm.id = ti.itemid
                        WHERE tg.$tagfield $havent_tags_sql
                        GROUP BY ti.itemid
                    ) AS have_forbidden_tags
                    ON have_forbidden_tags.itemid = cm.id
                ";
                $where[] = 'have_forbidden_tags.id IS NULL';
            }

            $sql = "SELECT cm.*, m.name AS `modname`, 
                  CONCAT_WS(',', $select_name) AS `name`
                FROM {course_modules} cm
                LEFT JOIN {modules} m
                  ON m.id = cm.module  
            ";
            $where = !empty($where) ? ("\nWHERE (" . join(') AND (', $where) . ')') : '';
            $join = !empty($join) ? join("\n", $join) : '';
            $groupby = "\nGROUP BY cm.id";
            $orderby = "\nORDER BY cm.course, cm.id";

            $this->_cms = $DB->get_records_sql($sql . $join . $where . $groupby . $orderby, $params) ?: [];
        }

        return $this->_cms;
    }

    /**
     * Return all courses from form data, or only one course by id
     *
     * @param null $id
     *
     * @return array | \core_course_list_element | \stdClass | null
     */
    public function get_data_courses($id=null){
        if (is_null($this->_courses)){
            $data = $this->get_data();
            if (!isset($data->coursecategory)){
                return $id ? null : [];
            }
            if ($data->coursecategory){
                $courseids =
                    \core_course_category::get($data->coursecategory, MUST_EXIST, true)->
                    get_courses(['recursive' => true, 'idonly' => true]);
                /** @var int $courseid */
                foreach ($courseids as $courseid){
                    $this->_courses[$courseid] = get_course($courseid);
                }
            } else {
                $this->_courses = get_courses();
            }

        }
        return $id ? ($this->_courses[$id] ?? null) : $this->_courses;
    }

    /**
     * See add_completion_rules function in *_mod_form forms
     *
     * @param $modname
     *
     * @return array
     */
    public function add_completion_rules($modname){
        $mform = $this->_form;
        $items = [];

        $n = function($name) use ($modname){
            return $modname.'_'.$name;
        };
        $str = function($identifier, $a=null) use ($modname){
            return get_string($identifier, $modname, $a);
        };

        switch ($modname){
            case static::MOD_ASSIGN:
                $mform->addElement('advcheckbox', $n('completionsubmit'), '', $str('completionsubmit'));
                // Enable this completion rule by default.
                $mform->setDefault($n('completionsubmit'), 1);
                $items[] = $n('completionsubmit');
                break;
            case static::MOD_QUIZ:
                $group = array();
                $group[] = $mform->createElement('advcheckbox', $n('completionpass'), null, $str('completionpass'),
                    array('group' => 'cpass'));
                $mform->disabledIf($n('completionpass'), $n('completionusegrade'), 'notchecked');
                $group[] = $mform->createElement('advcheckbox', $n('completionattemptsexhausted'), null,
                    $str('completionattemptsexhausted'),
                    array('group' => 'cattempts'));
                $mform->disabledIf($n('completionattemptsexhausted'), $n('completionpass'), 'notchecked');
                $mform->addGroup($group, $n('completionpassgroup'), $str('completionpass'), ' &nbsp; ', false);
                $mform->addHelpButton($n('completionpassgroup'), 'completionpass', 'quiz');
                $items[] = $n('completionpassgroup');
                break;
            case static::MOD_FORUM:
                $group = array();
                $group[] =& $mform->createElement('advcheckbox', $n('completionpostsenabled'), '', $str('completionposts'));
                $group[] =& $mform->createElement('text', $n('completionposts'), '', array('size'=>3));
                $mform->setType($n('completionposts'),PARAM_INT);
                $mform->setDefault($n('completionposts'), 1);
                $mform->addGroup($group, $n('completionpostsgroup'), $str('completionpostsgroup'), array(' '), false);
                $mform->disabledIf($n('completionposts'),$n('completionpostsenabled'),'notchecked');

                $group = array();
                $group[] =& $mform->createElement('advcheckbox', $n('completiondiscussionsenabled'), '', $str('completiondiscussions'));
                $group[] =& $mform->createElement('text', $n('completiondiscussions'), '', array('size'=>3));
                $mform->setType($n('completiondiscussions'),PARAM_INT);
                $mform->setDefault($n('completiondiscussions'), 1);
                $mform->addGroup($group, $n('completiondiscussionsgroup'), $str('completiondiscussionsgroup'), array(' '), false);
                $mform->disabledIf($n('completiondiscussions'),$n('completiondiscussionsenabled'),'notchecked');

                $group = array();
                $group[] =& $mform->createElement('advcheckbox', $n('completionrepliesenabled'), '', $str('completionreplies'));
                $group[] =& $mform->createElement('text', $n('completionreplies'), '', array('size'=>3));
                $mform->setType($n('completionreplies'),PARAM_INT);
                $mform->setDefault($n('completionreplies'), 1);
                $mform->addGroup($group, $n('completionrepliesgroup'), $str('completionrepliesgroup'), array(' '), false);
                $mform->disabledIf($n('completionreplies'),$n('completionrepliesenabled'),'notchecked');

                $items = array_merge($items,
                    [$n('completiondiscussionsgroup'),$n('completionrepliesgroup'),$n('completionpostsgroup')]);
                break;
        }

        return $items;
    }
}
