<?php
/**
 * user_extension
 *
 * @package    block_ned_teacher_tools
 * @category   deadline_manager
 * @copyright  2019 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools\form;

defined('MOODLE_INTERNAL') || die();

use block_ned_teacher_tools\deadline_manager as DM;
use block_ned_teacher_tools as NED;
use block_ned_teacher_tools\shared_lib as SH;
use block_ned_teacher_tools\utils;

/** @var \stdClass $CFG */
require_once($CFG->libdir.'/formslib.php');

/**
 * user_extension form
 *
 * @package    block_ned_teacher_tools
 */
class user_extension extends \moodleform {
    /** @var \block_ned_teacher_tools\deadline_manager_entity */
    public $dm_entity;
    /** @var \cm_info */
    public $cm;
    public $userid;

    /**
     * The constructor function calls the abstract function definition() and it will then
     * process and clean and attempt to validate incoming data.
     *
     * @param mixed $action the action attribute for the form. If empty defaults to auto detect the
     *              current url. If a moodle_url object then outputs params as hidden variables.
     * @param mixed $customdata if your form defintion method needs access to data such as $course
     *              $cm, etc. to construct the form definition then pass it in this array. You can
     *              use globals for somethings.
     * @param string $method if you set this to anything other than 'post' then _GET and _POST will
     *               be merged and used as incoming data to the form.
     * @param string $target target frame for form submission. You will rarely use this. Don't use
     *               it if you don't need to as the target attribute is deprecated in xhtml strict.
     * @param mixed $attributes you can pass a string of html attributes here or an array.
     *               Special attribute 'data-random-ids' will randomise generated elements ids. This
     *               is necessary when there are several forms on the same page.
     *               Special attribute 'data-double-submit-protection' set to 'off' will turn off
     *               double-submit protection JavaScript - this may be necessary if your form sends
     *               downloadable files in response to a submit button, and can't call
     *               \core_form\util::form_download_complete();
     * @param bool $editable
     * @param array $ajaxformdata Forms submitted via ajax, must pass their data here, instead of relying on _GET and _POST.
     */
    public function __construct($action=null, $customdata=null, $method='post', $target='', $attributes=null, $editable=true, $ajaxformdata=null){
        $this->userid = $customdata['userid'];
        $cmid = $customdata['cmid'];
        $cm = SH::get_cm_by_cmid($cmid, null, $this->userid);
        if ($cm){
            $this->cm = $cm;
            $this->dm_entity = DM::get_dm_entity($cm->course, null, $this->userid);
        } else {
            $this->error('Wrong form data!');
        }

        parent::__construct($action, $customdata, $method, $target, $attributes, $editable, $ajaxformdata);
    }


    /**
     * Form definition
     */
    public function definition() {
        if (empty($this->cm)) return;

        $data  = $this->_customdata;
        $context = $data['context'];
        $add_new =  $data['add_new'] ?? false;

        $mform = $this->_form;
        $mform->_attributes['class'] = $mform->_attributes['class'] ?? '';
        $mform->_attributes['class'].= ' '.SH::$PLUGIN_NAME;

        $blocksettings = SH::get_tt_block_config($this->cm->course);
        if (!($blocksettings->enabledeadlineextension ?? false)){
            $this->error(SH::str('activitynotdm'));
            return;
        }

        $dma = $this->dm_entity->get_dm_activity($this->cm->id);
        if (!$dma->is_enabled()){
            $this->error(SH::str('activitynotdm'));
            return;
        }

        [, $numberofextensions, $can_add_extension] = $dma->get_activity_extension_data();
        if ($numberofextensions > 0 && !$add_new){
            if (!DM::can_view_extension($context)){
                $this->error(SH::str('notallowed'));
                return;
            }

            $this->show_extension($blocksettings);
        } else {
            if (!$can_add_extension){
                $this->error(SH::str('notallowed'));
                return;
            }

            $this->add_extension($blocksettings, $numberofextensions);
        }
    }

    /**
     * Just show some info about extension
     *
     * @param \stdClass $blocksettings
     */
    public function show_extension($blocksettings){
        global $OUTPUT, $DB;
        $extension = DM::get_extension($this->userid, $this->cm);
        if (!$extension){
            $this->error('no extension');
            return;
        }

        $mform = $this->_form;
        $dma = $this->dm_entity->get_dm_activity($this->cm->id);

        $maxextensionperactivity = $blocksettings->maxextensionperactivity ?? 0;
        $maxextensionperactivity_txt = $maxextensionperactivity ?: SH::str('unlimited');

        $mform->addElement('static', 'extension',
            SH::span(SH::str('activityextension'), 'red'),
            "{$extension->number}/$maxextensionperactivity_txt");

        $deadline = DM::format_deadline($extension->duedate, $this->dm_entity->dm_timezone);
        if ($dma->is_overruled(DM::SOURCE_EXTENSION)){
            $deadline .= DM::icon_overruled();
        }
        $mform->addElement('static', 'extendedduedate', SH::span(SH::str('extendedduedate'), 'red'), $deadline);

        $original_source = $dma->get_previous_source(DM::SOURCE_EXTENSION);
        $original_deadline = $extension->duedateorig ?? $dma->get_deadline($original_source);
        $mform->addElement('static', 'originalduedate', SH::span(SH::str('originalduedate'), 'red'),
            DM::format_deadline($original_deadline, $this->dm_entity->dm_timezone) . " (".DM::format_source($original_source).")");

        $mform->addElement('html', SH::span(SH::str('reason'), 'bold'));
        $user_overridenby = $DB->get_record('user', ['id' => $extension->overridenby]);
        $userpicture = $OUTPUT->user_picture($user_overridenby, ['size' => 35]);

        $template = SH::div_start('comment-message');
        $template .= SH::div_start('comment-message-meta m-r-3');
        $template .= SH::span($userpicture, 'picture');
        $template .= SH::span(SH::q_user_link($this->userid)) . ' - ';
        $template .= SH::span(userdate($extension->timecreated), 'time');
        $template .= SH::div_end(); // .comment-message-meta
        $template .= SH::div($extension->reason, 'text');
        $template .= SH::div_end(); // .comment-message
        $mform->addElement('html', $template);
    }

    /**
     * Show form to add new (or edit existing) extension
     *
     * @param \stdClass $blocksettings
     * @param int $numberofextensions
     */
    public function add_extension($blocksettings, $numberofextensions=0){
        $mform = $this->_form;
        $dma = $this->dm_entity->get_dm_activity($this->cm->id);

        $maxextensionperactivity = $blocksettings->maxextensionperactivity ?? 0;
        $maxextensionperactivity_txt = $maxextensionperactivity ?: SH::str('unlimited');
        $numberofextensions++;

        $timezone = '';
        if ($schoolid = SH::get_user_school($this->userid, true)) {
            $school = new \local_schoolmanager\school($schoolid);
        }
        if (!empty($school) && $cohort = $school->get_cohort()) {
            $timezone = $cohort->timezone;
        }

        $mform->addElement('static', 'extension_i', SH::span(SH::str('activityextension')),
            "{$numberofextensions}/$maxextensionperactivity_txt");

        $mform->addElement('hidden', 'userid', $this->userid);
        $mform->setType('userid', PARAM_INT);

        $mform->addElement('hidden', 'cmid', $this->cm->id);
        $mform->setType('hidden', PARAM_INT);

        $mform->addElement('hidden', 'duedateorig', $dma->get_deadline($dma->get_previous_source(DM::SOURCE_EXTENSION)));
        $mform->setType('duedateorig', PARAM_INT);

        $mform->addElement('hidden', 'timezone', $timezone);
        $mform->setType('timezone', PARAM_RAW);

        $dma->form_add_element($mform, DM::SOURCE_EXTENSION);

        $mform->addElement('textarea', 'reason', SH::str('reason'),
            ['wrap' => 'virtual',  'rows' => "5", 'cols' => "50",
                'placeholder' => SH::str('reasonforextension') . '...']);
        // all checks are server now, also see validation()
        $mform->addRule('reason', get_string('required'), 'required', get_string('required'), 'server');

        $extension = DM::get_extension($this->userid, $this->cm);
        if ($extension){
            $this->set_data($extension);
        }
    }

    /**
     * If there are errors return array of errors ("fieldname"=>"error message"),
     * otherwise true if ok.
     * Server side rules do not work for uploaded files, implement serverside rules here if needed.
     *
     * WARNING: As this form load by js, you should double your checks also in externallib and throw errors from it
     * @see block_ned_teacher_tools_external::submit_user_extension_form()
     *
     * @param array $data array of ("fieldname"=>value) of submitted data
     * @param array $files array of uploaded files "element_name"=>tmp_file_path
     *
     * @return array of "element_name"=>"error_description" if there are errors,
     *         or an empty array if everything is OK (true allowed for backwards compatibility too).
     */
    public function validation($data, $files) {
        $errors = parent::validation($data, $files);
        $wrong_form_data = function() use (&$errors){
            $errors['duedate'] = SH::str('error_wrongformdata');
            return $errors;
        };

        if (empty($this->cm) || empty($this->userid)){
            return $wrong_form_data();
        }

        $this->dm_entity->form_load_data($data, DM::SOURCE_EXTENSION, true);
        $deadline_errors = $this->dm_entity->form_get_validation(DM::SOURCE_EXTENSION, true);
        $form_dl_errors = array_intersect_key($deadline_errors, $data);
        if (!empty($form_dl_errors)){
            return array_merge($errors, $form_dl_errors);
        }

        return $errors;
    }

    /**
     * Render & return form as html
     *
     * @return string
     */
    public function draw() {
        //finalize the form definition if not yet done
        if (!$this->_definition_finalized) {
            $this->_definition_finalized = true;
            $this->definition_after_data();
        }

        return $this->_form->toHtml();
    }

    /**
     * Show text error
     *
     * @param $text - plain error text
     */
    public function error($text){
        $this->_form->addElement('html', SH::div(
            SH::span(get_string('error').': ', 'error-label').
            SH::span($text, 'error-text'),
            'error'));
    }

}
