<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Deadline Manager Header
 *
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */


namespace block_ned_teacher_tools\mod;
use block_ned_teacher_tools\deadline_manager as DM;
use block_ned_teacher_tools\grading_tracker as GT;
use block_ned_teacher_tools\shared_lib as SH;
use block_ned_teacher_tools as TT;
use \local_ned_controller\support\ned_dm_deadline_record;

defined('MOODLE_INTERNAL') || die();

/**
 * Class deadline_manager_mod
 * @package block_ned_teacher_tools\mod
 */
abstract class deadline_manager_mod {
    const TABLE = 'mod_overrides';
    const MODNAME = 'mod';
    const FIELD_MOD = 'modid';
    const FIELD_TIME = 'time';
    const FIELD_OVERRULE = 'overrule';

    /**
     * @var static[] List of deadline_manager_mods for cache
     */
    static protected $_deadline_manager_mod = [];
    /**
     * @var \stdClass[] List of override records
     */
    static protected $_overrides = [];

    /**
     * @var \cm_info
     */
    protected $_cm;
    /**
     * @var int
     */
    protected $_courseid;
    /**
     * @var object - DB mod record
     */
    protected $_instance;
    /**
     * @var object - object of mod
     */
    protected $_module;

    /**
     * @var bool|null
     */
    protected $_enabled = null;

    /**
     * @var bool|null
     */
    protected $_can_use_extension = null;

    /**
     * @var bool|null
     */
    protected $_is_proxy = null;

    /**
     * deadline_manager_mod constructor
     *
     * @param \cm_info $cm
     *
     * @throws \dml_exception
     */
    public function __construct(\cm_info $cm){
        global $DB;

        $this->_cm = $cm;
        $this->_courseid = $cm->course;
        $this->_instance = $DB->get_record($cm->modname, array('id' => $cm->instance), '*', MUST_EXIST);
        static::$_deadline_manager_mod[$cm->id] = &$this;
    }

    /**
     * Check and update (or delete) connected NGC records
     *
     * @param int $userid
     */
    protected function _check_ngc_records($userid){
        if (empty($userid)) return;

        $NGC = SH::$ned_grade_controller;
        $NGC::check_and_fix_records_by_params($this->_cm->id ?? 0, $userid, ['reason' => $NGC::REASON_SUBMISSION]);
    }

    /**
     * Check and update (or delete) connected GT records
     *
     * @param int $userid
     */
    protected function _check_gt_records($userid){
        if (empty($userid)) return;

        $gt_records = GT::get_record_by_cmid_userid($this->_cm, $userid, true);
        if (!empty($gt_records)){
            GT::update_gt_records($gt_records);
        }
    }

    /**
     * Checked, can this course module use extension
     * NOTE: Here we check none capabilities, only cm settings
     *
     * @return bool
     */
    public function can_use_extension(){
        if (is_null($this->_can_use_extension)){
            $this->_can_use_extension = DM::can_cm_use_extension($this->_cm, $this->_cm->course);
        }

        return $this->_can_use_extension && $this->is_enabled();
    }

    /**
     * @param \cm_info|int|string $cm_or_id
     * @param object|numeric      $courseorid - Optional course object (or its id) if loaded, improves optimization if $cm_or_id is represented as ID
     *
     * @return static
     */
    public static function get_dmm_by_cm($cm_or_id, $courseorid=null){
        $cmid = SH::get_id($cm_or_id);
        if (!isset(static::$_deadline_manager_mod[$cmid])){
            $cm = SH::get_cm_by_cmorid($cm_or_id, $courseorid);
            return new static($cm);
        }
        return static::$_deadline_manager_mod[$cmid];
    }

    /**
     * Return delays for the base deadline values
     *
     * @return array [field => delay]
     */
    static public function get_delay_array(){
        return [static::FIELD_TIME => 0];
    }

    /**
     * @return int|null
     */
    public function get_default_deadline(){
        return $this->_instance->{static::FIELD_TIME} ?: null;
    }

    /**
     * Get time of submission by user
     *
     * @param numeric|object|null $user_or_id - optional user, by whom submitted time need, uses global USER by default
     *
     * @return int|false
     */
    public function get_user_submission_time($user_or_id){
        return SH::get_submitted_time_by_cm($this->_cm, $user_or_id);
    }

    /**
     * Return instance object to update by moodle
     * Update $this->instance with all necessary data, before call this method
     *
     * @return object
     */
    public function process_mod_instance_before_update(){
        $instance = $this->_instance;
        $instance->coursemodule = $this->_cm->id;
        $instance->instance = $this->_instance->id;
        $instance->modulename = $this->_cm->modname;

        $deadline = $instance->{static::FIELD_TIME};
        $deadline_delays = static::get_delay_array();
        foreach ($deadline_delays as $field => $delay){
            $instance->$field = $deadline ? ($deadline + $delay) : 0;
        }

        return $instance;
    }

    /**
     * @param int $deadline - false will remove deadline
     *
     * @return bool
     */
    public function set_default_deadline($deadline=null){
        $deadline = $deadline ?: 0;
        do {
            if ($this->get_default_deadline() == $deadline) break;

            $deadline_delays = static::get_delay_array();
            foreach ($deadline_delays as $field => $delay){
                $this->_instance->$field = $deadline ? ($deadline + $delay) : 0;
            }

            if (!SH::db()->update_record($this->_cm->modname, $this->_instance)) break;
/*
            $course = $this->cm->get_course();
            list($cm, $context, $module, $data, $cw) = SH::get_moduleinfo_data($this->cm, $course);

//            $deadline = $instance->{static::FIELD_TIME};
            $deadline_delays = static::get_delay_array();
            foreach ($deadline_delays as $field => $delay){
                $this->instance->$field = $deadline ? ($deadline + $delay) : 0;
            }

            SH::update_moduleinfo($cm, $data, $course);
            return true;

            $modname = $this->cm->modname;
            if (!SH::require_file("/mod/$modname/lib.php")){
                break;
            }
            $updateinstancefunction = $modname."_update_instance";
            if (!function_exists($updateinstancefunction)){
                break;
            }

            $this->instance->{static::FIELD_TIME} = $deadline;
            $new_instance = $this->process_mod_instance_before_update();

            if (!$updateinstancefunction($new_instance, null)){
                break;
            }

            $this->instance = $new_instance;
*/
            \core\event\course_module_updated::create_from_cm($this->_cm, $this->_cm->context)->trigger();

            return true;
        } while (false);

        return false;
    }

    /**
     * Alias for @see \block_ned_teacher_tools\mod\deadline_manager_mod::set_default_deadline
     *
     * @return bool
     */
    public function remove_default_deadline(){
        return $this->set_default_deadline(false);
    }

    /**
     * @return mixed
     */
    public function get_activity_name(){
        return $this->_instance->name;
    }

    /**
     * @return \moodle_url
     * @throws \moodle_exception
     */
    public function get_activity_url(){
        $url = new \moodle_url("/mod/{$this->_cm->modname}/view.php", array('id' => $this->_cm->id));
        return $url;
    }

    /**
     * @return \moodle_url
     */
    public function get_activity_icon_url(){
        return $this->_cm->get_icon_url();
    }

    /**
     * Return all activity deadlines (and overrides)
     *
     * @param bool $include_default - (optional) if true, include record with id '0' with default activity deadline
     * @param bool $include_instanceid - (optional) if true, include into the records field "instanceid"
     *                                      (which will be the same for all records)
     *
     * @return object|object[]|ned_dm_deadline_record|ned_dm_deadline_record[]
     */
    public function get_all_mod_deadlines($include_default=false, $include_instanceid=false){
        $field_deadline = static::FIELD_TIME;
        $fields = ["id", "groupid", "userid", "$field_deadline AS deadline"];
        if ($include_instanceid){
            $fields[] = static::FIELD_MOD." AS instanceid";
        }

        $res = [];
        if ($include_default){
            $def = (object)['id' => 0, 'groupid' => null, 'userid' => null, 'deadline' => $this->get_default_deadline()];
            if ($include_instanceid){
                $def->instanceid = $this->_instance->id;
            }
            $res[0] = $def;
        }

        $params = [static::FIELD_MOD => $this->_instance->id];
        $res += SH::db()->get_records(static::TABLE, $params, '', join(', ', $fields));

        return $res;
    }

    /**
     * Set new deadlines by the records from the get_all_mod_deadlines() of other module
     *
     * @param object[]|ned_dm_deadline_record[]     $records
     * @param int                                   $add_delay - seconds to add for deadlines from the records
     * @param bool                                  $remove_old - if true, remove all old overrides
     *
     * @return bool
     */
    public function set_deadlines_by_records($records, $add_delay=0, $remove_old=false){
        if ($remove_old){
            $this->delete_all_overrides(false);
        }

        // default deadline for activity will be with id = 0
        $default = null;
        foreach ($records as $record){
            if (!$record->id){
                $default = $record;
                continue;
            }

            $deadline = $record->deadline;
            $set_unlimited = !$deadline;
            $deadline += $add_delay;

            $this->set_override($record->groupid, $record->userid, $deadline, $record->overrule ?? null, $set_unlimited);
        }

        if ($default){
            $deadline = $default->deadline ? ($default->deadline + $add_delay) : 0;
            $this->set_default_deadline($deadline);
        } elseif ($remove_old){
            $this->remove_default_deadline();
        }

        return true;
    }

    /**
     * Return overrides (group if $group, user otherwise)
     *
     * @param bool $group
     *
     * @return array
     */
    public function get_list_overrides($group=true){
        global $DB;

        $select_field = $group ? 'groupid' : 'userid';
        $field_mod = static::FIELD_MOD;
        $field_deadline = static::FIELD_TIME;
        $fields = "id, $select_field, $field_deadline AS deadline";
        $select = "$field_mod = :instanceid AND $select_field <> ''";
        $params = ['instanceid' => $this->_instance->id];

        return $DB->get_records_select(static::TABLE, $select, $params, '', $fields);
    }

    /**
     * @return array
     */
    public function get_group_overrides(){
        return $this->get_list_overrides(true);
    }

    /**
     * @return array
     */
    public function get_user_overrides(){
        return $this->get_list_overrides(false);
    }

    /**
     * Return override record by $groupid or $userid, or override date, if $date = true
     *
     * @param numeric $groupid
     * @param numeric $userid
     * @param bool    $get_date
     *
     * @return \stdClass|int|null - return null, if don't find override
     */
    public function get_override($groupid=null, $userid=null, $get_date=false){
        global $DB;
        if (!$groupid && !$userid){
            return null;
        }

        if (!isset(static::$_overrides[$this->_cm->id][$groupid][$userid])){
            $params = [static::FIELD_MOD => $this->_instance->id];
            if ($userid){
                $params['userid'] = $userid;
            } else {
                $params['groupid'] = $groupid;
            }

            static::$_overrides[$this->_cm->id][$groupid][$userid] = $DB->get_record(static::TABLE, $params) ?: null;
        }

        $record = static::$_overrides[$this->_cm->id][$groupid][$userid];

        if ($get_date){
            return $record->{static::FIELD_TIME} ?? null;
        }
        return $record;
    }

    /**
     * Return override record by $groupid or $userid, or override date, if $date = true
     *
     * @param numeric $groupid
     * @param numeric $userid
     *
     * @return bool
     */
    public function get_overrule($groupid=null, $userid=null){
        $record = $this->get_override($groupid, $userid);
        return $record->{static::FIELD_OVERRULE} ?? false;
    }

    /**
     * @param $groupid
     *
     * @return \stdClass|null
     */
    public function get_group_override($groupid){
        return $this->get_override($groupid);
    }
    /**
     * @param $userid
     *
     * @return \stdClass|int|null
     */
    public function get_user_override($userid){
        return $this->get_override(null, $userid);
    }

    /**
     * @param $groupid
     *
     * @return int|null
     */
    public function get_group_override_date($groupid){
        return $this->get_override($groupid, null, true);
    }

    /**
     * @param $userid
     *
     * @return int
     */
    public function get_user_override_date($userid){
        return $this->get_override(null, $userid, true);
    }

    /**
     * Get user deadline including all sources
     *
     * @param numeric|object $user_or_id
     *
     * @return int|null - return null, if find nothing
     */
    public function get_user_final_deadline($user_or_id){
        $userid = SH::get_id($user_or_id);
        $user_override = $this->get_user_override_date($userid);
        if (!is_null($user_override)){
            return $user_override;
        }

        // Check group overrides.
        $group_override = $this->get_user_group_deadline($userid);
        if (!is_null($group_override)){
            return $group_override;
        }

        return $this->get_default_deadline();
    }

    /**
     * Get group deadline (override or activity default)
     *
     * @param numeric|object $group_or_id
     *
     * @return int|null - return null, if find nothing
     */
    public function get_group_final_deadline($group_or_id){
        $groupid = SH::get_id($group_or_id);

        // Check group overrides.
        $group_override = $this->get_group_override_date($groupid);
        if (!is_null($group_override)){
            return $group_override;
        }

        return $this->get_default_deadline();
    }

    /**
     * Update override record and return its id
     *
     * @param object $override - new override record
     *
     * @return int - override id
     */
    protected function _update_override_object($override){
        $mid = $this->_cm->modname.'id';
        $params = [
            'context' => $this->_cm->context,
            'other' => [
                $mid => $this->_cm->instance,
            ]
        ];

        if ($override->id ?? false){
            SH::db()->update_record(static::TABLE, $override);
        } else {
            $override->id = SH::db()->insert_record(static::TABLE, $override);
        }

        if (empty($override->id)){
            return 0;
        }

        // Determine which override updated event to fire.
        $params['objectid'] = $override->id;
        $event_class = "\\mod_{$this->_cm->modname}\\event\\";
        if ($override->groupid ?? false){
            $params['other']['groupid'] = $override->groupid;
            $event_class .= "group_override_updated";
        } else {
            $params['relateduserid'] = $override->userid;
            $event_class .= "user_override_updated";
        }

        if (class_exists($event_class)){
            // Trigger the override updated event.
            $event_class::create($params)->trigger();
        }

        return $override->id;
    }

    /**
     * Return override object, if insert/update required, true if record was deleted,
     *  false if nothing changed and null if can do nothing
     *
     * @param int|null  $groupid
     * @param int|null  $userid
     * @param int|null  $time - if zero, and $set_unlimited is false - remove override
     * @param bool      $set_overrule - set overrule flag, if null - don't change it
     * @param bool      $set_unlimited - ignore $time and set it to 0
     *
     * @return bool|null|object - if result is object, you can continue override process
     */
    protected function _prepare_override($groupid=null, $userid=null, $time=null, $set_overrule=null, $set_unlimited=false){
        if (!$groupid && !$userid){
            return null;
        }

        if ($set_unlimited){
            $time = 0;
        } elseif (!$time){
            if (!$this->delete_override($groupid, $userid)){
                return null;
            }
            return true;
        }

        $new = $change = false;
        $override = $this->get_override($groupid, $userid);
        if (!$override){
            $override = new \stdClass();
            $override->{static::FIELD_MOD} = $this->_cm->instance;
            if ($groupid){
                $override->groupid = $groupid;
            } else {
                $override->userid = $userid;
            }
            $override->{static::FIELD_OVERRULE} = $set_overrule ?: false;
            $new = true;
        } else {
            if (!is_null($set_overrule) && $override->{static::FIELD_OVERRULE} != $set_overrule){
                $override->{static::FIELD_OVERRULE} = $set_overrule;
                $change = true;
            }
        }

        $deadline_delays = static::get_delay_array();
        foreach ($deadline_delays as $key => $delay){
            $new_time = $time ? ($time+$delay) : 0;
            if ($new || $override->$key != $new_time){
                $override->$key = $new_time;
                $change = true;
            }
        }

        if (!$new && !$change){
            return false;
        }

        return $override;
    }

    /**
     * Process $override in mod classes
     * It should also call @see _update_override_object() inside
     *
     * @param object $override
     *
     * @return object - processed override
     */
    abstract protected function _process_override($override);
    /**
     * Return override id, if insert/update was successful,
     *  true if record was deleted, false if nothing to change, or null if could do nothing
     *
     * @param int|null  $groupid
     * @param int|null  $userid
     * @param int|null  $time - if zero, and $set_unlimited is false - remove override
     * @param bool      $set_overrule - set overrule flag, if null - don't change it
     * @param bool      $set_unlimited - ignore $time and set it to 0
     *
     * @return bool|int|null
     */
    public function set_override($groupid=null, $userid=null, $time=null, $set_overrule=null, $set_unlimited=false){
        if ($groupid && $userid){
            $groupid = null;
        }
        if (SH::check_event_loop(['DM: set_override ', $this->_cm->id, $groupid, $userid])){
            return false;
        }

        $override = $this->_prepare_override($groupid, $userid, $time, $set_overrule, $set_unlimited);
        if (!is_object($override)){
            return $override;
        }

        $override = $this->_process_override($override);
        if (empty($override->id)){
            return false;
        }

        $this->_check_ngc_records($userid);
        $this->_check_gt_records($userid);
        SH::dm_point_update_proxy($this->_cm, $groupid, $userid, $time, false);

        return $override->id;
    }

    /**
     * @param int       $groupid
     * @param int|null  $time - if zero, and $set_unlimited is false - remove override
     * @param bool      $set_overrule - set overrule flag, if null - don't change it
     * @param bool      $set_unlimited - ignore $time and set it to 0
     *
     * @return bool|int|null
     */
    public function set_group_override($groupid, $time, $set_overrule=null, $set_unlimited=false){
        return $this->set_override($groupid, null, $time, $set_overrule, $set_unlimited);
    }

    /**
     * @param int       $userid
     * @param int|null  $time - if zero, and $set_unlimited is false - remove override
     * @param bool      $set_overrule - set overrule flag, if null - don't change it
     * @param bool      $set_unlimited - ignore $time and set it to 0
     *
     * @return bool|int|null
     */
    public function set_user_override($userid, $time, $set_overrule=null, $set_unlimited=false){
        return $this->set_override(null, $userid, $time, $set_overrule, $set_unlimited);
    }

    /**
     * Process override deleting in mod classes
     * Called from the @see delete_override()
     *
     * @param object $override
     *
     * @return bool - true if $override was deleted
     */
    abstract protected function _process_delete_override($override);

    /**
     * Return true if record was deleted, false if couldn't find it
     *
     * @param int|null $groupid
     * @param int|null $userid
     *
     * @return bool
     */
    public function delete_override($groupid=null, $userid=null){
        if (SH::check_event_loop(['DM: delete_override ', $this->_cm->id, $groupid, $userid])){
            return false;
        }

        $override = $this->get_override($groupid, $userid);
        if (!$override){
            return false;
        }

        if (!$this->_process_delete_override($override)){
            return false;
        }

        unset(static::$_overrides[$this->_cm->id][$groupid][$userid]);
        $this->_check_ngc_records($userid);
        $this->_check_gt_records($userid);
        SH::dm_point_update_proxy($this->_cm, $groupid, $userid, 0, true);

        return true;
    }

    /**
     * Process all overrides deleting in mod classes
     * Called from the @see delete_all_overrides()
     *
     * @return bool
     */
    abstract protected function _process_delete_all_overrides();

    /**
     * Deletes all overrides from the database and clears any corresponding calendar events
     * Also deleted all extensions
     *
     * @param bool $including_default - if true, remove default deadline too
     *
     * @return bool - success
     */
    public function delete_all_overrides($including_default=false){
        if (SH::check_event_loop('DM: delete_all_overrides '.$this->_cm->id)){
            return false;
        }

        if ($including_default){
            $this->remove_default_deadline();
        }

        if (!$this->_process_delete_all_overrides()){
            return false;
        }

        unset(static::$_overrides[$this->_cm->id]);
        $this->delete_all_extensions();

        SH::dm_point_delete_proxy_deadlines($this->_cm, $including_default);

        return true;
    }

    /**
     * @param $groupid
     *
     * @return bool
     */
    public function delete_group_override($groupid){
        return $this->delete_override($groupid);
    }

    /**
     * @param $userid
     *
     * @return bool
     */
    public function delete_user_override($userid){
        return $this->delete_override(null, $userid);
    }

    /**
     * @return array
     */
    public function get_mod_deadlines(){
        return array(
            'mod' => $this->get_default_deadline(),
            'groupoverrides' => $this->get_group_overrides(),
            'useroverrides' => $this->get_user_overrides()
        );
    }

    /**
     * @param $userid
     * @return mixed
     */
    abstract public function get_user_effective_access($userid);

    /**
     * @return mixed
     */
    abstract public static function cron();

    /**
     * @param $groupid
     *
     * @return bool
     */
    public static function group_has_any_overrides($groupid){
        global $DB;
        return $DB->record_exists(static::TABLE, ['groupid' => $groupid]);
    }

    /**
     * @param $userid
     * @param $courseid
     *
     * @return bool
     */
    public static function user_has_any_overrides($userid, $courseid){
        global $DB;
        $table = static::TABLE;
        $mod_table = static::MODNAME;
        $field = static::FIELD_MOD;
        $sql = "SELECT override.*
                  FROM {$table} override
                  JOIN {$mod_table} mod
                    ON override.$field = mod.id
                 WHERE override.userid = :userid
                   AND mod.course = :courseid";

        return $DB->record_exists_sql($sql, ['userid' => $userid, 'courseid' => $courseid]);
    }

    /**
     * Return true, if user have already submitted (it doesn't check grade)
     *
     * @param numeric|object|null   $user_or_id - optional user, by whom submitted time need, uses global USER by default
     *
     * @return bool
     */
    function is_submitted($user_or_id=null){
        return SH::get_submitted_time_by_cm($this->_cm, $user_or_id, false) > 0;
    }

    /**
     * @param $userid
     * @return string|null
     * @throws \dml_exception
     */
    public function get_grade($userid){
        global $DB;

        $sql = "SELECT gg.id, gg.finalgrade, gg.rawgrademax
                  FROM {grade_items} gi 
                  JOIN {grade_grades} gg 
                    ON gi.id = gg.itemid
                 WHERE gi.itemtype = 'mod' 
                   AND gi.itemmodule = ?
                   AND gi.iteminstance = ? 
                   AND gg.userid = ?";

        if ($grade = $DB->get_record_sql($sql, [$this->_cm->modname, $this->_cm->instance, $userid])){
            $gradeval = grade_floatval($grade->finalgrade);

            if (!is_null($gradeval)){
                $gradestr = round($gradeval, 2).'/'.round($grade->rawgrademax, 2);
                return $gradestr;
            }
        }

        return null;
    }

    /**
     * @param $userids
     * @return int
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public function get_number_of_grades($userids){
        global $DB;

        if (empty($userids)){
            return 0;
        }

        list($insql, $params) = $DB->get_in_or_equal($userids, SQL_PARAMS_NAMED);
        $params['itemmodule'] = $this->_cm->modname;
        $params['iteminstance'] = $this->_cm->instance;

        $sql = "SELECT COUNT(1) 
                  FROM {grade_items} gi 
                  JOIN {grade_grades} gg 
                    ON gi.id = gg.itemid
                 WHERE gi.itemtype = 'mod' 
                   AND gi.itemmodule = :itemmodule
                   AND gi.iteminstance = :iteminstance 
                   AND gg.userid {$insql}
                   AND gg.finalgrade IS NOT NULL";

        return $DB->count_records_sql($sql, $params);
    }

    /**
     * @param $userids
     * @return int
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public function get_number_of_zeros($userids){
        global $DB;

        if (empty($userids)){
            return 0;
        }

        list($insql, $params) = $DB->get_in_or_equal($userids, SQL_PARAMS_NAMED);
        $params['itemmodule'] = $this->_cm->modname;
        $params['iteminstance'] = $this->_cm->instance;

        $sql = "SELECT COUNT(1) 
                  FROM {grade_items} gi 
                  JOIN {grade_grades} gg 
                    ON gi.id = gg.itemid
                 WHERE gi.itemtype = 'mod' 
                   AND gi.itemmodule = :itemmodule
                   AND gi.iteminstance = :iteminstance 
                   AND gg.userid {$insql}
                   AND gg.finalgrade = 0";

        return $DB->count_records_sql($sql, $params);
    }


    /**
     * @param numeric $userid
     *
     * @return string|null
     */
    public function get_grade_url($userid=null){
        return null;
    }

    /**
     * Return minimum group deadline by user id
     * @param $userid
     *
     * @return int|null - return null, if don't find override
     */
    public function get_user_group_deadline($userid){
        $group_ids = SH::get_user_groupids($this->_courseid, $userid);
        if (count($group_ids) > 1) {
            $course = $this->_cm->get_course();
            list($allstudents, $groups) = TT\get_users_and_groups($course, true, true);
        }
        $group_deadlines = [];
        foreach ($group_ids as $groupid){
            if (!$groupid) continue;
            if (isset($groups[$groupid]) && empty($groups[$groupid]->users)) {
                continue;
            }
            $temp = $this->get_group_override_date($groupid);
            if ($temp !== false){
                $group_deadlines[]= $temp;
            }
        }

        return SH::max($group_deadlines);
    }

    /**
     * @return bool
     */
    public function is_enabled(){
        if (is_null($this->_enabled)){
            $this->_enabled = DM::is_cm_enabled($this->_cm, $this->_courseid);
        }

        return $this->_enabled;
    }

    /**
     * Return extension data by current module & userid
     *
     * Extension rules:
     * • grader must have capability to add it
     * • student must have non-empty deadline
     * • student must not submit yet
     *
     * Grader also must have capability to ignore extensions limits,
     *  or next rules also applied:
     * + student get on course extensions less than $maxextensionperstudent
     * + student get in activity extensions less than $maxextensionperactivity
     * + Current deadline should pass, but not more than {@see \local_ned_controller\shared\C::EXTENSION_DELAY}
     *
     * Note: if you update this logic, also update/check extension-icon method
     *  {@see \local_ned_controller\shared\grade_util::can_add_dm_extension_to_ned_grade_icon()}
     *
     * @param $user_or_id
     * @param $maxextensionperstudent
     * @param $maxextensionperactivity
     * @param $canaddextension
     * @param $canignoredmlimits
     * @param int|null $deadline       - current deadline, if already loaded
     * @param bool $check_submitted    - if false, ignore submitted user activity or not
     * @param bool $load_by_course     - if true, load all data by courseid, otherwise by courseid and userid
     *
     * @return array($numberofextensions, $can_add_extension, $showextensionicon)
     */
    public function get_extension_data($user_or_id, $maxextensionperstudent, $maxextensionperactivity, $canaddextension, $canignoredmlimits,
        $deadline=null, $check_submitted=true, $load_by_course=false){
        $can_add_extension = $canaddextension;
        $userid = SH::get_id($user_or_id);
        $cmid = $this->_cm->id;

        if (!$this->can_use_extension()){
            $numberofextensions = $can_add_extension = $showextensionicon = null;
            return array($numberofextensions, $can_add_extension, $showextensionicon);
        }

        $numberofextensions = DM::get_number_of_extensions($userid, $cmid, $this->_courseid, $load_by_course);

        if ($can_add_extension && $numberofextensions > 0){
            if (!$canignoredmlimits){
                if ($maxextensionperactivity && $numberofextensions >= $maxextensionperactivity){
                    $can_add_extension = false;
                }
            }
        }

        do {
            if (!$can_add_extension){
                break;
            }
            // OK, but now we should pass all other tests
            $can_add_extension = false;

            if (is_null($deadline)){
                $deadline = $this->get_user_final_deadline($user_or_id);
            }

            if (!$deadline){
                break;
            }

            // Submitted?
            if ($check_submitted){
                if ($this->is_submitted($userid)){
                    break;
                }
            }

            // If we can ignore limits, than we don't need to check them
            if ($canignoredmlimits){
                $can_add_extension = true;
                break;
            }

            // Check maximum extension number.
            if (!empty($maxextensionperstudent)
                && $maxextensionperstudent <= DM::get_number_of_extensions_in_course($userid, $this->_courseid, $load_by_course)){
                break;
            }

            $now = time();
            $passed = $now > ($deadline + SH::EXTENSION_DELAY);
            if ($passed){
                break;
            }

            // Done
            $can_add_extension = true;
        } while(false);

        $showextensionicon = $can_add_extension || ($numberofextensions > 0);
        return array($numberofextensions, $can_add_extension, $showextensionicon);
    }

    /**
     * Get user extension
     * Alias for @see \block_ned_teacher_tools\deadline_manager::get_extension()
     *
     * @param numeric|object $user_or_id
     * @param bool           $load_by_course - if true, load all data by courseid, otherwise by courseid and userid
     *
     * @return \block_ned_teacher_tools\support\dm_extensions_record|object|false
     */
    public function get_extension($user_or_id, $load_by_course=false){
        return DM::get_extension($user_or_id, $this->_cm, $this->_cm->course, $load_by_course);
    }

    /**
     * Remove user extension
     * This function doesn't check any capabilities or existing extensions
     * Alias for @see \block_ned_teacher_tools\deadline_manager::delete_extension()
     *
     * @param numeric|object $user_or_id
     */
    public function delete_extension($user_or_id){
        DM::delete_extension($user_or_id, $this->_cm, $this->_cm->course);
    }

    /**
     * Remove all activity extensions
     * This function doesn't check any capabilities or existing extensions
     * Alias for @see \block_ned_teacher_tools\deadline_manager::delete_all_extensions_by_activity()
     */
    public function delete_all_extensions(){
        DM::delete_all_extensions_by_activity($this->_cm, $this->_cm->course);
    }


    /**
     * It checks proxy submission plugin
     *
     * @return bool
     */
    public function is_proxy(){
        if (is_null($this->_is_proxy)){
            $this->_is_proxy = SH::dm_is_proxy_activity($this->_cm);
        }

        return $this->_is_proxy;
    }
}