<?php
/**
 * Override external assign API from mod_assign
 *
 * @package    block_ned_teacher_tools
 * @subpackage NED
 * @copyright  2020 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools;
use local_ned_controller\mod_assign\assign as ned_assign;

defined('MOODLE_INTERNAL') || die;

/** @var \stdClass $CFG */
require_once("$CFG->dirroot/mod/assign/externallib.php");

class mod_assign_external extends \mod_assign_external {
    static protected $_replace_assign = false;

    /**
     * Generate a warning in a standard structure for a known failure.
     *
     * @param int $assignmentid - The assignment
     * @param string $warningcode - The key for the warning message
     * @param string $detail - A description of the error
     *
     * @return array - Warning structure containing item, itemid, warningcode, message
     */
    protected static function generate_warning($assignmentid, $warningcode, $detail): array{
        $warningmessages = array(
            'couldnotlock'=>'Could not lock the submission for this user.',
            'couldnotunlock'=>'Could not unlock the submission for this user.',
            'couldnotsubmitforgrading'=>'Could not submit assignment for grading.',
            'couldnotrevealidentities'=>'Could not reveal identities.',
            'couldnotgrantextensions'=>'Could not grant submission date extensions.',
            'couldnotrevert'=>'Could not revert submission to draft.',
            'invalidparameters'=>'Invalid parameters.',
            'couldnotsavesubmission'=>'Could not save submission.',
            'couldnotsavegrade'=>'Could not save grade.'
        );

        $message = $warningmessages[$warningcode];
        if (empty($message)) {
            $message = 'Unknown warning type.';
        }

        return array('item' => s($detail),
            'itemid'=>$assignmentid,
            'warningcode'=>$warningcode,
            'message'=>$message);
    }

    /**
     * Utility function for validating an assign.
     *
     * @param int $assignid assign instance id
     * @return array array containing the assign, course, context and course module objects
     * @since  Moodle 3.2
     */
    protected static function validate_assign($assignid): array{
        if (!static::$_replace_assign){
            return parent::validate_assign($assignid);
        }

        global $DB;

        // Request and permission validation.
        $assign = $DB->get_record('assign', array('id' => $assignid), 'id', MUST_EXIST);
        [$course, $cm] = get_course_and_cm_from_instance($assign, 'assign');

        $context = \context_module::instance($cm->id);
        // Please, note that is not required to check mod/assign:view because is done by validate_context->require_login.
        static::validate_context($context);
        // Use our class instead moodle assign
        $assign = new ned_assign($context, $cm, $course);

        return array($assign, $course, $cm, $context);
    }

    /**
     * Submit the logged in users assignment for grading.
     *
     * @param int $assignmentid The id of the assignment
     * @param int $userid The id of the user the submission belongs to.
     * @param string $jsonformdata The data from the form, encoded as a json array.
     * @return array of warnings to indicate any errors.
     * @since Moodle 3.1
     */
    public static function submit_grading_form($assignmentid, $userid, $jsonformdata) {
        global $CFG, $USER;

        require_once($CFG->dirroot . '/mod/assign/locallib.php');
        require_once($CFG->dirroot . '/mod/assign/gradeform.php');

        $params = static::validate_parameters(static::submit_grading_form_parameters(),
            array(
                'assignmentid' => $assignmentid,
                'userid' => $userid,
                'jsonformdata' => $jsonformdata
            ));

        /** @var ned_assign $assignment */
        static::$_replace_assign = true;
        [$assignment, $course, $cm, $context] = static::validate_assign($params['assignmentid']);
        static::$_replace_assign = false;

        $serialiseddata = json_decode($params['jsonformdata']);

        $data = array();
        parse_str($serialiseddata, $data);

        $warnings = array();

        $options = array(
            'userid' => $params['userid'],
            'attemptnumber' => $data['attemptnumber'],
            'rownum' => 0,
            'gradingpanel' => true
        );

        if (WS_SERVER) {
            // Assume form submission if coming from WS.
            $USER->ignoresesskey = true;
            $data['_qf__mod_assign_grade_form_'.$params['userid']] = 1;
        }

        $customdata = (object) $data;
        $formparams = array($assignment, $customdata, $options);

        // Data is injected into the form by the last param for the constructor.
        $mform = new \mod_assign_grade_form(null, $formparams, 'post', '', null, true, $data);
        $validateddata = $mform->get_data();

        if ($validateddata) {
            $assignment->save_grade($params['userid'], $validateddata);
        } else {
            $warnings[] = static::generate_warning($params['assignmentid'],
                'couldnotsavegrade',
                'Form validation failed.');
        }

        return $warnings;
    }
}