<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Deadline Manager Header
 *
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools\output;
use block_ned_teacher_tools as NED;
use block_ned_teacher_tools\shared_lib as SH;
use block_ned_teacher_tools\deadline_manager as DM;
use block_ned_teacher_tools\output\menu_bar as MB;

defined('MOODLE_INTERNAL') || die();

/**
 * Class deadline_manager_header
 *
 * @package block_ned_teacher_tools\output
 * @property-read string $pagetype;
 * @property-read \stdClass $course;
 * @property-read \stdClass $group;
 * @property-read \stdClass $user;
 * @property-read array $pageparams;
 * @property-read \context|\context_course $context;
 */
class deadline_manager_header implements \renderable, \templatable {

    /** @var \string $_pagetype */
    protected $_pagetype;
    /** @var \stdClass $_course */
    protected $_course;
    /** @var \stdClass $_group */
    protected $_group;
    /** @var \stdClass $_user */
    protected $_user;
    /** @var array $_pageparams */
    protected $_pageparams;
    /** @var \context|\context_course $_context */
    protected $_context;

    protected $groupid;
    protected $show_inactive;

    /**
     * deadline_manager_header constructor.
     *
     * @param       $pagetype
     * @param       $course
     * @param       $group
     * @param       $user
     * @param array $pageparams
     * @param null  $context
     */
    public function __construct($pagetype, $course, $group, $user, $pageparams=[], $context=null) {
        $this->_course = $course;
        $this->_pagetype = $pagetype;
        $this->_group = isset($group->id) ? $group :null;
        $this->_user = isset($user->id) ? $user : null;
        $this->_pageparams = $pageparams ?: [];
        $this->_context = $context ?? \context_course::instance($course->id);
    }

    public function __get($name){
        $name = '_' . $name;
        return property_exists($this, $name) ? $this->$name : null;
    }

    /**
     * Export page contents for template
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output) {
        $ctx = $this->_context;
        $course = $this->_course;

        $d = function($date){
            return $date ? date('j M Y', $date) : '';
        };
        $users_by_role = function($name, $groupid) use ($ctx){
            return implode(', ', DM::get_role_users_fullnames($name, $ctx, $groupid, true));
        };

        $blocksettings = block_ned_teacher_tools_get_block_config($course->id);
        $managegroupdeadlines = DM::can_edit_group_deadline($ctx);
        $manageuserdeadlines = DM::can_edit_user_deadline($ctx);

        $user_page = false;
        switch ($this->_pagetype){
            case DM::PAGE_USER:
            case DM::PAGE_USER_OVERVIEW:
            case DM::PAGE_USER_EDIT:
                $user_page = true;
        }

        $data = new \stdClass();
        $data->coursename = $course->shortname;
        $data->contextid = $ctx->id;
        $data->issiteadmin = is_siteadmin();

        if ($this->_group || $this->_user){
            $school = ($user_page && $this->_user) ? SH::get_user_school($this->_user) : DM::get_school_cohort($this->_group);
            $school_time_zone = false;
            $now = time();
            if (class_exists('\local_schoolmanager\school') && $school instanceof \local_schoolmanager\school){
                $timezone = $school_time_zone = $school->get_timezone();
            } elseif (!empty($school->timezone)){
                $timezone = $school_time_zone = $school->timezone;
            } elseif (!empty($this->_user->timezone)){
                $timezone = $this->_user->timezone;
            } else {
                $timezone = SH::NED_TIMEZONE;
            }

            $time_zone_info = [];
            if ($school_time_zone){
                $msg_key = 'timezone_info_school';
            } elseif (!empty($this->_user->timezone)){
                $msg_key = 'timezone_info_student';
            } else {
                $msg_key = 'timezone_info_server';
            }
            $tz_name = \core_date::get_user_timezone($timezone);
            $time_zone_info[] = SH::div(SH::str($msg_key, ['timezone' => $tz_name, 'now' => SH::ned_date_tz($now, $timezone)]));
            $viewer_tz = \core_date::get_user_timezone(null);
            if ($viewer_tz != $tz_name){
                $time_zone_info[] = SH::div(SH::str('timezone_info_user',
                    ['timezone' => $viewer_tz, 'now' => SH::ned_date_tz($now, null)]));
            }
            $data->time_zone_info = SH::arr2str($time_zone_info);
        }

        if ($user_page){
            $group = $this->_group;
            $user = $this->_user;
            if (empty($group) && !empty($user->group)){
                $group = $user->group;
            }

            if (!empty($group->id)){
                $data->groupid = $group->id;
                $data->groupname = $group->name;
                $data->groupstartend = "{$d($group->startdate)} - {$d($group->enddate)}";
                $data->groupdaysleft = $this->_daysleft($group->startdate, $group->enddate);
                $data->onlineteacher = $users_by_role('online-teacher', $group->id);
                $data->classroomteacher = $users_by_role('classroom-teacher', $group->id);
                if (isset($group->users)){
                    $data->numberofstudent = count($group->users);
                } else {
                    $data->numberofstudent = count(SH::get_group_users($group->id, true));
                }
                $data->addstudenturl = SH::url('/group/members.php', ['group' => $group->id])->out(false);

                $DM = new DM($course->id);
                if ($midtermdate = $DM->get_midterm_date($group->id)){
                    $data->midtermdategroup = $d($midtermdate);
                }
            }

            if ($user){
                $data->studentid = $user->id;
                $data->studentname = SH::q_user_link($user, $course);
                $data->studenthighlight = true;
                $data->currentgrade = SH::get_course_grade($course->id, $user->id);

                if ($this->_pagetype == DM::PAGE_USER && $manageuserdeadlines){
                    $data->managedeadlinesurl = $this->_make_url(MB::PAGE_DM_EDIT, [MB::PAR_SUBTYPE => DM::PAGE_USER_EDIT]);
                }

                if ($blocksettings->enabledeadlineextension ?? false){
                    $data->enabledeadlineextension = true;
                    $maxextensionperstudent = $blocksettings->maxextensionperstudent ?? 0;
                    $data->maxextensionperstudent = $maxextensionperstudent ?: NED\str('unlimited');
                    $data->numberofextensions = DM::get_number_of_extensions_in_course($user->id, $course->id);
                }

                if ($userextension = SH::db()->get_record('block_ned_teacher_tools_usrd', ['userid' => $user->id, 'courseid' => $course->id])){
                    $startdate = $userextension->startdate ?? 0;
                    $enddate = $userextension->enddate ?? 0;
                } else {
                    $startdate = $group->startdate ?? 0;
                    $enddate = $group->enddate ?? 0;
                }

                $data->studentstartend = $d($startdate).' - '.$d($enddate);
                $data->extendclassstudentenddate = DM::can_extend_group_user_enddate($ctx);
                $data->studentdaysleft = $this->_daysleft($startdate, $enddate);

                $DM = new DM($course->id);
                if ($midtermdate = $DM->get_midterm_date(null, $user->id)){
                    $data->midtermdateuser = $d($midtermdate);
                }
            }
        } else { // group page
            if ($group = $this->_group){
                $data->groupid = $group->id;
                $data->groupname = $group->name;
                $data->groupstartend = "{$d($group->startdate)} - {$d($group->enddate)}";
                $data->groupdaysleft = $this->_daysleft($group->startdate, $group->enddate);
                $data->groupediturl = SH::url(SH::PAGE_DM,
                    ['courseid' => $course->id, 'page' => 0, 'group' => NED\GROUP_ALL])->out(false);

                $data->grouphighlight = true;
                $data->extendclassstudentenddate = DM::can_extend_group_user_enddate($ctx);

                $data->numberofstudent = count($group->users);
                $data->addstudenturl = SH::url('/group/members.php', ['group' => $this->_group->id])->out(false);
                $data->onlineteacher = $users_by_role('online-teacher', $group->id);
                $data->classroomteacher = $users_by_role('classroom-teacher', $group->id);

                if ($this->_pagetype == DM::PAGE_GROUP && $managegroupdeadlines){
                    $data->managedeadlinesurl = $this->_make_url(MB::PAGE_DM_EDIT, [MB::PAR_SUBTYPE => DM::PAGE_GROUP_EDIT]);
                }

                /** @var \report_ghs\task\adhoc_ghs_group_enrollment_report $report */
                $report = '\report_ghs\task\adhoc_ghs_group_enrollment_report';
                if (class_exists($report)){
                    $data->numberofconflicts = $report::count_dm_conflict($group);
                }

                $DM = new DM($course->id);
                if ($midtermdate = $DM->get_midterm_date($group->id)){
                    $data->midtermdategroup = $d($midtermdate);
                }

                // Cohort.
                if ($this->_pagetype == DM::PAGE_GROUP){
                    if (!empty($school) && !empty($school_time_zone)){
                        if (class_exists('\local_schoolmanager\school') && $school instanceof \local_schoolmanager\school){
                            $cohort = $school->get_cohort();
                        } else {
                            $cohort = $school;
                        }
                        $data->hasdifferenttimezone = DM::has_different_timezone_users_in_school($group, $cohort);
                    }
                }
            }
        }

        $pagetypes = [
            DM::PAGE_GROUP => NED\str('class'),
            DM::PAGE_USER  => NED\str('student'),
        ];
        $selected = $user_page ? DM::PAGE_USER : DM::PAGE_GROUP;
        $data->pselector =
            NED\single_select($this->_make_url(MB::PAGE_DM), MB::PAR_SUBTYPE, $pagetypes, $selected, '', 'fn_select_p');

        return $data;
    }

    /**
     * Return link to page by page name and class page params (and merged params)
     *
     * @param string $page
     * @param array  $merge_params
     *
     * @return string
     */
    protected function _make_url($page, $merge_params=[]){
        $url = MB::get_page_moodle_url($page, array_merge($this->_pageparams, $merge_params));
        return $url->out(false);
    }

    /**
     * Return "{days left} / {all days}" based on start and end timestamps
     *
     * @param int $startdate
     * @param int $enddate
     *
     * @return string
     */
    protected function _daysleft($startdate, $enddate){
        if (empty($enddate) || empty($startdate)) return '-';

        if (time() > $enddate) {
            return '0';
        }

        return floor(($enddate - time()) / DAYSECS).' / '.floor(($enddate - $startdate) / DAYSECS);
    }
}
