<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Deadline Manager Header
 *
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */


namespace block_ned_teacher_tools\output;

defined('MOODLE_INTERNAL') || die();

use block_ned_teacher_tools as NED;
use block_ned_teacher_tools\deadline_manager as DM;
use block_ned_teacher_tools\output\menu_bar as MB;
use block_ned_teacher_tools\shared_lib as SH;

/**
 * Class deadline_manager_report_user_overview - overview by the all provided user
 *
 * @package block_ned_teacher_tools\output
 */
class deadline_manager_report_user_overview extends deadline_manager_report {

    /**
     * Export page contents for template
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output) {
        global $PAGE, $OUTPUT, $DB;

        $page      = optional_param('page', 0, PARAM_INT);
        $perpage   = optional_param('perpage', 20, PARAM_INT);

        $data = new \stdClass();
        $totalcount = 0;
        $studentsurl_params = ['contextid' => $this->context->id, 'id' => $this->course->id, 'roleid' => 5];


        if ($users = $this->_enrolledusers) {
            $totalcount = count($users);
            $pagedusers = array_chunk($users, $perpage);
            $max_page = $totalcount/$perpage;
            $page = $page > $max_page ? $max_page : $page;
            $checked_user_count = count($pagedusers[$page]);

            foreach ($pagedusers[$page] as $user) {
                $row = new \stdClass();
                $row->username = SH::q_user_link($user, $this->course);
                $row->id = "S".$user->id;
                $row->userurl = $this->make_url(MB::PAGE_DM, [MB::PAR_SUBTYPE => DM::PAGE_USER, MB::PAR_USER => $user->id]);
                if (isset($user->group)) {
                    $row->group = $user->group->name;
                    $row->students = $user->group->users;
                    $row->onlineteacher = $this->get_users_by_role(SH::ROLE_OT, $user->group);
                    $row->classroomteacher = $this->get_users_by_role(SH::ROLE_CT, $user->group);
                }

                $userdates = $DB->get_record('block_ned_teacher_tools_usrd', ['userid' => $user->id, 'courseid' => $this->course->id]);
                $userdatesid = ($userdates->id ?? 0).'_'.$user->id.'_'.$this->course->id;

                // Start Date.
                $startdate = self::strdate($userdates->startdate ?? null);
                $tmplstartdate = new \core\output\inplace_editable(NED\PLUGIN_NAME, 'userstartdate', $userdatesid,
                    true, $startdate, $startdate);
                $row->startdate = $OUTPUT->render($tmplstartdate);

                // End Date.
                $enddate = self::strdate($userdates->enddate ?? null) ;
                $tmplenddate = new \core\output\inplace_editable(NED\PLUGIN_NAME, 'userenddate', $userdatesid,
                    true, $enddate, $enddate);
                $row->enddate = $OUTPUT->render($tmplenddate);


                $studentsurl_params['group'] = $user->id;
                $row->studentsurl = (new \moodle_url('/user/index.php', $studentsurl_params))->out(false);

                $row->extensionsgranted = DM::get_number_of_extensions_in_course($user, $this->course, $checked_user_count > 99);

                $data->users[] = $row;
            }
        }

        $data->hasusers = !empty($data->users);

        $paginurl = clone $PAGE->url;

        $pagingbar = new \paging_bar($totalcount, $page, $perpage, $paginurl, 'page');
        $data->pagingbar = $OUTPUT->render($pagingbar);

        $data->perpagebar = NED\utils::perpage_bar($totalcount, $paginurl, 'page', true);

        return $data;
    }

}

