<?php
/**
 * @package    block_ned_teacher_tools
 * @subpackage NED
 * @copyright  2020 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 *
 * @noinspection PhpUnnecessaryCurlyVarSyntaxInspection
 */

namespace block_ned_teacher_tools\output;
use block_ned_teacher_tools\grading_tracker as GT;
use block_ned_teacher_tools\output\menu_bar as MB;
use block_ned_teacher_tools\shared_lib as SH;

/**
 * @property-read int $status = 0;
 * @property-read int $period = 0;
 * @property-read int $view = self::VIEW_SUMMARY;
 * @property-read string $download = null;
 * @property-read bool $showhidden = false;
 *
 * @property-read int $static_data;
 */
class grading_tracker_render extends \local_ned_controller\output\ned_base_table_page_render {
    use \local_ned_controller\base_empty_class;

    //region Config to set up base class work
    protected const _PLUGIN = SH::TT;
    protected const _USE_COURSE_VIEW = false;
    //endregion

    //region SQL data
    protected const _SQL_TABLE = GT::TABLE;
    protected const _SQL_ALIAS = 'gt';
    protected const _SQL_GET_GRADERNAME = "IF(u2.id IS NOT NULL, CONCAT(u2.firstname, ' ', u2.lastname), 'None')";
    protected const _SQL_ALIAS_NGC = 'ngc';
    protected const _SQL_ALIAS_AI = 'ai';
    protected const _SQL_UIF_GT_TYPE = 'gt_status';
    //endregion

    //region Params
    const PAR_PERIOD = 'period';
    const PAR_VIEW = 'view';
    const PAR_TYPE = 'type';
    const PAR_SHOW_HIDDEN = 'showhidden';
    const PAR_ATTEMPT = 'attempt';
    const PAR_SELFGRADED = 'sg';
    const PAR_DOWNLOAD_UNLIMITED = 'du';

    const PARAMS = [
        SH::PAR_ID, SH::PAR_IDS, SH::PAR_FILTER_ID, SH::PAR_FILTER_IDS,

        SH::PAR_COURSE, SH::PAR_CM, SH::PAR_USER, SH::PAR_GROUP,
        SH::PAR_GRADER, SH::PAR_SCHOOL,

        SH::PAR_ACTIVE, SH::PAR_STATUS,
        SH::PAR_ACTION, SH::PAR_DOWNLOAD,

        SH::PAR_PAGE, SH::PAR_PERPAGE, SH::PAR_SORT, SH::PAR_SORT_DESC,

        self::PAR_PERIOD, self::PAR_VIEW, self::PAR_TYPE, self::PAR_SHOW_HIDDEN, self::PAR_ATTEMPT,
        self::PAR_SELFGRADED, self::PAR_DOWNLOAD_UNLIMITED,
    ];

    /**
     * By default, params uses _PARAM_TYPE_DEFAULT and _PARAM_VALUE_DEFAULT, but you can set defaults to others here
     * Possible keys for rewrite array:
     *  • type - change default type of param
     *  • default - change default value of param
     *  • property - load raw param value for $this as $this->{$property}
     * Also {@see \local_ned_controller\shared\C::PARAM_DATA}
     *
     * @var array[] - keys from PARAMS, value is array
     */
    const PARAM_DATA = [
        SH::PAR_GRADER        => ['type' => PARAM_INT, 'default' => self::GRADER_ALL],
        self::PAR_SHOW_HIDDEN => ['type' => PARAM_BOOL, 'default' => null],
        self::PAR_SELFGRADED => ['type' => PARAM_INT, 'default' => self::SELFGRADED_HIDE],
        self::PAR_DOWNLOAD_UNLIMITED => ['type' => PARAM_BOOL, 'default' => true, 'property' => '_download_unlimited'],
    ];
    //endregion

    //region Const Status
    const STATUS_ALL = SH::ALL;
    const STATUS_WAIT = 1;
    const STATUS_ONTIME = 2;
    const STATUS_LATE = 3;
    const STATUS_UNCOUNTED = 4;
    const STATUS_BUG = 5;
    const STATUS_UPCOMING = 7;
    const STATUS_NOT_SUBMIT = 8;
    const STATUS_PAUSED = 10;
    const STATUS_GRADED = 11;
    const STATUS_AIV_UNAPPROVED = 20;
    // All Statuses = STATUSES + STATUSES_PLUS + STATUSES_ADMIN
    const STATUSES = [
        self::STATUS_WAIT => 'waitingforgrade',
        self::STATUS_UPCOMING => 'waitingforgrade_upcoming',
        self::STATUS_ONTIME => 'ontime',
        self::STATUS_LATE => 'late',
        self::STATUS_GRADED => 'allgraded',
        self::STATUS_NOT_SUBMIT => 'didnotsubmit',
        self::STATUS_AIV_UNAPPROVED => 'aivunapproved',
    ];
    const STATUSES_PLUS = [
        self::STATUS_UNCOUNTED => 'uncounted',
        self::STATUS_PAUSED => 'gtpaused',
        self::STATUS_BUG => 'bugreport',
    ];
    const STATUSES_ADMIN = [];

    // Statuses counted for the all records
    const COUNTED_STATUSES = [
        self::STATUS_WAIT,
        self::STATUS_UPCOMING,
        self::STATUS_ONTIME,
        self::STATUS_LATE,
        self::STATUS_AIV_UNAPPROVED,
    ];
    const STATUSES_ORDER = [
        self::STATUS_WAIT,
        self::STATUS_UPCOMING,

        self::STATUS_ONTIME,
        self::STATUS_LATE,
        self::STATUS_UNCOUNTED,
        self::STATUS_GRADED,

        self::STATUS_NOT_SUBMIT,
        self::STATUS_PAUSED,
        self::STATUS_BUG,
        self::STATUS_AIV_UNAPPROVED,
    ];
    //endregion

    //region Const Active
    const ACTIVE_ALL = 0;
    const ACTIVE_YES = 1;
    const ACTIVE_NONE = 2;
    const ACTIVES = [
        self::ACTIVE_ALL => 'all',
        self::ACTIVE_YES => 'activeusers',
        self::ACTIVE_NONE => 'suspendedusers',
    ];
    //endregion

    //region Const Action
    const ACTION_UPDATE = 1;
    const ACTION_REFRESH = 2;
    const ACTION_USER_UPDATE = 3;
    //endregion

    //region Const Period
    const PERIOD_ALL = 0;
    const PERIOD_1WEEK = 1;
    const PERIOD_2WEEK = 2;
    const PERIOD_1MONTH = 3;
    const PERIOD_2MONTH = 4;
    const PERIOD_6MONTH = 5;
    const PERIODS = [self::PERIOD_ALL, self::PERIOD_1WEEK, self::PERIOD_2WEEK, self::PERIOD_1MONTH, self::PERIOD_2MONTH, self::PERIOD_6MONTH];
    //endregion

    //region Const View
    const VIEW_EXPANDED = 1;
    const VIEW_COMPACT = 2;
    const VIEW_OVERVIEW = 3;
    const VIEWS = [
        self::VIEW_EXPANDED => 'expanded',
        self::VIEW_COMPACT => 'compact',
        self::VIEW_OVERVIEW => 'overview',
    ];
    const VIEWS_MENU = [
        self::VIEW_EXPANDED => self::VIEWS[self::VIEW_EXPANDED],
        self::VIEW_COMPACT => self::VIEWS[self::VIEW_COMPACT],
        self::VIEW_OVERVIEW => 'gradingstatistics',
    ];
    //endregion

    //region Const Attempts
    const ATTEMPT_ALL = 0;
    const ATTEMPT_LAST = 1;
    const ATTEMPT_NOT_LAST = 2;
    const ATTEMPT_NOT_FIRST = 3;
    const ATTEMPTS = [
        self::ATTEMPT_ALL => 'all',
        self::ATTEMPT_LAST => 'gt:attempts_last',
        self::ATTEMPT_NOT_LAST => 'gt:attempts_not_last',
        self::ATTEMPT_NOT_FIRST => 'gt:attempts_not_first',
    ];
    //endregion

    //region Const Columns
    const COLUMN_IDS = 1;
    const COLUMN_COURSE = 2;
    const COLUMN_CLASS = 3;
    const COLUMN_GRADERNAME = 4;
    const COLUMN_STUDENTNAME = 5;
    const COLUMN_LASTCHECK = 6;
    const COLUMN_ACTIVITY = 7;
    const COLUMN_TYPE = 8;
    const COLUMN_DEADLINE = 9;
    const COLUMN_SUBMITTED = 10;
    const COLUMN_OT_GRADE_DUE = 11;
    const COLUMN_OT_GRADE_AWARDED = 12;
    const COLUMN_REASON = 13;
    const COLUMN_BUGREPORT = 14;
    const COLUMN_NOTE = 15;
    const COLUMN_COUNTDOWN = 16;
    // from the overview page
    const COLUMN_STUDENT_COUNT = 17;
    const COLUMN_WAITING_FOR_GRADE = 18;
    const COLUMN_OVERDUE = 19;
    const COLUMN_OVERDUE36 = 20;
    const COLUMN_LATE = 21;
    const COLUMN_LATE36 = 22;
    const COLUMN_ONTIME = 23;
    const COLUMN_GPD = 24;
    const COLUMN_RATE = 25;
    // for download file
    const COLUMN_OT_GRADE_AWARDED_TIME = 26;
    const COLUMN_GRADING_WINDOW = 27;
    const COLUMN_GT_TYPE = 28;

    // COLUMN => [data key, string lang key, (optional) sql sort key]
    const COLUMNS_PARAMETERS = [
        self::COLUMN_IDS              => ['id', 'ID'],
        self::COLUMN_COURSE           => ['course_name', 'course'],
        self::COLUMN_CLASS            => ['classname', 'class'],
        self::COLUMN_GRADERNAME       => ['gradername', 'gradername'],
        self::COLUMN_STUDENTNAME      => ['student', 'studentname', 'gt.userid'],
        self::COLUMN_LASTCHECK        => ['last_check', 'lastcheck_short', 'timemodified'],
        self::COLUMN_ACTIVITY         => ['show_cm_name', 'activity', 'cm.id'],
        self::COLUMN_TYPE             => ['tags', 'type'],
        self::COLUMN_DEADLINE         => ['deadline', 'activitydeadline'],
        self::COLUMN_SUBMITTED        => ['timestart', 'submitted'],
        self::COLUMN_OT_GRADE_DUE     => ['timeend', 'otgradedue'],
        self::COLUMN_OT_GRADE_AWARDED => ['timegrade', 'otgradeawarded'],
        self::COLUMN_REASON           => ['uncounted_reason', 'reason'],
        self::COLUMN_BUGREPORT        => ['bug_report', 'bugreport'],
        self::COLUMN_NOTE             => ['note', 'note'],
        self::COLUMN_COUNTDOWN        => ['str_status', 'countdown', 'status_sort'],
        // from the overview page
        self::COLUMN_STUDENT_COUNT     => ['students', 'students', ''],
        self::COLUMN_WAITING_FOR_GRADE => ['waitingforgrade', 'waitingforgrade_grade', ''],
        self::COLUMN_OVERDUE           => ['overdue', 'totaloverdue', ''],
        self::COLUMN_OVERDUE36         => ['overdue36', 'overdue36', ''],
        self::COLUMN_LATE              => ['late', 'totalgradedlate', ''],
        self::COLUMN_LATE36            => ['late36', 'late_grade_36hrs', ''],
        self::COLUMN_ONTIME            => ['ontime', 'ontime_grade', ''],
        self::COLUMN_GPD               => ['gpd', 'average_gpd', ''],
        self::COLUMN_RATE              => ['rate', 'successrate', ''],
        // for download file
        self::COLUMN_OT_GRADE_AWARDED_TIME => ['timegrade_time', 'otgradeawardedtime', ''],
        self::COLUMN_GRADING_WINDOW        => ['workdays', 'gradingwindow', ''],
        self::COLUMN_GT_TYPE               => ['gt_type', 'gt_type', ''],
    ];
    //endregion

    //region Const Types
    const TYPE_ALL = 0;
    const TYPE_SUMMATIVE = 1;
    const TYPE_FORMATIVE = 2;
    const TYPE_QF = 3;
    const TYPE_MIDTERM = 4;
    const TYPE_UNIT_TEST = 5;
    const TYPE_FINAL_EXAM = 6;
    const TYPE_PROXY = 7;
    const TYPE_ASSIGN = 9;
    const TYPE_QUIZ = 10;
    const TYPE_FORUM = 11;

    const TYPE_TAGS = [
        self::TYPE_SUMMATIVE    => SH::TAG_SUMMATIVE,
        self::TYPE_FORMATIVE    => SH::TAG_FORMATIVE,
        self::TYPE_QF           => SH::TAG_QUICK_FEEDBACK,
        self::TYPE_MIDTERM      => SH::TAG_MIDTERM,
        self::TYPE_UNIT_TEST    => SH::TAG_UNIT_TEST,
        self::TYPE_FINAL_EXAM   => SH::TAG_FINAL_EXAM,
        self::TYPE_PROXY        => SH::TAG_PROXY,
    ];
    const TYPE_MODS = [
        self::TYPE_ASSIGN   => SH::MOD_ASSIGN,
        self::TYPE_QUIZ     => SH::MOD_QUIZ,
        self::TYPE_FORUM    => SH::MOD_FORUM,
    ];
    //endregion

    //region Const Selfgraded
    const SELFGRADED_ALL = 0;
    const SELFGRADED_HIDE = 1;
    const SELFGRADED_ONLY = 2;
    const SELFGRADED_OPTIONS = [
        self::SELFGRADED_ALL => 'gt:selfgraded:all',
        self::SELFGRADED_HIDE => 'gt:selfgraded:hide',
        self::SELFGRADED_ONLY => 'gt:selfgraded:only',
    ];
    //endregion

    //region Other Consts
    const PERPAGES = [100, 250, 500, 1000];
    const URL = SH::PLUGIN_DIRS[SH::TT].'/grading_tracker.php';
    const TITLE_KEY = 'gradingtracker';

    const CRON_WORK = 'cron_work';
    //endregion

    //region Static Properties
    static protected $_uif_gt_type_id = null;
    static protected $_uif_gt_type_def = null;
    //endregion

    //region Properties
    public $sort;
    public $sort_desc;

    protected $_status = 0;
    protected $_status_plus = 0;
    protected $_period = 0;
    protected $_view = self::VIEW_COMPACT;
    protected $_type = 0;
    protected $_attempt = 0;
    protected $_active = self::ACTIVE_YES;
    /** @var string|bool - string means type of download file, e.g. 'csv' */
    protected $_download = null;
    protected $_download_unlimited = false;
    protected $_showhidden = false;
    protected $_selfgraded = self::SELFGRADED_HIDE;
    //endregion

    //region Init methods
    /**
     * @constructor grading_tracker_render
     */
    public function __construct(){
        parent::__construct();

        $this->_cap_view = GT::get_capability($this->_ctx, $this->_viewerid);
        if ($this->_cap_view <= GT::CAP_CANT_VIEW){
            /** @noinspection PhpUnnecessaryStopStatementInspection */
            return;
        }
    }

    /**
     * Return path to AI infraction class, or false
     *
     * @return false|string|\local_academic_integrity\infraction
     */
    protected static function _ai_infraction(){
        static $_data = null;

        if (is_null($_data)){
            $_data = false;
            /** @var \local_academic_integrity\infraction|string $cls */
            $cls = '\local_academic_integrity\infraction';
            if (SH::is_ai_exists() && method_exists($cls, 'get_user_unapplied_records')){
                $_data = $cls;
            }
        }

        return $_data;
    }

    /**
     * Check loaded params
     * Normally called by {@see set_and_check_params()} after param loading
     */
    protected function _check_params(){
        if (!$this->can_see()) return;

        if (!$this->can_see_all()){
            unset($this->_params[SH::PAR_GRADER]);
            $this->_graderid = $this->_viewerid;
        } else {
            // we need to check grader before other options
            $options = $this->get_graders_options();
            $this->_graderid = SH::isset_key($options, $this->_params[SH::PAR_GRADER], static::GRADER_ALL);
        }

        $this->_view = SH::isset_key(static::VIEWS, $this->_params[SH::PAR_VIEW], static::VIEW_EXPANDED);

        if ($this->_view == static::VIEW_COMPACT){
            unset($this->_params[SH::PAR_COURSE]);
            $this->_courseid = SH::ALL;
            unset($this->_params[SH::PAR_USER]);
            $this->_userid = SH::ALL;
            unset($this->_params[SH::PAR_SCHOOL]);
            $this->_schoolid = SH::ALL;

            unset($this->_params[static::PAR_SELFGRADED]);
            $this->_selfgraded = static::SELFGRADED_HIDE;
        } else {
            $this->_selfgraded = SH::isset_key(static::SELFGRADED_OPTIONS, $this->_params[static::PAR_SELFGRADED], static::SELFGRADED_HIDE);
        }

        $this->_check_base_params();

        $this->_status = static::STATUS_ALL;
        $this->_download = $this->_params[SH::PAR_DOWNLOAD] ?? false;
        unset($this->_params[SH::PAR_DOWNLOAD]);

        if ($this->_view != static::VIEW_OVERVIEW){
            $statuses = static::get_list_as_options($this->get_statuses(), true);
            $this->_status = SH::isset_key($statuses, $this->_params[SH::PAR_STATUS], static::STATUS_UPCOMING);
            $this->_status_plus = isset(static::STATUSES[$this->_status]) ? false : $this->_status;

            //for sorting on not-overview page
            $this->sort = SH::isset_key(static::COLUMNS_PARAMETERS, $this->_params[SH::PAR_SORT], static::COLUMN_COUNTDOWN);
            $this->sort_desc = (bool)($this->_params[SH::PAR_SORT_DESC]);
        }

        $this->_period = SH::isset_in_list(static::PERIODS, $this->_params[static::PAR_PERIOD], static::PERIOD_1MONTH);
        $this->_type = SH::isset_key(static::get_types(), $this->_params[static::PAR_TYPE], static::TYPE_ALL);
        $this->_attempt = SH::isset_key(static::ATTEMPTS, $this->_params[static::PAR_ATTEMPT], static::ATTEMPT_ALL);

        switch($this->_action){
            default:
                $this->_action = null;
                break;
            case static::ACTION_UPDATE:
                if (!$this->_is_admin){
                    $this->_action = null;
                }
                break;
            case static::ACTION_USER_UPDATE:
                if (!$this->is_admin || !$this->_userid || !$this->_courseid){
                    $this->_action = null;
                }
                break;
            case static::ACTION_REFRESH:
                break;
        }
        // we don't want to add it in other links
        unset($this->_params[SH::PAR_ACTION]);

        /**
         * If admin - we can set id and that's all
         * If not - only can set id as filter for other 'where' conditions, and not allowed as later url params
         */
        if ($this->_is_admin){
            $param_ids = $this->_params[SH::PAR_IDS] ?? $this->_params[SH::PAR_ID] ?? null;
            $this->_ids = $param_ids ? explode(',', $param_ids) : [];

            $this->_showhidden = (bool)$this->_params[static::PAR_SHOW_HIDDEN];
        } else {
            unset($this->_params[SH::PAR_IDS]);
            $this->_ids = null;
            $this->_showhidden = false;
        }

        // we don't use simple id for now
        unset($this->_params[SH::PAR_ID]);
        $this->_id = null;

        $param_filter_ids = $this->_params[SH::PAR_FILTER_IDS] ?? $this->_params[SH::PAR_FILTER_ID] ?? null;
        $this->_filter_ids = $param_filter_ids ? explode(',', $param_filter_ids) : [];
        unset($this->_params[SH::PAR_FILTER_ID]);
        unset($this->_params[SH::PAR_FILTER_IDS]);

        if ($this->_action === static::ACTION_REFRESH){
            if (empty($this->_filter_ids)){
                $this->_action = null;
            } elseif (count($this->_filter_ids) > 1){
                // allow refresh only single record at a time
                $this->_filter_ids = [reset($this->_filter_ids)];
            }
        }

        $this->_active = SH::isset_key(static::ACTIVES, $this->_params[SH::PAR_ACTIVE], static::ACTIVE_ALL);

        $this->_params_init = true;
    }
    //endregion

    //region Capabilities methods
    /**
     * Return true, if current user is able to see this page
     *
     * @return bool
     */
    public function can_see(){
        return $this->_cap_view > GT::CAP_CANT_VIEW;
    }

    /**
     * Return true, if current user is able to see all GT data
     *
     * @return bool
     */
    public function can_see_all(){
        return $this->_cap_view >= GT::CAP_SEE_ALL;
    }
    //endregion

    //region Setup Page methods
    /**
     * @return string
     */
    public function get_page_title(){
        return SH::str(static::TITLE_KEY);
    }

    /**
     * Return downloadable file, if 'download' param was set
     * If successfully return file, then exit script
     *
     * @param array|\Iterable $data - if null, get data by {@see get_table_data()} method
     *
     * @return void
     */
    public function check_download($data=null){
        if (!$this->_download) return;

        if ($this->_download_unlimited){
            $this->_page = $this->_perpage = 0;
            SH::server_ask_more_resources(true, true, true);
        }

        $data = $data ?? $this->get_table_data();
        if (empty($data)) return;

        $this->_before_start_export();
        $this->content->data = $data;
        $raw_columns = $this->get_columns();
        if (empty($raw_columns)) return;

        // There is no more returns, next only file downloading,
        // now deleting redundant data and ask more server resources
        $this->clear_cache(true);

        $filename = $this->get_page_title().'['.SH::str(static::VIEWS[$this->_view]).']';
        $columns = [];
        foreach ($raw_columns as $raw_column){
            $data_key = static::get_column_data_key($raw_column);
            if (empty($data_key)) continue;

            $columns[$data_key] = static::get_column_name($raw_column) ?: $raw_column;
        }

        SH::try_download_data($filename, $this->_download, $columns, $data);
        die;
    }

    /**
     * Call this method before header output, then you can redirect here
     */
    public function before_header_output(){
        if (!$this->can_see()) return;

        if ($this->_download){
            $this->check_download();
            return;
        } elseif ($this->_view == static::VIEW_OVERVIEW) {
            return;
        }

        $url = $this->get_my_url();
        // refreshing single record
        if ($this->_action === static::ACTION_REFRESH && !empty($this->_filter_ids)){
            $records = $this->get_main_table_data();
            if (!empty($records)){
                $upd_statuses = GT::update_gt_records($records);
                SH::redirect($url, SH::str('checked_updated_records', $upd_statuses), null, SH::NOTIFY_SUCCESS);
                die;
            }
        } elseif ($this->_action === static::ACTION_UPDATE && $this->_is_admin){
            // we need data here for refreshing
            $records = $this->get_main_table_data();
            if (!empty($records)){
                // refreshing all records on the page
                $upd_statuses = GT::update_gt_records($records);
                SH::redirect($url, SH::str('checked_updated_records', $upd_statuses), null, SH::NOTIFY_SUCCESS);
                die;
            }
        } elseif ($this->_action === static::ACTION_USER_UPDATE && $this->_is_admin){
            // check/update single user on the course
            $res = GT::check_and_update_course($this->_courseid, $this->_userid);
            $url = static::get_url([SH::PAR_COURSE => $this->_courseid, SH::PAR_USER => $this->_userid]);
            if (!empty($res[GT::UPD_ERROR])){
                SH::redirect($url, $res[GT::UPD_ERROR], null, SH::NOTIFY_ERROR);
            } else {
                $res['coursename'] = SH::q_course_link($this->_courseid);
                $res['username'] = SH::q_user_link($this->_userid, $this->_courseid);
                SH::redirect($url, SH::str('gt_checked_user_res', $res), null, SH::NOTIFY_SUCCESS);
            }
            die;
        }
    }
    //endregion

    //region Getters & Setters
    /**
     * Read or write "count data" value
     *
     * @param array|null $set_data - if null, read data
     *
     * @return int[]|array
     */
    protected function _rw_count_data($set_data=null){
        $def = [0, 0, 0, 0];
        if (!is_null($set_data)){
            $this->_static_data[__FUNCTION__] = ($set_data + $def);
        } elseif (!isset($this->_static_data[__FUNCTION__])){
            debugging("You probably didn't set count data");
            return $def;
        }

        return $this->_static_data[__FUNCTION__];
    }

    /**
     * Get count data for record statistics
     *
     * @return array [$all, $uncounted, $notsubmit, $usual]
     */
    public function get_count_data(){
        return $this->_rw_count_data();
    }

    /**
     * Set count data for record statistics
     *
     * @param int $all
     * @param int $uncounted
     * @param int $notsubmit
     * @param int $usual
     */
    public function set_count_data($all=0, $uncounted=0, $notsubmit=0, $usual=0){
        $this->_rw_count_data([$all, $uncounted, $notsubmit, $usual]);
    }
    //endregion

    //region SQL menu filter methods
    /**
     * Add additional view limitation for {@see ned_base_table_page_render::_sql_menu_filter()}
     *  changing $where and $params for sql query
     *
     * @param array $joins
     * @param array $where
     * @param array $params
     *
     * @return void
     */
    protected function _sql_add_view_limitation(&$joins=[], &$where=[], &$params=[]){
        if ($this->_graderid > static::GRADER_ALL){
            $this->_sql_add_limit_grader($joins, $where, $params, null, false);
        }
    }

    /**
     * Load sql params for getting GT graders with capability "gradingtracker_seeown"
     *
     * @param array  $joins
     * @param array  $where
     * @param array  $params
     * @param string $sql_graderid    = gt graderid
     * @param bool   $load_user_table - if true, load user table here
     * @param string $userid_sql      - sql to check user id (u.id by default)
     * @param string $u_table_prefix  - prefix for loading user table (if $load_user_table - true)
     */
    public function sql_get_grader_query(&$joins=[], &$where=[], &$params=[], $sql_graderid=null,
        $load_user_table=true, $userid_sql='', $u_table_prefix=''){

        $u = $u_table_prefix.'u';
        $userid = $userid_sql ?: $u.'.id';
        $sql_graderid = $sql_graderid ?: static::_sql_a(static::_SQL_GRADERID);
        if ($load_user_table){
            $joins[] = "LEFT JOIN {user} $u ON $userid = $sql_graderid";
        }

        $joins[] = "
            LEFT JOIN (
                SELECT rc.id, rc.roleid, ra.userid
                FROM {role_assignments} ra 
                JOIN {role_capabilities} rc 
                    ON rc.roleid = ra.roleid
                WHERE ra.contextid = :context AND rc.capability = :capability AND rc.permission = :allow
                GROUP BY ra.userid
            ) grader_permissions
            ON grader_permissions.userid = $userid
        ";

        $where[] = "(grader_permissions.id IS NOT NULL OR $userid IS NULL)";
        $params['capability'] = SH::get_full_capability('gradingtracker_seeown');
        $params['allow'] = CAP_ALLOW;
        $params['context'] = $this->_ctx->id;
    }
    //endregion

    //region SQL & Data utils methods
    /**
     * @param $status
     *
     * @return string
     */
    static public function get_status_condition($status){
        $time_end = static::_sql_a('timeend');
        $time_grade = static::_sql_a('timegrade');

        $ungraded_cond_postfix = '';
        $INF = static::_ai_infraction();
        if ($INF){
            $prop = static::_SQL_ALIAS_AI.".state";
            $val = $INF::ST_UNAPPROVED;
            $ungraded_cond_postfix = " AND ($prop IS NULL OR $prop <> $val)";
        }

        switch($status){
            /** @noinspection PhpMissingBreakStatementInspection */
            default:
                trigger_error('Unknown status for grade controller');
            case static::STATUS_WAIT:
                return "$time_grade = 0".$ungraded_cond_postfix;
            case static::STATUS_UPCOMING:
                return "$time_grade = 0 AND $time_end < UNIX_TIMESTAMP(TIMESTAMPADD(DAY, 10, CURRENT_TIMESTAMP))".$ungraded_cond_postfix;
            case static::STATUS_ONTIME:
                $uncounted = static::_sql_a('uncounted');
                return "$time_grade > 0 AND ($time_end >= $time_grade OR $uncounted = 1)";
            case static::STATUS_LATE:
                $uncounted = static::_sql_a('uncounted');
                return "$time_grade > 0 AND $time_grade > $time_end AND $uncounted = 0";
            case static::STATUS_GRADED:
                return "$time_grade > 0";
            case static::STATUS_NOT_SUBMIT:
                return static::_sql_a('timestart')." = 0";
            case static::STATUS_AIV_UNAPPROVED:
                if ($INF){
                    return static::_SQL_ALIAS_AI.'.state = '.$INF::ST_UNAPPROVED;
                }
                return SH::SQL_NONE_COND;
        }
    }

    /**
     * @param $period
     *
     * @return int
     */
    static public function get_period_in_days($period){
        switch($period){
            /** @noinspection PhpMissingBreakStatementInspection */
            default:
                trigger_error('Unknown period for grade controller', E_USER_NOTICE);
            case static::PERIOD_ALL:
                return 0;
            case static::PERIOD_1WEEK:
                return 7;
            case static::PERIOD_2WEEK:
                return 14;
            case static::PERIOD_1MONTH:
                return 30;
            case static::PERIOD_2MONTH:
                return 60;
            case static::PERIOD_6MONTH:
                return 182;
        }
    }

    /**
     * @param $period
     *
     * @return int
     */
    static public function get_period_time($period){
        $days = static::get_period_in_days($period);
        if (!$days){
            return 0;
        }

        return time() - ($days*DAYSECS);
    }

    /**
     * Load and get data for GT type user field: field ID and its default value
     * Return GT type field ID, can be false if there is no such field
     *
     * @return false|int
     * @noinspection PhpReturnDocTypeMismatchInspection
     */
    static public function uif_get_gt_type_id(){
        if (is_null(static::$_uif_gt_type_id)){
            $record = SH::db()->get_record(SH::TABLE_USER_INFO_FIELD, ['shortname' => static::_SQL_UIF_GT_TYPE], 'id, defaultdata');
            static::$_uif_gt_type_id = $record->id ?? false;
            static::$_uif_gt_type_def = $record->defaultdata ?? false;
        }

        return static::$_uif_gt_type_id;
    }

    /**
     * Load and get data for GT type user field: field ID and its default value
     * Return GT type field default, can be false if there is no such field
     *
     * @return mixed|false
     */
    static public function uif_get_gt_type_default(){
        if (is_null(static::$_uif_gt_type_id)){
            static::uif_get_gt_type_id();
        }

        return static::$_uif_gt_type_def;
    }
    //endregion

    //region Column's methods
    /**
     * @param int $column - const column key from the {@see static::COLUMNS_PARAMETERS}
     *
     * @return string - data key
     */
    static public function get_column_data_key($column){
        return static::COLUMNS_PARAMETERS[$column][0] ?? '';
    }

    /**
     * Return list of column data keys by columns (ids)
     * @param array|int $columns - list of the const column keys from the {@see static::COLUMNS_PARAMETERS}
     *
     * @return array - list of data keys
     */
    static public function get_list_column_data_keys($columns){
        $columns = SH::val2arr($columns);
        $res = [];
        foreach ($columns as $column){
            if (empty(static::COLUMNS_PARAMETERS[$column][0])) continue;

            $res[] = static::COLUMNS_PARAMETERS[$column][0];
        }
        return $res;
    }

    /**
     * @param int $column - const column key from the {@see static::COLUMNS_PARAMETERS}
     *
     * @return string - sql sort value
     */
    static public function get_column_sort($column){
        return static::COLUMNS_PARAMETERS[$column][2] ?? (static::COLUMNS_PARAMETERS[$column][0] ?? '');
    }

    /**
     * @param int $column - const column key from the {@see static::COLUMNS_PARAMETERS}
     *
     * @return string - translated column name
     */
    static public function get_column_name($column){
        $key = static::COLUMNS_PARAMETERS[$column][1] ?? '';
        return SH::str($key);
    }

    /**
     * Get data from object by using key from the required column
     *
     * @param object|array $object - object, from where we need data
     * @param int          $column - const column key from the {@see static::COLUMNS_PARAMETERS}
     *
     * @return mixed|null
     */
    static public function get_obj_data_by_column(&$object, $column){
        if (empty(static::COLUMNS_PARAMETERS[$column][0])) return null;

        $key = static::COLUMNS_PARAMETERS[$column][0];
        if (is_object($object)) return $object->$key ?? null;
        elseif (is_array($object)) return $object[$key] ?? null;
        return null;
    }

    /**
     * Save data to object by using key from the required column
     *
     * @param object|array $object - object, from where we need data
     * @param int          $column - const column key from the {@see static::COLUMNS_PARAMETERS}
     * @param mixed        $data - data to save in object
     *
     * @return object|array - saved $object
     */
    static public function set_obj_data_by_column(&$object, $column, $data=null){
        if (empty(static::COLUMNS_PARAMETERS[$column][0])) return $object;

        $key = static::COLUMNS_PARAMETERS[$column][0];
        if (is_object($object)) $object->$key = $data;
        elseif (is_array($object)) $object[$key] = $data;

        return $object;
    }

    /**
     * Return data table columns,
     *  column keys from the {@see static::COLUMNS_PARAMETERS} of columns, which should be on the current page
     *
     * Please, watch, that this structure will be the same as in
     * @template block_ned_teacher_tools/grading_tracker_render
     * @template block_ned_teacher_tools/grading_tracker_render_rows
     *
     * @return array|int[]
     */
    public function get_columns(){
        $c = SH::stdClass2($this->content);
        $columns = [];
        if ($c->main_table){
            $columns[] = static::COLUMN_COURSE;
            if ($c->show_ids){
                $columns[] = static::COLUMN_IDS;
            }
            $columns[] = static::COLUMN_CLASS;
            $columns[] = static::COLUMN_GRADERNAME;
            $columns[] = static::COLUMN_STUDENTNAME;
            if ($c->expanded){
                $columns[] = static::COLUMN_LASTCHECK;
            }
            $columns[] = static::COLUMN_ACTIVITY;
            $columns[] = static::COLUMN_TYPE;
            if ($c->expanded){
                $columns[] = static::COLUMN_DEADLINE;
                $columns[] = static::COLUMN_SUBMITTED;
            }
            $columns[] = static::COLUMN_OT_GRADE_DUE;
            if ($c->expanded && $c->show_timegrade){
                $columns[] = static::COLUMN_OT_GRADE_AWARDED;
                if ($this->_download){
                    $columns[] = static::COLUMN_OT_GRADE_AWARDED_TIME;
                }
            }
            if ($c->show_uncounted_reason){
                $columns[] = static::COLUMN_REASON;
            }
            if ($c->show_bugreport){
                $columns[] = static::COLUMN_BUGREPORT;
            }
            if ($c->show_note){
                $columns[] = static::COLUMN_NOTE;
            }
            $columns[] = static::COLUMN_COUNTDOWN;

            if ($this->_download){
                $columns[] = static::COLUMN_GRADING_WINDOW;
                if (static::uif_get_gt_type_id()){
                    $columns[] = static::COLUMN_GT_TYPE;
                }
            }
        } elseif ($c->overview){
            $columns = [
                static::COLUMN_GRADERNAME,
                static::COLUMN_STUDENT_COUNT,
                static::COLUMN_WAITING_FOR_GRADE,
                static::COLUMN_OVERDUE,
                static::COLUMN_OVERDUE36,
                static::COLUMN_LATE,
                static::COLUMN_LATE36,
                static::COLUMN_ONTIME,
            ];
            if ($c->show_gpd){
                $columns[] = static::COLUMN_GPD;
            }
            $columns[] = static::COLUMN_RATE;
        }

        return $columns;
    }
    //endregion

    //region Get main table data methods
    /**
     * Get and render data page table
     *
     * @param bool $only_keys - if true, return only records key (normally it will be a list of record ids)
     *
     * @return array - list of records or keys
     */
    public function get_table_data($only_keys=false){
        if (!$this->can_see()) return [];

        if ($this->_view == static::VIEW_OVERVIEW){
            $records = $this->get_overview_table_data();
        } else {
            $records = $this->get_main_table_data($only_keys);
        }

        if ($only_keys){
            return $records;
        }

        return array_values($records);
    }

    /**
     * Table data for main GT type of pages
     *
     * @param bool $only_keys - if true, return only list of ids
     *
     * @return array
     */
    public function get_main_table_data($only_keys=false){
        if (!$this->can_see()) return [];

        if (isset($this->_static_data[__FUNCTION__])){
            $records = $this->_static_data[__FUNCTION__];
            if ($only_keys){
                return array_keys($records);
            }

            return $records;
        }

        $t = static::_SQL_ALIAS;
        $group_t = static::_SQL_GROUP_ALIAS;
        $params = [];
        $status_cond = '';
        foreach (static::COUNTED_STATUSES as $status){
            $status_cond .= 'IF('.static::get_status_condition($status).', '.$status.', ';
        }
        $status_cond .= static::STATUS_ALL . str_repeat(')', count(static::COUNTED_STATUSES));
        $e12 = 10**12;
        /** @var $c_key - alias to get key for object data by column name */
        $c_key = function($column) { return static::get_column_data_key($column); };

        $select = [
            "$t.*",
            "$t.hidden AS gthidden",
            "$status_cond AS status",
            "tg.tags",
            "$group_t.group_ids",
            "$group_t.group_names AS ".$c_key(static::COLUMN_CLASS),
            "IF($t.timegrade = 0, $t.timeend, $e12 + $t.timeend - $t.timegrade) as status_sort",
            static::get_status_condition(static::STATUS_NOT_SUBMIT).' AS notsubmit',
            static::_SQL_GET_GRADERNAME." AS ".$c_key(static::COLUMN_GRADERNAME),
            "c.shortname AS ".$c_key(static::COLUMN_COURSE),
            "c.fullname AS course_fullname",
            "c.id AS courseid",
            "m.name AS `module`",
        ];

        if (static::_ai_infraction()){
            $select[] = static::_SQL_ALIAS_AI.".id AS ai_id";
        }

        $joins = ["
            JOIN {course} c
                ON c.id = ".static::_sql_a(static::_SQL_COURSEID)."
            JOIN {course_modules} cm
                ON cm.id = ".static::_sql_a(static::_SQL_CMID)."
            JOIN {modules} m
                ON cm.module = m.id
            LEFT JOIN {user} u
                ON u.id = ".static::_sql_a(static::_SQL_USERID)."
        "];

        if ($this->_download){
            $uif_gt_type_id = static::uif_get_gt_type_id();
            if ($uif_gt_type_id){
                $sql_graderid = static::_sql_a(static::_SQL_GRADERID);
                $select[] = "uid.data AS ".$c_key(static::COLUMN_GT_TYPE);
                $joins[] = "
                    LEFT JOIN {user_info_data} uid
                        ON uid.fieldid = :uif_gt_type
                        AND uid.userid = $sql_graderid
                ";
                $params['uif_gt_type'] = $uif_gt_type_id;
            }
        }

        $where = [];
        $orderby = static::get_column_sort($this->sort);
        if (!empty($orderby)){
            $orderby .= ' ' . ($this->sort_desc ? 'DESC' : 'ASC');

            if ($this->sort != static::COLUMN_COUNTDOWN){
                $orderby .= ', '.static::get_column_sort(static::COLUMN_COUNTDOWN).' ASC';
            }
        }
        $set_limit = $this->_perpage;

        if ($this->_graderid > static::GRADER_ALL){
            static::_sql_add_limit_grader($joins, $where, $params, null, false);
        }

        if ($this->_view == static::VIEW_EXPANDED){
            $select[] = "$t.deadline AS deadline";
        }

        if ($this->_status && isset(static::STATUSES[$this->_status])){
            $where[] = static::get_status_condition($this->_status);
        }

        $records = $this->get_db_table_data($select, $joins, $where, $params, static::_sql_a('id'), $orderby, $set_limit);
        if ($only_keys){
            return array_keys($records);
        }

        $this->_static_data[__FUNCTION__] = $this->_process_table_raw_records($records);
        return $this->_static_data[__FUNCTION__];
    }

    /**
     * Table data for overview (Grading Statistics) page
     *
     * @return array
     */
    public function get_overview_table_data(){
        if (!$this->can_see()) return [];

        if (isset($this->_static_data[__FUNCTION__])){
            return $this->_static_data[__FUNCTION__];
        }

        $hrs36 = HOURSECS*36;
        $time_start = static::_sql_a('timestart');
        $time_end = static::_sql_a('timeend');
        $deadline = static::_sql_a('deadline');
        $time_grade = static::_sql_a('timegrade');
        /** @var $c_key - closure alias to get key for object data by column name */
        $c_key = function($column) { return static::get_column_data_key($column); };

        $joins = $where = $params = [];
        $select = [
            static::_sql_a(static::_SQL_GRADERID),
            static::_SQL_GET_GRADERNAME." AS ".$c_key(static::COLUMN_GRADERNAME),
            "COUNT(DISTINCT ".static::_sql_a(static::_SQL_USERID).") AS ".$c_key(static::COLUMN_STUDENT_COUNT),
            "SUM($time_grade = 0 AND UNIX_TIMESTAMP() > $time_end) AS ".$c_key(static::COLUMN_OVERDUE),
            "SUM($time_grade = 0 AND (UNIX_TIMESTAMP() - $time_end) > $hrs36) AS ".$c_key(static::COLUMN_OVERDUE36),
            "SUM($time_grade > 0 AND $time_grade > ($time_end + $hrs36)) AS ".$c_key(static::COLUMN_LATE36),
        ];
        foreach (static::STATUSES as $status => $name){
            $select[] = 'SUM('.static::get_status_condition($status).') AS '.$name;
        }

        if ($this->_cap_view == GT::CAP_SEE_OWN){
            static::_sql_add_limit_grader($joins, $where, $params, $this->_viewerid, false);
        } elseif ($this->_graderid > static::GRADER_ALL && $this->_download == static::CRON_WORK){
            static::_sql_add_limit_grader($joins, $where, $params, $this->_graderid, false);
        } else {
            $this->sql_get_grader_query($joins, $where, $params);
        }

        if ($this->content->statisticdate ?? 0){
            $where[] = "GREATEST($time_start, $time_grade, IF($time_start = 0 AND $time_grade = 0, $deadline, 0)) >= :statisticdate";
            $params['statisticdate'] = $this->content->statisticdate;
        }

        $records = $this->get_db_table_data($select, $joins, $where, $params, static::_sql_a(static::_SQL_GRADERID), 'u2.firstname');
        $this->_static_data[__FUNCTION__] = $this->_process_overview_table_raw_records($records);

        return $this->_static_data[__FUNCTION__];
    }

    /**
     * Process raw records from the DB and return rendered result
     *
     * @param array $raw_records
     *
     * @return array
     */
    protected function _process_table_raw_records(&$raw_records=[]){
        $count_data = ['all' => 0, 'uncounted' => 0, 'notsubmit' => 0, 'usual' => 0];

        if (empty($raw_records)){
            $this->set_count_data(...array_values($count_data));
            return [];
        }

        $INF = static::_ai_infraction();
        $can_see_ai_unapproved = false;
        if ($INF){
            $can_see_ai_unapproved = \local_academic_integrity\shared_lib::can_view_unapproved();
        }

        $now = time();
        $my_url = $this->get_my_url();
        $time_columns = [static::COLUMN_DEADLINE, static::COLUMN_SUBMITTED, static::COLUMN_OT_GRADE_DUE, static::COLUMN_OT_GRADE_AWARDED];
        $uif_gt_type_def = static::uif_get_gt_type_default();
        $str_days_short = SH::str('days_short');
        $str_nosubmission = SH::str('nosubmission');
        $str_proxy_deadline = SH::str('gt:proxy_deadline');

        /** @var object $record - var from $raw_records loop  */
        $record = null;
        // closure aliases to get/set data from/to $record by column
        $get = function($column) use (&$record) { return static::get_obj_data_by_column($record, $column); };
        $set = function($column, $data) use (&$record) { return static::set_obj_data_by_column($record, $column, $data); };
        $records = [];

        foreach ($raw_records as $record){
            $class_status = [];
            $class_row = [];

            /// data need for all
            if (!$this->_is_admin){
                $record->gthidden = null;
                $record->note = null;
            }

            $cm = SH::get_cm_by_cmid($record->cmid, $record->courseid);
            $cm_name = SH::q_cm_name($cm, null, null, true);
            $record->cm_name = $cm_name;
            $max_len = 5;
            $pos = min(stripos($cm_name, ' ') ?: $max_len, stripos($cm_name, ':') ?: $max_len, $max_len);
            $set(static::COLUMN_ACTIVITY, substr($cm_name, 0, $pos));

            $set(static::COLUMN_STUDENTNAME, SH::q_user_link($record->userid, $record->courseid, true, $this->_download));

            if ($this->_view == static::VIEW_EXPANDED){
                if ($record->timemodified){
                    $record->last_check_raw = max($now - $record->timemodified, 1);
                    $last_check = SH::time_diff_to_str_max($record->last_check_raw, 0, 1);
                } else {
                    $record->last_check_raw = 0;
                    $last_check = '-';
                }
                $set(static::COLUMN_LASTCHECK, $last_check);
            }

            switch ($record->status){
                case static::STATUS_WAIT:
                case static::STATUS_UPCOMING:
                    $timeend = $get(static::COLUMN_OT_GRADE_DUE);
                    $time_hm = SH::time_diff_to_str_max($timeend, null, 2);
                    if ($now > $timeend){
                        // overdue
                        if ($record->uncounted){
                            $str_status = SH::str('overdue_uncounted_x', $time_hm);
                        } else {
                            $str_status = SH::str('overdue_x', $time_hm);
                            $class_status[] = 'red bold';
                        }
                    } else {
                        // wait
                        $str_status = SH::str('timeleft', $time_hm);
                        $class_status[] = 'green';
                        if (!GT::time_longer_h($timeend,24) && !$record->uncounted) {
                            $class_status[] = 'red bold';
                        }
                    }
                    break;
                case static::STATUS_ONTIME:
                    $str_status = SH::str('ontime');
                    if ($record->uncounted){
                        $timeend = $get(static::COLUMN_OT_GRADE_DUE);
                        $timegrade = $get(static::COLUMN_OT_GRADE_AWARDED);
                        if ($timegrade > $timeend){
                            $str_status_postfix = '*';
                            if (!$this->_download){
                                $str_status_postfix = SH::span($str_status_postfix, 'red font-weight-bold h5');
                            }
                            $str_status .= $str_status_postfix;
                        }
                    }
                    break;
                case static::STATUS_LATE:
                    $str_status = SH::str('late');
                    break;
                case static::STATUS_AIV_UNAPPROVED:
                    $text = SH::str('aivunapproved');
                    if ($can_see_ai_unapproved && !$this->_download && !empty($record->ai_id)){
                        $text .= SH::fa('fa-external-link mx-1');
                        $str_status = SH::link([SH::PAGE_AI_VIEW, [SH::PAR_ID => $record->ai_id]], $text);
                    } else {
                        $str_status = $text;
                    }
                    break;
                default:
                    $str_status = '';
            }
            $set(static::COLUMN_COUNTDOWN, $str_status);

            /// DOWNLOAD only
            if ($this->_download){
                // show only important tags
                $type = $get(static::COLUMN_TYPE);
                if (!empty($type)){
                    $tag_keys = '';
                    $check_tags = [
                        'S' => SH::TAG_SUMMATIVE,
                        'F' => SH::TAG_FORMATIVE,
                        'M' => SH::TAG_MIDTERM,
                    ];
                    foreach ($check_tags as $key => $check_tag){
                        if (SH::str_has($type, $check_tag, 0, false)){
                            $tag_keys.= $key;
                        }
                    }

                    $set(static::COLUMN_TYPE, $tag_keys);
                }

                // change date format
                foreach ($time_columns as $t_column){
                    $none_data = '-';
                    if ($t_column == static::COLUMN_SUBMITTED){
                        $none_data = $str_nosubmission;
                    }

                    $raw_data = $get($t_column);
                    $data = SH::ned_date($raw_data, $none_data, null, SH::DT_FORMAT_DATE_MDY);
                    $set($t_column, $data);

                    if ($t_column == static::COLUMN_OT_GRADE_AWARDED){
                        $data = SH::ned_date($raw_data, $none_data, null, SH::DT_FORMAT_TIME24);
                        $set(static::COLUMN_OT_GRADE_AWARDED_TIME, $data);
                    }
                }

                // humanize grading window
                $workdays = $get(static::COLUMN_GRADING_WINDOW);
                if (!$record->uncounted && $workdays){
                    $workdays = $workdays.$str_days_short;
                } else {
                    $workdays = '';
                }
                $set(static::COLUMN_GRADING_WINDOW, $workdays);

                if (!empty($record->proxy)){
                    $set(static::COLUMN_DEADLINE, $str_proxy_deadline);
                }

                if ($uif_gt_type_def){
                    $gt_type = $get(static::COLUMN_GT_TYPE);
                    if (is_null($gt_type)){
                        $set(static::COLUMN_GT_TYPE, $uif_gt_type_def);
                    }
                }

                // for download, it's enough -> go to next record
                $records[$record->id] = $record;
                continue;
            }

            /// all other data
            $count_data['all']++;
            $usual = true;
            if ($record->uncounted){
                $count_data['uncounted']++;
                $usual = false;
            }
            if ($record->notsubmit){
                $count_data['notsubmit']++;
                $usual = false;
            }

            if ($usual) {
                $count_data['usual']++;
            }

            $record->class_status = '';
            $courseid = $record->courseid;
            if (isset($record->group_ids)){
                $groupids = SH::str2arr($record->group_ids, '', ',');
                $record->groupid = reset($groupids);
            } else {
                $record->groupid = 0;
            }

            $record->course_url = MB::get_page_moodle_url(MB::PAGE_DM,
                [MB::PAR_COURSE => $courseid, MB::PAR_GROUP => $record->groupid]);
            $record->activity_url = SH::cm_get_grader_url($cm, $record->userid, $cm->course, [SH::PAR_USE_KICA_QUIZ => 1]);

            if ($record->graderid){
                $record->gradername_url = new \moodle_url('/user/view.php', ['id' => $record->graderid, 'course' => $courseid]);
            }
            if ($this->_view == static::VIEW_EXPANDED){
                if (empty($record->proxy)){
                    $record->deadline_url = MB::get_page_moodle_url(MB::PAGE_DM,
                        [MB::PAR_COURSE => $courseid, MB::PAR_SETUSER => $record->userid, MB::PAR_SUBTYPE => 'user']);
                }
            }

            $record->mod_url = $cm->url;

            foreach (['bug', 'suspended', 'uncounted', 'gthidden'] as $key){
                if ($record->$key ?? false){
                    $class_row[] = $key;
                }
            }

            if (empty($get(static::COLUMN_SUBMITTED))){
                $class_row[] = static::STATUSES[static::STATUS_NOT_SUBMIT];
            }

            $record->class_status = join(' ', $class_status);
            $record->class_row = join(' ', $class_row);
            $record->action_menu = $this->get_gt_action_menu($record, $my_url);

            $records[$record->id] = $record;
        }

        $this->set_count_data(...array_values($count_data));

        return $records;
    }

    /**
     * Process raw records for overview (Grading Statistics) page
     *
     * @param array $raw_records
     *
     * @return array
     */
    protected function _process_overview_table_raw_records(&$raw_records=[]){
        if (empty($raw_records)) return [];

        $records = [];
        $period_days = static::get_period_in_days($this->_period);
        $r = reset($raw_records);
        $r = get_object_vars($r);
        unset($r['graderid']);
        unset($r['gradername']);
        $keys = array_keys($r);
        $total = array_fill_keys($keys, 0);

        /** @var object $record */
        foreach ($raw_records as $record){
            /// data need for all
            $graded = $record->ontime + $record->late;
            $record->rate = $graded ? round(100 * $record->ontime / $graded) : 0;
            $record->gpd = $period_days ? round($graded/$period_days, 2) : 0;
            foreach ($keys as $key){
                $total[$key] += $record->$key ?? 0;
            }

            /// data need for download
            if ($this->_download){
                $record->rate .= '%';

                $records[$record->id] = $record;
                continue;
            }

            /// other data
            $record->class_status = '';
            if ($record->graderid == $this->_graderid){
                $record->active = true;
            }
            if ($record->graderid > 0){
                $record->gradername_url = $this->get_my_url(
                    [SH::PAR_GRADER => $record->graderid, static::PAR_VIEW => static::VIEW_COMPACT]);
            }

            $records[$record->graderid] = $record;
        }

        $count_records = count($records);
        if ($count_records > 1){
            $total = (object)$total;
            $graded = $total->ontime + $total->late;
            $total->rate = $graded ? round(100 * $total->ontime / $graded) : 0;
            $total->gpd = $period_days ? round($graded/$period_days, 2) : 0;
            $total->gradername = SH::str('total');
            if ($this->_download){
                $total->rate .= '%';
            } else {
                $total->class_status = 'total nosort';
            }
            $records[] = $total;
        }

        return $records;
    }

    /**
     * Get base joins for {@see _db_process_sql_query()}
     *
     * @param array $add_joins  - add additional joins to it
     * @param array $params     - array to save additional params
     * @param array $base_joins - joins for adding to the beginning of the join list
     *
     * @return array
     */
    protected function _db_get_base_join($add_joins=[], &$params=[], $base_joins=[]){
        $base_joins = parent::_db_get_base_join($add_joins,$params, $base_joins);

        // join NGC
        $ngc_table = SH::$ned_grade_controller::TABLE;
        $ngc_alias = static::_SQL_ALIAS_NGC;
        $base_joins[] = "
            LEFT JOIN {{$ngc_table}} AS $ngc_alias
                ON $ngc_alias.cmid = ".static::_sql_a(static::_SQL_CMID)."
                AND $ngc_alias.userid = ".static::_sql_a(static::_SQL_USERID)."
        ";
        // join AI
        $INF = static::_ai_infraction();
        if ($INF){
            $ai_table = $INF::TABLE;
            $ai_alias = static::_SQL_ALIAS_AI;
            $base_joins[] = "
                LEFT JOIN {{$ai_table}} AS $ai_alias
                    ON $ai_alias.cmid = ".static::_sql_a(static::_SQL_CMID)."
                    AND $ai_alias.student = ".static::_sql_a(static::_SQL_USERID)."
            ";
        }
        // join graders
        $base_joins[] = "
            LEFT JOIN {user} u2
                ON u2.id = ".static::_sql_a(static::_SQL_GRADERID)."
        ";

        // join tags
        $types = [];
        foreach (static::TYPE_TAGS as $type => $tag_name){
            $types[] = "SUM(DISTINCT tg.rawname = '$tag_name') AS type_$type";
        }
        $types = join(',', $types);
        $base_joins[] = "
                LEFT JOIN (
                    SELECT ti.itemid as itemid, $types,
                    GROUP_CONCAT(DISTINCT tg.rawname SEPARATOR ' ') AS tags
                    FROM {tag} tg 
                    LEFT JOIN {tag_instance} ti
                        ON ti.tagid = tg.id
                        AND ti.itemtype = :tiitemtype 
                        AND ti.component = :ticomponent
                    LEFT JOIN {course_modules} cm
                        ON cm.id = ti.itemid
                    GROUP BY ti.itemid
                ) AS tg
                    ON tg.itemid = ".static::_sql_a(static::_SQL_CMID)."
            ";
        $params['tiitemtype'] = 'course_modules';
        $params['ticomponent'] = 'core';

        return $base_joins;
    }

    /**
     * Process SQL query for the getting data from the DB for table view
     *
     * @param array|string $select
     * @param array|string $joins
     * @param array|string $where
     * @param array        $params
     * @param array|string $groupby
     * @param array|string $orderby
     *
     * @return bool - saves result in the params, return true if you can continue query
     */
    protected function _db_process_sql_query(&$select=[], &$joins=[], &$where=[], &$params=[], &$groupby=[], &$orderby=[]){
        [$select, $joins, $where, $groupby, $orderby] = SH::val2arr_multi(true, $select, $joins, $where, $groupby, $orderby);
        if (empty($select)){
            $select[] = static::_sql_a('*');
        }
        $joins = $this->_db_get_base_join($joins, $params);

        if ($this->_is_admin && !empty($this->_ids)){
            $ids = $this->_ids;
            // if we have filter, we should filter even admin call
            if (!empty($this->_filter_ids)){
                $ids = array_intersect($ids, $this->_filter_ids);
                if (empty($ids)){
                    return false;
                }
            }
            SH::sql_add_get_in_or_equal_options(static::_sql_a('id'), $ids, $where, $params);
        } else {
            static::_sql_set_simple_filters($where, $params,
                $this->_courseid, $this->_cmid, $this->_userid, $this->_groupid, null, $this->_schoolid,
                $this->_filter_ids, $this->_filter_id
            );

            $sql_now = SH::SQL_NOW;
            if ($this->_period){
                $time_start = static::_sql_a('timestart');
                $time_end = static::_sql_a('timeend');
                $time_grade = static::_sql_a('timegrade');
                $where[] = "(
                    (GREATEST($time_start, $time_grade) >= :timestart) OR
                    ($time_grade = 0 AND $time_end BETWEEN :timestart1 AND $sql_now)
                )";
                $params['timestart'] = $params['timestart1'] = static::get_period_time($this->_period);
            }

            if ($this->_active != static::ACTIVE_ALL){
                static::_sql_add_equal('suspended', ($this->_active == static::ACTIVE_NONE) ? 1 : 0, $where, $params);
            }

            if ($this->_view == static::VIEW_OVERVIEW){
                static::_sql_add_equal('uncounted', 0, $where, $params);
            } elseif ($this->_status_plus){
                switch ($this->_status_plus){
                    case static::STATUS_BUG:
                        static::_sql_add_equal('bug', 1, $where, $params);
                        break;
                    case static::STATUS_UNCOUNTED:
                        static::_sql_add_equal('uncounted', 1, $where, $params);
                        break;
                    case static::STATUS_PAUSED:
                        $where[] = static::_SQL_ALIAS_NGC.".status IN (:paused1, :paused2)";
                        $params['paused1'] = SH::$ned_grade_controller::ST_PAUSED;
                        $params['paused2'] = SH::$ned_grade_controller::ST_OBSOLETED;
                        break;
                }
            }

            if (!$this->_is_admin || !$this->_showhidden){
                static::_sql_add_equal('hidden', 0, $where, $params);
            }

            if ($this->_type){
                if (isset(static::TYPE_TAGS[$this->_type])){
                    $tag_select = 'type_'.$this->_type;
                    $where[] = "tg.$tag_select > 0";
                } elseif (isset(static::TYPE_MODS[$this->_type])){
                    $where[] = "m.name = :mod_name";
                    $params['mod_name'] = static::TYPE_MODS[$this->_type];
                }
            }

            if ($this->_attempt){
                switch ($this->_attempt){
                    case static::ATTEMPT_LAST:
                        static::_sql_add_equal('latest', 1, $where, $params);
                        break;
                    case static::ATTEMPT_NOT_LAST:
                        static::_sql_add_equal('latest', 0, $where, $params);
                        break;
                    case static::ATTEMPT_NOT_FIRST:
                        $where[] = static::_sql_a('attempt')." > 0";
                        break;
                    default: break;
                }
            }

            switch ($this->_selfgraded){
                default:
                case static::SELFGRADED_ALL:
                    break;
                case static::SELFGRADED_HIDE:
                    $where[] = static::_sql_a(static::_SQL_GRADERID)." <> ".static::_sql_a(static::_SQL_USERID);
                    break;
                case static::SELFGRADED_ONLY:
                    $where[] = static::_sql_a(static::_SQL_GRADERID)." = ".static::_sql_a(static::_SQL_USERID);
                    break;
            }
        }

        return true;
    }
    //endregion

    //region Get filter options [get_*_options()]
    /**
     * @return array
     */
    public function get_graders_options(){
        if (!$this->can_see_all()) return [];

        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $key = static::_sql_a(static::_SQL_GRADERID);
            $val = "IF(u.id IS NOT NULL, CONCAT(u.firstname, ' ', u.lastname), 'None')";
            $this->sql_get_grader_query($joins, $where, $params, $key);
            $data = $this->_sql_menu_filter($key, $val, $joins, $where, $params, static::GRADER_ALL, true);
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array
     */
    static public function get_period_options(){
        return [
            static::PERIOD_ALL => get_string('all'),
            static::PERIOD_1WEEK => SH::str('days_x', 7),
            static::PERIOD_2WEEK => SH::str('days_x', 14),
            static::PERIOD_1MONTH => SH::str('days_x', 30),
            static::PERIOD_2MONTH => SH::str('days_x', 60),
            static::PERIOD_6MONTH => SH::str('months_x', 6),
        ];
    }

    /**
     * Return all available statuses for current user
     *
     * @param bool $sort
     *
     * @return string[]
     */
    public function get_statuses($sort=false){
        $statuses = static::STATUSES;
        $statuses += static::STATUSES_PLUS;
        if ($this->_is_admin){
            $statuses += static::STATUSES_ADMIN;
        }

        if ($sort){
            $sorted = [];
            foreach (static::STATUSES_ORDER as $st){
                if (!isset($statuses[$st])) continue;

                $sorted[$st] = $statuses[$st];
            }
            // insert all others to the end
            $statuses = $sorted + $statuses;
        }

        return $statuses;
    }

    /**
     * Return all available types
     *
     * @return string[]
     */
    static public function get_types(){
        return [static::TYPE_ALL => 'all'] + static::TYPE_TAGS + static::TYPE_MODS;
    }
    //endregion

    //region Secondary render methods
    /**
     * Render this class element
     *
     * @param      $id
     * @param bool $return
     *
     * @return string
     */
    public function render_main_row($id, $return=false){
        $this->_filter_ids = [$id];
        $data = array_values($this->get_main_table_data());
        if (empty($data)){
            return '';
        }

        $this->_before_start_export();
        $this->content->data = [reset($data)];
        $this->_before_finish_export();
        $res = SH::render_from_template('grading_tracker_render_rows', $this->content);
        if (!$return){
            echo $res;
        }
        return $res;
    }

    /**
     * Get GT secondary menu for single record or bulk menu
     *
     * @param null|\stdClass $record - record, for which menu is
     * @param \moodle_url    $url    - current moodle_url
     * @param bool           $render - if true, return string, otherwise return action_menu
     *
     * @return \action_menu|string
     */
    public function get_gt_action_menu($record=null, $url=null, $render=true){
        $menu = new \action_menu();
        $menu->set_menu_trigger(SH::fa('fa-caret-down'));

        $menu->attributes['class'] .= ' ned-actionmenu';
        $m_add = function($call_form_or_name, $add_url=null, $class=[]) use (&$menu){
            $url = $add_url ?: ['#'];
            $class = SH::val2arr($class);
            $class[] = $call_form_or_name;
            if (!$add_url){
                $class[] = 'callform';
            }

            $menu->add(SH::link($url, $call_form_or_name, $class));
        };

        // action-menu which call modal with form
        if ($record){
            $m_add($record->bug ? 'unreportbug' : 'reportbug');
            $m_add($record->uncounted ? 'setcounted' : 'setuncounted');
        } else {
            $m_add('reportbug');
            $m_add('unreportbug');
            $m_add('setuncounted');
            $m_add('setcounted');
        }

        if ($this->_is_admin){
            if ($record){
                $m_add($record->gthidden ? 'gtshow' : 'gthide');
            } else {
                $m_add('gthide');
                $m_add('gtshow');
            }
        }

        $url = $url ? clone($url) : $this->get_my_url();
        if ($record->id ?? false){
            $url->param(SH::PAR_ACTION, static::ACTION_REFRESH);
            $url->param(SH::PAR_FILTER_ID, $record->id);
            $m_add('refreshnow', $url);
        } else {
            $url->param(SH::PAR_FILTER_IDS, '{ids}');
            $m_add('showonlychosenrecords', $url, 'link');
        }

        if ($render){
            return $this->_o->render($menu);
        }

        return $menu;
    }

    /**
     * Render one header cell
     *
     * @param int                $column          - const column key from the COLUMNS_PARAMETERS
     * @param string|\moodle_url $url             - optional, url of the page
     * @param string             $additional_info - optional HTML, added to the column name
     * @param null|bool|int      $sort_desc       - current sort direction, null if sorted by another column
     *
     * @return string
     */
    protected static function _render_column_cell($column, $url='', $additional_info='', $sort_desc=null){
        if (empty($url)){
            $url = static::get_url();
        } else {
            $url = new \moodle_url($url);
        }

        $th_class = '';
        $url->param(SH::PAR_SORT, $column);
        if (!is_null($sort_desc)){
            $url->param(SH::PAR_SORT_DESC, !$sort_desc);
            $th_class = 'sort-' . ($sort_desc ? 'down' : 'up');
        } else {
            $url->remove_params(SH::PAR_SORT_DESC);
        }

        $rendered_div = SH::div(static::get_column_name($column), 'tablesorter-header-inner');
        $link = SH::link($url, $rendered_div, 'sort_link', ['tabindex' => 1]);

        return SH::tag('th', $additional_info.$link, $th_class, ['scope' => 'col']);
    }
    //endregion

    //region Main render page content methods
    /**
     * Init some content values before {@see export_for_template()}
     * Normally calling from the {@see export_for_template()}
     */
    protected function _before_start_export(){
        parent::_before_start_export();

        $this->content->contextid = $this->_ctx->id;
        $statuses = $this->get_statuses();
        $this->content->table_class = [$statuses[$this->_status] ?? 'all-statuses'];

        $this->content->notifications = [];
        $this->content->admin_links = [];
        $this->content->control_panel1 = [];
        $this->content->control_panel2 = [];
        $this->content->control_panel3 = [];

        $this->content->show_timegrade = ($this->_status != static::STATUS_WAIT &&
            $this->_status != static::STATUS_UPCOMING && $this->_status != static::STATUS_NOT_SUBMIT);
        $this->content->show_uncounted_reason = $this->_status == static::STATUS_UNCOUNTED;
        $this->content->show_bugreport = $this->_status == static::STATUS_BUG;
        $this->content->statisticdate = GT::get_date_from_text(GT::get_config('gradingtracker_statisticdate'));

        $this->content->is_admin = $this->_is_admin;
        if ($this->_is_admin){
            $this->content->show_note = $this->_showhidden;
            $this->content->show_ids = !empty($this->_ids);
        }

        $this->content->{static::VIEWS[$this->_view]} = true;
        $this->content->main_table = ($this->content->expanded ?? 0) || ($this->content->compact ?? 0);
    }

    /**
     * Render page content
     * Normally calling from the {@see export_for_template()}
     * You can rewrite this method and not changing original {@see export_for_template()}
     * Please, do not use redirect methods here, normally they should be called from {@see before_header_output()}
     * @noinspection DuplicatedCode
     */
    protected function _render_page_content(){
        $this->content->data = $this->get_table_data();

        if ($this->_view == static::VIEW_OVERVIEW){
            $this->content->show_gpd = $this->_period != static::PERIOD_ALL;
            SH::$C::js_call_amd('add_sorter','add_sort', ['table.nedtable']);
        } else {
            // admin refreshing link
            if ($this->_is_admin){
                // we need data here for updating admin link
                if (!empty($this->content->data)){
                    [$records_count, $uncounted_c, $notsubmit_c, $usual_c] = $this->get_count_data();
                    $statuses = $this->get_statuses();
                    $additional_info = '';
                    if ( ($uncounted_c || $notsubmit_c) && ($usual_c || ($uncounted_c && $notsubmit_c)) ){
                        $additional_info = [];
                        $check_count = [static::STATUS_UNCOUNTED => $uncounted_c, static::STATUS_NOT_SUBMIT => $notsubmit_c, 'usual' => $usual_c];
                        foreach ($check_count as $name => $value){
                            if (!$value) continue;

                            $status_c = $statuses[$name] ?? $name;
                            $additional_info[] = SH::str($status_c) .' '.$value;
                        }
                        $additional_info = ' {'.join(', ', $additional_info).'}';
                    }

                    $this->content->admin_links[] = SH::link(
                        [$this->get_my_url([SH::PAR_ACTION => static::ACTION_UPDATE])],
                        SH::str('update_records_x', ['count' => $records_count, 'additional_info' => $additional_info]), '',
                        ['onclick' => 'return confirm("' . SH::str('messagerecalculation') . '");']
                    );
                }
            }

            $this->content->headers = $this->render_table_header();
        }

        $url = $this->get_my_url();
        $this->content->pager = $this->r_pager();

        $grader_selector = $this->r_grader_selector();
        if (!empty($grader_selector)){
            $this->content->control_panel1[] = $grader_selector;
        }

        if ($this->_view != static::VIEW_COMPACT){
            $school_selector = $this->r_school_selector();
            if (!empty($school_selector)){
                $this->content->control_panel2[] = $school_selector;
            }

            if (!$this->_course_view){
                $this->content->control_panel2[] = $this->r_course_selector();
            }

            if ($this->_courseid > 0){
                $this->content->control_panel2[] = $this->r_cm_selector();
                $this->content->control_panel2[] = $this->r_group_selector();
            }

            $this->content->control_panel2[] = $this->r_users_selector();
        }

        if ($this->_is_admin){
            $this->content->control_panel3[] = SH::single_checkbox($url, $this->_showhidden, static::PAR_SHOW_HIDDEN);
        }

        if ($this->_view != static::VIEW_OVERVIEW){
            $this->content->control_panel3[] = $this->r_perpage_selector();

            $statuses = $this->get_statuses(true);
            $options = static::get_list_as_options($statuses, true);
            $this->content->control_panel1[] =
                SH::single_select($url, SH::PAR_STATUS, $options, $this->_status, SH::str('status'));

            if (!empty($this->content->data)){
                $this->content->bulk_menu = $this->get_gt_action_menu();
            }
        }

        if ($this->_view != static::VIEW_COMPACT){
            $options = static::get_period_options();
            $this->content->control_panel1[] =
                SH::single_select($url, static::PAR_PERIOD, $options, $this->_period, SH::str('period'));
        }

        $options = static::get_list_as_options(static::get_types(), false);
        $this->content->control_panel1[] =
            SH::single_select($url, static::PAR_TYPE, $options, $this->_type, SH::str('type'));

        if ($this->_view != static::VIEW_COMPACT){
            $options = static::get_list_as_options(static::ACTIVES);
            $this->content->control_panel1[] =
                SH::single_select($url, SH::PAR_ACTIVE, $options, $this->_active, SH::str('active'));
        }

        $options = static::get_list_as_options(static::ATTEMPTS, false);
        $this->content->control_panel1[] =
            SH::single_select($url, static::PAR_ATTEMPT, $options, $this->_attempt, SH::str('gt:attempts'));

        if ($this->_view != static::VIEW_COMPACT){
            $options = static::get_list_as_options(static::SELFGRADED_OPTIONS, false);
            $this->content->control_panel1[] =
                SH::single_select($url, static::PAR_SELFGRADED, $options, $this->_selfgraded, SH::str('gt:selfgraded'));
        }

        // PAR_VIEW should be last
        $view_url = $this->_view == static::VIEW_EXPANDED ? new \moodle_url(static::URL) : $url;
        $options = static::get_list_as_options(static::VIEWS_MENU);
        $this->content->control_panel1[] =
            SH::single_select($view_url, static::PAR_VIEW, $options, $this->_view, SH::str('view'));

        $this->content->download_element = SH::download_dataformat_selector('exportdataas', $this->get_my_url());
        $this->content->download_unlimited = $this->render_du_element();

        SH::js_call_amd('add_modal','add_modal', ['.information .about']);
    }

    /**
     * Render table header
     * Require to call function @see _before_start_export() before.
     *
     * @return array
     */
    public function render_table_header(){
        if (empty($this->content->data)) return [];

        $columns = $this->get_columns();
        $url = $this->get_my_url();
        $result = [];

        // master checkbox
        $checkbox = SH::tag('input', '', 'ned_tags_checkbox',
            ['type' => 'checkbox', 'name' => 'head_checkbox', 'value' => 0]);
        $additional_info = SH::span($checkbox,'checkbox-container');

        foreach ($columns as $column){
            $sort_desc = null;
            if ($column == $this->sort){
                $sort_desc = $this->sort_desc;
            }

            $result[] = static::_render_column_cell($column, $url, $additional_info, $sort_desc);

            // reset $additional_info after first column
            $additional_info = '';
        }

        return $result;
    }

    /**
     * Render "download unlimited" control element
     *
     * @return string - HTML element, checkbox to switch download_unlimited property
     */
    public function render_du_element(){
        $on = SH::fa('fa-sitemap color-green', '', SH::str('gt:download_unlimited:off'), ['data-toggle' => 'tooltip']);
        $off = SH::fa('fa-sitemap color-grey', '', SH::str('gt:download_unlimited:on'), ['data-toggle' => 'tooltip']);
        return SH::single_checkbox2($this->get_my_url(), $this->_download_unlimited, static::PAR_DOWNLOAD_UNLIMITED, $on, $off, ' mx-1');
    }
    //endregion

    //region Main rendering class/page methods
    /**
     * @param       $gtid
     * @param array $params
     *
     * @return string
     */
    static public function GTR_get_rendered_row($gtid, $params=[]){
        $GTR = new static();
        $GTR->init(false, $params, false);
        return $GTR->render_main_row($gtid, true);
    }

    /**
     * Get not-rendered overview data
     *
     * @param bool $raw - if true, get data as for downloading
     * @param array $params - additional page params
     *
     * @return array
     */
    static public function GTR_get_overview_GT_data($raw=true, $params=[]){
        $params = SH::val2arr($params);
        $params[static::PAR_VIEW] = static::VIEW_OVERVIEW;
        $params[SH::PAR_DOWNLOAD] = $raw ? static::CRON_WORK : false;

        $GTR = new static();
        $GTR->init(false, $params, false);
        return $GTR->get_overview_table_data();
    }

    /**
     * Check (and filter) GT ids for current user and calling params
     *
     * @param array|int|string $gt_ids
     * @param array $params
     *
     * @return array
     */
    static public function GTR_check_gt_ids($gt_ids, $params=[]){
        $GTR = new static();
        $GTR->init(false, $params, false);
        $GTR->set_filter_ids($gt_ids);

        return $GTR->get_main_table_data(true);
    }
    //endregion
}
