<?php
/**
 * @package    block_ned_teacher_tools
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools;
use block_ned_teacher_tools\shared_lib as SH;
use local_ned_controller\tt_config_manager as CM;
use PhpOffice\PhpSpreadsheet\Exception;

defined('MOODLE_INTERNAL') || die();

/**
 * Class shared_lib
 *
 * @package block_ned_teacher_tools
 */
class shared_lib extends \local_ned_controller\shared\base_class {
    use \local_ned_controller\shared\tt_and_sm;

    const CONF_DM_X_DAYS_BETWEEN_DL_QUIZ = 3;
    const CONF_DM_X_DAYS_BETWEEN_DL_NON_QUIZ = 2;
    const CONF_DM_ALLOW_QUIZ_OTHER_DL_IN_ONE_DAY = 0;

    /**
     * @var \block_ned_teacher_tools\course_users_status @readonly - course_users_status class
     */
    static public $course_users_status = '\block_ned_teacher_tools\course_users_status';
    /**
     * @var \block_ned_teacher_tools\deadline_manager @readonly - deadline_manager class
     */
    static public $DM = '\block_ned_teacher_tools\deadline_manager';

    /**
     * Return user preference block_ned_teacher_tools_dm_timezone or NED default timezone
     *
     * @param bool $user_timezone - if true, return USER timezone
     *
     * @return int|string
     */
    static public function get_user_dm_timezone($user_timezone=false){
        global $USER;
        if ($user_timezone){
            return $USER->timezone ?? SH::NED_TIMEZONE;
        }
        return get_user_preferences('block_ned_teacher_tools_dm_timezone', SH::NED_TIMEZONE);
    }

    /**
     * Check TT capability 'view_timezone_menu' and return user preference block_ned_teacher_tools_dm_timezone
     *  or NED default timezone
     *
     * @param \context $context
     *
     * @return int|string
     */
    static public function get_user_dm_timezone_ctx($context=null){
        return static::get_user_dm_timezone(!static::has_capability('view_timezone_menu', $context));
    }

    /**
     * @param object|numeric      $courseorid
     *
     * @return array
     */
    static public function get_course_metadata($courseorid) {
        $courseid = SH::get_id($courseorid);
        $handler = \core_customfield\handler::get_handler('core_course', 'course');
        $datas = $handler->get_instance_data($courseid, true);
        $metadata = [];
        foreach ($datas as $data) {
            if (empty($data->get_value())) {
                continue;
            }
            $metadata[$data->get_field()->get('shortname')] = [
                'export_value' => $data->export_value(),
                'value' => $data->get_value()
            ];
        }

        return $metadata;
    }

    /**
     * @param object|numeric  $courseorid
     * @param int|mixed       $status
     *
     * @return bool
     */
    static public function get_check_permission_view_grade($courseorid, $status) {
        $courseid = SH::get_id($courseorid);
        $course_metadata = SH::get_course_metadata($courseid);
        $coursetype = $course_metadata['course_type']['export_value'] ?? '';
        if (SH::has_capability('seecoursegradewhilewaitingforfinalgrade', \context_course::instance($courseid))) {
            if ($coursetype == course_users_status::COURSE_TYPE_AUTO_SET_WAITING_FOR_GRADE) {
                if (in_array($status, [course_users_status::STATUS_WAITING_FOR_FINAL_GRADE, course_users_status::STATUS_WAITING_FOR_INTERNAL_REVIEW])) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * @return string
     */
    public static function render_course_completed_key(){
        $r = function($icon, $text){ return SH::row([SH::fa($icon), SH::str($text)]); };

        $table = SH::html_table(' ');
        $table->caption = SH::str('coursecompletion');
        $table->data[] = $r('tt-status-icon tt-si-inprogress', 'courseinprogress');
        $table->data[] = $r('tt-status-icon tt-si-waitingfinalgrade', 'coursecompletewffg');
        $table->data[] = $r('tt-status-icon tt-si-waitingforinternalreview', 'coursecompletewffir');
        $table->data[] = $r('tt-status-icon tt-si-pass', 'coursecompletepass');
        $table->data[] = $r('tt-status-icon tt-si-fail', 'coursecompletefail');

        return SH::render_table($table, 'coursecompleted_keys');
    }

    /**
     * @param string $caption
     * @param array  $statuses
     * @param array  $texts     (optional) if not texts, it will use statuses as text too
     *
     * @return string - html table
     */
    public static function render_class_progress_legend_table($caption='', $statuses=[], $texts=[]){
        $table = SH::html_table(' ');
        if (!empty($caption)){
            $table->caption = SH::str($caption);
        }

        foreach ($statuses as $i => $status){
            $table->data[] = SH::row([
                SH::div(SH::get_ned_grade_element(null, $status)),
                SH::$C::str($texts[$i] ?? $status)]);
        }

        return SH::render_table($table, $caption);
    }

    /**
     * Get legend with NGC zeros statuses
     *
     * @return string - html table
     */
    public static function render_AZ_legend_table(){
        $table = SH::html_table(' ');
        $NGC = SH::$ned_grade_controller;
        $add2table= function($icons, $texts, $add=null) use (&$table){
            if ($add){
                $texts[] = SH::$C::str($add);
                $icons = [$add => $add] + $icons;
            }
            $table->data[] = SH::row([
                SH::div(SH::get_ned_grade_element($icons)),
                SH::arr2str($texts, '', ' - ')
            ]);
        };

        $table->caption = SH::$C::str($NGC::GRADE_TYPES[$NGC::GT_AWARD_ZERO]);
        $reasons = SH::array_remove_by_values($NGC::REASONS_GT[$NGC::GT_AWARD_ZERO], $NGC::REASON_OTHER);
        foreach ($reasons as $reason){
            if ($reason == $NGC::REASON_FILE) continue;

            $texts = [SH::$C::str($NGC::REASONS[$reason])];
            $icons = $NGC::get_grade_status($NGC::GT_AWARD_ZERO, $reason);
            if ($reason === $NGC::REASON_AI){
                $add2table($icons, $texts);
            } else {
                foreach (SH::ST_ZEROS as $st_zero){
                    $add2table($icons, $texts, $st_zero);
                }
            }
        }

        return SH::render_table($table, SH::STATUS_NGC_ZERO);
    }

    /**
     * @param bool $has_kica
     * @param bool $course_status_controller
     *
     * @return string
     */
    public static function render_progress_report_legend_table($has_kica=false, $course_status_controller=false){
        global $PAGE;
        $output = [];
        $url = $PAGE->url;
        $output[] = SH::div(SH::link((new \moodle_url($url, ['export' => 1]))->out(), SH::str('downloadclasslist')), 'export-link mt-3 mr-5');
        if ($course_status_controller){
            $output[] = SH::render_course_completed_key();
        }

        if ($has_kica) {
            $S_F = [];
            $S_F[] = SH::render_class_progress_legend_table('summativeactivities', SH::ST_SUMMATIVE);
            $S_F[] = SH::render_class_progress_legend_table('formativeactivities', SH::ST_FORMATIVE);
            $output[] = SH::div($S_F);
            $output[] = SH::render_class_progress_legend_table('general', SH::ST_GENERAL);

        } else {
            $output[] = SH::render_class_progress_legend_table('activities',
                array_merge(SH::ST_NOT_KICA, SH::ST_GENERAL), SH::ST_FORMATIVE);
        }

        $output[] = SH::render_AZ_legend_table();

        return SH::div(SH::div($output, 'row'), 'container class-progress-legend');
    }

    /**
     * Get extension popover attributes, if can
     *
     * @param \cm_info|int|string      $cm_or_id        - it's better to use cm_info, if you already loaded it
     * @param object|int|string        $student_or_id
     * @param object|numeric|null      $grader_or_id    (optional) Uses current $USER by default
     * @param \context_course|\context $context         (optional) Uses $course context by default
     * @param int|null                 $deadline        (optional) current deadline, if already loaded
     * @param bool                     $check_submitted (optional) if false, ignore submitted user activity or not
     * @param bool                     $load_by_course  (optional) if true, load all data by courseid, otherwise by courseid and userid
     * @param object|numeric           $courseorid      (optional) course object (or its id) if loaded,
     *                                                  improves optimization if $cm_or_id is represented as ID
     *
     * @return array - attributes for html element
     */
    public static function get_class_progress_extension_popover_attributes($cm_or_id, $student_or_id, $grader_or_id=null, $context=null,
        $deadline=null, $check_submitted=true, $load_by_course=false, $courseorid=null){
        $attributes = [];
        [$use_extension, $numberofextensions, $can_add_extension, $showextensionicon] =
            SH::$DM::get_activity_extension_data($cm_or_id, $student_or_id, $grader_or_id, $context,
                $deadline, $check_submitted, $load_by_course, $courseorid);
        if ($use_extension && $can_add_extension){
            [$cmid, $studentid] = SH::get_ids($cm_or_id, $student_or_id);
            $attributes = [
                'data-bind' => 'popover',
                'data-content' => SH::render_from_template('class_progress_extension_popover', [
                    'userid' => $studentid,
                    'cmid' => $cmid,
                    'numberofextensions' => $numberofextensions,
                    'showextensionicon' => $showextensionicon,
                    'can_add_extension' => $can_add_extension,
                ]),
            ];
        }

        return $attributes;
    }

    /**
     * {@see tt_config_manager::is_resubmission_enabled()}
     *
     * @param int $courseid
     *
     * @return bool are resubmissions enabled in course
     */
    public static function is_resubmission_enabled($courseid){
        return CM::is_resubmission_enabled($courseid);
    }

    /**
     * @param $students
     * @return void
     * @throws Exception
     * @throws \PhpOffice\PhpSpreadsheet\Writer\Exception
     */
    public static function export_class_list($students) {
        global $CFG;

        require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");

        ob_start();
        set_time_limit(300);
        raise_memory_limit(MEMORY_EXTRA);

        $table = new \stdClass();
        $table->head = ['firstname', 'lastname'];

        $counter = 0;
        $filename = 'Class_List_'.(date('Y-m-d'));
        $downloadfilename = clean_filename($filename);

        $workbook = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $myxls = $workbook->setActiveSheetIndex(0);

        $numberofcolumns = count($table->head);

        // Header row.
        foreach ($table->head as $key => $heading) {
            $cell = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($key + 1) . '1'; // A1 cell address.
            $myxls->setCellValue($cell, str_replace("\n", ' ', htmlspecialchars_decode(strip_tags(nl2br(self::str($heading))))));
        }

        $rownum = 2;
        foreach ($students as $student) {
            $row = array();
            $columnum = 1;
            foreach ($table->head as $column) {
                $cell = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($columnum) . $rownum; // A2 cell address.
                $myxls->setCellValue($cell, $student->$column);
                $columnum++;
            }
            $rownum++;
        }

        // Set active sheet index to the first sheet, so Excel opens this as the first sheet
        $workbook->setActiveSheetIndex(0);

        $objWriter = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($workbook);

        if (ob_get_length()) {
            ob_end_clean();
        }
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="'.$downloadfilename.'.xlsx"');
        header('Cache-Control: max-age=0');

        ob_end_clean();
        $objWriter->save('php://output');
        exit;
    }
}

shared_lib::init();
