<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use block_ned_teacher_tools\shared_lib as SH;

defined('MOODLE_INTERNAL') || die();
require_once(dirname(__FILE__) . '/../../config.php');
require_once($CFG->dirroot . '/blocks/ned_teacher_tools/lib.php');

/**
 * Simple ned_teacher_tools block config form definition
 *
 * @package    contrib
 * @subpackage block_ned_teacher_tools
 * @copyright  2011 MoodleFN
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * Simple ned_teacher_tools block config form class
 *
 * @copyright 2011 MoodleFN
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnused
 */

class block_ned_teacher_tools_edit_form extends block_edit_form {

    const PAR_ACTION = 'tt_action';

    const ACTION_NONE = 0;
    const ACTION_CHECK_CTA = 1;

    /**
     * @param MoodleQuickForm $mform - quickform object definition
     */
    protected function specific_definition($mform) {
        $this->check_actions();

        // Useful functions
        $mform->_attributes['class'] = 'mform block_ned_teacher_tools_edit_form';
        $config = get_config(SH::$PLUGIN_NAME);

        $check_vars = function($name, $visiblename=null, $defaultsetting=null, $defdef='') use ($config) {
            $visiblename = SH::str_check(empty($visiblename) ? $name : $visiblename);
            $defaultsetting = is_null($defaultsetting) ? ($config->$name ?? $defdef) : $defaultsetting;
            $name = 'config_' . $name;
            return [$name, $visiblename, $defaultsetting];
        };

        $configheading = function($name, $heading='') use ($mform){
            $heading = $heading ?: $name;
            $mform->addElement('header', $name, SH::str_check($heading));
        };
        $configstatic = function($name, $visiblename='', $text='') use ($mform){
            $visiblename = empty($visiblename) ? SH::str_check($name, '') : SH::str_check($visiblename);
            $mform->addElement('static', $name, $visiblename, SH::str_check($text));
        };
        $configselect = function($name, $visiblename='', $defaultsetting=null, $choices=[]) use ($mform, $check_vars){
            [$name, $visiblename, $defaultsetting] = $check_vars($name, $visiblename, $defaultsetting, 0);
            $mform->addElement('select', $name, $visiblename, $choices);
            $mform->getElement($name)->setSelected($defaultsetting);
        };
        $configyesno = function($name, $visiblename='', $defaultsetting=null, $attrib = []) use ($mform, $check_vars){
            [$name, $visiblename, $defaultsetting] = $check_vars($name, $visiblename, $defaultsetting, 0);
            $mform->addElement('selectyesno', $name, $visiblename, $attrib);
            $mform->setDefault($name, $defaultsetting);
        };
        $configtext = function($name, $visiblename='', $defaultsetting=null, $paramtype=PARAM_RAW) use ($mform, $check_vars){
            [$name, $visiblename, $defaultsetting] = $check_vars($name, $visiblename, $defaultsetting);
            $mform->addElement('text', $name, $visiblename);
            $mform->setType($name, $paramtype);
            $mform->setDefault($name, $defaultsetting);
        };
        $configtextarea = function($name, $visiblename='', $defaultsetting=null, $paramtype=PARAM_RAW) use ($mform, $check_vars){
            [$name, $visiblename, $defaultsetting] = $check_vars($name, $visiblename, $defaultsetting);
            $mform->addElement('textarea', $name, $visiblename, 'wrap="virtual" rows="10" cols="100"');
            $mform->setType($name, $paramtype);
            $mform->setDefault($name, $defaultsetting);
        };
        $configeditor = function($name, $visiblename='', $defaultsetting='', $paramtype=PARAM_RAW) use ($mform, $check_vars){
            [$name, $visiblename, $defaultsetting] = $check_vars($name, $visiblename, $defaultsetting);
            $mform->addElement('editor', $name, $visiblename);
            $mform->setType($name, $paramtype);
            $mform->setDefault($name, $defaultsetting);
        };
        $configtags = function($name, $visiblename='', $defaultsetting=[]) use ($mform, $check_vars){
            [$name, $visiblename, $defaultsetting] = $check_vars($name, $visiblename, $defaultsetting);
            $tag_options = ['itemtype' => 'course_modules', 'component' => 'core'];
            $tags_selector = $mform->addElement('tags', $name, $visiblename, $tag_options);
            $tags_selector->_attributes['ned_tags_selector'] = '';
            $mform->setDefault($name, $defaultsetting);
        };
        $config_hide_if = function($element_name, $dependent, $condition=SH::COND_KEY_NEQ, $val=1) use ($mform){
            // default $condition and $val will hide $element_name if $dependent value != 1
            $mform->hideif('config_'.$element_name, 'config_'.$dependent, $condition, $val);
        };

        $config_add_help = function($name, $identifier_without_help='') use ($mform, $check_vars) {
            $identifier_without_help = $identifier_without_help ?: $name;
            [$name,,] = $check_vars($name);
            $mform->addHelpButton($name, $identifier_without_help, SH::TT);
        };

        $get_link = function($t_url='', $text='', $param_url=[], $attr=['target' => '_blank']){
            $m_url = new moodle_url($t_url, $param_url);
            if (!empty($text)) {
                $text = SH::str_check($text);
            } else {
                $text = $t_url;
            }
            return html_writer::link($m_url, $text, $attr);
        };

        $hideshow = array(0 => get_string('hide'), 1 => get_string('show'));

        // Configuring a NED Teacher Tools.
        $pluginname = $config->blocktitlecourselevel ?: SH::str('blocktitle');

        $configheading('configheader', SH::str('blockconfiguring', $pluginname));
        // General
        $configtext('block_title', '', $pluginname);
        $configstatic('blockinfo', '',
            $get_link('/local/ned_controller/index.php', get_string('pluginname', 'local_ned_controller')));

        $configstatic('blocksettings', SH::str('blockconfiguring', $pluginname),
            $get_link('/admin/settings.php', 'opensitesettingspage', ['section' => 'blocksettingned_teacher_tools'])
        );

        if ($this->block->page->course->id == SITEID) return;

        // continue General section
        $configyesno('excludewithoutsla');
        // Manage activities
        $manage_url = SH::url('~/manage_activities_settings.php', ['id' => $this->page->course->id]);
        $configstatic('config_manage', 'manage',
            $get_link($manage_url, 'manage_activities_settings', [], ['class' => 'btn'])
        );
        // Participation power
        $configheading('participationpower');
        $configyesno('participationpower_option');
        // Resubmissions Manager
        $configheading('resubmissionsmanager');
        $configyesno('enableresubmissions',0);
        $configselect('resubmissioncriteria',0, null,
            [0 => SH::str('none'), 1 => SH::str('participationpower')]);
        $configtext('participationpowerrequired','', null,PARAM_INT);
        $config_hide_if('participationpowerrequired', 'resubmissioncriteria');
        $configtext('maximumattemptsincourse','', null,PARAM_INT);
        $configtext('maximumattempts','', null,PARAM_INT);
        $configtext('daysforresubmission','', null,PARAM_INT);

        // Deadline Manager
        $configheading('deadlinemanager');

        $configyesno('enabledeadlinemanager', '', 0);
        $configyesno('enabledeadlineextension', '', 0);

        $configyesno('classstudentenddateextension', '', 1);
        $configyesno('allowexpireddeadlineupdate', '', 0);
        $configtext('maxdaysbetween', '', 90, PARAM_INT);

        $configyesno('syncdmenddateswithenrolment', '', 1);
        $configtext('setenrolmentenddayduration', '', 15, PARAM_INT);

        $roles = \block_ned_teacher_tools\utils::get_assignable_course_roles();
        $configselect('enrolmentenddaterole', '', 5,  [0 => SH::str('none')] + $roles);
        $configtext('maxextensionperstudent', '', 20, PARAM_INT);
        $configtext('maxextensionperactivity', '', 3, PARAM_INT);

        $configyesno('forcequizsequence', '', 1);
        $configtext('x_days_between_dl_quiz', '', SH::CONF_DM_X_DAYS_BETWEEN_DL_QUIZ, PARAM_INT);
        $configtext('x_days_between_dl_other', '', SH::CONF_DM_X_DAYS_BETWEEN_DL_NON_QUIZ, PARAM_INT);
        $configyesno('allow_quiz_other_dl_in_one_day', null, SH::CONF_DM_ALLOW_QUIZ_OTHER_DL_IN_ONE_DAY);
        $configselect('x_days_apply_to_all', '', 0, [0 => SH::str('group'), 1 => SH::str('all')]);
        $configyesno('enablemidtermpoint', '', 0);

        // Final Evaluation
        $configheading('finalevaluation');
        $configyesno('fe_coursecompletion', '', 0);
        $configtags('fe_tags');
        $configyesno('allowextentionsafterccompletion', '', 0);

        // Course Completion Status Controller
        $configheading('coursecompletionstatuscontroller');
        $configyesno('autosetwfgstatus', '', 0);
        $configyesno('autosetwfirstatus', '', 0);

        // Marking manager
        $configheading('markingmanager');
        $configyesno('keepseparate', '', 1);
        $configyesno('showtopmessage');

        $configeditor('topmessage');

        // Student Progress
        $configheading('studentprogress');

        $configyesno('showgeneralinfosection','', 1);
        $configyesno('showsummativeactivitiessection','', 1);
        $configyesno('showformativeactivitiessection','', 1);
        $configyesno('showformativesummativeicons','', 1);
        $configyesno('showduedatecolumn','', 1);
        $configyesno('showtagsfilter','', 1);

        // CTA
        $menu_def = [-1 => get_string('none'), 0 => 'default']; // '0' value will be replaced lately in code
        $configheading('ctatracker');
        $configyesno('ctatracker_enable', '', 0);
        $config_add_help('ctatracker_enable');

        /** @var grade_scale[] $scales */
        $scales = (grade_scale::fetch_all_global() ?: []) + (grade_scale::fetch_all_local($this->page->course->id) ?: []);
        $scale_options = $menu_def;
        foreach ($scales as $scale){
            if (!$scale->is_hidden()){
                $scale_options[$scale->id] = $scale->get_name();
            }
        }
        $scale_def = SH::get_config('cta_scale') ?: -1;
        $scale_options[0] = SH::$C::str('sitedefault_x', $scale_options[$scale_def] ?? $scale_options[-1]);
        $configselect('cta_scale', '', 0, $scale_options);

        $role_ids = get_roles_for_contextlevels(CONTEXT_MODULE); // some_id => roleid
        $where = [];
        $params = [];
        SH::sql_add_get_in_or_equal_options('id', $role_ids, $where, $params);
        $role_objects = SH::db()->get_records_select('role', SH::sql_condition($where), $params);
        $role_options = $menu_def + role_fix_names($role_objects, null, ROLENAME_ALIAS, true);
        $role_def = SH::get_config('cta_role') ?: -1;
        $role_options[0] = SH::$C::str('sitedefault_x', $role_options[$role_def] ?? $role_options[-1]);
        $configselect('cta_role', '', 0, $role_options);
        $config_add_help('cta_role');

        $configstatic('cta_check_button', SH::HTML_SPACE,
            $get_link(SH::page()->url, 'cta_graders_update', [static::PAR_ACTION => static::ACTION_CHECK_CTA], ['class' => 'btn'])
        );

        $cta_elems = ['cta_scale', 'cta_role'];
        foreach ($cta_elems as $cta_elem){
            $config_hide_if($cta_elem, 'ctatracker_enable');
        }

        // Custom links.
        $configheading('customlinks');

        $mform->addElement('textarea', 'config_customlinks', '', 'wrap="virtual" rows="10" cols="100"');
        $configstatic('config_customlinks_desc', '', 'customlinksconfigdesc');

        // Site links.
        $mform->addElement('html', html_writer::div('Site links', 'ned_controller_configsubtitle'));
        if ($numberoflinks = $config->numberoflinks ?? 0) {
            for ($i = 1; $i <= $numberoflinks; $i++) {
                $sitelinkgroup=array();
                $sitelinkgroup[] = $mform->createElement('select', 'config_sitelink_'.$i, '', $hideshow);
                $sitelinkgroup[] = $mform->createElement('static', 'config_sitelink_title_'.$i, '',
                    html_writer::div(get_config('block_ned_student_menu', 'customlinkstitle_' . $i),
                        'block_ned_teacher_tools_sitelink-title'));
                $mform->addGroup($sitelinkgroup, 'sitelinkgroup'.$i, '', array(' '), false);
                $mform->setDefault('config_sitelink_'.$i, 1);
            }
        }

        // General Settings.
        $configheading('generalsettings');
        $configyesno('showlinktogradebook','showlinktogradebook', true);
        $configyesno('include_orphaned');
    }

    /**
     * Check action and redirect, if there is any action
     *
     * @return void
     */
    protected function check_actions(){
        $action = optional_param(static::PAR_ACTION, static::ACTION_NONE, PARAM_INT);
        switch ($action){
            default:
            case static::ACTION_NONE:
                // do nothing, just return
                return;

            case static::ACTION_CHECK_CTA:
                try {
                    \block_ned_teacher_tools\cta_tracker::update_cta_graders_by_course($this->page->course);
                    SH::notification_add(SH::str('cta_graders_update_success'), SH::NOTIFY_SUCCESS);
                } catch (\Throwable $e){
                    $e_info = get_exception_info($e);
                    SH::notification_add(SH::str('cta_graders_update_error', $e_info->message), SH::NOTIFY_ERROR);
                }
                break;
        }

        $url = SH::page()->url;
        $url->remove_params(static::PAR_ACTION);
        SH::redirect($url);
        die;
    }

    /**
     * Load in existing data as form defaults. Usually new entry defaults are stored directly in
     * form definition (new entry form); this function is used to load in data where values
     * already exist and data is being edited (edit entry form).
     *
     * @param $defaults
     */
    public function set_data($defaults){
        parent::set_data($defaults);
    }
}
