<?php

// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * External API
 *
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die;

use block_ned_teacher_tools\deadline_manager as DM;
use block_ned_teacher_tools\shared_lib as SH;

require_once($CFG->libdir . '/externallib.php');

/**
 * Class block_ned_teacher_tools_external
 */
class block_ned_teacher_tools_external extends external_api {
    /**
     * Describes the parameters for submit_user_extension_form webservice.
     * @return external_function_parameters
     */
    public static function submit_user_extension_form_parameters() {
        return new external_function_parameters(
            array(
                'contextid' => new external_value(PARAM_INT, 'The context id for the course'),
                'jsonformdata' => new external_value(PARAM_RAW, 'The data from the create group form, encoded as a json array')
            )
        );
    }

    /**
     * Submit the create group form.
     *
     * @param int $contextid The context id for the course.
     * @param string $jsonformdata The data from the form, encoded as a json array.
     * @return int new group id.
     */
    public static function submit_user_extension_form($contextid, $jsonformdata) {
        global $CFG;
        require_once($CFG->dirroot . '/group/lib.php');
        require_once($CFG->dirroot . '/group/group_form.php');

        // We always must pass webservice params through validate_parameters.
        $params = self::validate_parameters(self::submit_user_extension_form_parameters(),
            ['contextid' => $contextid, 'jsonformdata' => $jsonformdata]);

        $context = context::instance_by_id($params['contextid'], MUST_EXIST);

        // We always must call validate_context in a webservice.
        self::validate_context($context);
        if (!DM::can_add_extension($context)){
            throw new moodle_exception('nopermissions', SH::$PLUGIN_NAME);
        }

        list($ignored, $course) = get_context_info_array($context->id);
        $blocksettings = SH::get_tt_block_config($course->id);
        if (!($blocksettings->enabledeadlineextension ?? false)){
            return 0;
        }

        $serialiseddata = json_decode($params['jsonformdata']);
        $data = array();
        parse_str($serialiseddata, $data);
        $userid = $data['userid'];
        $cmid = $data['cmid'];

        $dm_entity = DM::get_dm_entity($course, null, $userid);
        $source = DM::SOURCE_EXTENSION;
        $dm_entity->form_load_data($data, $source, true);
        $dm_entity->check_conflicts(true);
        $dma = $dm_entity->get_dm_activity($cmid);
        if ($conflict = $dma->get_date_conflict()){
            throw new moodle_exception($conflict, SH::$PLUGIN_NAME);
        }

        $extension = DM::set_extension($userid, $cmid, $dma->get_deadline($source), $dma->get_overrule($source),
            trim($data['reason'] ?? ''), $data['duedateorig'], static::get_cohortid($userid), $course);
        if ($extension === false){
            throw new moodle_exception('error_wrongformdata', SH::$PLUGIN_NAME);
        }

        return $extension->id ?? 0;
    }

    /**
     * Returns description of method result value.
     *
     * @return external_description
     * @since Moodle 3.0
     */
    public static function submit_user_extension_form_returns() {
        return new external_value(PARAM_INT, 'id');
    }

    /**
     * Describes the parameters for submit_class_extension_form webservice.
     * @return external_function_parameters
     */
    public static function submit_class_extension_form_parameters() {
        return new external_function_parameters(
            array(
                'contextid' => new external_value(PARAM_INT, 'The context id for the course'),
                'jsonformdata' => new external_value(PARAM_RAW, 'The data from the create group form, encoded as a json array')
            )
        );
    }

    /**
     * Submit the create group form.
     *
     * @param int $contextid The context id for the course.
     * @param string $jsonformdata The data from the form, encoded as a json array.
     * @return array
     */
    public static function submit_class_extension_form($contextid, $jsonformdata) {
        global $CFG, $USER, $DB;

        require_once($CFG->dirroot . '/group/lib.php');
        require_once($CFG->dirroot . '/group/group_form.php');

        $warnings = [];

        // We always must pass webservice params through validate_parameters.
        $params = self::validate_parameters(self::submit_class_extension_form_parameters(),
            ['contextid' => $contextid, 'jsonformdata' => $jsonformdata]);

        $context = context::instance_by_id($params['contextid'], MUST_EXIST);

        // We always must call validate_context in a webservice.
        self::validate_context($context);

        list($ignored, $course) = get_context_info_array($context->id);
        $serialiseddata = json_decode($params['jsonformdata']);

        $data = array();
        parse_str($serialiseddata, $data);

        $enddate = make_timestamp(
            $data['enddate']['year'],
            $data['enddate']['month'],
            $data['enddate']['day'],
            0,
            0,
            0,
            usertimezone(),
            true);

        $group = $DB->get_record('groups', array('id' => $data['groupid']), '*', MUST_EXIST);

        $overriderestrictions = DM::can_override_restrictions($context);
        $config = block_ned_teacher_tools_get_block_config($course->id);

        if (!$overriderestrictions) {
            if (($enddate - $group->startdate) > ($config->maxdaysbetween * DAYSECS)) {
                $warnings[] = array(
                    'item' => 'group',
                    'itemid' => $group->id,
                    'warningcode' => '1',
                    'message' => get_string('maxdayerror', 'ned_teacher_tools', $config->maxdaysbetween)
                );
                $result = [];
                $result['status'] = false;
                $result['warnings'] = $warnings;
                return $result;
            }
        }

        $rec = new stdClass();
        $rec->id = $group->id;
        $rec->enddate = $enddate;

        $DB->update_record('groups', $rec);
        cache_helper::invalidate_by_definition('core', 'groupdata', array(), array($group->courseid));

        // Insert log.
        $log = new stdClass();
        $log->submitter = $USER->id;
        $log->cohortid = self::get_cohortid($USER->id);
        $log->changetype = 'classenddate';
        $log->userid = 0;
        $log->groupid = $group->id;
        $log->courseid = $group->courseid;
        $log->enddateorig = $group->enddate;
        $log->enddate = $enddate;
        $log->reason =  $data['reason'];
        $log->timecreated = time();
        $DB->insert_record('block_ned_teacher_tools_cued', $log);

        $result = [];
        $result['status'] = true;
        $result['warnings'] = $warnings;
        return $result;
    }

    /**
     * Returns description of method result value.
     *
     * @return external_description
     * @since Moodle 3.0
     */
    public static function submit_class_extension_form_returns() {
        return new external_single_structure(
            array(
                'status' => new external_value(PARAM_BOOL, 'status: true if success'),
                'warnings' => new external_warnings()
            )
        );
    }

    /**
     * Describes the parameters for submit_student_extension_form webservice.
     * @return external_function_parameters
     */
    public static function submit_student_extension_form_parameters() {
        return new external_function_parameters(
            array(
                'contextid' => new external_value(PARAM_INT, 'The context id for the course'),
                'jsonformdata' => new external_value(PARAM_RAW, 'The data from the create group form, encoded as a json array')
            )
        );
    }

    /**
     * Submit the create group form.
     *
     * @param int $contextid The context id for the course.
     * @param string $jsonformdata The data from the form, encoded as a json array.
     * @return array
     */
    public static function submit_student_extension_form($contextid, $jsonformdata) {
        global $CFG, $USER, $DB;

        require_once($CFG->dirroot . '/group/lib.php');
        require_once($CFG->dirroot . '/group/group_form.php');

        $warnings = [];

        // We always must pass webservice params through validate_parameters.
        $params = self::validate_parameters(self::submit_student_extension_form_parameters(),
            ['contextid' => $contextid, 'jsonformdata' => $jsonformdata]);

        $context = context::instance_by_id($params['contextid'], MUST_EXIST);

        // We always must call validate_context in a webservice.
        self::validate_context($context);

        list($ignored, $course) = get_context_info_array($context->id);
        $serialiseddata = json_decode($params['jsonformdata']);

        $data = array();
        parse_str($serialiseddata, $data);

        $enddate = make_timestamp(
            $data['enddate']['year'],
            $data['enddate']['month'],
            $data['enddate']['day'],
            0,
            0,
            0,
            usertimezone(),
            true);

        $userid = $data['studentid'];
        $groupid = $data['groupid'];
        $group = SH::get_group($groupid);
        if(empty($group)){
            $groups = SH::get_all_user_course_groups($course->id, $userid);
            $group = reset($groups);
        }
        $userextension = $DB->get_record('block_ned_teacher_tools_usrd', array('userid' => $userid, 'courseid' => $course->id));

        $overriderestrictions = DM::can_override_restrictions($context);
        $config = block_ned_teacher_tools_get_block_config($course->id);

        $enddateorig = $userextension->enddate ?? ($group->enddate ?? 0);

        if (!$overriderestrictions) {
            $startdate = $userextension->startdate ?: $group->startdate;
            if (($enddate - $startdate) > ($config->maxdaysbetween * DAYSECS)) {
                $warnings[] = array(
                    'item' => 'group',
                    'itemid' => $group->id,
                    'warningcode' => '1',
                    'message' => get_string('maxdayerror', 'ned_teacher_tools', $config->maxdaysbetween)
                );
                $result = [];
                $result['status'] = false;
                $result['warnings'] = $warnings;
                return $result;
            }
        }

        if ($userextension) {
            $rec = new stdClass();
            $rec->id = $userextension->id;
            $rec->enddate = $enddate;
            $DB->update_record('block_ned_teacher_tools_usrd', $rec);
        } else {
            $rec = new stdClass();
            $rec->userid = $data['studentid'];
            $rec->courseid = $course->id;
            $rec->enddate = $enddate;
            $rec->timecreated = time();
            $DB->insert_record('block_ned_teacher_tools_usrd', $rec);
        }

        // Insert log.
        $log = new stdClass();
        $log->submitter = $USER->id;
        $log->cohortid = self::get_cohortid($USER->id);
        $log->changetype = 'studentenddate';
        $log->userid = $data['studentid'];
        $log->groupid = $group->id ?? 0;
        $log->courseid = $course->id;
        $log->enddateorig = $enddateorig;
        $log->enddate = $enddate;
        $log->reason =  $data['reason'];
        $log->timecreated = time();
        $DB->insert_record('block_ned_teacher_tools_cued', $log);

        $result = [];
        $result['status'] = true;
        $result['warnings'] = $warnings;
        return $result;
    }

    /**
     * Returns description of method result value.
     *
     * @return external_description
     * @since Moodle 3.0
     */
    public static function submit_student_extension_form_returns() {
        return new external_single_structure(
            array(
                'status' => new external_value(PARAM_BOOL, 'status: true if success'),
                'warnings' => new external_warnings()
            )
        );
    }

    /**
     * @param int $userid
     * @return int Cohort ID
     * @throws dml_exception
     */
    public static function get_cohortid(int $userid) {
        global $DB;

        $context = context_system::instance();

        $sql = "SELECT c.* FROM {cohort} c JOIN {cohort_members} m ON c.id = m.cohortid
                 WHERE c.contextid = ? AND m.userid = ? AND c.name NOT LIKE '*%'";

        if ($cohort = $DB->get_record_sql($sql, [$context->id, $userid], IGNORE_MISSING)) {
            return $cohort->id;
        }
        return 0;
    }
}