<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnnecessaryCurlyVarSyntaxInspection
 */

use local_ned_controller\marking_manager\marking_manager as MM;
use local_ned_controller\ned_grade_controller as NGC;
use block_ned_teacher_tools\output\menu_bar as MB;
use block_ned_teacher_tools\shared_lib as SH;

require_once(dirname(__FILE__) . '/../../config.php');
require_once(__DIR__ . '/lib.php');

const PAR_GRADE_VIEW = 'gv';
const PAR_GRADE_STATS = 'gs';

const GV_COMPLETION = 0;
const GV_GRADES = 1;
const GV_PERCENTAGE = 2;
const GV_OPTIONS = [
    GV_COMPLETION => 'completionview',
    GV_GRADES => 'gradesview',
    GV_PERCENTAGE => 'percentageview',
];

const GS_NONE = 0;
const GS_TERM_AVG = 1;
const GS_OPTIONS = [
    GS_NONE => 'selectgradestats',
    GS_TERM_AVG => 'termaverages',
];

/**
 * Rows (users) per table, before it becomes scrollable
 * Defined by the current styles and other options, so in case of other changes, you should be sure to change this data
 */
const PR_ROWS_PER_TABLE = 15;

class_exists('\local_tem\helper') && \local_tem\helper::tem_submission_notify_or_redirect_check();

$get_finalgrade_from_mm = function($mm_data){
    /** @var object|\local_ned_controller\marking_manager\mm_data_by_activity_user $mm_data */
    if (!isset($mm_data->finalgrade)) return null;

    $finalgrade = $mm_data->finalgrade;
    if (empty($finalgrade)) return $finalgrade;

    if (!empty($mm_data->ngc_grade_change) && ($mm_data->ngc_grade_type ?? 0) == NGC::GT_DEDUCTION && ($mm_data->ngc_status ?? 0) == NGC::ST_DONE){
        $finalgrade *= (100 - $mm_data->ngc_grade_change)/100;
    }

    return $finalgrade;
};

$show = optional_param(MB::PAR_PR_SHOW, MM::ST_ALL, PARAM_TEXT);
$view = optional_param(PAR_GRADE_VIEW, GV_COMPLETION, PARAM_INT);
$grade_stats = optional_param(PAR_GRADE_STATS, GS_NONE, PARAM_INT);
$export = optional_param('export', 0, PARAM_INT);

$showopts = SH::strings2menu(MM::MM_SELECT_STATUS, true, '&nbsp;&nbsp;');
$viewopts = SH::strings2menu(GV_OPTIONS, false, '&nbsp;&nbsp;');
$gs_opts = SH::strings2menu(GS_OPTIONS, false, '&nbsp;&nbsp;');

$show = SH::isset_key($showopts, $show);
$view = SH::isset_key(GV_OPTIONS, $view, GV_COMPLETION);

$MB = new MB(MB::PAGE_CP, [MB::PAR_PR_SHOW => $show, PAR_GRADE_VIEW => $view]);
$MB->show_error_for_not_teacher();
$courseid = $MB->courseid;
$coursecontext = SH::ctx($courseid);

$can_view_gs = SH::has_capability('viewgradestats', $coursecontext);
if ($can_view_gs){
    $grade_stats = SH::isset_key(GS_OPTIONS, $grade_stats, GS_NONE);
    $MB->set_pageparam(PAR_GRADE_STATS, $grade_stats);
} else {
    $grade_stats = GS_NONE;
}

$course = $MB->course;
$tag = $MB->tag_id;
$base_moodle_url = $MB->get_url();

$PAGE->set_url($base_moodle_url);
SH::js_call_amds(['progress_report', 'reviewmodalmessage']);
SH::js_call_amd('userextension', 'add', ['body', $coursecontext->id]);
$PAGE->add_body_class(GV_OPTIONS[GV_GRADES]);
$tablecontainer_relative_classes = ['tablecontainer-relative'];
$tablecontainer_overflow_classes = ['tablecontainer-overflow'];

// Print header.
$page_title = SH::str('simplegradebook');
$PAGE->navbar->add($page_title, new moodle_url(''));
$PAGE->set_title($page_title);
$PAGE->set_heading($course->fullname . ': ' . $page_title);

$completed_form = new \block_ned_teacher_tools\form\complete_course(null, array_merge($MB->get_url()->params(),
    ['url' => $MB->get_url()]), 'post', '', ['id' => 'complete_course_form']);
$allowed_to_complete_courses = false;
if (SH::has_capability('managecompletionstatus', $coursecontext)){
    $allowed_to_complete_courses = true;
}

$plugincfg = SH::get_config();
/** @var \block_ned_teacher_tools\course_users_status $course_users_status */
$course_users_status = new SH::$course_users_status($courseid);
$enable_course_status_controller_on_this_course = $course_users_status->enable_on_course();
$tt_config = SH::get_block_config($courseid, SH::TT_NAME);
$pp_on = $tt_config->participationpower_option ?? false;
$course_status_controller = $plugincfg->enablecoursecompletionstatuscontroller && $enable_course_status_controller_on_this_course;
$course_status_controller_edit = $course_status_controller && $allowed_to_complete_courses;
if ($course_status_controller){
    $tablecontainer_overflow_classes[] = 'pl-4';
}

if ($completed_form->is_submitted()){
    if ($course_status_controller_edit){
        $users_fail = [[], []];
        $error_msg = ['errorchengestatustopass', 'errorpermissionschengestatustopass'];
        $data = $completed_form->get_data();
        if (!empty($data->complete_course_data)){
            $complete_course_data = explode(',', $data->complete_course_data);
            foreach($complete_course_data as $data_userid){
                if (!$course_users_status->add_user_status($data_userid, $data->complete_course_select)){
                    $error_type = $course_users_status->get_error_type() - 1;
                    if (isset($users_fail[$error_type])){
                        $users_fail[$error_type][] = $data_userid;
                    }
                }
            }
        }

        foreach ($users_fail as $error_type => $item){
            if (empty($item)){
                continue;
            }

            list($sqluser, $paramuser) = $DB->get_in_or_equal($item, SQL_PARAMS_NAMED, 'usr');
            $uf_permissions = $DB->get_records_sql_menu(
                "SELECT u.id, CONCAT(u.firstname, ' ', u.lastname) fullname FROM {user} u WHERE u.id {$sqluser}",
                $paramuser
            );
            $str_note = SH::str($error_msg[$error_type]);
            $str_note .= implode(', ', $uf_permissions);
        }

        if (!empty($str_note)){
            redirect($data->url, $str_note, null, SH::NOTIFY_WARNING);
        }

        redirect($data->url);
    }
}

// Export.
if ($export && !(empty($MB->students ?? 0))) {
    SH::export_class_list($MB->students);
    die;
}

echo $OUTPUT->header();
echo SH::render($MB);

if ($MB->groupid == SH::GROUP_NONE || (empty($MB->students ?? 0))){
    if (empty($MB->allstudents ?? 0)){
        echo $OUTPUT->heading(get_string("nostudentsyet"));
    }
    echo $OUTPUT->footer($course);
    return;
}

$show_all_groups = $MB->groupid == SH::GROUP_ALL;
$activities = SH::get_important_activities_for_users($courseid, $MB->students, true);
$MM = MM::get_MM_by_params($MB->parameters_for_MM(['studuserid' => 0]));
if (SH::get_kica_enabled($courseid)){
    SH::kg_get_grades_by_course($courseid, array_keys($MB->students), false, true);
}

$filter = [
    MM::BY_ACTIVITY, MM::SORT, MM::GET_TAGS,
    MM::NOT_ZERO => true, MM::CHECK_TAG => $tag, $show
];
$mod_data = $MM->get_status_data($filter, MM::MOD_ALL);

$filter = [MM::USE_KICA, MM::BY_ACTIVITY_USER, MM::SORT, MM::ST_ALL, MM::GET_TAGS, MM::COUNT_MISSED_DEADLINE];
$MM->filter_hide_grades_before_midn_if_need($filter);
/** @var \local_ned_controller\marking_manager\mm_data_by_activity_user[][] $data */
$data = $MM->get_status_data($filter, MM::MOD_ALL);
$suspended_userids = $MB->show_inactive ? get_suspended_userids($MB->context) : [];
$fn_base_url = MB::get_page_url(MB::PAGE_MM);
$fn_base_param_url = [
    MB::PAR_COURSE => $courseid, MB::PAR_GROUP => $MB->groupid, MM::P_STATUS => 'unmarked', MB::PAR_TAG => $tag,
    'dir' => 'DESC', 'sort' => 'date', 'unsubmitted' => 0, 'activity_type' => 0, 'view' => 'less'
];

// table content
$table = SH::html_table('generaltable simplegradebook progress_report', 'datatable');
$table->head[] = SH::cell(SH::img('i/course', ''), 'grade-cell mod-icon sorter-true',
    ['title' => get_string('courseavg', 'grades')]);

// student column
$name_head = SH::str('student');
$name_head .= SH::span('('.SH::str('firstname').' ↑)', 'px-1 font-weight-normal sort-info sort-info-up');
$name_head .= SH::span('('.SH::str('lastname').' ↑)', 'px-1 font-weight-normal sort-info sort-info-down');
$name_head_classes = ['name', 'mod-icon', 'sorter-true'];

if ($view == GV_GRADES){
    $name_head .= ' \ '. SH::str('maxgrade');
    $name_head_classes[] = 'name-max-grade';
}
$table->head[] = SH::cell($name_head, $name_head_classes);

if ($show_all_groups){
    $table->head[] = SH::cell(get_string('group'), 'mod-icon sorter-true');
}

$grademax_terms = array_fill_keys(SH::TERMS, 0);
$head_activities = [];
$cm_kica_links = [];
// Create table head and filter which activities we will show
foreach ($activities as $cm){
    $mod = $mod_data[$cm->id] ?? null;

    $grademax = 0;
    if ($grade_stats == GS_TERM_AVG){
        $has_any_term = false;
        $has_term = array_fill_keys(SH::TERMS, false);
        if ($mod){
            if (!empty($mod->is_summative)){
                $grademax = $mod->grademax ?? 0;
                foreach (SH::TERMS as $term){
                    $has_term[$term] = $mod->{'is_'.$term} ?? false;
                    $has_any_term = $has_any_term || $has_term[$term];
                }
            }
        } else {
            $tags = SH::cm_get_tags($cm);
            if (in_array(SH::TAG_SUMMATIVE, $tags)){
                foreach (SH::TERMS as $term_tag => $term){
                    $has_term[$term] = in_array($term_tag, $tags);
                    $has_any_term = $has_any_term || $has_term[$term];
                }

                if ($has_any_term){
                    $gi = SH::get_grade_item($cm, $courseid);
                    if ($gi){
                        $grademax = $gi->grademax;
                    }
                }
            }
        }

        if ($has_any_term && $grademax > 0){
            foreach (SH::TERMS as $term){
                if ($has_term[$term]){
                    $grademax_terms[$term] += $grademax;
                }
            }
        }
    }

    if (!$mod) continue;

    $add_head = '';
    $head_class = ['mod-icon', 'add-tag-icon'];
    if ($mod->is_midterm){
        $add_head .= SH::div('', 'activity-midterm-point', ['title' => SH::str('midtermpoint:title')]);
        $head_class[] = 'column-midterm-point';
    }
    if (!$cm->available){
        $head_class[] = 'column-restricted-activity';
    }

    $displayname = $full_cm_name = $cm->get_formatted_name();
    $mod_link_params = ['data-title' => $full_cm_name, 'data-toggle' => 'tooltip', 'data-placement' => 'top'];

    if ($plugincfg->truncateactivitynames){
        $displayname = substr($displayname, 0, (int)$plugincfg->truncateactivitynames);
    }
    $displayname = shorten_text($displayname, 30);
    $shorten_link_cm = SH::cm_get_text_icon_link($mod, $courseid, null, true, true, false,
        true, $displayname, null, null, 'text-decoration-none', ['title' => '']);
    $cm_name = SH::div(SH::span($shorten_link_cm, 'cm-name-rotated'),'cm-name');

    $main_tag = null;
    foreach (SH::IMPORTANT_TAGS as $tag_key => $tag){
        $is_tag = 'is_'.$tag_key;
        if ($mod->$is_tag ?? false){
            $main_tag = $tag;
            break;
        }
    }

    if ($main_tag){
        $mod_link_params['data-tags'] = $main_tag;
    }


    $prefix = '';
    $postfix = '';
    switch ($view){
        case GV_GRADES:
            $grademax = $mod->grademax ?? 0;
            /** @noinspection PhpConditionAlreadyCheckedInspection */
            if ($grademax > 0){
                $prefix = SH::span(round($grademax), 'activity-maxgrade');
            }
            break;
        case GV_PERCENTAGE:
            $postfix = ' %';
            break;
    }

    $cm_kica_links[$cm->id] = false;
    if ($cm->modname === 'quiz' && !empty($mod->tags)){
        $tags = explode(',', $mod->tags);
        $cm_kica_links[$cm->id] = in_array('KICA', $tags);
    }

    $cm_icon = SH::cm_get_text_icon_link($mod, $courseid, null, false, true, true,
        true, null, null, null, null, ['title' => '']);
    $head_activities[] = SH::cell($add_head . $cm_name . $prefix . $cm_icon . $postfix, $head_class, $mod_link_params);
}

if ($grade_stats == GS_TERM_AVG){
    foreach (SH::TERMS as $term){
        $postfix = '';
        switch ($view){
            case GV_COMPLETION:
            case GV_PERCENTAGE:
                $postfix = '%';
                break;
            case GV_GRADES:
                $postfix = '/'.round($grademax_terms[$term]);
        }

        $table->head[] = SH::cell(SH::str($term.'short').$postfix, 'mod-icon sorter-true');
    }
}

if ($pp_on){
    $table->head[] = SH::cell(SH::fa('fa-rocket'), 'mod-icon sorter-true', ['title' => SH::str('participationpower')]);
}

$table->head = array_merge($table->head, $head_activities);

$users_statuses = $course_users_status->get_users_statuses(null, true);
$courseobj = get_course($courseid);
$completioninfo = new \completion_info($courseobj);

foreach ($MB->students as $studentid => $student){
    $studentclass = [];
    $divname_class = ['row-toggle'];

    if ($MB->show_inactive){
        $supendedenrollment = $student->suspended || isset($suspended_userids[$studentid]);
        $simplegradebook[$studentid]['suspended'] = $supendedenrollment;
        if ($supendedenrollment){
            $studentclass[] = 'suspended';
        }
    } else {
        $simplegradebook[$studentid]['suspended'] = false;
    }

    if ($MB->userid == $studentid){
        $studentclass[] = 'highlight';
        $divname_class[] = 'highlighted';
    }

    $row = SH::row(null, join(' ', $studentclass), ['data-userid' => $studentid]);
    // other cells
    $cell_pp = null;
    $cell_terms = array_fill_keys(SH::TERMS, []);

    // course grade
    $is_view_grade = true;
    if ($course_status_controller && !is_siteadmin()){
        if (array_key_exists($studentid, $users_statuses) && !empty($users_statuses[$studentid]) && !empty($users_statuses[$studentid]->status)){
            $is_view_grade = SH::get_check_permission_view_grade($courseid, $users_statuses[$studentid]->status);
        }
    }
    if ($is_view_grade){
        $avg = SH::get_course_grade($courseid, $studentid);
        if ($avg != '-'){
            $percentage = round($avg, 0);
            $percentagecls = $avg >= 50 ? 'green' : 'red';
        } else {
            $percentage = ' - ';
            $percentagecls = 'red';
        }

        if ($MM->has_kica){
            $grade_link = [SH::PAGE_KICA, [SH::PAR_SETUSER => $studentid, SH::PAR_COURSE => $courseid]];
        } else {
            $grade_link = ['/grade/report/user/index.php', ['userid' => $studentid, 'id' => $courseid]];
        }
        $row->cells[] = SH::cell(SH::link($grade_link, $percentage, '', ['target' => '_blank']), 'grade-cell '.$percentagecls);
    } else {
        $row->cells[] = SH::cell(
            SH::fa("tt-status-icon tt-si-waitingfinalgrade"),
            'review-modal-message-text'
        );
    }

    // username
    // don't use student menu here, it will be added in the separate sell due style problems with small table (i.e. with one record)
    $course_status_css_class = ['name'];
    $icon_status_css_class = [];
    if ($course_status_controller){
        $course_status = 'inprogress';
        $icon_status_css_class = ['fa', 'tt-status-icon'];
        if ($completioninfo->is_course_complete($studentid)){
            $icon_status_css_class[] = 'tt-si-course-completed';
        }

        if (array_key_exists($studentid, $users_statuses) && !empty($users_statuses[$studentid]) && !empty($users_statuses[$studentid]->status)){
            switch($users_statuses[$studentid]->status){
                default:
                case 1: break; // $course_status = 'inprogress';
                case 2: $course_status = 'waitingfinalgrade'; break;
                case 3: $course_status = 'pass'; break;
                case 4: $course_status = 'fail'; break;
                case 5: $course_status = 'waitingforinternalreview'; break;
            }
        }

        $icon_status_css_class[] = 'tt-si-'.$course_status;
        $course_status_css_class[] = 'status-'.$course_status;
    }

    $username = SH::q_student_link($studentid, $courseid, 'ned-username', false, true, false);
    $sort_username = $student->firstname.'||'.$student->lastname;
    $row->cells[] = SH::cell(
        SH::div(SH::div('', $icon_status_css_class), $divname_class) . $username,
        $course_status_css_class, ['nowrap' => 'nowrap', 'data-sort-value' => $sort_username]
    );

    // group
    if ($show_all_groups){
        $groupname = $student->group->name ?? '-';
        $row->cells[] = SH::cell(substr($groupname, 0, 4), 'groupname', ['data-sort-value' => $groupname, 'title' => $groupname]);
    }

    // Term cells
    if ($grade_stats == GS_TERM_AVG){
        foreach (SH::TERMS as $term){
            $row->cells[] = ''; // for term_x later
            $cell_terms[$term] = count($row->cells) - 1;
        }
    }

    if ($pp_on){
        $row->cells[] = ''; // for pp later
        $cell_pp = count($row->cells) - 1;
    }

    $st_data_count = 0;
    $data_terms = array_fill_keys(SH::TERMS, 0);
    $graded_terms = array_fill_keys(SH::TERMS, true);
    $userurl = new moodle_url($fn_base_url, array_merge($fn_base_param_url, [MB::PAR_USER => $studentid]));
    $saved_url = $userurl;

    /** @var \block_ned_teacher_tools\activity_status $user_activity_status */
    $user_activity_status = new SH::$activity_status();
    foreach ($activities as $cm){
        $userurl = $saved_url;
        $newuserurl = null;
        $any_mod = $data[$cm->id] ?? null;
        /** @var object|\local_ned_controller\marking_manager\mm_data_by_activity_user $st_mod */
        $st_mod = $any_mod[$studentid] ?? false;
        $add_class = ['icon'];
        $link_class = [];
        $status = SH::$activity_status::STATUS_NOTATTEMPTED;
        if ($st_mod){
            $st_mod->course = $courseid;
            $status = $user_activity_status->check_activity($st_mod, $studentid, false, $MM->has_kica, $st_mod);

            if ($grade_stats == GS_TERM_AVG && !empty($st_mod->is_summative) && !empty($st_mod->grademax)){
                foreach (SH::TERMS as $term){
                    if ($st_mod->{'is_'.$term} ?? false){
                        if (isset($st_mod->finalgrade)){
                            $data_terms[$term] += $get_finalgrade_from_mm($st_mod);
                        } else {
                            $graded_terms[$term] = false;
                        }
                    }
                }
            }
        }

        /**
         * Warning: all global calculations (based on data from the all activities) should be before this line
         */

        $mod = $mod_data[$cm->id] ?? null;
        if (!$mod){
            continue;
        }

        if ($st_mod->$show ?? false){
            $st_data_count++;
        } else {
            $status = '-';
        }

        if (!SH::get_cm_visibility_by_user($cm, $studentid, false, false)){
            $row->cells[] = SH::cell(SH::fa('fa-eye-slash text-align-center w-100', '', SH::str('hidden')), 'disabled');
            continue;
        }

        // set url param
        $sort = -1;
        switch ($status){
            case SH::STATUS_COMPLETED:
            case SH::STATUS_INCOMPLETED:
            case SH::STATUS_INCOMPLETED_SUMMATIVE:
            case SH::STATUS_INCOMPLETED_FORMATIVE:
            case SH::STATUS_PASSED:
            case SH::STATUS_GRADED_KICA_SUMMATIVE:
            case SH::STATUS_GRADED_KICA_FORMATIVE:
            case SH::STATUS_KICA_ZEROGRADE:
                $sort = 1;
                $userurl->param(MM::P_STATUS, MM::ST_MARKED);
                break;
            case SH::STATUS_NGC_ZERO:
                $sort = 0;
                //remove the link for the hard zero status, it will be with an icon
                $userurl = null;
                break;
            case SH::STATUS_NOTATTEMPTED:
                $sort = -2;
                $userurl->param(MM::P_STATUS, MM::ST_UNSUBMITTED);
                break;
            case SH::STATUS_DRAFT:
                $sort = -3;
                $userurl->param(MM::P_STATUS, MM::ST_DRAFT);
                break;
            case SH::STATUS_WAITINGFORGRADE:
            case SH::STATUS_UNGRADED_KICA_SUMMATIVE:
            case SH::STATUS_UNGRADED_KICA_FORMATIVE:
            case SH::STATUS_UNGRADED_KICA:
                $sort = -10;
                $userurl->param(MM::P_STATUS, MM::ST_UNMARKED);
                break;
            case SH::STATUS_EXCLUDED:
            default:
                $userurl->param(MM::P_STATUS, MM::ST_ALL);
        }
        if (!empty($userurl)){
            $userurl->param(MM::P_MID, $cm->id);
            $newuserurl = clone $userurl;
        }

        if (SH::is_kica_exists() && $cm_kica_links[$cm->id] ?? false){
            if ($kicagradingurl = \local_kicaquizgrading\helper::get_kica_grader_link_from_cm($cm, $studentid)){
                $newuserurl = $kicagradingurl;
            }
        }

        $icons = [];
        if ($view != GV_COMPLETION){
            $text = '';
            $finalgrade = $get_finalgrade_from_mm($st_mod);
            $sort = $finalgrade ?? $sort;
            if (!is_null($finalgrade)){
                if ($st_mod->shouldpass){
                    if ($st_mod->completed_grade_successfully ?? false){
                        $link_class[] = 'green';
                    } else {
                        $link_class[] = 'red';
                        $add_class[] = 'red';
                    }
                }

                switch ($view){
                    case GV_GRADES:
                        $text = round($finalgrade);
                        break;
                    case GV_PERCENTAGE:
                        $text = !empty($st_mod->grademax) ? round($finalgrade / $st_mod->grademax * 100) : '';
                        break;
                }
            }
        } else {
            $icons = SH::get_ned_grade_icon($st_mod, $status, true, $show_all_groups, $cm);
            $text = SH::get_ned_grade_element($icons, $status);
        }

        $attrs = [];
        if (!SH::get_cm_visibility_by_user($cm, $studentid, true, false)){
            // student see it, but hasn't access
            $add_class[] = 'column-restricted-activity';
            $attrs['title'] = SH::str('restricted');
        } else {
            if ($mod->is_midterm ?? false){
                $add_class[] = 'column-midterm-point';
            }

            if ($icons[SH::STATUS_EXTENSION] ?? false){
                $attrs = SH::get_class_progress_extension_popover_attributes($cm, $studentid,
                    null, $coursecontext, $st_mod->duedate ?? 0, false, true, $courseid);
            }
        }

        if (!empty($newuserurl)){
            $text = SH::link([$newuserurl->out(false)], $text, $link_class);
        }

        $row->cells[] = SH::cell($text, $add_class, $attrs + ['data-sort-value' => $sort]);
    }

    if ($st_data_count < 1){
        continue;
    }

    /**
     * Insert all earlier data
     */
    if ($grade_stats == GS_TERM_AVG){
        foreach (SH::TERMS as $term){
            if (empty($cell_terms[$term])) continue;

            if (!empty($graded_terms[$term]) && !empty($grademax_terms[$term])){
                $val = $data_terms[$term] / $grademax_terms[$term];
                switch ($view){
                    case GV_COMPLETION:
                    case GV_PERCENTAGE:
                        $str_val = round($val*100);
                        break;
                    case GV_GRADES:
                        $str_val = round($data_terms[$term]);
                        break;
                }
            } else {
                $val = -1;
                $str_val = '-';
            }

            $row->cells[$cell_terms[$term]] = SH::cell($str_val, $term.'-cell', ['data-sort-value' => $val]);
        }
    }

    if ($pp_on && $cell_pp){
        list($summative_score, $formative_score, $participationpower, $divided_score, $text_status) = $user_activity_status->get_full_pp_data();
        if ($summative_score){
            $pp_status = 'pp-' . $text_status;
            $pp_value = SH::link(['#'],
                SH::div(SH::fa('fa-rocket') . " ($participationpower)", $pp_status, ['title' => $participationpower]),
                'participationpower',
                [
                    'title'  => $participationpower,
                    'data-course' => $course->shortname,
                    'data-data' => "$summative_score $formative_score $divided_score $participationpower $pp_status",
                ]
            );
        } else {
            $pp_value = '-';
            $participationpower = -1;
        }

        $row->cells[$cell_pp] = SH::cell($pp_value, 'participationpower-cell', ['data-sort-value' => $participationpower]);
    }

    // user menu, should be last cell in the row
    $user_menu = SH::get_profile_context_menu($studentid, $courseid);
    if (!empty($user_menu)){
        $row->cells[] = SH::cell(SH::div($user_menu, 'ned-student-menu'), 'menu-cell');
    }

    $table->data[] = $row;
}

if (!empty($table->data) && count($table->data) > PR_ROWS_PER_TABLE){
    $tablecontainer_relative_classes[]= 'y-scrollable';
}

// Filter form.
$filterform = '';
// view-select
$filterform .= SH::div(
    SH::single_select($base_moodle_url, PAR_GRADE_VIEW, $viewopts, $view, ''),
    'groupselector prview-selector'
);

$filterform .= $MB->render_tag_selector('', true, 'groupselector tag-selector', '&nbsp;&nbsp;');

// status-select
$filterform .= SH::div(
    SH::single_select($base_moodle_url, MB::PAR_PR_SHOW, $showopts, $show, ''),
    'groupselector status-selector'
);

// grade stats select
if ($can_view_gs){
    $filterform .= SH::div(
        SH::single_select($base_moodle_url, PAR_GRADE_STATS, $gs_opts, $grade_stats, ''),
        'groupselector grade-stats-selector'
    );
}

echo SH::div($filterform, 'ned-teacher_tools-menu-wrapper');
if (!empty($table->data)){
    echo SH::div(SH::div(SH::render_table($table), $tablecontainer_overflow_classes), $tablecontainer_relative_classes);
}
// Permission.
if ($course_status_controller_edit){
    $completed_form->display();
}
echo SH::render_progress_report_legend_table($MM->has_kica, $course_status_controller);
echo block_ned_teacher_tools_footer($plugincfg);
echo $OUTPUT->footer($course);
