<?php

namespace local_academic_integrity;

use local_academic_integrity\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/user/profile/lib.php');

/**
 * Class ai_flag
 *
 * @package local_academic_integrity
 */
class ai_flag {
    const TRANSFER_STUDENT_FIELD = 'transfer_student';

    const ST_NONE = 0;
    const ST_GOOD_STANDING = 1;
    const ST_FIRST_OFFENSE = 2;
    const ST_MULTIPLE_OFFENSES = 3;
    const STATUSES = [
        self::ST_NONE => 'ai-none',
        self::ST_GOOD_STANDING => 'ai-good-standing',
        self::ST_FIRST_OFFENSE => 'ai-first-offense',
        self::ST_MULTIPLE_OFFENSES => 'ai-multiple-offenses',
    ];

    /**
     * Return HTML element containing the academic integrity flag by the specified status.
     *
     * @param int $status - AI status, one of the {@see static::STATUSES}
     *
     * @return string the HTML fragment (<span class="..."> </span>")
     */
    public static function get_html_flag_by_status($status=self::ST_NONE) {
        if (empty($status)) return '';

        $st_class = static::STATUSES[$status] ?? static::STATUSES[static::ST_NONE];
        return NED::span('', 'fa fa-flag fa-fw academic-integrity '.$st_class, ['title' => static::title($status)]);
    }

    /**
     * Return HTML element containing the academic integrity flag for the
     * specified student.
     *
     * @param object|numeric|null $user_or_id The user to get the academic integrity for, uses current $USER if null
     *
     * @return string the HTML fragment ("<span class="..."></span>") if we can view the flag or "" if we can't.
     */
    public static function flag($user_or_id){
        $flag = "";

        if ($status = static::status($user_or_id)){
            $flag = static::get_html_flag_by_status($status);

            $profile = profile_user_record(NED::get_userid_or_global($user_or_id));
            if (!empty($profile->{static::TRANSFER_STUDENT_FIELD})){
                $flag .= NED::span('', 'fa fa-exchange fa-fw academic-integrity transfer-student',
                    ['title' => NED::str('transferstudent')]);
            }
        }

        return $flag;
    }

    /**
     * Get the academic integrity of the specified user.
     * It can be one of the {@see static::STATUSES}, or translated string (e.g. "Good standing")
     * If this isn't the current user then a value is only returned if the current user has the
     *  capability to see the specified user's value, otherwise return false
     *
     * @param object|numeric|null $user_or_id The user to get the academic integrity for, uses current $USER if null
     * @param bool                $translated - if true, return translated string
     *
     * @return int|string
     */
    public static function status($user_or_id=null, $translated=false) {
        $userid = NED::get_userid_or_global($user_or_id);
        if (empty(NED::get_config('showacademicintegrityflags'))) return static::ST_NONE;
        if (!NED::can_view_user_ai($userid) || !NED::ai_is_student($userid)) return static::ST_NONE;

        $aiv = infraction::get_user_aiv_count($userid);
        if (empty($aiv)){
            $status = static::ST_GOOD_STANDING;
        } elseif ($aiv === 1){
            $status = static::ST_FIRST_OFFENSE;
        } else {
            $status = static::ST_MULTIPLE_OFFENSES;
        }

        if ($translated){
            return static::str_status($status);
        }

        return $status;
    }

    /**
     * Translate status value to the human string
     *
     * @param int $status - one of the {@see static::STATUSES}
     *
     * @return string
     */
    public static function str_status($status=self::ST_NONE){
        return NED::str(static::STATUSES[$status] ?? static::STATUSES[static::ST_NONE]);
    }

    /**
     * Return a value for use as the title="" attribute for the flag for the given academic integrity.
     *
     * @param string|int $ai_status Text status (e.g. "Good standing") or one of the int {@see static::STATUSES}
     *
     * @return string - For example "Academic integrity: Good standing".
     */
    private static function title($ai_status) {
        $ai_status = static::str_status($ai_status) ?: $ai_status;
        return NED::str('local_academic_integrity_title', $ai_status);
    }
}
