<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Infraction Form
 *
 * @package    local_academic_integrity
 * @copyright  2021 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_academic_integrity\form;

defined('MOODLE_INTERNAL') || die();

use local_academic_integrity\infraction as INF;
use local_academic_integrity\shared_lib as NED;

/**
 * Class infraction
 * @package local_academic_integrity\form
 */
class infraction_form extends \local_ned_controller\form\base_form {
    const REQUIRED_FIELDS = ['cmid', 'penalty', 'reason', 'grader', 'activitytype', 'note'];

    public function _main_definition() {
        $FE = static::$FE;
        $mform = $this->_form;
        $customdate = $this->_customdata;
        $studentid = $customdate[NED::PAR_USER] ?? 0;
        $courseid = $customdate[NED::PAR_COURSE] ?? 0;
        $cmid = $customdate[NED::PAR_CM] ?? 0;
        $pleaseselect = [0 => NED::str('pleaseselect')];

        if ($courseid){
            $courseid = NED::check_grader_courseid($courseid, 0, $studentid) ? $courseid : 0;
            $customdate[NED::PAR_COURSE] = $courseid;
        }

        $this->addHiddens($customdate);
        $this->add($FE::hidden('id', 0, PARAM_INT));

        $this->add($FE::header(NED::str('academicintegrityinfraction')));
        $this->add($FE::st('studentname', NED::str('student'), NED::q_user_link($studentid, $courseid)));
        if ($courseid){
            $this->add($FE::st('coursename', NED::str('course'), NED::q_course_link($courseid)));
        }
        $this->add($FE::st('schoolname', NED::str('school'), NED::get_user_school_names($studentid)));

        if (!$courseid){
            $url = new \moodle_url($this->get_action(), $customdate);
            $courses = NED::get_grader_courses(0, $studentid);
            $courses = $pleaseselect + $courses;
            $this->add($FE::html(NED::course_autocomplete($url, 0, $courses)));

            $this->disable_form_change_checker();
        } else {
            if ($cmid){
                $this->add($FE::st('cm_name', NED::str('activity'), NED::q_cm_link($cmid, $courseid)));
                $this->add($FE::hidden('cmid', $cmid));
            } else {
                $cms = NED::get_course_activities($courseid);
                $options = $pleaseselect + NED::cms2menu($cms);
                $this->add($FE::autocomplete('cmid', NED::str('activity'), $options));
            }

            $def_ot_id = $def_ct_id = 0;
            $infr = null;
            if (!empty($this->_customdata[NED::PAR_ID])){
                $infr = INF::get_record($this->_customdata[NED::PAR_ID]) ?: null;
            }
            if (!$infr && $studentid){
                $def_ot_id = NED::get_graderid_by_studentid($studentid, $courseid, $cmid, NED::ROLE_OT);
                $def_ct_id = NED::get_graderid_by_studentid($studentid, $courseid, $cmid, NED::ROLE_CT);
            }

            $mform->addElement('date_selector', 'infractiondate', NED::str('dateofinfraction'));

            $options  = $pleaseselect + INF::get_penalty_list();
            $mform->addElement('select', 'penalty', NED::str('penalty'), $options);
            $this->add($FE::st('penalty_desc', '&nbsp;', NED::str('penalty_desc')));

            $options  = $pleaseselect + INF::get_reason_list();
            $mform->addElement('select', 'reason', NED::str('reason'), $options);

            $options = $pleaseselect + INF::get_grader_list($courseid, $cmid);
            $mform->addElement('select', 'grader', NED::str('grader'), $options);
            $mform->setDefault('grader', $def_ot_id);

            $options = $pleaseselect + INF::get_ct_list($courseid, $cmid);
            $mform->addElement('select', 'ctid', NED::str('ct'), $options);
            $mform->setDefault('ctid', $def_ct_id);

            $options  = $pleaseselect + INF::get_activity_type_list();
            $mform->addElement('select', 'activitytype', NED::str('activitytype'), $options);

            $mform->addElement('textarea', 'note', NED::str('infractiondescription'), 'wrap="virtual" rows="5" cols="40"');
            $this->add($FE::st('infraction_note_desc', NED::HTML_SPACE,
                NED::html_i(NED::str('infraction_note_desc'), 'd-flex')));

            $options = $this->upload_file_options();
            $mform->addElement('filemanager', 'uploadedfiles', NED::str('uploadedfiles'), null, $options);

            $options = INF::get_states_list(true);
            $mform->addElement('select', 'state', NED::str('status'), $options);
            $mform->setDefault('state', INF::ST_UNAPPROVED);
            if (!NED::can_edit_state()){
                $mform->disabledIf('state', 'penalty', 'neq', '-100'); // should be always true
            }

            foreach (static::REQUIRED_FIELDS as $field) {
                $mform->addRule($field, get_string('required'),'required');
            }

            if ($infr){
                NED::save_author_and_editor_to_obj($infr, $infr);
                if ($infr->show_changed_by ?? false){
                    $this->add($FE::st('editor', '',
                        NED::$C::str('changedby', $infr->editor).NED::HTML_SPACE.$infr->timemodified_str));
                }
                if ($infr->timecreated_str ?? false){
                    $this->add($FE::st('author', '',
                        NED::$C::str('createdby', $infr->author).NED::HTML_SPACE.$infr->timecreated_str));
                }
            }

            $this->add_action_buttons_alt(true, NED::str('saveandclose'));
        }
    }

    /**
     * @param array $data
     * @param array $files
     * @return array
     */
    public function validation($data, $files){
        $errors = array();
        if (!empty($data['id'])){
            $ngc_record = INF::get_infraction_ngc_record($data['id']);
            if ($ngc_record){
                if ($ngc_record->grade_type != INF::penalty2ngc($data['penalty'])){
                    $errors['penalty'] = NED::str('penalty:error');
                }
            }
        }

        foreach (static::REQUIRED_FIELDS as $field){
            if (empty($data[$field])){
                $errors[$field] = get_string('required');
            }
        }

        return $errors;
    }

    /**
     * @return array
     */
    public function upload_file_options () {
        global $CFG;

        return array(
            'maxfiles' => 50,
            'maxbytes' => $CFG->maxbytes,
            'subdirs' => 0,
            'accepted_types' => '*'
        );
    }

    /**
     * Return true, if successfully saved form data, false otherwise
     * It doesn't check user permission for adding/editing AIVs
     *
     * @return bool
     */
    public function save_data(){
        $formdata = $this->get_data();
        if (empty($formdata->cmid)){
            return false;
        }

        $studentid = $formdata->{NED::PAR_USER};
        //check user for existing
        $student = NED::get_user($studentid);
        if (!$student){
            return false;
        }

        $rec = new \stdClass();
        $rec->id = null;
        $prev_infraction = null;
        $editorid = NED::get_userid_or_global();
        if (!empty($formdata->id)){
            $prev_infraction = INF::get_record($formdata->id);
            if ($prev_infraction){
                $rec->id = $prev_infraction->id;
                $rec->authorid = $prev_infraction->authorid ?? $editorid;
            }
        }

        $rec->student = $student->id;
        $rec->courseid = $formdata->{NED::PAR_COURSE};
        $rec->cmid = $formdata->cmid;
        $rec->reason = $formdata->reason;
        $rec->penalty = $formdata->penalty;
        $rec->ctid = $formdata->ctid;
        $rec->infractiondate = $formdata->infractiondate;
        $rec->grader = $formdata->grader;
        $rec->activitytype = $formdata->activitytype;
        $rec->note = $formdata->note;
        $rec->authorid = $rec->authorid ?? $editorid;
        $rec->editorid = $editorid;

        if (NED::can_edit_state()){
            $rec->state = $formdata->state;
        } else {
            $rec->state = $prev_infraction->state ?? INF::ST_UNAPPROVED;
        }

        $rec = INF::save_record($rec);
        NED::notification_add('successfullysavedinfraction', NED::NOTIFY_SUCCESS);

        file_save_draft_area_files($formdata->uploadedfiles, NED::ctx()->id, NED::$PLUGIN_NAME, 'uploadedfiles',
            $rec->id, $this->upload_file_options());

        $NGC = NED::$ned_grade_controller;
        $ngc_record = INF::get_infraction_ngc_record($rec->id);
        if ($rec->state == INF::ST_UNAPPROVED){
            if ($ngc_record){
                $NGC::check_and_delete($ngc_record, false);
            }
        } else {
            // $rec->state != INF::ST_UNAPPROVED
            if ($ngc_record){
                // upd NGC status
                $infr = new INF($rec->id);
                $infr->update_ngc_status($rec->editorid);
            } else {
                // check NGC creation // $rec->state != INF::ST_UNAPPROVED && empty($ngc_record)
                $grade_type = INF::penalty2ngc($formdata->penalty);
                if (!is_null($grade_type)){
                    $grade_change = 0;
                    if ($grade_type == $NGC::GT_DEDUCTION){
                        $grade_change = 20;
                    }

                    $infr = new INF($rec->id);
                    if ($infr->create_ngc_record($grade_type, $grade_change, $rec->authorid)){
                        NED::notification_add('successfullysave_ngc', NED::NOTIFY_SUCCESS);
                    } else {
                        NED::notification_add('failedsavedaz', NED::NOTIFY_ERROR);
                    }
                }
            }
        }

        return true;
    }
}
