<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Infraction
 *
 * @package    local_academic_integrity
 * @copyright  2021 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_academic_integrity;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/lib/questionlib.php');

use local_academic_integrity\output\datatable_infractions;
use local_academic_integrity\shared_lib as NED;

/**
 * Class infraction
 *
 * @package local_academic_integrity
 *
 * @property-read bool $view_all_schools = false
 * From local_academic_integrity_inf table record:
 * @property-read int id
 * @property-read int courseid
 * @property-read int cmid
 * @property-read int state
 * @property-read int student // student id
 * @property-read string school
 * @property-read string ct
 * @property-read int infractiondate
 * @property-read string reason
 * @property-read string penalty
 * @property-read int grader // grader id
 * @property-read string activitytype
 * @property-read string note
 * @property-read int timecreated
 * @property-read int timemodified
 * @property-read bool get_raw_data
 * @property-read \context context;
 */
class infraction {
    const TABLE = 'local_academic_integrity_inf';

    const PENALTY_NONE = 0;
    const PENALTY_MINOR_PLAGIARISM = 1;
    const PENALTY_MAJOR_PLAGIARISM = 2;
    const PENALTY_CHEATING = 3;
    const PENALTIES = [
        self::PENALTY_MINOR_PLAGIARISM => 'minorplagiarism',
        self::PENALTY_MAJOR_PLAGIARISM => 'majorplagiarism',
        self::PENALTY_CHEATING         => 'cheating',
    ];

    // Reasons
    const REASON_ASSIGN_INTERNET = 10;
    const REASON_ASSIGN_STUDENT = 11;
    const REASON_ASSIGN_SELF = 12;
    const REASON_ASSIGN_CITATIONS = 13;
    const REASON_ASSIGN_GENAI = 14;
    const REASON_EXAM_INTERNET = 20;
    const REASON_EXAM_STUDENT = 21;
    const REASON_EXAM_AIDS = 22;
    const REASON_EXAM_NOT_SUPPORTED = 23;
    const REASON_EXAM_NOT_MATCH = 24;
    const REASON_OTHER = 99;
    const REASONS = [
        self::REASON_ASSIGN_INTERNET  => 'inf:reason:assign:internet',
        self::REASON_ASSIGN_STUDENT   => 'inf:reason:assign:student',
        self::REASON_ASSIGN_SELF      => 'inf:reason:assign:self',
        self::REASON_ASSIGN_CITATIONS => 'inf:reason:assign:citations',
        self::REASON_ASSIGN_GENAI     => 'inf:reason:assign:genai',

        self::REASON_EXAM_INTERNET      => 'inf:reason:exam:internet',
        self::REASON_EXAM_STUDENT       => 'inf:reason:exam:student',
        self::REASON_EXAM_AIDS          => 'inf:reason:exam:aids',
        self::REASON_EXAM_NOT_SUPPORTED => 'inf:reason:exam:not_supported',
        self::REASON_EXAM_NOT_MATCH     => 'inf:reason:exam:not_match',

        // should be the latest in the list
        self::REASON_OTHER              => 'inf:reason:other',
    ];

    const ST_UNAPPROVED = 0;
    const ST_ACTIVE = 1;
    const ST_PAUSE_SHOW = 2;
    const ST_PAUSE_HIDE = 3;
    const STATES = [
        self::ST_UNAPPROVED => 'inf:state:unapproved',
        self::ST_ACTIVE     => 'inf:state:active',
        self::ST_PAUSE_SHOW => 'inf:state:pause_show',
        self::ST_PAUSE_HIDE => 'inf:state:pause_hide',
    ];

    protected $_view_all_schools = false;
    protected $_get_raw_data = false;
    protected $_context = null;
    /**
     * @var int Infraction ID
     */
    public $id;

    /**
     * @var object Infraction data object
     */
    public $instance;

    /**
     * infraction constructor.
     * @param $id
     */
    public function __construct($id) {
        $this->id = $id;
        $this->instance = static::get_record($this->id, MUST_EXIST);
        $this->_view_all_schools = NED::cap_view_all_schools();
        $courseid = $this->instance->courseid ?? 0;
        $this->_context = NED::ctx($courseid);
    }

    /**
     * @param bool $raw
     */
    public function set_get_raw_data($raw=true){
        $this->_get_raw_data = $raw;
    }

    /**
     * @throws \moodle_exception
     */
    static protected function _print_permission_error(){
        NED::print_module_error('nopermissions', 'error', '', get_string('checkpermissions', 'core_role'));
    }

    /**
     * @param numeric|object $user_or_id
     *
     * @throws \moodle_exception
     */
    static public function show_error_if_can_not_add_aiv($user_or_id){
        if (!NED::can_add_user_ai($user_or_id)){
            static::_print_permission_error();
        }
    }

    /**
     * @param     $id
     * @param int $strictness
     *
     * @return null|\stdClass
     */
    static public function get_record($id, $strictness=IGNORE_MISSING){
        return NED::db()->get_record(static::TABLE, ['id' => $id], '*', $strictness) ?: null;
    }

    /**
     * @param \stdClass $record
     *
     * @return null|\stdClass
     */
    static public function save_record($record){
        global $DB;
        $t = time();
        if (empty($record->editorid)){
            $record->editorid = NED::get_userid_or_global();
            $record->authorid = $record->authorid ?? $record->editorid;
        }

        if ($record->id ?? false){
            $record->timemodified = $t;
            $DB->update_record(static::TABLE, $record);
        } else {
            $record->timecreated = $record->timecreated ?? $t;
            $record->timemodified = $record->timemodified ?? $t;
            $record->id = $DB->insert_record(static::TABLE, $record);
        }
        return $record;
    }

    /**
     * Delete AIV records and connected penalties
     * It should be endpoint for every infraction deleting
     *
     * @param array $params
     *
     * @return bool
     */
    static public function delete_records_by_params($params=[]){
        if (empty($params)) return false;

        $records = NED::db()->get_records(static::TABLE, $params);
        if (empty($records)) return true;

        $ngc_records = [];
        foreach ($records as $record){
            $ngc_records[] = static::get_infraction_ngc_record($record->id);
        }

        if (NED::db()->delete_records(static::TABLE, $params)){
            NED::$ned_grade_controller::check_and_delete($ngc_records, false);
            return true;
        }

        return false;
    }

    /**
     * @param numeric $cmid
     * @param numeric $userid
     * @param array   $other_params
     *
     * @return bool
     */
    static public function delete_records_by_userid_cmid($cmid, $userid, $other_params=[]){
        if (!$cmid && !$userid){
            return false;
        }

        $params = NED::val2arr($other_params);
        if ($cmid){
            $params['cmid'] = $cmid;
        }
        if ($userid){
            $params['student'] = $userid;
        }

        return static::delete_records_by_params($params);
    }

    /**
     * @param $id
     *
     * @return bool
     */
    static public function delete_record($id){
        return static::delete_records_by_params(['id' => $id]);
    }

    /**
     * @return bool
     */
    public function can_view() {
        static $_data = null;

        if (is_null($_data)){
            if ($this->_view_all_schools){
                $view_user = true;
            } else {
                $view_user = NED::can_view_user_ai($this->instance->student);
            }

            $_data = $view_user &&
                ($this->instance->state != static::ST_UNAPPROVED || $this->can_view_unapproved()) &&
                ($this->instance->state != static::ST_PAUSE_HIDE || $this->can_view_hidden()) &&
                !NED::grade_is_hidden_now_before_midn($this->instance->cmid, $this->instance->student);
        }

        return $_data;
    }

    /**
     * @return bool
     */
    public function can_view_hidden(){
        return $this->can_view_unapproved() || $this->can_edit();
    }

    /**
     * @return bool
     */
    public function can_view_unapproved(){
        static $_data = null;

        if (is_null($_data)){
            $_data = NED::can_view_unapproved();
        }

        return $_data;
    }

    /**
     * @return bool
     */
    public function can_edit() {
        static $_data = null;

        if (is_null($_data)){
            if ($this->can_view()){
                $_data = NED::can_edit_user_ai($this->instance->student, true);
            } else {
                $_data = false;
            }
        }

        return $_data;
    }

    /**
     * @return bool
     */
    public function can_edit_state(){
        static $_data = null;

        if (is_null($_data)){
            $_data = NED::can_edit_state();
        }

        return $_data;
    }

    /**
     * @throws \moodle_exception
     */
    public function show_error_if_can_not_view() {
        if (!$this->can_view()){
            static::_print_permission_error();
        }
    }

    /**
     * @throws \moodle_exception
     */
    public function show_error_if_can_not_edit() {
        if (!$this->can_edit()){
            static::_print_permission_error();
        }
    }

    /**
     * @param $name
     *
     * @return mixed|null
     */
    public function __get($name){
        if ($this->_get_raw_data){
            $pr_name = '_'.$name;
            if (property_exists($this, $pr_name)){
                return $this->$pr_name;
            }

            if (property_exists($this->instance, $name)) {
                return $this->instance->$name;
            }

            $method = 'get_' . $name;
            if (method_exists($this, $method)){
                return $this->$method();
            }
        } else {
            $method = 'get_' . $name;
            if (method_exists($this, $method)){
                return $this->$method();
            }

            $pr_name = '_'.$name;
            if (property_exists($this, $pr_name)){
                return $this->$pr_name;
            }

            if (property_exists($this->instance, $name)) {
                $counter = 0;
                return datatable_infractions::get_row_data($this->instance, $name, $counter, null);
            }
        }

        return null;
    }

    /**
     * @param string $name
     *
     * @return bool
     */
    public function __isset($name){
        $val = $this->$name ?? null;
        return isset($val);
    }

    /**
     * Get saved DB record from class object
     *
     * @return object|\stdClass|null
     */
    public function get_instance(){
        return clone($this->instance);
    }

    /**
     * @return bool
     */
    public function delete() {
        return static::delete_record($this->id);
    }

    /**
     * Try to create & save NGC record from current infraction object
     * Note: also set _get_raw_data in true
     *
     * @param int                 $grade_type
     * @param int                 $grade_change
     * @param numeric|object|null $editor_or_id - if null, uses global user
     *
     * @return bool - true, if NGC record saved successfully
     */
    public function create_ngc_record($grade_type, $grade_change=0, $editor_or_id=null){
        try {
            $NGC = NED::$ned_grade_controller;
            /** @var object|\local_ned_controller\support\ned_grade_controller_record $data */
            $data = new \stdClass();
            $this->set_get_raw_data(true);
            $data->courseid = $this->courseid;
            $data->cmid = $this->cmid;
            $data->userid = $this->student;
            $data->grade_type = $grade_type;
            $data->grade_change = $grade_change;
            $data->reason = $NGC::REASON_AI;
            $data->relatedid = $this->id;
            $data->status = $NGC::ST_PAUSED;
            if ($NGC::check_and_save($data, $editor_or_id, false)){
                $this->update_ngc_status($editor_or_id);
                return true;
            }

            return false;
        } catch (\Throwable $exception){
            return false;
        }
    }

    /**
     * Update NGC status by the current infraction state
     *
     * @param numeric|object|null $editor_or_id - if not null, update editor for all changed records
     *
     * @return void
     */
    public function update_ngc_status($editor_or_id=null){
        $NGC = NED::$ned_grade_controller;
        $ngc_record = static::get_infraction_ngc_record($this->id);
        if (empty($ngc_record)) return;
        if ($ngc_record->status == $NGC::ST_ERROR || $ngc_record->status == $NGC::ST_WAIT) return;

        $this->set_get_raw_data(true);
        $new_ngc_status = static::state2ngc($this->state);
        if ($new_ngc_status == $ngc_record->status) return;

        if ($new_ngc_status == $NGC::ST_DONE){
            $set = false;
            $state = $ngc_record->status;
        } else {
            $set = true;
            $state = $new_ngc_status;
        }

        $NGC::check_and_change_about_pause_state($state, $ngc_record, $set, true, true, $editor_or_id);
    }

    /**
     * @param bool $translate - if true, return translated strings
     *
     * @return array|string[]
     */
    public static function get_states_list($translate=true){
        return $translate ? NED::strings2menu(static::STATES) : static::STATES;
    }

    /**
     * @param bool $translate - if true, return translated strings
     *
     * @return array|string[]
     */
    public static function get_penalty_list($translate=true){
        return $translate ? NED::strings2menu(static::PENALTIES) : static::PENALTIES;
    }

    /**
     * @param bool $translate - if true, return translated strings
     *
     * @return array|string[]
     */
    public static function get_reason_list($translate=true){
        return $translate ? NED::strings2menu(static::REASONS) : static::REASONS;
    }

    /**
     * @param bool $translate - if true, return translated strings
     *
     * @return array|string[]
     */
    public static function get_activity_type_list($translate=true){
        $str_keys = ['assignment', 'unittest', 'exam', 'other'];
        return $translate ? NED::strings2menu($str_keys, true) : $str_keys;
    }

    /**
     * @param numeric           $courseid
     * @param numeric|bool|null $cmid - course module id, null to check nothing, false to check all, true - check all by course, or id
     *
     * @return array
     */
    public static function get_grader_list($courseid=null, $cmid=null) {
        $uids = NED::get_online_teachers_ids($courseid, null, $cmid ?: null, true, null, true);
        $users = NED::get_user_list($uids);
        $user_menu = NED::users2menu($users);
        asort($user_menu);

        return $user_menu;
    }

    /**
     * @param numeric           $courseid
     * @param numeric|bool|null $cmid - course module id, null to check nothing, false to check all, true - check all by course, or id
     *
     * @return array
     */
    public static function get_ct_list($courseid=null, $cmid=null){
        $uids = NED::get_classroom_teachers_ids($courseid, null, $cmid ?: null, true, null, true);
        $users = NED::get_user_list($uids);
        $user_menu = NED::users2menu($users);
        asort($user_menu);

        return $user_menu;
    }

    /**
     * Get count of student AIVs
     * If all time variables ($startdate, $enddate, $lastdays) are null, uses time period of the current NED school year
     *
     * @param object|numeric      $user_or_id   - student
     * @param object|numeric|null $course_or_id - filter by some course (otherwise count for all site)
     * @param numeric|null        $startdate    - count only after some date (UNIX time)
     * @param numeric|null        $enddate      - count only before some date (UNIX time)
     * @param numeric|null        $lastdays     - count only for some last days (num of days)
     * @param bool                $count_hidden - if true, count all AIVs, otherwise count only shown AIVs
     *
     * @return int - count of the AIVs
     */
    public static function get_user_aiv_count($user_or_id, $course_or_id=null, $startdate=null, $enddate=null, $lastdays=null, $count_hidden=false){
        $where = $params = [];
        NED::sql_add_equal('student', NED::get_userid_or_global($user_or_id), $where, $params);
        if ($course_or_id){
            NED::sql_add_equal('courseid', NED::get_courseid_or_global($course_or_id), $where, $params);
        }

        if (is_null($startdate) && is_null($enddate) && is_null($lastdays)){
            $startdate = NED::$C::get_config('ned_school_year_start') ?: 0;
        } else {
            if ($enddate){
                NED::sql_add_condition('infractiondate', $enddate, $where, $params, NED::SQL_COND_LTE, 'enddate');
            }
            if ($lastdays){
                NED::sql_add_condition('infractiondate', (time() - $lastdays * DAYSECS), $where, $params, NED::SQL_COND_GTE, 'lastdays');
            }
        }

        if ($startdate){
            NED::sql_add_condition('infractiondate', $startdate, $where, $params, NED::SQL_COND_GTE, 'startdate');
        }

        if (!$count_hidden){
            NED::sql_add_get_in_or_equal_options('state', [static::ST_ACTIVE, static::ST_PAUSE_SHOW], $where, $params);
        }

        return NED::db()->count_records_select(static::TABLE, NED::sql_condition($where), $params, 'COUNT(1)');
    }

    /**
     * Get student unapplied AIVs
     *
     * @param object|numeric      $user_or_id   - student
     * @param object|numeric|null $course_or_id - filter by some course
     * @param object|numeric|null $cm_or_id     - filter by some activity
     * @param bool                $return_first - if true, return only one record (or null, if found nothing)
     *
     * @return array|object|null - AIV record(s)
     */
    public static function get_user_unapplied_records($user_or_id, $course_or_id=null, $cm_or_id=null, $return_first=false){
        $where = $params = [];
        NED::sql_add_equal('student', NED::get_userid_or_global($user_or_id), $where, $params);
        NED::sql_add_equal('state', static::ST_UNAPPROVED, $where, $params);

        if ($course_or_id){
            NED::sql_add_equal('courseid', NED::get_courseid_or_global($course_or_id), $where, $params);
        }
        if ($cm_or_id){
            NED::sql_add_equal('cmid', NED::get_id($cm_or_id), $where, $params);
        }

        $records = NED::db()->get_records_select(static::TABLE, NED::sql_condition($where), $params);
        if ($return_first){
            return empty($records) ? null : reset($records);
        }
        return $records;
    }

    /**
     * Return NGC record connected to the infraction record by id, or null
     *
     * @param int $infraction_id
     *
     * @return \local_ned_controller\support\ned_grade_controller_record|object|null
     */
    public static function get_infraction_ngc_record($infraction_id=0){
        $NGC = NED::$ned_grade_controller;
        $record = $NGC::get_record_by_related_id($infraction_id);
        if ($record && $record->reason == $NGC::REASON_AI){
            return $record;
        }
        return null;
    }

    /**
     * Change state for all records by ids
     * Note: it doesn't update NGC statuses (as it is designed to be called from the NGC)
     *
     * @param int       $ngc_status - one of the {@see \local_ned_controller\ned_grade_controller::STATUSES}
     * @param int|array $ids        - ID's of AI record to update
     *
     * @return void
     */
    public static function update_states_by_ngc_status($ngc_status, $ids=[]){
        $state = static::ngc_status_to_infraction_state($ngc_status);
        if (is_null($state)) return;

        $where = $params = [];
        NED::sql_add_get_in_or_equal_options('id', $ids, $where, $params);
        NED::db()->set_field_select(static::TABLE, 'state', $state, NED::sql_condition($where), $params);
    }

    /**
     * Return NGC grade type by penalty
     *
     * @param int $penalty - one of the {@see static::PENALTIES}
     *
     * @return int|null - one of the {@see \local_ned_controller\ned_grade_controller::GRADE_TYPES} or null
     */
    public static function penalty2ngc($penalty=self::PENALTY_NONE){
        $NGC = NED::$ned_grade_controller;
        switch($penalty){
            default:
            case static::PENALTY_NONE:
                return null;
            case static::PENALTY_MINOR_PLAGIARISM:
                return $NGC::GT_DEDUCTION;
            case static::PENALTY_MAJOR_PLAGIARISM:
            case static::PENALTY_CHEATING:
                return $NGC::GT_AWARD_ZERO;
        }
    }

    /**
     * Return NGC status by infraction state
     *
     * @param int $state - one of the {@see static::STATES}
     *
     * @return int|null
     */
    public static function state2ngc($state=self::ST_ACTIVE){
        $NGC = NED::$ned_grade_controller;
        switch ($state){
            default:
            case static::ST_UNAPPROVED:
                return $NGC::ST_ERROR;
            case static::ST_ACTIVE:
                return $NGC::ST_DONE;
            case static::ST_PAUSE_SHOW:
                return $NGC::ST_OBSOLETED;
            case static::ST_PAUSE_HIDE:
                return $NGC::ST_PAUSED;
        }
    }

    /**
     * Return infraction state by NGC status
     *
     * @param int $status - one of the {@see \local_ned_controller\ned_grade_controller::STATUSES}
     *
     * @return int|null
     */
    public static function ngc_status_to_infraction_state($status){
        $NGC = NED::$ned_grade_controller;
        switch ($status){
            default:
            case $NGC::ST_WAIT:
            case $NGC::ST_ERROR:
                return null;
            case $NGC::ST_DONE:
                return static::ST_ACTIVE;
            case $NGC::ST_PAUSED:
                return static::ST_PAUSE_HIDE;
            case $NGC::ST_OBSOLETED:
                return static::ST_PAUSE_SHOW;
        }
    }
}
