<?php
/**
 * datatable_infractions - infractions render
 *
 * @package    local_academic_integrity
 * @subpackage NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_academic_integrity\output;
use local_academic_integrity\infraction as INF;
use local_academic_integrity\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/local/academic_integrity/lib.php');

/**
 * datatable_infractions
 * Useful methods:
 *  Render class for page: {@see datatable_infractions::render_full_page()}
 *
 *  Main render content method: {@see datatable_infractions::_render_page_content()}
 *  Main get_data method: {@see datatable_infractions::get_table_data()}
 *  Main get_data SQL method: {@see datatable_infractions::get_db_table_data()}
 *
 *  Start init point: {@see datatable_infractions::init()}
 *  Main check params: {@see datatable_infractions::_check_params()}
 *  Export content for template: {@see datatable_infractions::export_for_template()}
 *
 * @package    local_academic_integrity\output
 *
 * @property-read int    $state
 * @property-read int    $penalty
 * @property-read int    $reason
 * @property-read string $sort
 * @property-read bool   $dir
 */
class datatable_infractions extends \local_ned_controller\output\ned_base_table_page_render {
    protected const _PLUGIN = NED::AI;
    protected const _USE_NED_SCHOOL_YEAR_FILTER = true;

    //region SQL data
    protected const _SQL_TABLE = INF::TABLE;
    protected const _SQL_ALIAS = 'inf';

    protected const _SQL_USERID = 'student';
    protected const _SQL_GRADERID = 'grader';
    //endregion

    //region Params
    const PAR_STATE = 'state';
    const PAR_PENALTY = 'penalty';
    const PAR_REASON = 'reason';

    const PARAMS = [
        NED::PAR_FILTER_ID,
        NED::PAR_COURSE,
        NED::PAR_COURSE_VIEW,
        NED::PAR_CM,
        NED::PAR_USER,
        NED::PAR_GRADER,
        NED::PAR_GROUP,
        NED::PAR_SCHOOL,
        NED::PAR_SCHOOL_YEAR,
        NED::PAR_ACTION,
        NED::PAR_PAGE,
        NED::PAR_PERPAGE,

        self::PAR_STATE,
        self::PAR_PENALTY,
        self::PAR_REASON,

        NED::PAR_SORT,
        NED::PAR_DIR,
    ];

    /**
     * By default, params uses _PARAM_TYPE_DEFAULT and _PARAM_VALUE_DEFAULT, but you can set defaults to others here
     * Possible keys for rewrite array:
     *  • type - change default type of param
     *  • default - change default value of param
     *  • property - load raw param value for $this as $this->{$property}
     * Also {@see \local_ned_controller\shared\C::PARAM_DATA}
     *
     * @var array[] - keys from PARAMS, value is array
     */
    const PARAM_DATA = [
        self::PAR_STATE   => ['type' => PARAM_INT, 'default' => null, 'property' => '_state'],
        self::PAR_PENALTY => ['type' => PARAM_INT, 'default' => NED::ALL, 'property' => '_penalty'],
        self::PAR_REASON  => ['type' => PARAM_INT, 'default' => NED::ALL, 'property' => '_reason'],

        NED::PAR_GRADER => ['type' => PARAM_INT, 'default' => self::GRADER_ALL],
        NED::PAR_SORT   => ['type' => PARAM_ALPHANUMEXT, 'default' => ''],
        NED::PAR_DIR    => ['type' => PARAM_BOOL, 'default' => false],
    ];
    //endregion

    //region Other Consts
    const URL = NED::PAGE_AI_INFRACTIONS;
    const ACTION_DOWNLOAD_CSV = 1;
    const STATE_ALL = -1;

    const TABLE_COLUMNS = [
        'infractiondate', 'school', 'courseid', 'grader', 'ctid', 'student', 'cmid',
        'activitytype', 'penalty', 'reason', 'note', 'state',
    ];
    const PROFILE_COLUMNS = [
        'infractiondate', 'course', 'activity', 'penalty', 'reason', 'note'
    ];
    //endregion

    //region Properties
    protected $_state;
    protected $_penalty;
    protected $_reason;

    protected $_sort = '';
    protected $_dir = false;
    //endregion

    //region Init methods
    /**
     * @constructor
     *
     */
    public function __construct() {
        parent::__construct();

        $this->_view_all_schools = $this->_is_admin || NED::cap_view_all_schools();
        if (!$this->_view_all_schools && NED::cap_view_own_schools()){
            $this->_limit_schoolids = NED::get_user_schools($this->_viewer, false, true);
        }

        $this->_cap_view = $this->_view_all_schools || !empty($this->_limit_schoolids);
        $this->_cap_edit = NED::cap_manage_violations();
    }

    /**
     * Base check of the base params
     * You may not call it at all, if you make all params check by yourself
     * Normally calling from the {@see _check_params()}
     */
    protected function _check_base_params(){
        parent::_check_base_params();

        $this->_filter_id = $this->_params[NED::PAR_FILTER_ID];

        $options = static::get_state_options();
        $this->_state = NED::isset_key($options, $this->_state, NED::can_edit_state() ? INF::ST_UNAPPROVED : static::STATE_ALL);

        $options = static::get_penalty_options();
        $this->_penalty = NED::isset_key($options, $this->_penalty, NED::ALL);

        $options = static::get_reason_options();
        $this->_reason = NED::isset_key($options, $this->_reason, NED::ALL);

        $sort_option = static::get_table_columns(false);
        $this->_sort = NED::isset_in_list($sort_option, $this->_params[NED::PAR_SORT], 'infractiondate');

        $this->_dir = (bool)$this->_params[NED::PAR_DIR];
    }
    //endregion

    //region Setup Page methods
    /**
     * Setup PAGE data
     *
     * @param bool $set_header
     */
    public function setup_page($set_header=true){
        if ($this->can_see() && $this->_action && $this->_action == static::ACTION_DOWNLOAD_CSV){
            $this->download_csv();
            die;
        }

        parent::setup_page($set_header);
    }

    /**
     * Download CSV file and die
     */
    public function download_csv(){
        if (!$this->can_see()) return;

        $fp_csv = NED::csv_prepare_file_download('infractions_'.date('Y-m-d'));
        $fputcsv = NED::csv_get_data_fn($fp_csv);

        // Output the column headings.
        $column_data = static::get_table_columns(false, true);
        $fputcsv($column_data);
        $counter = 0;

        $this->_db_process_sql_query($select, $joins, $where, $params, $groupby, $orderby);
        $sql = static::_sql_generate_sql($select, $joins, $where, $groupby, $orderby);
        $rs = NED::db()->get_recordset_sql($sql, $params);
        foreach ($rs as $record) {
            $row = array();
            foreach ($column_data as $column => $name){
                $row[] = static::get_row_data($record, $column, $counter, [], true);
            }
            $fputcsv($row);
        }
        $rs->close();

        fclose($fp_csv);
        die;
    }

    /**
     * @return string
     */
    public function get_page_title(){
        return NED::str('academicintegrityinfractions');
    }
    //endregion

    //region SQL menu filter methods
    /**
     * Add limitation by grader, changing $where and $params for sql query
     *
     * @param array $joins
     * @param array $where
     * @param array $params
     *
     * @return void
     */
    protected function _sql_add_view_limitation(&$joins=[], &$where=[], &$params=[]){
        if (!$this->_view_all_schools){
            if (empty($this->_limit_schoolids)){
                $where[] = NED::SQL_NONE_COND;
            } else {
                $this->_sql_add_limit_school($joins, $where, $params, $this->_limit_schoolids);
            }
        }
    }
    //endregion

    //region Get main table data methods
    /**
     * Process raw records from the DB and return rendered result
     * Normally called from the {@see get_table_data()}
     * @abstract
     *
     * @param array $raw_records
     *
     * @return array
     */
    protected function _process_table_raw_records(&$raw_records=[]){
        foreach ($raw_records as $key => $r_record){
            if (NED::grade_is_hidden_now_before_midn($r_record->cmid, $r_record->student)){
                unset($raw_records[$key]);
            }
        }

        return $raw_records;
    }

    /**
     * Process SQL query for the getting data from the DB for main table
     * @abstract
     *
     * @param array|string $select
     * @param array|string $joins
     * @param array|string $where
     * @param array        $params
     * @param array|string $groupby
     * @param array|string $orderby
     *
     * @return bool - saves result in the params, return true if you can continue query
     */
    protected function _db_process_sql_query(&$select=[], &$joins=[], &$where=[], &$params=[], &$groupby=[], &$orderby=[]){
        [$select, $joins, $where, $groupby, $orderby] = NED::val2arr_multi(true, $select, $joins, $where, $groupby, $orderby);

        $school_t = static::_SQL_SCHOOL_ALIAS;
        $group_t = static::_SQL_GROUP_ALIAS;
        $params = $params ?: [];
        if (empty($select)){
            $select = [
                static::_sql_a('*'),
                "$group_t.group_ids AS group_ids",
                "$group_t.group_names AS group_names",
                "$school_t.school_ids AS school_ids",
                "$school_t.school_names AS school_names",
                "$school_t.school_codes AS school_codes",
            ];
        }
        $joins = $this->_db_get_base_join($joins, $params);

        $simple_school_id_filter = $this->_sql_provide_school_limitation($joins,$where,$params,$groupby);
        static::_sql_set_simple_filters($where, $params,
            $this->_courseid, $this->_cmid, $this->_userid, $this->_groupid, $this->_graderid, $simple_school_id_filter,
            null, $this->_filter_id, $this->_school_year
        );

        if (isset($this->_state) && $this->_state > static::STATE_ALL){
            static::_sql_add_equal('state', $this->_state, $where, $params);
        } else {
            $not_states = [];
            if (!NED::can_view_unapproved()){
                $not_states[] = INF::ST_UNAPPROVED;
            }
            if (!NED::can_view_hidden()){
                $not_states[] = INF::ST_PAUSE_HIDE;
            }
            if (!empty($not_states)){
                static::_sql_add_simple_limit('state', $not_states, $where, $params, false, 'param', false);
            }
        }
        if ($this->_penalty){
            static::_sql_add_equal('penalty', $this->_penalty, $where, $params);
        }
        if (isset($this->_reason) && $this->_reason > NED::ALL){
            static::_sql_add_equal('reason', $this->_reason, $where, $params);
        }

        if ($this->_sort){
            if ($this->_sort == 'school'){
                $sort = "$school_t.school_codes";
            } else {
                $sort = static::_sql_a($this->_sort);
            }
            $orderby = [$sort.' '.($this->_dir ? 'DESC' : 'ASC')];
        }

        return true;
    }
    //endregion

    //region Render utils methods, including r_*_selector() functions (filter selectors)
    /**
     * Get icon of the state
     *
     * @param int $state - one of the {@see \local_academic_integrity\infraction::STATES}
     *
     * @return string
     */
    static public function r_state_icon($state=INF::ST_ACTIVE){
        $add2eye = '';
        $add2gavel = '';
        $content = '';

        switch ($state){
            default:
                $content = NED::fa('fa-question');
                break;
            case INF::ST_UNAPPROVED:
                $content = NED::fa('fa-ellipsis-h');
                break;
            case INF::ST_ACTIVE:
                $state = INF::ST_ACTIVE;
                break;
            case INF::ST_PAUSE_SHOW:
                $add2gavel .= ' inf-crossed';
                break;
            case INF::ST_PAUSE_HIDE:
                $add2eye .= ' inf-crossed';
                $add2gavel .= ' inf-crossed';
                break;
        }

        $title = INF::get_states_list()[$state] ?? '?';

        if (empty($content)){
            $content =
                NED::fa('fa-eye'.$add2eye).
                NED::span(NED::HTML_SPACE.'+'.NED::HTML_SPACE).
                NED::fa('fa-gavel'.$add2gavel);
        }
        return NED::div($content, 'inf-state-icon cursor-help', ['title' => $title, 'data-toggle' => 'tooltip']);

    }
    //endregion

    //region Get filter options [get_*_options()]
    /**
     * @return array
     */
    public function get_schools_options(){
        if ($this->_view_all_schools) return parent::get_schools_options();

        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $data = NED::records2menu(NED::get_user_schools($this->_viewer));
            $data = [NED::ALL => get_string('all')] + $data;

            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array|string[]
     */
    public function get_state_options(){
        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $data = [static::STATE_ALL => get_string('all')] + INF::get_states_list();
            if (!NED::can_view_unapproved()){
                unset($data[INF::ST_UNAPPROVED]);
            }
            if (!NED::can_view_hidden()){
                unset($data[INF::ST_PAUSE_HIDE]);
            }
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array|string[]
     */
    public function get_penalty_options(){
        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $data = [NED::ALL => get_string('all')] + INF::get_penalty_list();
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array|string[]
     */
    public function get_reason_options(){
        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $data = [NED::ALL => get_string('all')] + INF::get_reason_list();
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }
    //endregion

    //region Main render page content methods

    /**
     * Render page content
     * Normally calling from the {@see export_for_template()}
     * You can rewrite this method and not changing original {@see export_for_template()}
     *
     * @noinspection DuplicatedCode
     */
    protected function _render_page_content(){
        $this->content->reports_header = NED::render(new \local_schoolmanager\output\reports_header());
        $this->content->filterid = $this->_filter_id;
        $this->content->page_title = $this->get_page_title();
        if (!empty($this->_limit_schoolids) && count($this->_limit_schoolids) == 1){
            $school = NED::school_get_school(reset($this->_limit_schoolids));
            if (isset($school->name)){
                $this->content->school_name = $school->name;
            }
        }
        $this->content->download_csv_url = $this->get_my_url([NED::PAR_ACTION => static::ACTION_DOWNLOAD_CSV])->out(false);

        $columns = static::get_table_columns();
        $column_data = [];
        $url = $this->get_my_url();
        foreach ($columns as $column) {
            $string[$column] = NED::str($column);
            if ($this->_sort != $column) {
                $column_icon = "";
                $column_dir = false;
            } else {
                $column_dir = !$this->_dir;
                $column_icon = $this->_dir ? 'sort_desc' : 'sort_asc';
                $column_icon = $this->_o->pix_icon('t/'.$column_icon, $column, 'moodle', ['class' => 'iconsort']);
            }
            if (($column == 'rowcount') || ($column == 'action') || ($column == 'kicadiff')) {
                $column_data[$column] = $string[$column];
            } else {
                $url->params([NED::PAR_SORT => $column, NED::PAR_DIR => $column_dir]);
                $column_data[$column] = NED::link($url->out(false), $string[$column]).' '.$column_icon;
            }
        }
        // reset $url params
        $url = $this->get_my_url();

        $counter = 0;
        $table = static::fill_and_get_table($column_data, $this->get_table_data(), $counter, $this->_params);
        $this->content->table = NED::render_table($table);
        $this->content->pager = $this->r_pager();

        $grader_selector = $this->r_grader_selector();
        if (!empty($grader_selector)){
            $this->content->control_panel1[] = $grader_selector;
        }

        $this->content->control_panel1[] = NED::single_select($url, static::PAR_PENALTY, $this->get_penalty_options(),
            $this->_penalty, NED::str('penalty'));
        $this->content->control_panel1[] = NED::single_select($url, static::PAR_REASON, $this->get_reason_options(),
            $this->_reason, NED::str('reason'));

        $school_selector = $this->r_school_selector();
        if (!empty($school_selector)){
            $this->content->control_panel2[] = $school_selector;
        }

        if (!$this->_course_view){
            $this->content->control_panel2[] = $this->r_course_selector();
        }

        if ($this->_courseid > 0){
            $this->content->control_panel2[] = $this->r_cm_selector();
            $this->content->control_panel2[] = $this->r_group_selector();
        }

        $this->content->control_panel2[] = $this->r_users_selector();

        $this->content->control_panel2[] = NED::single_select($url, static::PAR_STATE, $this->get_state_options(),
            $this->_state, NED::str('state'));
        $this->content->control_panel2[] = $this->r_school_year_selector();

        if ($this->_courseid > 0){
            $this->content->buttons[] = $this->r_course_view_button();
        }

        if (!empty($url->params()) || !empty($this->_filter_ids) || !empty($this->_ids)){
            $this->content->buttons[] = $this->r_reset_button();
        }

        $this->content->control_panel3[] = $this->r_perpage_selector();
    }

    /**
     * Some last content changes at final {@see export_for_template()}
     * Normally calling from the {@see export_for_template()}
     */
    protected function _before_finish_export(){
        parent::_before_finish_export();
        $this->content->has_data = true;
    }

    /**
     * Get AIV records for the one student
     *
     * @param numeric|null $userid
     * @param numeric|null $school_year - {@see static::SCHOOL_YEAR_OPTIONS}
     * @param numeric|null $courseid
     * @param bool         $show_hidden - if true, get all AIVs, otherwise get only shown AIVs
     *
     * @return array|object[]
     */
    static public function get_student_records($userid=null, $school_year=null, $courseid=null, $show_hidden=false){
        $where = $params = [];
        // don't call params, which requires join other tables
        static::_sql_set_simple_filters($where, $params, $courseid, null, $userid, null, null, null,
            null, null, $school_year);

        if (!$show_hidden){
            NED::sql_add_get_in_or_equal_options('state', [INF::ST_ACTIVE, INF::ST_PAUSE_SHOW], $where, $params);
        }

        $sql = static::_sql_generate_sql([], [], $where);
        return NED::db()->get_records_sql($sql, $params);
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param null $pageparams
     * @param bool $export
     *
     * @return false|string
     */
    public static function get_row_data($data, $column, &$counter=0, $pageparams=[], $export=false) {
        global $USER;

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y g:i A", $data->$column);
                }
                break;
            case 'infractiondate':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("m/d/Y", $data->$column);
                }
                break;
            case 'student':
            case 'grader':
            case 'ctid':
                $var = NED::q_user_link($data->$column, $data->courseid ?? 0, 'inline-block', $export);
                break;
            case 'infraction':
            case 'infractiontype':
            case 'penalty':
                $var = INF::get_penalty_list()[$data->penalty] ?? '?';
                break;
            case 'reason':
                $var = INF::get_reason_list()[$data->reason] ?? '?';
                break;
            case 'activitytype':
                $var = '-';
                if (!empty($data->$column)) {
                    $var = NED::str($data->$column);
                }
                break;
            case 'state':
                if ($export){
                    $var = INF::get_states_list()[$data->$column] ?? '';
                } else {
                    $var = static::r_state_icon($data->$column);
                }
                break;
            case 'action':
                $var = '-';
                if ($export) break;

                $actions = array();
                $infraction = new \local_academic_integrity\infraction($data->id);

                if ($infraction->can_view()) {
                    $params = [NED::PAR_ID => $data->id, NED::PAR_COURSE => $pageparams[NED::PAR_COURSE] ?? 0,
                        NED::PAR_RETURN_URL => $pageparams[NED::PAR_RETURN_URL] ?? ''];
                    $edit = $infraction->can_edit();

                    if ($edit){
                        $actions[] = array(
                            'url' => NED::url('~/infraction.php', $params),
                            'icon' => new \pix_icon('t/edit', get_string('edit')),
                            'attributes' => array('class' => 'action-edit')
                        );
                    }

                    $actions[] = array(
                        'url' =>  NED::url('~/infraction_view.php', $params),
                        'icon' => new \pix_icon('view', get_string('view'), NED::$PLUGIN_NAME),
                        'attributes' => array('class' => 'action-view')
                    );

                    if ($edit){
                        $actions[] = array(
                            'url' =>  NED::url('~/infraction_delete.php', $params),
                            'icon' => new \pix_icon('i/delete', get_string('delete')),
                            'attributes' => array('class' => 'action-delete')
                        );
                    }
                }
                if (!empty($actions)) {
                    $actionshtml = array();
                    foreach ($actions as $action) {
                        $action['attributes']['role'] = 'button';
                        $actionshtml[] = NED::O()->action_icon($action['url'], $action['icon'], null, $action['attributes']);
                    }
                    if ($html = NED::span(implode('', $actionshtml), 'class-item-actions item-actions')) {
                        $var = $html;
                    }
                }
                break;
            case 'course':
            case 'courseid':
                $var = $data->coursename ?? NED::q_course_link($data->courseid, true, $export);
                break;
            case 'activity':
            case 'cmid':
                if (isset($data->cm_name)){
                    $var = $data->cm_name;
                } else {
                    if ($export || $USER->id == $data->student){
                        $var = NED::q_cm_link($data->cmid, $data->courseid ?? 0, null, $export);
                    } else {
                        $var = NED::q_cm_grade_link($data->cmid, $data->student, $data->courseid ?? 0, true, true);
                    }
                }
                break;
            case 'school':
                $var = null;
                if (isset($data->school_names)){
                    if (isset($data->school_codes) && !$export){
                        $var = NED::span($data->school_codes, '', ['title' => $data->school_names]);
                    } else {
                        $var = $data->school_names;
                    }
                }
                $var = $var ?? NED::get_user_school_names($data->student);
                break;
            case 'note':
                if ($export){
                    $var = $data->$column ?? '';
                } else {
                    $var = NED::fa('fa-info-circle color-lightblue cursor-help', '', $data->$column, ['data-toggle' => 'tooltip']);
                }
                break;
            default:
                $var = $data->$column ?? '';
        }

        return $var;
    }

    /**
     * @param array|string[] $column_data
     * @param array|object[] $tablerows
     * @param int            $counter
     * @param array          $pageparams
     *
     * @return \html_table
     */
    public static function fill_and_get_table($column_data, $tablerows, $counter=0, $pageparams=[]){
        $columns = array_keys($column_data);
        $counter = $counter ?? 0;

        $table = new \html_table();
        $table->head = array();
        $table->wrap = array();
        $table->attributes = ['class' => 'nedtable fullwidth'];

        foreach ($column_data as $column => $c_data){
            $table->head[$column] = $c_data;
            $table->wrap[$column] = '';
        }

        // Override cell wrap.
        $table->wrap['action'] = 'nowrap';

        foreach ($tablerows as $tablerow){
            $row = NED::row([]);
            $row->id = $tablerow->id;

            foreach ($columns as $column){
                $c_class = [];
                if ($column == 'state' && $tablerow->state == INF::ST_UNAPPROVED){
                    $c_class[] = 'aiv-unapproved-cell';
                }
                $row->cells[$column] = NED::cell(static::get_row_data($tablerow, $column, $counter, $pageparams), $c_class);
            }

            $table->data[] = $row;
        }

        return $table;
    }

    /**
     * Return table columns for the infraction table
     *
     * @param bool $with_action - true by default
     * @param bool $with_translate - false by default
     *
     * @return array|string[]
     */
    public static function get_table_columns($with_action=true, $with_translate=false){
        $columns = static::TABLE_COLUMNS;
        if ($with_action){
            $columns[] = 'action';
        }

        if ($with_translate){
            return NED::strings2menu($columns, true);
        }

        return $columns;
    }

    /**
     * Return table columns for the infraction table in the user profile
     *
     * @param bool $with_action - true by default
     * @param bool $with_translate - false by default
     *
     * @return array|string[]
     */
    public static function get_user_profile_columns($with_action=true, $with_translate=false){
        $columns = static::PROFILE_COLUMNS;
        if (NED::cap_view_unapproved()){
            $columns[] = 'state';
        }
        if ($with_action){
            $columns[] = 'action';
        }

        if ($with_translate){
            return NED::strings2menu($columns, true);
        }

        return $columns;
    }
    //endregion
}
