<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Assign role to cohort form.
 *
 * @package    tool_cohortroles
 * @copyright  2018 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kica\form;

defined('MOODLE_INTERNAL') || die();

use \local_kica as kica;
use local_kica\helper;
use local_kica\shared_lib as NED;

/** @var \stdClass $CFG */
require_once($CFG->libdir . '/formslib.php');

/**
 * KICA course setting form.
 *
 * @package    local_kica
 * @copyright  2018 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class kicaactivities_edit extends \moodleform {
    /**
     * @var int
     */
    public $grademax = 0;
    /**
     * @var null
     */
    public $area = null;

    /**
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public function definition() {
        $mform = $this->_form;

        $customdata = $this->_customdata;
        $gradeitem = $customdata['gradeitem'];
        $courseid = $customdata['courseid'];
        $this->area = $customdata['area'];

        $this->grademax = $gradeitem->grademax;

        $instance = NED::get_module_instance($gradeitem->iteminstance, $gradeitem->itemmodule);
        $mform->addElement('header', '', $instance->name, '');
        $mform->addElement('static', 'grademaxinfo', NED::str('grademax'), round($gradeitem->grademax));

        $kicagroups = NED::get_kica_groups($courseid);
        $mform->addElement('select', 'kicagroup', NED::str('kicagroup'), $kicagroups);

        $haskicalabel = false;
        if ($gradeitem->itemmodule === 'quiz') {
            $cm = NED::get_cm_by_kica_or_grade_item($gradeitem);
            $haskicalabel = helper::has_kica_tag($cm->id);
        }

        foreach (NED::KICA_KEYS as $key){
            if (is_null($this->area) && !$haskicalabel) {
                $mform->addElement('text', $key, NED::str($key), ['size' => '10']);
                $mform->setType($key, PARAM_FLOAT);
                $mform->addRule($key, null, 'required', null, 'client');
            } else {
                $mform->addElement('static', $key, NED::str($key));
            }
        }

        $mform->addElement('hidden', 'id');
        $mform->setType('id', PARAM_INT);

        $mform->addElement('hidden', 'courseid');
        $mform->setType('courseid', PARAM_INT);

        $this->add_action_buttons(true, NED::str('submit'));
    }

    /**
     * @param array $data
     * @param array $files
     * @return array
     * @throws \coding_exception
     */
    public function validation($data, $files) {
        $errors = array();

        if (!is_null($this->area)) {
            return $errors;
        }
        if (isset($data['knowledge']) && isset($data['inquiry']) && isset($data['communication']) && isset($data['application'])) {
            if (($data['knowledge'] + $data['inquiry'] + $data['communication'] + $data['application']) > $this->grademax) {
                $errors['knowledge'] = get_string('kicadistributiongreater', 'local_kica', round($this->grademax));
            }

            if (($data['knowledge'] + $data['inquiry'] + $data['communication'] + $data['application']) < $this->grademax) {
                $errors['knowledge'] = get_string('kicadistributionless', 'local_kica', round($this->grademax));
            }
        }
        return $errors;
    }
}