<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Contains helper class for KICA area.
 *
 * @package    local_kica
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kica;

defined('MOODLE_INTERNAL') || die();

use local_kica\shared_lib as NED;

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');

/**
 * Helper class for KICA.
 *
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class helper {
    /**
     * @param $fromform
     * @return array
     */
    public static function get_activity_cmids_from_form($fromform) {
        $cmids = [];
        foreach ($fromform as $key => $value) {
            if (strpos($key, 'activity') === 0) {
                if ((int)$value > 0) {
                    $cmids[$value] = $value;
                }
            }
        }
        return $cmids;
    }

    /**
     * @param $formdata
     * @return array array[itemid][userid][knowledge] = grade
     */
    public static function parse_grade($formdata) {
        $var = array();

        foreach ($formdata as $name => $value) {
            $arr = explode('_', $name);
            if (count($arr) != 3) {
                continue;
            }
            if (!in_array($arr[0], NED::KICA_KEYS)) {
                continue;
            }
            $var[$arr[1]][$arr[2]][$arr[0]] = $value;
        }
        return $var;
    }

    /**
     * @param $itemid
     * @param $userid
     * @param $grades
     */
    public static function save_grade($itemid, $userid, $grades) {
        $kicaitem = kica_item::get_by_id($itemid);
        $rec = new \stdClass();

        foreach (NED::KICA_KEYS as $item) {
            if (!(isset($grades[$item]) && 0 <= $grades[$item] && round($grades[$item], 4) <= $kicaitem->get_section($item, false, 4))) {
                $grades[$item] = null;
            }

            $rec->$item = grade_floatval($grades[$item]);
        }

        $rec->finalgrade = grade_floatval(array_sum($grades));
        $rec->flag = 0;

        $current_grade_record = kica_grade::get_raw_records_by_itemids_userids($itemid, $userid, true);
        if ($current_grade_record) {
            $rec->id = $current_grade_record->id;
        } else {
            $rec->courseid = $kicaitem->courseid;
            $rec->userid = $userid;
            $rec->itemid = $itemid;
        }

        kica_grade::save_record($rec);
    }

    /**
     * @param numeric|object $course_or_id
     *
     * @return bool
     */
    public static function iskicaready($course_or_id){
        return NED::is_kica_ready($course_or_id);
    }

    /**
     * @param $courseid
     *
     * @return bool
     */
    public static function has_kica_item($courseid){
        return \local_kica\kica_item::course_has_kica_item($courseid);
    }

    /**
     * @param \local_kica\kica_item $kicaitem
     * @param string                $column
     * @param int                   $counter
     * @param int                   $courseid
     *
     * @return false|string
     */
    public static function kicaactivities_data($kicaitem, $column, &$counter, $courseid=0)
    {
        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'timecreated':
            case 'timemodified':
                $var = '-';
                if ($kicaitem->$column > 0) {
                    $var = userdate($kicaitem->$column);
                }
                break;
            case 'itemname':
                $var = $kicaitem->cm ? NED::q_cm_link($kicaitem->cm) : '-';
                break;
            case 'grademax':
            case 'knowledge':
            case 'inquiry':
            case 'communication':
            case 'application':
                $var = '-';
                if ($kicaitem->$column) {
                    $var = round($kicaitem->$column, 2);
                }
                break;

            case 'actions':
                $var = '-';
                if (!$kicaitem || !$kicaitem->cm) break;

                $actions = array();
                if ($kicaitem->is_graded() || (($kicaitem->itemmodule === 'quiz') && helper::has_kica_tag($kicaitem->cm->id))){
                    $icon = new \pix_icon('t/locked', get_string('edit'));
                    $lockedcls = ' item-locked';
                } else {
                    $icon = new \pix_icon('t/edit', get_string('edit'));
                    $lockedcls = '';
                }
                $actions[] = [
                    'url' => NED::url('~/kicaactivities_edit.php', ['id' => $kicaitem->id, 'courseid' => $courseid]),
                    'icon' => $icon,
                    'attributes' => ['class' => 'action-edit'.$lockedcls]
                ];
                $actions[] = [
                    'url' => NED::url('~/kicaactivities_delete.php', ['id' => $kicaitem->id, 'courseid' => $courseid]),
                    'icon' => new \pix_icon('t/delete', get_string('delete')),
                    'attributes' => ['class' => 'action-delete']
                ];
                if (!empty($actions)) {
                    $actionshtml = array();
                    foreach ($actions as $action) {
                        $action['attributes']['role'] = 'button';
                        $actionshtml[] = NED::O()->action_icon($action['url'], $action['icon'], null, $action['attributes']);
                    }
                    if ($html = NED::span($actionshtml, 'class-item-actions item-actions')) {
                        $var = $html;
                    }
                }
                if ($kicaitem->itemmodule === 'quiz') {
                    if (helper::has_kica_tag($kicaitem->cm->id)) {
                        $var .= NED::span(NED::fa('fa-bar-chart'), 'text-blue');
                    } else {
                        $var .= NED::span(NED::fa('fa-bar-chart'), 'muted');
                    }
                }
                break;
            default:
                $var = $kicaitem->$column;
        }

        return $var;
    }

    /**
     * @param $iteminstance
     * @param $userid
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_gradingform_instance_itemid($iteminstance, $userid) {
        global $DB;

        $sql = "SELECT ag.id,								 
                       ag.grader
                  FROM {assign_grades} ag, {assign_submission} s
                 WHERE s.assignment = ?
                   AND s.userid = ag.userid
                   AND s.latest = 1
                   AND s.attemptnumber = ag.attemptnumber										 
                   AND ag.assignment = s.assignment
                   AND ag.userid = ?
								";
        return $DB->get_field_sql($sql, array($iteminstance, $userid));
    }

    /**
     * @param $kicaitem
     * @param $userid
     * @param bool $controller
     * @return bool|null
     * @throws \dml_exception
     */
    public static function is_gradingform_needupdate(kica_item $kicaitem, $userid , $controller = false) {
        global $USER;
        if ($controller === false) {
            $controller = $kicaitem->get_grading_controller();
        }

        if (is_null($controller)) {
            return null;
        }
        if (!is_null($controller) && $itemid = static::get_gradingform_instance_itemid($kicaitem->iteminstance, $userid)) {
            $currentinstance = $controller->get_current_instance($USER->id, $itemid);
            return $currentinstance && ($currentinstance->get_status() == \gradingform_instance::INSTANCE_STATUS_NEEDUPDATE);
        }
        return false;
    }

    /**
     * Checks KICA tag in activity
     *
     * @param $cmid
     * @return bool
     * @throws \dml_exception
     */
    public static function has_kica_tag($cmid) {
        global $DB;
        $sql = "SELECT t.* 
                  FROM {tag_instance} ti 
                  JOIN {tag} t 
                    ON ti.tagid = t.id
                 WHERE ti.component = 'core'
                   AND ti.itemtype = 'course_modules' 
                   AND ti.itemid = ? 
                   AND t.name = 'kica'";
        if ($DB->record_exists_sql($sql, [$cmid])) {
            return true;
        }
        return false;
    }

    /**
     * @param $kicaitemid
     * @return void
     * @throws \dml_exception
     */
    public static function clear_activity_kica_flag($kicaitemid) {
        global $DB;

        $DB->execute('UPDATE {local_kica_grade_grades} SET flag = 0 WHERE itemid = ?', [$kicaitemid]);
    }
}