<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Contains helper class for KICA area.
 *
 * @package    local_kica
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kica;
use local_kica\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/**
 * Class kica_grade
 * @package local_kica
 */
class kica_grade {
    const SPLIT_7030 = 1;
    const SPLIT_702010 = 2;
    const CALCULATION_WEIGHTED = 1;
    const CALCULATION_NATURAL = 2;
    const TABLE = 'local_kica_grade_grades';
    const GRADE_TOLERANCE = 0.00004;

    /**
     * @var array id => kica grade
     */
    static protected $_data = [];
    /**
     * @var array [userid][itemid] => kica grade
     */
    static protected $_data_userid_itemid = [];
    /**
     * @var array [courseid][userid] => bool
     */
    static protected $_data_course_userid = [];

    /**
     * @var int $id
     */
    public $id = null;

    /**
     * @var int $courseid
     */
    public $courseid;

    /**
     * @var float $userid
     */
    public $userid;
    /**
     * @var float $itemid
     */
    public $itemid;
    /**
     * @var float $finalgrade
     */
    public $finalgrade = null;
    /**
     * @var float $finalgradebook
     */
    public $finalgradebook = null;
    /**
     * @var float $knowledge
     */
    public $knowledge = null;
    /**
     * @var float $inquiry
     */
    public $inquiry = null;
    /**
     * @var float $communication
     */
    public $communication = null;
    /**
     * @var float $application
     */
    public $application = null;
    /**
     * @var int $pullfromgradebook
     */
    public $pullfromgradebook = 0;
    /**
     * @var bool
     */
    public $area = false;
    /**
     * @var int $flag
     */
    public $flag = 0;
    /**
     * @var int $excluded
     */
    public $excluded = 0;
    /**
     * @var int $timecreated
     */
    public $timecreated = 0;
    /**
     * @var int $timemodified
     */
    public $timemodified = 0;
    /**
     * @var \local_kica\kica_item $kica_grade_item
     */
    public $kica_grade_item = null;

    /**
     * @var \local_ned_controller\support\ned_grade_controller_record $_ngc_record
     */
    protected $_ngc_record = null;
    /**
     * @var bool|null $_is_grade_hidden - please, access it only through the {@see is_grade_hidden()}
     */
    protected $_is_grade_hidden = null;

    /**
     * kica grade constructor
     * @constructor
     *
     * @param numeric|object                    $user_or_id
     * @param kica_item|numeric|object|array    $item_or_id
     * @param bool                              $get_empty_object If true, will load/check nothing
     */
    public function __construct($user_or_id, $item_or_id, $get_empty_object=false) {
        if ($get_empty_object) return;

        $kica_item = kica_item::get_by_item_or_id($item_or_id);
        if (empty($kica_item->id)) return;

        $this->update_from_db($user_or_id, $kica_item);
    }

    /**
     * Update object from the DB tables and update local cache
     *
     * @param numeric|object                    $user_or_id
     * @param kica_item                         $kica_item
     * @param object|null                       $kica - (optional) course kica record
     * @param object|\grade_grade|false|null    $grade_grade - (optional) grade_grade record, if null - try to load it
     * @param object|false|null                 $grade_recorde - (optional) kica grade record, if null - try to load it
     * @param object|\local_ned_controller\support\ned_grade_controller_record $ngc_record - (optional) ngc record, if null - try to load it
     *
     */
    public function update_from_db($user_or_id, $kica_item, $kica=null, $grade_grade=null, $grade_recorde=null, $ngc_record=null){
        if (empty($user_or_id) || empty($kica_item) || !is_object($kica_item)){
            return;
        }

        $userid = NED::get_id($user_or_id);
        $this->userid = $userid;
        $this->itemid = $kica_item->id;
        $this->courseid = $kica_item->courseid;
        $this->kica_grade_item = $kica_item;

        if ($kica_item->areaid) {
            if ($area = NED::get_grading_area($kica_item->areaid)) {
                $this->area = true;
            }
        }

        $kica = $kica ?: NED::get_kica($this->courseid);
        $this->pullfromgradebook = $kica->pullfromgradebook ?? null;

        $gradebook = $grade_grade ?? NED::get_grade_grade($kica_item->cm, $userid);
        $this->excluded = $gradebook->excluded ?? 0;

        if ($this->pullfromgradebook) {
            if ($gradebook) {
                $this->finalgradebook = $gradebook->finalgrade;
            } else {
                $this->finalgradebook = null;
            }
        }

        $grade = $grade_recorde ?? static::get_raw_records_by_itemids_userids($this->itemid, $this->userid, true);
        if ($this->excluded){
            if (!empty($grade->id)){
                static::delete_records(['id' => $grade->id]);
            }
        } else {
            if ($grade){
                $this->import($grade);
            }

            if (is_null($ngc_record)){
                $ngc_record = NED::$ned_grade_controller::get_records_by_params_done($kica_item->cm->id, $userid);
                $ngc_record = reset($ngc_record);
            }
            $this->_ngc_record = $ngc_record;

            if (empty($this->id) && $this->has_ngc_zero(false)){
                $this->set_null();
                $this->id = $this->update_record();
            }
        }

        $this->_update_cache();
    }

    /**
     * Alias for getting empty object from the constructor
     * @constructor
     *
     * WARNING: you can get errors, if try use this object in raw way
     *
     * @return static
     */
    static public function new_empty_object(){
        return new static(null, null, true);
    }

    /**
     * Save/update data of kica_grade in static cache
     */
    protected function _update_cache(){
        if ($this->id){
            static::$_data[$this->id] = $this;
        }

        static::$_data_userid_itemid[$this->userid][$this->kica_grade_item->id] = $this;
    }

    /**
     * Get original finalgrade for kica grade
     *
     * @param bool      $as_string - (optional) if true, return string result
     * @param bool|int  $round - (optional) if true, round result, if number - round with such precision
     *                  NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return float|string|null
     */
    public function get_original_finalgrade($as_string=false, $round=false){
        if ($this->is_grade_hidden()){
            $val = null;
        } elseif ($this->pullfromgradebook){
            $val = $this->finalgradebook;
        } else {
            $val = $this->finalgrade;
        }

        return NED::grade_val($val, $as_string, $round);
    }

    /**
     * Get finalgrade for kica grade
     *
     * @param bool      $as_string - (optional) if true, return string result
     * @param bool|int  $round - (optional) if true, round result, if number - round with such precision
     *                  NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return float|string|null
     */
    public function get_finalgrade($as_string=false, $round=false){
        if ($this->is_grade_hidden()){
            $val = null;
        } elseif ($this->has_ngc_zero()){
            $val = 0;
        } else {
            $val = $this->get_original_finalgrade(false);
            if (!is_null($val) && $this->has_deduction()){
                $val = $val*$this->get_deduction_ratio();
            }
        }

        return NED::grade_val($val, $as_string, $round);
    }

    /**
     * Get format finalgrade as html text
     *
     * @param bool|int              $round - (optional) if true, round result, if number - round with such precision
     *                              NOTE: 0 (zero) $round is interpreted as TRUE
     * @param string|\moodle_url    $url - (optional) if set, return html link with this url
     * @param string                $maskicon - (optional) if set, rewrite grade result
     *
     * @return string - html text
     */
    public function get_format_finalgrade($round=2, $url=null, $maskicon=null){
        $res = [];
        $prefix = '';
        if ($this->has_ngc_zero()){
            $prefix = NED::get_ned_grade_icon_by_ngc($this->_ngc_record);
            $res[] = $this->get_format_max_section(NED::FINALGRADE, $round);
        } else {
            $max = $this->kica_grade_item->get_grademax(true, $round);
            if ($maskicon){
                $res[] = $maskicon;
            } elseif ($max){
                $f_grade = $this->get_finalgrade(true, $round);
                if ($this->has_deduction()){
                    $orig_grade = $this->get_original_finalgrade(true, $round);
                    if ($orig_grade != '' && $orig_grade != $f_grade){
                        $res[] = NED::tag('del', $orig_grade, 'dimmed_text');
                        $res[] = NED::HTML_SPACE;
                    }
                }
                if (!is_null($this->get_finalgrade())){
                    $res[] = NED::span($f_grade, 'kica-grade');
                }
            } else {
                $res[] = '-';
            }

            if ($max){
                $res[] = $this->get_format_max_section(NED::FINALGRADE, $round, $max);
            }
        }

        $postfix = $this->get_finalgrade_postfix();
        if (!empty($postfix)){
            $postfix = NED::HTML_SPACE.$postfix;
        }

        $res = join('', $res);
        $class = 'kica-grade kica-'.NED::FINALGRADE;
        if ($url){
            $result = NED::link($url, $res, '', ['target' => '_blank']);
        } else {
            $result = NED::span($res);
        }

        return NED::span($prefix . $result . $postfix, $class);
    }

    /**
     * Get kica grade percentage
     *
     * @param bool          $as_string - (optional) if true, return string result, with % symbol
     *                      NOTE: if true, $round default value will be 2
     * @param bool|int      $round - (optional) if true, round result, if number - round with such precision
     *                      NOTE: 0 (zero) $round is interpreted as TRUE
     * @param bool          $null_if_ungraded - (optional) if true, return null (or '-' if string) when it's ungraded, 0 (zero) otherwise
     * @param string|mixed  $def_string_value - (optional) default string to return when value is NULL
     *
     * @return float|string|null
     */
    public function get_grade_percentage($as_string=false, $round=false, $null_if_ungraded=false, $def_string_value='-'){
        $kicagrade = 0;
        $kica_max = 0;
        if ($this->is_graded() && $this->kica_grade_item && $kica = NED::get_kica($this->courseid)){
            foreach (NED::KICA_KEYS as $item) {
                if (!empty($kica->$item) && $this->kica_grade_item->has_section($item)) {
                    $max = $this->kica_grade_item->get_section($item);
                    $kicagrade += ($this->$item / $max) * $kica->$item;
                    $kica_max += $kica->$item;
                }
            }
        }

        if ($kica_max){
            if ($this->has_ngc_zero()){
                $kicagrade = 0;
            } else {
                $kicagrade = $kicagrade * 100 / $kica_max;
            }

            if ($this->has_deduction()){
                $kicagrade = $kicagrade * $this->get_deduction_ratio();
            }
        } else {
            $kicagrade = $null_if_ungraded ? null : 0;
        }

        if ($as_string){
            if (is_null($kicagrade)){
                return $def_string_value;
            } else {
                $round = ($round === false) ? 2 : $round;
                return NED::grade_val($kicagrade, true, $round ?: 2).'%';
            }
        } else {
            return NED::grade_val($kicagrade, false, $round);
        }
    }

    /**
     * Get kica grade percentage
     *
     * @param bool|null     $kica_calculation_weighted - does kica course calculation is weighted, if null - load value by itself
     * @param bool          $as_string - (optional) if true, return string result, with % symbol
     *                      NOTE: if true, $round default value will be 2 if $kica_calculation_weighted is true
     * @param bool|int      $round - (optional) if true, round result, if number - round with such precision
     *                      NOTE: 0 (zero) $round is interpreted as TRUE
     * @param bool          $null_if_ungraded - (optional) if true, return null (or '-' if string) when it's ungraded, 0 (zero) otherwise
     *
     * @return float|string|null
     */
    public function get_grade_percentage_or_finalgrade($kica_calculation_weighted=null, $as_string=false, $round=false, $null_if_ungraded=false){
        if (is_null($kica_calculation_weighted)){
            $kica = NED::get_kica($this->courseid);
            $kica_calculation_weighted = ($kica == static::CALCULATION_WEIGHTED);
        }

        if ($kica_calculation_weighted){
            return $this->get_grade_percentage($as_string, $round, $null_if_ungraded);
        } else {
            return $this->get_finalgrade($as_string, $round);
        }
    }

    /**
     * Get grade value on of the KICA sections or "finalgrade"
     *
     * @param string    $name - one of the KICA sections or "finalgrade"
     * @param bool      $as_string - (optional) if true, return string result
     * @param bool|int  $round - (optional) if true, round result, if number - round with such precision
     *                  NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return float|string|null
     */
    public function get_section($name, $as_string=false, $round=false){
        $val = null;
        if (!$this->is_grade_hidden()){
            if ($name == NED::FINALGRADE){
                return $this->get_finalgrade($as_string);
            } elseif (in_array($name, NED::KICA_KEYS)){
                $val = $this->$name;
            }
        }

        return NED::grade_val($val, $as_string, $round);
    }

    /**
     * Get format KICA section or finalgrade as html text
     *
     * @param string                $name - one of the KICA sections or "finalgrade"
     * @param bool|int              $round - (optional) if true, round result, if number - round with such precision
     *                              NOTE: 0 (zero) $round is interpreted as TRUE
     * @param string|\moodle_url    $url - (optional) if set, return html link with this url
     * @param string                $maskicon - (optional) if set, rewrite grade result
     *
     * @return string - html text
     */
    public function get_format_section($name, $round=2, $url=null, $maskicon=null){
        if ($name == NED::FINALGRADE){
            return $this->get_format_finalgrade($round, $url, $maskicon);
        } elseif (in_array($name, NED::KICA_KEYS)){
            $val = $this->$name;
            $max = $this->kica_grade_item->get_section($name, true, $round);
        } else {
            $val = null;
            $max = null;
        }

        $res = [];
        if ($maskicon){
            $res[] = $maskicon;
        } elseif ($max){
            if (!is_null($val)){
                $res[] = $this->get_section($name, true, $round);
            }
        } else {
            $res[] = '-';
        }

        if ($max){
            $res[] = ' / '.$max;
        }

        $res = join('', $res);
        $class = 'kica-grade kica-'.$name;
        if ($url){
            return NED::link($url, $res, $class, ['target' => '_blank']);
        } else {
            return NED::span($res, $class);
        }
    }

    /**
     * Get format max KICA section or finalgrade as html text
     *
     * @param string            $name - one of the KICA sections or "finalgrade", can be not provided, if you provide $max
     * @param bool|int          $round - (optional) if true, round result, if number - round with such precision
     *                              NOTE: 0 (zero) $round is interpreted as TRUE
     * @param int|string|null   $max  - max value, if not provided - get it by $name section
     * @param string            $delimiter - string delimiter, '/' by default
     *
     * @return string - html text
     */
    public function get_format_max_section($name, $round=2, $max=null, $delimiter='/'){
        if (empty($delimiter)){
            return '';
        }

        $max = $max ?? $this->kica_grade_item->get_section($name, true, $round);
        if (empty($max)){
            return '';
        }

        return NED::HTML_SPACE.$delimiter.NED::HTML_SPACE.$max;
    }

    /**
     * Check, has this grade such flag
     *
     * @param string $name - one of the KICA item sections
     *
     * @return bool
     */
    public function has_grade_flag($name) {
        $flag = $this->flag;
        if (!$flag || !$name) {
            return false;
        }

        $range_flag_value = kica_item::KICA_FLAGS[$name] ?? null;
        if (!empty($range_flag_value)) {
            if ($flag & $range_flag_value) {
                return true;
            }
        }

        return false;
    }

    /**
     * @return bool
     */
    public function grade_mismatch(){
        if (!$this->is_gradable()) {
            return false;
        }

        $totalgrade = grade_floatval($this->knowledge + $this->inquiry + $this->communication + $this->application);

        if ($this->area) {
            if (grade_floatval($this->finalgradebook) != $totalgrade) {
                $totalgrade = round($totalgrade, 0);
            }
        }

        $finalgradebook = grade_floatval($this->finalgradebook);

        if (is_null($finalgradebook) && is_null($totalgrade)) {
            return false;
        } else if (is_null($finalgradebook) || is_null($totalgrade)) {
            return true;
        }

        return abs($finalgradebook - $totalgrade) > self::GRADE_TOLERANCE;
    }

    /**
     * Return, that global $USER can't view grade data
     *
     * @return bool - false, if $USER can view grade, otherwise return true
     */
    public function is_grade_hidden(){
        if (is_null($this->_is_grade_hidden)){
            $this->_is_grade_hidden = NED::grade_is_hidden_now_before_midn($this->kica_grade_item->cm, $this->userid) ?: false;
        }

        return $this->_is_grade_hidden;
    }

    /**
     * It checks whether activity is gradeble or not. If he gets finalgrade from gradebook and gradebook grade is null,
     * it will not be gradable
     *
     * @return bool
     */
    public function is_gradable(){
        if ($this->excluded)                return false;
        if ($this->area)                    return false;
        if ($this->is_grade_hidden())       return false;
        if ($this->has_ngc_zero())          return false;
        if (!$this->pullfromgradebook)      return true;
        if ($this->is_graded())             return true;
        if (is_null($this->finalgradebook)) return false;

        return true;
    }

    /**
     * @return bool
     */
    public function is_graded(){
        if ($this->is_grade_hidden())                   return false;
        if ($this->has_ngc_zero())                      return true;
        if (is_null($this->get_original_finalgrade()))  return false;

        foreach (NED::KICA_KEYS as $kica_key){
            if (!is_null($this->$kica_key)){
                return true;
            }
        }

        return false;
    }

    /**
     * Base check for NGC records
     *
     * @param int  $type - type of NGC record
     * @param bool $get_obsoleted - if true, check obsoleted record, "done" otherwise
     *
     * @return bool
     */
    protected function _has_check_ngc_record($type, $get_obsoleted=false){
        if ($this->_ngc_record && $this->_ngc_record->grade_type == $type){
            if ($get_obsoleted){
                return $this->_ngc_record->status == NED::$ned_grade_controller::ST_OBSOLETED;
            } else {
                return $this->_ngc_record->status == NED::$ned_grade_controller::ST_DONE;
            }
        }

        return false;
    }

    /**
     * @param bool $get_obsoleted - if true, check obsoleted record, "done" otherwise
     *
     * @return bool
     */
    public function has_ngc_zero($get_obsoleted=false){
        if ($this->is_grade_hidden()) return false;
        return $this->_has_check_ngc_record(NED::$ned_grade_controller::GT_AWARD_ZERO, $get_obsoleted);
    }

    /**
     * @param bool $get_obsoleted - if true, check obsoleted record, "done" otherwise
     *
     * @return bool
     */
    public function has_deduction($get_obsoleted=false){
        if ($this->is_grade_hidden()) return false;
        if (empty($this->_ngc_record->grade_change)) return false;
        return $this->_has_check_ngc_record(NED::$ned_grade_controller::GT_DEDUCTION, $get_obsoleted);
    }

    /**
     * Return deduction ratio for grade
     *
     * @return float - number between 0 and 1
     * @noinspection PhpConditionAlreadyCheckedInspection
     */
    public function get_deduction_ratio(){
        if ($this->has_deduction()){
            $deduction = $this->_ngc_record->grade_change ?? 0;
            if ($deduction >= 100){
                return 0;
            } elseif ($deduction > 0 && $deduction < 100){
                return (100 - $deduction) / 100;
            }
        }

        return 1;
    }

    /**
     * Get icon deduction postfix for the current grade
     *
     * @return string - html (or empty)
     */
    public function get_finalgrade_postfix(){
        if ($this->has_deduction(false) || $this->has_deduction(true)){
            return NED::get_grade_deduction_postfix_by_ngc_or_reason($this->_ngc_record);
        } elseif ($this->has_ngc_zero(true)){
            return NED::get_ned_grade_icon_by_ngc($this->_ngc_record);
        }

        return '';
    }

    /**
     * Import $record to this object
     *
     * @param $record
     */
    public function import($record){
        foreach ($record as $key => $val){
            if (property_exists($this, $key)){
                $this->$key = $val;
            }
        }
    }

    /**
     * Export object for DB
     * @return \stdClass
     */
    public function to_object(){
        global $DB;
        $t_cols = $DB->get_columns(static::TABLE);
        $obj = new \stdClass();
        foreach ($t_cols as $col => $col_info){
            $obj->$col = $this->$col ?? null;
        }
        return $obj;
    }

    /**
     * Set zero grade
     *
     * @param bool $set_null - if true, set null, 0 (zero) otherwise
     */
    public function set_zero($set_null=false){
        $grade = $set_null ? null : 0;
        foreach (NED::KICA_KEYS as $key){
            if (($this->kica_grade_item->$key ?? 0) > 0){
                $this->$key = $grade;
            }
        }
        $this->finalgrade = $grade;
    }

    /**
     * Set null grade
     * Alias for @see kica_grade::set_zero()
     */
    public function set_null(){
        $this->set_zero(true);
    }

    /**
     * Update row in the DB table
     *
     * @return int|bool
     */
    public function update_record(){
        return static::save_record($this->to_object());
    }

    /**
     * Delete row in the DB table
     *
     * @return bool
     */
    public function delete_record(){
        if (!empty($this->id)){
            if (static::delete_records(['id' => $this->id])){
                $this->id = null;
                return true;
            }
        }

        return false;
    }

    /**
     * @param $itemid
     * @param $flag
     * @throws \dml_exception
     */
    public static function set_flag($itemid, $flag) {
        NED::db()->execute("UPDATE {".static::TABLE."} SET flag = flag | ? WHERE itemid = ?", [$flag, $itemid]);
    }

    /**
     * Save or update record in/to DB
     *
     * @param object|array $record
     *
     * @return bool|int
     */
    public static function save_record($record){
        $record = (object)$record;
        $record->usermodified = NED::get_userid_or_global();
        if ($record->id ?? false){
            $record->timemodified = time();

            return NED::db()->update_record(static::TABLE, $record);
        } else {
            $check_keys = ['courseid', 'userid', 'itemid'];
            foreach ($check_keys as $key){
                if (!($record->$key ?? false)){
                    NED::print_error("$key can't be empty");
                }
            }

            $record->timecreated = time();
            $record->timemodified = $record->timecreated;

            return NED::db()->insert_record(static::TABLE, $record);
        }
    }

    /**
     * Delete records in the DB table
     *
     * @param array $params
     *
     * @return bool
     */
    public static function delete_records($params=[]){
        if (empty($params)){
            return false;
        }

        return NED::db()->delete_records(static::TABLE, $params);
    }

    /**
     * Return raw record(s) from the db
     *
     * @param array $params
     * @param bool  $get_one_record - if true (false by default), return only first record (or null)
     *
     * @return array|object[]|object|null
     */
    static public function get_raw_records($params=[], $get_one_record=false){
        $records = NED::db()->get_records(static::TABLE, $params);
        if ($get_one_record){
            return empty($records) ? null : reset($records);
        }

        return $records;
    }

    /**
     * Return raw record(s) from the db by kica item ids and user ids
     *  If $by_id true - result array will be by ids
     *  Otherwise result array will be by itemids and userids
     * You can provide only $itemids or only $userids - but you need provide at least something
     *
     * @param numeric|array $itemids - id(s) of kica items; if empty - load for all items
     * @param numeric|array $userids - user id(s); if empty - load for all users
     * @param bool  $get_one_record  - if true (false by default), return only first record (or null)
     * @param bool  $by_id           - if true (false by default), return array by id (as keys), otherwise by itemid and userid
     *
     * @return array|object[]|object|null object, or [id => object], or [itemid => [userid => object]]
     */
    static public function get_raw_records_by_itemids_userids($itemids=[], $userids=[], $get_one_record=false, $by_id=false){
        if (empty($itemids) && empty($userids)){
            // do not return whole table
            return [];
        }

        $itemids = NED::val2arr($itemids);
        $userids = NED::val2arr($userids);

        [$where, $params] = NED::sql_get_in_or_equal_options(['itemid' => $itemids, 'userid' => $userids]);
        $records = NED::db()->get_records_select(static::TABLE, $where, $params);

        if ($get_one_record){
            return empty($records) ? null : reset($records);
        } elseif ($by_id){
            return $records;
        }

        $res = [];
        foreach ($records as $record){
            $res[$record->itemid][$record->userid] = $record;
        }

        return $res;
    }

    /**
     * Get kica grade by id
     * @constructor
     *
     * @param numeric $id
     *
     * @return static|null
     */
    public static function get_by_id($id){
        if (empty($id)){
            return null;
        }

        if (!isset(static::$_data[$id])){
            $record = static::get_raw_records(['id' => $id], true);
            if ($record){
                new kica_grade($record->userid, $record->itemid);
            }
        }

        return static::$_data[$id] ?? null;
    }

    /**
     * Get kica grade by userid and itemid
     * @constructor
     *
     * @param numeric|object                    $user_or_id
     * @param kica_item|numeric|object|array    $item_or_id
     *
     * @return static|null
     */
    public static function get_by_userid_itemid($user_or_id, $item_or_id){
        if (empty($user_or_id) || empty($item_or_id)){
            return null;
        }

        $userid = NED::get_id($user_or_id);
        $itemid = NED::get_id($item_or_id);
        if (!isset(static::$_data_userid_itemid[$userid][$itemid])){
            new kica_grade($userid, $item_or_id);
        }

        return static::$_data_userid_itemid[$userid][$itemid] ?? null;
    }

    /**
     * Get kica grade by userid and cmid
     * @constructor
     *
     * @param numeric|object            $user_or_id
     * @param \cm_info|object|numeric   $cm_or_id
     *
     * @return static|null
     */
    public static function get_by_userid_cmid($user_or_id, $cm_or_id){
        if (empty($user_or_id) || empty($cm_or_id)){
            return null;
        }

        $userid = NED::get_id($user_or_id);
        $kica_item = NED::ki_get_by_cm($cm_or_id);
        if (empty($kica_item->id)){
            return null;
        } elseif (!isset(static::$_data_userid_itemid[$userid][$kica_item->id])){
            new kica_grade($userid, $kica_item);
        }

        return static::$_data_userid_itemid[$userid][$kica_item->id] ?? null;
    }

    /**
     * Get kica grade by userid and itemid
     * @constructor
     *
     * @param numeric|object    $course_or_id
     * @param array|int[]|int   $userids
     * @param bool              $by_userid_itemid - (optional) if true, result array will be by userid and itemid
     * @param bool              $only_load - (optional) if true, return empty array, if there is no need to load more users grades
     *
     * @return array|object[][]|static[][] - [itemid => [userid => kica_grade]] or [userid => [itemid => kica_grade]]
     */
    public static function get_grades_by_course($course_or_id, $userids, $by_userid_itemid=false, $only_load=false){
        $courseid = NED::get_id($course_or_id);
        $userids = NED::val2arr($userids);
        $kica_items = kica_item::get_all_items_by_course($courseid);
        if (empty($userids) || empty($kica_items)){
            return [];
        }

        $users_to_load = [];
        foreach ($userids as $userid){
            if (empty(static::$_data_course_userid[$courseid][$userid])){
                $users_to_load[] = $userid;
            }
        }

        $kica = NED::get_kica($courseid);
        if (empty($users_to_load) && $only_load){
            return [];
        }

        if (!empty($users_to_load)){
            $cmids = [];
            $ki_ids = [];
            $gi_ids = [];
            foreach ($kica_items as $kica_item){
                $cmids[] = $kica_item->cm->id;
                $ki_ids[] = $kica_item->id;
                $gi_ids[] = $kica_item->itemid;
            }
            $kg_records = static::get_raw_records_by_itemids_userids($ki_ids, $users_to_load);
            $ngc_records = NED::$ned_grade_controller::get_records_by_cmids_userids($cmids, $users_to_load);
            // load ggs in cache
            NED::get_grade_grade_by_itemids_userids($gi_ids, $users_to_load);
        }

        $res = [];
        foreach ($userids as $userid){
            foreach ($kica_items as $ki_id => $kica_item){
                if (empty($ki_id) || empty($kica_item->cm->id)) continue;

                if (empty(static::$_data_userid_itemid[$userid][$ki_id])){
                    $kg = static::new_empty_object();
                    $gg = NED::get_grade_grade_cached($kica_item->cm, $userid) ?: false;
                    $kg_record = $kg_records[$ki_id][$userid] ?? false;
                    $ngc_record = $ngc_records[$kica_item->cm->id][$userid] ?? false;
                    $kg->update_from_db($userid, $kica_item, $kica, $gg, $kg_record, $ngc_record);
                } else {
                    $kg = static::$_data_userid_itemid[$userid][$ki_id];
                }

                if ($by_userid_itemid){
                    $res[$userid][$ki_id] = $kg;
                } else {
                    $res[$ki_id][$userid] = $kg;
                }
            }

            static::$_data_course_userid[$courseid][$userid] = true;
        }

        return $res;
    }

    /**
     * Get average grade by KICA section, or array with averages result
     *
     * @param numeric|object        $course_or_id
     * @param numeric|object        $user_or_id
     * @param null|string           $section - KICA section or 'finalgrade', if null|false - return all averages in array
     * @param null|numeric|string   $kicagroup - KICA group, if null|false - return all course kica groups in array
     * @param bool|int              $round - int precision or false, if not need it; NOTE: 0 (zero) $round is interpreted as TRUE
     * @param array|static[]|null   $kica_grades - kica grades to average calculate from, if null - load it
     * @param null|string           $modaverage - Mod base average ie assign, quiz
     *
     * @return array [kica_group => [key => average(or max)]],
     *                  where kica_group is int group or string "max" with max list
     *                  and keys - one of the KICAF keys (KICA keys + "finalgrade")
     */
    public static function get_kica_average($course_or_id, $user_or_id, $section=null, $kicagroup=null, $round=false, $kica_grades=null, $modaverage=null){
        $numerator = [];
        $denominator = [];
        $courseid = NED::get_id($course_or_id);
        $userid = NED::get_id($user_or_id);
        $kica = NED::get_kica($courseid);
        if (!$kica){
            return [];
        }
        if ($section && (!in_array($section, NED::KICA_KEYS) && $section != NED::FINALGRADE)){
            return [];
        }

        $kicagrouppending = false;

        $calc_natural = ($kica->calculation == kica_grade::CALCULATION_NATURAL);
        $kicagroups = $kicagroup ? [$kicagroup] : NED::get_kica_groups($course_or_id, true);
        $k_keys = $section ? [$section] : NED::KICAF_KEYS;
        foreach ($kicagroups as $kgroup){
            $numerator[$kgroup] = [];
            $denominator[$kgroup] = [];
            foreach ($k_keys as $k_key){
                $numerator[$kgroup][$k_key] = [];
                $denominator[$kgroup][$k_key] = [];
            }
        }

        if (is_null($kica_grades)){
            $kica_grades = NED::kg_get_grades_by_course($courseid, $userid, true)[$userid] ?? [];
        }

        /** @noinspection PhpArrayIndexImmediatelyRewrittenInspection */
        $modstat = [
            'assigncompleted' => 0,
            'assignavailable' => 0,
            'testcompleted' => 0,
            'testavailable' => 0,
            'test' => [],
            'assign' => [],
        ];

        foreach ($kica_grades as $kica_grade){
            if (!empty($modaverage)) {
                if ($modaverage === 'assign') {
                    if ($kica_grade->kica_grade_item->cm->modname !== 'assign') {
                        continue;
                    }
                    if (!NED::cm_is_test_or_assignment($kica_grade->kica_grade_item->cm)) {
                        continue;
                    }
                    $modstat['assignavailable']++;
                } else if ($modaverage === 'quiz') {
                    if ($kica_grade->kica_grade_item->cm->modname !== 'quiz') {
                        continue;
                    }
                    if (!NED::cm_is_test_or_assignment($kica_grade->kica_grade_item->cm)) {
                        continue;
                    }
                    $modstat['testavailable']++;
                }
            }

            if (empty($kica_grade->kica_grade_item->id)) continue;
            $ki = $kica_grade->kica_grade_item;
            if (!in_array($ki->kicagroup, $kicagroups)) continue;
            if (!$ki->is_visible($userid)) continue;
            if ($kica_grade->excluded) continue;
            if (empty($kica_grade->id) || !$kica_grade->is_graded() || $kica_grade->grade_mismatch()) {
                if ($ki->kicagroup == 30) {
                    $kicagrouppending = true;
                }
                continue;
            }

            foreach ($k_keys as $k_key){
                if (!$ki->has_section($k_key)) continue;
                if (!empty($modaverage)) {
                    if ($modaverage === 'assign') {
                        $modstat['assign'][$kica_grade->id] = 1;
                    } else if ($modaverage === 'quiz') {
                        $modstat['test'][$kica_grade->id] = 1;
                    }
                }
                $numerator[$ki->kicagroup][$k_key][] = $kica_grade->get_section($k_key);
                $denominator[$ki->kicagroup][$k_key][] = $ki->get_section($k_key);
            }
        }

        if ($kicagrouppending && isset($kgroup)){
            foreach ($k_keys as $k_key){
                $numerator[$kgroup][$k_key] = [];
                $denominator[$kgroup][$k_key] = [];
            }
        }

        $modstat['assigncompleted'] = count($modstat['assign']);
        unset($modstat['assign']);
        $modstat['testcompleted'] = count($modstat['test']);
        unset($modstat['test']);

        $average = [];
        $max = [];
        foreach ($kicagroups as $kgroup){
            foreach ($k_keys as $k_key){
                if (empty($denominator[$kgroup][$k_key])){
                    $average[$kgroup][$k_key] = null;
                    $max[$kgroup][$k_key] = null;

                    continue;
                }

                $sum_denominator = array_sum($denominator[$kgroup][$k_key]);
                if ($calc_natural && $k_key != NED::FINALGRADE){
                    $avg = array_sum($numerator[$kgroup][$k_key]);
                } else {
                    $multi = $kica->$k_key ?? 100;
                    $avg = $sum_denominator ? ((array_sum($numerator[$kgroup][$k_key]) / $sum_denominator) * $multi) : 0;
                }
                $average[$kgroup][$k_key] = NED::grade_val($avg, false, $round);
                $max[$kgroup][$k_key] = NED::grade_val($sum_denominator, false, $round);
            }
        }

        $average['max'] = $max;
        if (!empty($modaverage)) {
            $average['stat'] = $modstat;
        }
        return $average;
    }

    /**
     * Get average grade by KICA section, form of result related of params
     * Alias @see kica_grade::get_kica_average() - you also can get raw data by this function
     *
     * @param numeric|object        $course_or_id
     * @param numeric|object        $user_or_id
     * @param null|string           $section - KICA section or 'finalgrade', if null|false - return all averages in array
     * @param null|numeric|string   $kicagroup
     * @param bool                  $get_max - if true, also return max values
     * @param bool|int              $round - int precision or false, if not need it; NOTE: 0 (zero) $round is interpreted as TRUE
     * @param array|static[]|null   $kica_grades - kica grades to average calculate from, if null - load it
     *
     * @return array|float|null :
     *          • list($res, $max) - if $get_max is true, otherwise only $res, where $res and $max are:
     *              • float|null - if $section and $kicagroup are provided;
     *              • array(key => average(or max)) - if $kicagroup is provided and $section is not;
     *              • array(kica_group => average(or max)) - if $section is provided and $kicagroup is not;
     *              • array(kica_group => [key => average(or max)]) - if $section and $kicagroup are not provided;
     *          Where key - one of the KICAF keys (KICA keys + "finalgrade")
     */
    public static function get_kica_average_by_section($course_or_id, $user_or_id, $section=null, $kicagroup=null,
        $get_max=false, $round=false, $kica_grades=null){
        $res = static::get_kica_average($course_or_id, $user_or_id, $section, $kicagroup, $round, $kica_grades);
        $max = $res['max'] ?? [];
        unset($res['max']);

        if ($kicagroup && $section){
            $res = $res[$kicagroup][$section] ?? null;
            $max = $max[$kicagroup][$section] ?? null;
        } elseif ($kicagroup){
            $res = $res[$kicagroup] ?? [];
            $max = $max[$kicagroup] ?? [];
        } elseif ($section){
            $kicagroups = NED::get_kica_groups($course_or_id, true);
            foreach ($kicagroups as $kgroup){
                $res[$kgroup] = $res[$kgroup][$section] ?? null;
                $max[$kgroup] = $max[$kgroup][$section] ?? null;
            }
        }

        if ($get_max){
            return [$res, $max];
        } else {
            return $res;
        }
    }

    /**
     * Get course average grade by KICA section, or array with averages result
     *
     * @param numeric|object        $course_or_id
     * @param numeric|object        $user_or_id
     * @param null|string           $section - KICA section or 'finalgrade', if null|false - return all averages in array
     * @param bool|int              $round - int precision or false, if not need it; NOTE: 0 (zero) $round is interpreted as TRUE
     * @param array|float[][]|null  $average_data - average kica grades [kica_group => [key => average(or max)]],
     *                                          you can get them from @see kica_grade::get_kica_average()
     * @param array|static[]|null   $kica_grades  - kica grades to average calculate from,
     *                                          make sense to send them only if $average_data is null, if $kica_grades null - load it
     *
     * @return array|float|null - float|null value for the provided $section, otherwise array [key => value],
     *                              where keys - one of the KICAF keys (KICA keys + "finalgrade")
     */
    public static function get_course_average($course_or_id, $user_or_id, $section=null, $round=false, $average_data=null, $kica_grades=null){
        if ($section && (!in_array($section, NED::KICA_KEYS) && $section != NED::FINALGRADE)){
            return [];
        }

        $k_keys = $section ? [$section] : NED::KICAF_KEYS;
        $calc_natural = NED::is_kica_calculation_natural($course_or_id);
        $kica_groups = NED::get_kica_groups($course_or_id, true);
        if (empty($kica_groups)){
            return $section ? null : array_fill_keys($k_keys, null);
        }

        $sum_groups = array_sum($kica_groups);
        if (!$sum_groups && (!$calc_natural || ($section && $section == NED::FINALGRADE))){
            return $section ? null : array_fill_keys($k_keys, null);
        }

        if (is_null($average_data)){
            $average_data = static::get_kica_average($course_or_id, $user_or_id, $section, null, false, $kica_grades);
        }

        $res = [];
        foreach ($k_keys as $k_key){
            $numerator = 0;
            $max = 0;
            $isnull = true;
            foreach ($kica_groups as $k_group){
                $group_average_grade = $average_data[$k_group][$k_key] ?? null;
                if (is_null($group_average_grade)) continue;

                $isnull = false;
                if ($calc_natural && $k_key != NED::FINALGRADE){
                    $max += $average_data['max'][$k_group][$k_key] ?? null;
                    $numerator += $group_average_grade;
                } else {
                    $max += $k_group;
                    $numerator += $k_group * $group_average_grade;
                }
            }

            if ($isnull || !$max){
                $res[$k_key] = null;
            } else {
                if ($calc_natural && $k_key != NED::FINALGRADE){
                    $val = 100 * $numerator/$max;
                } else {
                    $val = $numerator/$max;
                }
                $res[$k_key] = NED::grade_val($val, false, $round);
            }
        }

        return $section ? $res[$section] : $res;
    }

    /**
     * Get course average grade by KICA section, or array with averages result
     *
     * @param numeric|object  $course_or_id
     * @param array|float[][] $average_data - average kica grades [kica_group => [key => average(or max)]], from @see kica_grade::get_kica_average()
     * @param array           $calculated_average  - calculated grades from @see kica_grade::get_course_average()
     * @param bool|int        $round - int precision or false, if not need it; NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return array|string[][] - data for grade-calculation template
     */
    public static function get_course_average_calculation($course_or_id, $average_data, $calculated_average, $round=2){
        $kica = NED::get_kica($course_or_id);
        $calc_natural = ($kica->calculation == kica_grade::CALCULATION_NATURAL);
        $kica_groups = NED::get_kica_groups($course_or_id);
        $res = [];
        if (empty($kica_groups) || empty($average_data) || empty($calculated_average)){
            return [];
        }
        $skipcoursegarde = false;
        $course_calc = [];
        foreach ($kica_groups as $group_value => $group_title){
            // TODO change calculation view
            //if (is_null($average_data[$group_value][NED::FINALGRADE] ?? null)) continue;

            $group_str = ['title' => $group_title, 'items' => []];
            if (!$calc_natural){
                foreach (NED::KICA_KEYS as $kica_key){
                    $val = NED::grade_val($average_data[$group_value][$kica_key], true, $round, '-');
                    $group_str['items'][] = NED::str($kica_key).': '.$val;
                }
            }

            if ($group_value == 30 && is_null($average_data[$group_value][NED::FINALGRADE])) {
                $skipcoursegarde = true;
                $val = get_string('pending', 'local_kica');
            } else {
                $val = NED::grade_val($average_data[$group_value][NED::FINALGRADE], true, $round, '0');
            }
            $group_str['items'][] = NED::str('kicatotal').': '.$val;
            $res[] = $group_str;

            $denom_val = $group_value/100;
            $grade_val = ($average_data[$group_value][NED::FINALGRADE] ?? 0) * $denom_val;
            $denom = NED::grade_val($denom_val, true, $round, '0');
            $grade = NED::grade_val($grade_val, true, $round, '0');
            $course_calc[] = "($val x $denom = $grade)";
        }

        if (!$skipcoursegarde) {
            $course_str = ['title' => NED::str('coursegrade'), 'items' => []];
            $grade_val = ($calculated_average[NED::FINALGRADE] ?? 0);
            $grade = NED::grade_val($grade_val, true, $round, '0');
            $course_str['items'][] = join(' + ', $course_calc) . " = $grade ($grade %)";
            $res[] = $course_str;
        }

        return $res;
    }

    /**
     * Clear static caches for this class
     */
    static public function clear_grades_cache(){
        static::$_data = [];
        static::$_data_userid_itemid = [];
        static::$_data_course_userid = [];
    }
}