<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package     local_kica
 * @subpackage  NED
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kica;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/lib/gradelib.php');

use local_kica\shared_lib as NED;

/**
 * KICA Grade Item
 *
 * @package kica_grade_item
 */
class kica_item {
    /**
     * DB Table.
     * @var string $table
     */
    const TABLE = 'local_kica_grade_items';

    /**
     *
     */
    const FLAG_KNOWLEDGE = 1;
    /**
     *
     */
    const FLAG_INQUIRY = 2;
    /**
     *
     */
    const FLAG_COMMUNICATION = 4;
    /**
     *
     */
    const FLAG_APPLICATION = 8;

    const KICA_FLAGS = [
        NED::KICA_K => self::FLAG_KNOWLEDGE,
        NED::KICA_I => self::FLAG_INQUIRY,
        NED::KICA_C => self::FLAG_COMMUNICATION,
        NED::KICA_A => self::FLAG_APPLICATION,
    ];

    /**
     * @var array - id => kica_item
     */
    static protected $_ki_data = [];
    /**
     * @var array - grade_item_id => kica_item_id
     */
    static protected $_gi_ids = [];
    /**
     * @var array - cm_id => kica_item_id
     */
    static protected $_cm_ids = [];
    /**
     * @var array - course_id => [kica_item_ids]
     */
    static protected $_course_kis = [];

    /**
     * Array of required table fields, must start with 'id'.
     * @var array $required_fields
     */
    public $required_fields = array('id', 'itemid', 'knowledge', 'inquiry',
        'communication', 'application', 'kicagroup', 'timecreated', 'timemodified');

    /**
     * Array of optional table fields.
     * @var array $optional_fields
     */
    public $optional_fields = array('courseid', 'itemname', 'itemtype',
        'itemmodule', 'iteminstance', 'grademax', 'grademin', 'areaid');

    /**
     * The ID of this kica_item.
     * @var int $id
     */
    public $id = 0;

    /**
     * The ID of this grade_item.
     * @var int $courseid
     */
    public $itemid;

    /**
     * The course this grade_item belongs to.
     * @var int $courseid
     */
    public $courseid;

    /**
     * The name of this grade_item (pushed by the module).
     * @var string $itemname
     */
    public $itemname;

    /**
     * e.g. 'category', 'course' and 'mod', 'blocks', 'import', etc...
     * @var string $itemtype
     */
    public $itemtype;

    /**
     * The module pushing this grade (e.g. 'forum', 'quiz', 'assignment' etc).
     * @var string $itemmodule
     */
    public $itemmodule;

    /**
     * ID of the item module
     * @var int $iteminstance
     */
    public $iteminstance;

    /**
     * Maximum allowable grade.
     * @var float $grademax
     */
    public $grademax = 100;

    /**
     * Minimum allowable grade.
     * @var float $grademin
     */
    public $grademin = 0;

    /**
     * grade required to pass. (grademin <= gradepass <= grademax)
     * @var float $gradepass
     */
    public $gradepass = 0;

    /**
     * @var float $knowledge
     */
    public $knowledge = 0;

    /**
     * @var float $inquiry
     */
    public $inquiry = 0;

    /**
     * @var float $communication
     */
    public $communication = 0;
    /**
     * @var float $application
     */
    public $application = 0;

    /**
     * @var int $kicagroup
     */
    public $kicagroup = 0;

    /**
     * @var bool $areaid
     */
    public $areaid = 0;

    /**
     * @var bool $incomplete
     */
    public $incomplete = true;

    /**
     * @var null|object course module context instance
     */
    public $context = null;

    /**
     * @var \cm_info|\course_modinfo|null course module object
     */
    public $cm = null;

    /**
     * Constructor. Optionally (and by default) attempts to fetch corresponding row from the database
     * Note: Consider using static methods instead
     * @constructor
     *
     * @param array $params An array with required parameters for this grade object.
     * @param bool  $fetch  Whether to fetch corresponding row from the database or not,
     *                      optional fields might not be defined if false used
     * @param bool  $use_only_gradeitem If true and $params has id, it will be used for load gradeitem, not kica
     * @param bool  $get_empty_object If true, will load/check nothing
     *
     */
    public function __construct($params, $fetch=true, $use_only_gradeitem=false, $get_empty_object=false) {
        if ($get_empty_object){
            return;
        }

        if (!$use_only_gradeitem && isset($params['id'])) {
            $this->update_from_db($params['id']);
        } else {
            $this->update_from_db(null, NED::get_grade_item_by_id_or_params(null, $params));
        }

        if (isset($params['kicagroup'])) {
            $this->kicagroup = $params['kicagroup'];
        }

        $this->_update_cache();
    }

    /**
     * Save/update data of kica_item in static cache
     */
    protected function _update_cache(){
        if (!$this->id){
            return;
        }

        static::$_ki_data[$this->id] = $this;
        static::$_gi_ids[$this->itemid] = $this->id;
        static::$_cm_ids[$this->cm->id ?? 0] = $this->id;
    }

    /**
     * Alias for getting empty object from the constructor
     * @constructor
     *
     * WARNING: you can get errors, if try use this object in raw way
     *
     * @return static
     */
    static public function new_empty_object(){
        return new static(null, null, null, true);
    }

    /**
     * @return bool
     */
    public function update() {
        if (empty($this->id)) {
            debugging('Can not update grade object, no id!');
            return false;
        }

        $this->grademin      = grade_floatval($this->grademin);
        $this->grademax      = grade_floatval($this->grademax);
        $this->knowledge     = grade_floatval($this->knowledge);
        $this->inquiry       = grade_floatval($this->inquiry);
        $this->communication = grade_floatval($this->communication);
        $this->application   = grade_floatval($this->application);

        if ($flag = $this->get_flag()) {
            kica_grade::set_flag($this->id, $flag);
        }

        $data = $this->get_record_data();
        $data->timemodified = time();

        NED::db()->update_record(self::TABLE, $data);
        $this->_update_cache();

        return true;
    }

    /**
     * It flags changing KICA ranges
     *
     * @return int
     */
    public function get_flag() {
        $flag = 0;

        if ($rawdata = static::get_raw_records(['id' => $this->id], true)) {
            if ($this->knowledge != $rawdata->knowledge) {
                $flag |= self::FLAG_KNOWLEDGE;
            }
            if ($this->inquiry != $rawdata->inquiry) {
                $flag |= self::FLAG_INQUIRY;
            }
            if ($this->communication != $rawdata->communication) {
                $flag |= self::FLAG_COMMUNICATION;
            }
            if ($this->application != $rawdata->application) {
                $flag |= self::FLAG_APPLICATION;
            }
        }
        return $flag;
    }

    /**
     * Check, that kica_item has any grade
     *
     * @param array $userids - (optional) list of userid to filter grades
     *
     * @return bool
     */
    public function activity_has_grade($userids=[]) {
        if (empty($this->id)) {
            return false;
        }

        $options = ['itemid' => $this->id];
        if (!empty($userids)){
            $options['userid'] = $userids;
        }
        list($where, $params) = NED::sql_get_in_or_equal_options($options);

        return NED::db()->record_exists_select(kica_grade::TABLE, $where, $params);
    }

    /**
     * Check, has any grade flag from this kica_item and selected user ids
     *
     * @param array $userids
     *
     * @return bool
     */
    public function activity_has_flagged_grade($userids=[]) {
        if (empty($this->id) || empty($userids)) {
            return false;
        }

        $options = [
            'itemid' => $this->id,
            'userid' => $userids
        ];
        list($where, $params) = NED::sql_get_in_or_equal_options($options);
        $where .= ' AND flag > 0';

        return NED::db()->record_exists_select(kica_grade::TABLE, $where, $params);
    }

    /**
     * @return bool
     */
    public function insert() {
        if (empty($this->courseid)) {
            NED::print_module_error('cannotinsertgrade');
        }

        if (!empty($this->id)) {
            debugging("Grade object already exists!");
            return false;
        }

        $data = $this->get_record_data();
        $data->timecreated = $data->timemodified = time();
        $this->id = NED::db()->insert_record(self::TABLE, $data);

        // Set all object properties from real db data.
        $this->update_from_db();

        if (!$this->id) {
            debugging("Could not insert this grade_item in the database!");
            return false;
        }

        return true;
    }

    /**
     * Returns object with fields and values that are defined in database
     *
     * @return \stdClass
     */
    public function get_record_data() {
        $data = new \stdClass();

        foreach ($this as $var=>$value) {
            if (in_array($var, $this->required_fields) || in_array($var, $this->optional_fields)) {
                if (is_object($value) or is_array($value)) {
                    debugging("Incorrect property '$var' found when inserting grade object");
                } else {
                    $data->$var = $value;
                }
            }
        }
        return $data;
    }

    /**
     * Delete all grades and force_regrading of parent category.
     *
     * @return bool success
     */
    public function delete() {
        if (empty($this->id)) {
            debugging('Can not delete grade object, no id!');
            return false;
        }

        if (NED::db()->delete_records(static::TABLE, ['id' => $this->id])){
            NED::db()->delete_records(kica_grade::TABLE, ['itemid' => $this->id]);
            unset(static::$_ki_data[$this->id]);
            unset(static::$_gi_ids[$this->itemid]);
            unset(static::$_cm_ids[$this->cm->id ?? 0]);
            $this->id = null;
            return true;
        } else {
            return false;
        }
    }

    /**
     * Using this object's id field, fetches the matching record in the DB, and looks at
     * each variable in turn. If the DB has different data, the db's data is used to update
     * the object. This is different from the update() function, which acts on the DB record
     * based on the object.
     *
     * All parameters are optional, but you need provide at least something
     *
     * @param numeric|null                  $id                             - you can provide id here for new object
     * @param \grade_item|object|null       $grade_item_or_id               - grade item for current kica item
     * @param \course_modinfo|\cm_info|object|numeric|null  $modinfo_or_id  - course module, we can get grade item from here
     * @param object|null                   $full_kica_item                 - full kica item from the DB table, if provide
     * @param bool                          $ignore_mismatch                - if true, don't show error or debugging text
     *
     * @return bool True if successful
     */
    public function update_from_db($id=null, $grade_item_or_id=null, $modinfo_or_id=null, $full_kica_item=null, $ignore_mismatch=false) {
        $load_kica_record = function($params){
            $kica_i = static::get_raw_records($params, true);
            if ($kica_i){
                self::set_properties($this, $kica_i);
            }
            return $kica_i->id ?? null;
        };

        $fki_id = $full_kica_item->id ?? 0;
        if ($fki_id && ($fki_id == $id || $fki_id == $this->id || (!$id && !$this->id))){
            $id = $this->id = $fki_id;
            self::set_properties($this, $full_kica_item);
        } else {
            if ($id){
                $this->id = (int)$id;
            } else {
                $id = $this->id;
            }

            if ($id){
                $id = $load_kica_record(['id' => $id]);
            }
        }

        $cm = $course = null;
        if (!empty($modinfo_or_id)){
            if ($modinfo_or_id instanceof \course_modinfo){
                $course = $modinfo_or_id;
            } else {
                $cm = NED::get_cm_by_cmorid($modinfo_or_id);
            }
        }

        if ($grade_item_or_id){
            if (is_numeric($grade_item_or_id)){
                $grade_item = NED::get_grade_item_by_id_or_params($grade_item_or_id);
            } else {
                $grade_item = $grade_item_or_id;
            }
        } else {
            if ($this->itemid){
                $grade_item = NED::get_grade_item_by_id_or_params($this->itemid);
            } elseif ($cm){
                $grade_item = NED::get_grade_item($cm);
            } elseif ($course){
                $grade_item = NED::get_course_grade_item($course);
            }
        }

        /** @var \grade_item|null $grade_item */
        if (empty($grade_item)) {
            if (!CLI_SCRIPT && !$ignore_mismatch){
                debugging("The object could not be used as there is no grade item for it");
            }
            return false;
        }

        if (!$id){
            $id = $load_kica_record(['itemid' => $grade_item->id]);
        }

        if (!$cm && !$course){
            $cm = NED::get_cm_by_kica_or_grade_item($grade_item);
            if (!$cm){
                $course = NED::get_fast_modinfo($grade_item->courseid ?? null);
            }

            if (!$cm && !$course){
                if (!$ignore_mismatch){
                    debugging("The object could not be used as there is no such activity");
                }
                return false;
            }
        }

        $this->id = $id;
        $this->cm = $cm;
        $this->itemid = $grade_item->id;
        $this->courseid = $grade_item->courseid;
        $this->itemname = $grade_item->itemname;
        $this->itemtype = $grade_item->itemtype;
        $this->itemmodule = $grade_item->itemmodule;
        $this->iteminstance = $grade_item->iteminstance;
        $this->grademax = $grade_item->grademax;
        $this->grademin = $grade_item->grademin;
        $this->gradepass = $grade_item->gradepass;

        if (isset($this->cm->modname)){
            $this->context = \context_module::instance($cm->id);
        } else {
            $this->context = \context_course::instance($grade_item->courseid);
        }

        $this->incomplete = $this->is_incomplete();
        $this->_update_cache();

        return true;
    }

    /**
     * Given an associated array or object, cycles through each key/variable
     * and assigns the value to the corresponding variable in this object.
     *
     * @param static|\stdClass  $instance The object to set the properties on
     * @param array|object      $params An array of properties to set like $propertyname => $propertyvalue
     */
    static public function set_properties(&$instance, $params) {
        $params = (array) $params;
        foreach ($params as $var => $value) {
            if (in_array($var, $instance->required_fields) || in_array($var, $instance->optional_fields)) {
                $instance->$var = $value;
            }
        }
    }

    /**
     * Get raw section value
     *
     * @param string $name - one of the KICA sections or ("finalgrade", "grademax", "grademin", "gradepass")
     *
     * @return float|string|null
     */
    protected function _section_value($name){
        if (in_array($name, NED::KICA_KEYS)){
            $val = $this->$name;
        } elseif ($name == 'grademax' || $name == 'grademin' || $name == 'gradepass'){
            $val = $this->$name;
        } elseif ($name == NED::FINALGRADE){
            $val = $this->grademax;
        } else {
            $val = null;
        }

        return $val;
    }

    /**
     * Check, has object such not-null KICA item section or ("finalgrade", "grademax", "grademin", "gradepass")
     *
     * @param string    $name - one of the KICA sections or ("finalgrade", "grademax", "grademin", "gradepass")
     *
     * @return float|string|null
     */
    public function has_section($name){
        return !empty((float)$this->_section_value($name));
    }

    /**
     * Get value on of the KICA item sections or ("finalgrade", "grademax", "grademin", "gradepass")
     * It is also max value for kica grades section, where "grademax" is max for finalgrade
     *
     * @param string    $name - one of the KICA sections or "grademax", or "grademin", or "gradepass"
     * @param bool      $as_string - (optional) if true, return string result
     * @param bool|int  $round - (optional) if true, round result, if number - round with such precision
     *                  NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return float|string|null
     */
    public function get_section($name, $as_string=false, $round=false){
        return NED::grade_val($this->_section_value($name), $as_string, $round);
    }

    /**
     * Get value on of the KICA item knowledge
     * Alias for the @see \local_kica\kica_item::get_section()
     *
     * @param bool      $as_string - (optional) if true, return string result
     * @param bool|int  $round - (optional) if true, round result, if number - round with such precision
     *                  NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return float|string|null
     */
    public function get_knowledge($as_string=false, $round=false){
        return $this->get_section(NED::KICA_K, $as_string, $round);
    }

    /**
     * Get value on of the KICA item inquiry
     * Alias for the @see \local_kica\kica_item::get_section()
     *
     * @param bool      $as_string - (optional) if true, return string result
     * @param bool|int  $round - (optional) if true, round result, if number - round with such precision
     *                  NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return float|string|null
     */
    public function get_inquiry($as_string=false, $round=false){
        return $this->get_section(NED::KICA_I, $as_string, $round);
    }

    /**
     * Get value on of the KICA item communication
     * Alias for the @see \local_kica\kica_item::get_section()
     *
     * @param bool      $as_string - (optional) if true, return string result
     * @param bool|int  $round - (optional) if true, round result, if number - round with such precision
     *                  NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return float|string|null
     */
    public function get_communication($as_string=false, $round=false){
        return $this->get_section(NED::KICA_C, $as_string, $round);
    }

    /**
     * Get value on of the KICA item application
     * Alias for the @see \local_kica\kica_item::get_section()
     *
     * @param bool      $as_string - (optional) if true, return string result
     * @param bool|int  $round - (optional) if true, round result, if number - round with such precision
     *                  NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return float|string|null
     */
    public function get_application($as_string=false, $round=false){
        return $this->get_section(NED::KICA_A, $as_string, $round);
    }

    /**
     * Get value on of the grade item grademax
     * Alias for the @see \local_kica\kica_item::get_section()
     *
     * @param bool      $as_string - (optional) if true, return string result
     * @param bool|int  $round - (optional) if true, round result, if number - round with such precision
     *                  NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return float|string|null
     */
    public function get_grademax($as_string=false, $round=false){
        return $this->get_section('grademax', $as_string, $round);
    }

    /**
     * @return bool
     */
    public function is_incomplete(){
        if (empty($this->grademax)) {
            return true;
        }
        if (abs($this->grademax - ($this->knowledge + $this->inquiry + $this->communication + $this->application)) > kica_grade::GRADE_TOLERANCE) {
            return true;
        }
        return false;
    }

    /**
     * @return bool
     */
    public function is_graded(){
        global $DB;
        $table = kica_grade::TABLE;
        $sql = "SELECT gg.id
                  FROM {{$table}} gg
                 WHERE gg.itemid = ?
                   AND (gg.knowledge IS NOT NULL
                    OR gg.inquiry IS NOT NULL
                    OR gg.communication IS NOT NULL
                    OR gg.application IS NOT NULL)";

        if ($DB->record_exists_sql($sql, array($this->id))) {
            return true;
        }
        return false;
    }

    /**
     * @return \gradingform_controller|null
     * @throws \moodle_exception
     */
    public function get_grading_controller() {
        if (!$this->areaid) {
            return null;
        }
        $gradingmanager = get_grading_manager($this->context, 'mod_' . $this->itemmodule, 'submissions');
        $controller = null;
        if ($gradingmethod = $gradingmanager->get_active_method()) {
            return $gradingmanager->get_controller($gradingmethod);
        }

        return null;
    }

    /**
     * Checked visibility of kica item by course module
     *
     * @param numeric|object $user_or_id               - user, for whom check visibility (global by default)
     * @param bool           $unavailable_as_invisible - if true, than with false uservisible - return false, despite available info
     * @param bool           $check_global_visibility  - check (or not), that global user can see it too
     *
     * @return bool
     */
    public function is_visible($user_or_id=null, $unavailable_as_invisible=false, $check_global_visibility=false){
        if (empty($this->cm->id)) return false;

        return NED::get_cm_visibility_by_user($this->cm, $user_or_id, $unavailable_as_invisible, $check_global_visibility);
    }

    /**
     * Return raw record(s) from the db
     *
     * @param array $params
     * @param bool  $get_one_record - if true (false by default), return only first record (or null)
     *
     * @return array|object[]|object|null
     */
    static public function get_raw_records($params=[], $get_one_record=false){
        $records = NED::db()->get_records(static::TABLE, $params);
        if ($get_one_record){
            return empty($records) ? null : reset($records);
        }

        return $records;
    }

    /**
     * Return KICA item by its id or full DB item
     * @constructor
     *
     * @param                $id
     * @param \stdClass|array|null $full_item
     *
     * @return static|null
     */
    static public function get_by_id($id, $full_item=null){
        if (!$id && !$full_item){
            return null;
        }

        if (!isset(static::$_ki_data[$id])){
            if ($full_item){
                new kica_item((array)$full_item);
            } else {
                new kica_item(['id' => $id]);
            }
        }

        return static::$_ki_data[$id] ?? null;
    }

    /**
     * Return kica item from the object or its id
     *
     * @param static|numeric|object|array $item_or_id
     *
     * @return \local_kica\kica_item|static|null
     */
    static public function get_by_item_or_id($item_or_id){
        if (empty($item_or_id)){
            return null;
        } elseif ($item_or_id instanceof static){
            return $item_or_id;
        } elseif (is_numeric($item_or_id)){
            return static::get_by_id($item_or_id);
        } elseif (is_object($item_or_id) && isset($item_or_id->id)){
            return static::get_by_id($item_or_id->id, $item_or_id);
        } elseif (is_array($item_or_id) && isset($item_or_id['id'])){
            return static::get_by_id($item_or_id['id'], $item_or_id);
        } else {
            return null;
        }
    }

    /**
     * Return KICA item by its id, if it exists
     * @constructor
     *
     * @param numeric   $id
     * @param array     $params (optional) params for additional check
     *
     * @return static|null
     */
    static public function get_by_id_existing($id, $params=[]){
        if (!$id){
            return null;
        }

        if (!isset(static::$_ki_data[$id])){
            $record = static::get_raw_records(['id' => $id], true);
            if ($record){
                $ki = static::new_empty_object();
                if (!$ki->update_from_db($id, null, null, $record, true)){
                    return null;
                }
            }
        }

        $ki = static::$_ki_data[$id] ?? null;
        if ($ki && !empty($params)){
            foreach ($params as $param => $value){
                $ki_val = $ki->$param ?? null;
                if ($ki_val != $value){
                    return null;
                }
            }
        }

        return $ki;
    }

    /**
     * Get kica_item by grade_item or its id
     * @constructor
     *
     * @param numeric|object|\grade_item $gi_item_or_id
     *
     * @return static|null
     */
    static public function get_by_grade_item($gi_item_or_id){
        $gi_id = NED::get_id($gi_item_or_id);
        if (!$gi_id){
            return null;
        }

        if (!isset(static::$_gi_ids[$gi_id])){
            $ki = static::new_empty_object();
            $ki->update_from_db(null, $gi_item_or_id);
            return $ki;
        }

        return static::get_by_id(static::$_gi_ids[$gi_id] ?? 0);
    }

    /**
     * Get kica_item by course_module (cm) or its id
     * @constructor
     *
     * @param numeric|object|\cm_info $cm_or_id
     * @param bool                    $only_loaded - check only from already loaded kica items
     *                                             useful, if you have already loaded all items by course, for example
     *
     * @return static|null
     */
    static public function get_by_cm($cm_or_id, $only_loaded=false){
        $cmid = NED::get_id($cm_or_id);
        if (!$cmid){
            return null;
        }

        if (!isset(static::$_cm_ids[$cmid]) && !$only_loaded){
            $ki = static::new_empty_object();
            $ki->update_from_db(null, null, $cm_or_id);
            return $ki;
        }

        return static::get_by_id(static::$_cm_ids[$cmid] ?? 0);
    }

    /**
     * Return simple kica_item object, but load all of them for the course
     *  (so, if you will use all kica items by course, it will be quicker)
     * @constructor
     *
     * Also, if you need filter by user visibility or KICA group, check static::get_all_items_by_course()
     * @see get_all_items_by_course()
     *
     * @param numeric|object|\cm_info   $cm_or_id
     * @param numeric|object            $course_or_id (optional) course, if already loaded, otherwise it will use courseid from the cm
     *                                                it there is no $course_or_id, it will be better to sent course module (cm) object as $cm_or_id
     *
     * @return static|null
     */
    static public function get_by_cm_and_course($cm_or_id, $course_or_id=null){
        $courseid = NED::get_id($course_or_id);

        if (!$courseid){
            if (is_object($cm_or_id)){
                $courseid = $cm_or_id->course;
            } else {
                // it will be quicker to try get ki item by cm only first
                $res = static::get_by_cm($cm_or_id, true);
                if ($res){
                    return $res;
                } else {
                    $cm = NED::get_cm_by_cmorid($cm_or_id);
                    if ($cm){
                        $courseid = $cm->course;
                    } else {
                        return null;
                    }
                }
            }
        }

        if (is_null(static::$_course_kis[$courseid] ?? null)){
            static::get_all_items_by_course($courseid, null, null, true);
        }

        return static::get_by_cm($cm_or_id, true);
    }

    /**
     * Return kica_item for the course
     * @constructor
     *
     * @param numeric|object $course_or_id
     *
     * @return static
     */
    static public function get_course_item($course_or_id){
        $course_gi = NED::get_course_grade_item($course_or_id);
        $ki = static::new_empty_object();
        $ki->update_from_db(null, $course_gi, NED::get_fast_modinfo($course_or_id));
        return $ki;
    }

    /**
     * Return all kica_items by course at once
     *
     * @param object|numeric   $course_or_id - course (or id) by which should be loaded kica items
     * @param numeric          $kica_group   - (optional) filter kica_items by kica group
     * @param array|int|object $users_or_ids - (optional) list of users or userids (or single user/id), who should see them,
     *                                       if null - check only global (default), otherwise check global AND these user(s)
     * @param bool             $only_load_cache - if true, only load exists items in the cache, and return all found KI keys
     *                                          NOTE: with this options in true, function ignore $kica_group and $users_or_ids parameters
     *
     * @return array|static[]|int[] - array of kica_items or kica_items keys (if $only_load_cache is true)
     */
    static public function get_all_items_by_course($course_or_id, $kica_group=null, $users_or_ids=[], $only_load_cache=false){
        $courseid = NED::get_id($course_or_id);
        $ki_ids = static::$_course_kis[$courseid] ?? null;

        if (is_null($ki_ids)){
            static::$_course_kis[$courseid] = [];
            $ki_records = static::get_raw_records(['courseid' => $courseid]);
            if (empty($ki_records)) return [];

            $ki_ids_by_cmid = [];
            $cms = NED::get_course_cms($courseid);
            foreach ($ki_records as $ki_record){
                if (!$ki_record->itemid) continue;

                $gi_item = NED::get_grade_item_by_id_or_params($ki_record->itemid, [], $courseid);
                if (!$gi_item) continue;

                $cm_by_gi = NED::get_cm_by_kica_or_grade_item($gi_item);
                if (!isset($cms[$cm_by_gi->id ?? 0])) continue;

                $ki_item = static::new_empty_object();
                $ki_item->update_from_db($ki_record->id, $gi_item, $cm_by_gi, $ki_record);
                if ($ki_item->id ?? false){
                    $ki_ids_by_cmid[$cm_by_gi->id] = $ki_item->id;
                }
            }

            $ki_ids = [];
            if (!empty($ki_ids_by_cmid)){
                // sort kica items in activity order
                foreach ($cms as $cm){
                    if (!isset($ki_ids_by_cmid[$cm->id])) continue;

                    $ki_ids[] = $ki_ids_by_cmid[$cm->id];
                }
            }
            static::$_course_kis[$courseid] = $ki_ids;
        }

        if ($only_load_cache){
            return $ki_ids;
        }

        $ki_items = [];
        foreach ($ki_ids as $ki_id){
            /** @var static $ki_item */
            $ki_item = static::$_ki_data[$ki_id] ?? null;
            if (!$ki_item) continue;
            if ($kica_group && $ki_item->kicagroup != $kica_group) continue;
            if (!NED::get_cm_visibility_by_userlist($ki_item->cm, $users_or_ids)) continue;

            $ki_items[$ki_id] = $ki_item;
        }

        return $ki_items;
    }

    /**
     * Return kica_item_id if it exists, false otherwise
     *
     * @param $courseid
     * @param $gradeitem_id
     *
     * @return int|false
     */
    static public function get_kica_item_id($courseid, $gradeitem_id){
        global $DB;
        if (isset(static::$_gi_ids[$gradeitem_id])){
            return static::$_gi_ids[$gradeitem_id];
        }
        return $DB->get_field(self::TABLE, 'id', ['itemid' => $gradeitem_id, 'courseid' => $courseid]);
    }

    /**
     * Check, does this course have any KICA items
     *
     * @param object|numeric   $course_or_id
     *
     * @return bool
     */
    static public function course_has_kica_item($course_or_id){
        $courseid = NED::get_id($course_or_id);
        return NED::db()->record_exists(static::TABLE, ['courseid' => $courseid]);
    }

    /**
     * Return visibility data by kica items and users
     * Activity visibility checks much faster (around 10 times) when users - in outer circle, so,
     *  if in your case users should be in the inner circle, there will be more quickly to get visibility data in separate loop,
     *  for example by this function.
     *
     * @param array|object[]|static[]  $kicaitems                - list of kica items
     * @param array|object[]|numeric[] $users_or_ids             - list of users or users ids
     * @param bool                     $by_userid_itemid         - (optional) if true, result array will be by userid and itemid
     * @param bool                     $unavailable_as_invisible - (optional) if true, than with false uservisible - return false
     * @param bool                     $check_global_visibility  - (optional) check (or not), that global user can see it too
     *
     * @return array [$kicaitem_id => [$userid => true]] or [$userid => [$kicaitem_id => true]]
     */
    static public function get_visibility_data($kicaitems, $users_or_ids, $by_userid_itemid=false,
        $unavailable_as_invisible=false, $check_global_visibility=false){
        $visibility_data = [];
        foreach ($users_or_ids as $user_or_id){
            $userid = NED::get_id($user_or_id);
            foreach ($kicaitems as $kicaitem){
                if ($kicaitem->is_visible($userid, $unavailable_as_invisible, $check_global_visibility)){
                    if ($by_userid_itemid){
                        $visibility_data[$userid][$kicaitem->id] = true;
                    } else {
                        $visibility_data[$kicaitem->id][$userid] = true;
                    }
                }
            }
        }

        return $visibility_data;
    }

    /**
     * Clear static caches for this class
     */
    static public function clear_items_cache(){
        static::$_ki_data = [];
        static::$_gi_ids = [];
        static::$_cm_ids = [];
        static::$_course_kis = [];
    }
}