<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Contains class local_kica\output\renderer
 *
 * @package    local_kica
 * @copyright  2018 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kica\output;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/group/lib.php');

use local_kica as kica;
use local_kica\output\menu_bar as MB;
use local_kica\shared_lib as NED;

/**
 * Class renderer
 *
 * @package local_kica\output
 */
class renderer extends \plugin_renderer_base {
    /**
     * Render the completion form start HTML.
     * @param string $action The action URL.
     * @param array $hiddeninputs Name/value pairs of hidden inputs used by the form.
     * @return string The output for the page.
     */
    public function grade_formstart($action, $hiddeninputs=[]) {
        $output = '';
        $output .= \html_writer::start_tag('form', ['id' => 'grade_edit', 'method' => 'post', 'action' => $action]) . "\n";
        foreach ($hiddeninputs as $name => $value) {
            $output .= \html_writer::empty_tag('input', ['type' => 'hidden', 'name' => $name, 'value' => $value]) . "\n";
        }
        return $output;
    }

    /**
     * Render the completion form end HTML.
     * @param array $inputs Type/attribute array of inputs and values used by the form.
     * @return string The output for the page.
     */
    public function grade_formend($inputs=[]) {
        $output = '';
        foreach ($inputs as $type => $attributes) {
            $output .= \html_writer::empty_tag('input', array_merge(['type' => $type], $attributes)) . "\n";
        }
        $output .= \html_writer::end_tag('form') . "\n";
        return $output;
    }

    /**
     * Render the completion form control buttons.
     * @return string The output for the page.
     */
    public function grade_controlbuttons() {
        $output = '';
        $output .= \html_writer::start_div('grade-control-btns');
        $output .= \html_writer::empty_tag('input',
            array('type' => 'submit', 'name' => 'submit', 'class' => 'btn btn-primary', 'value' => NED::str('savechanges')));
        $output .= \html_writer::end_div();
        return $output;
    }

    /**
     * @param string $name Values are knowledge, inquiry, communication, application
     * @param        $itemid
     * @param        $userid
     * @param        $value
     * @param        $grademax
     * @param bool   $incomplete
     * @param bool   $gradable
     *
     * @return string
     */
    public function grade_input($name, $itemid, $userid, $value, $grademax, $incomplete=false, $gradable=true) {
        if ($incomplete || $grademax == 0 || !$gradable) {
            return \html_writer::empty_tag('input',
                array('type' => 'text', 'name' => $name . '_disabled_' . $itemid, 'class' => 'grade-input',
                    'value' => $value, 'size' => '4', 'disabled' => 'disabled', 'data-grademax' => $grademax)
            );
        } else {
            return \html_writer::empty_tag('input',
                array('type' => 'text', 'name' => $name . '_' . $itemid . '_' . $userid, 'class' => 'grade-input',
                    'value' => $value, 'size' => '4', 'data-grademax' => $grademax)
            );
        }
    }

    /**
     * @param $courseid
     * @param $urlroot
     * @param $selecteditemid
     *
     * @return string
     */
    public function gradeitem_menu($courseid, $urlroot, $selecteditemid) {
        $options = ['0' => NED::str('allactivities')];
        $kicaitems = NED::ki_get_all_by_course($courseid);
        $options += NED::records2menu($kicaitems, 'itemname');

        $itemselect = NED::single_select($urlroot, 'itemid', $options, $selecteditemid,
            NED::span('', 'icon fa fa-table', ['aria-hidden' => 'true']), [], 'ned_item');
        return NED::div($itemselect, 'groupuserselector');
    }

    /**
     * @param \moodle_url $url
     * @param $str
     * @return string
     * @throws \coding_exception
     */
    public function grade_action_button(\moodle_url $url, $str) {
        return NED::div(NED::link($url, NED::str($str), 'btn btn-default'), 'groupuserselector grade-action-btn');
    }

    /**
     * Render user table with KICA grades for one user
     *
     * @param             $courseid
     * @param \stdClass   $user
     * @param \moodle_url $pageurl
     * @param             $gradeview
     * @param             $mode
     * @param bool        $return
     *
     * @return string
     */
    public function print_grade_user($courseid, $user, $pageurl, $gradeview, $mode, $return=false){
        $ROUND = 2;
        $grader = $gradeview == kica\VIEW_GRADER;
        $view = $mode == kica\MODE_VIEW;
        $edit = $mode == kica\MODE_EDIT;
        $userid = $user->id ?? 0;
        $o = '';
        $KMB = MB::get_menu_bar_by_courseid($courseid);
        $kica = NED::get_kica($courseid);
        $calc_natural = $kica->calculation == kica\kica_grade::CALCULATION_NATURAL;

        $activityicons = [
            'assign' => NED::div(NED::fa('fa-circle-o'), 'float-right gray-icon'),
            'quiz' => NED::div(NED::fa('fa-square-o'), 'float-right gray-icon'),
        ];

        $viewgradedisparity = NED::has_capability('viewgradedisparity', NED::ctx($courseid));

        if ($grader || $KMB->isteacher){
            $grade_url = $KMB->get_url(MB::PAGE_MM, [MB::PAR_SETUSER => $userid, 'getshow' => 1]);
        } else {
            $grade_url = clone $pageurl;
        }

        $action_form = [];
        if ($gradeview != kica\VIEW_USER){

            $actionurl = clone $pageurl;
            $actionurl->param(kica\P_MODE, ($view ? kica\MODE_EDIT : kica\MODE_VIEW));
            if ($user && $grader) {
                $action_form[] =  $this->grade_action_button($actionurl, ($view ? 'editkicagrades' : 'cancel'));
            }
        }

        if (!empty($action_form)){
            $o .= NED::div(join('', $action_form), 'grade-action-form');
        }

        if ($grader && ($formdata = data_submitted()) && confirm_sesskey()) {
            $submittedgrades = kica\helper::parse_grade($formdata);
            $redirect = false;
            foreach ($submittedgrades as $gradeitemid => $grades) {
                foreach ($grades as $uid => $grade) {
                    kica\helper::save_grade($gradeitemid, $uid, $grade);
                    $redirect = true;
                }
            }
            if ($redirect){
                NED::redirect($KMB->get_url());
            }
        }

        $categoryicon = NED::img('i/folder');
        $kicagroups = NED::get_kica_groups($courseid);

        if ($user && $edit) {
            $o .= $this->grade_formstart($pageurl->out(false),
                ['participant' => $userid, 'userid' => $userid, 'sesskey' => sesskey()]
            );
            $o .= $this->grade_controlbuttons();
        }

        // Grade summary
        $grade_summary = new kica\output\grade_summary($courseid, $userid);
        $o .= $this->render($grade_summary);

        // Completed course message.
        if (NED::is_course_final_evaluation_completed($courseid, $userid)){
            $o .= NED::div(NED::str('coursecompletedmsg'), 'kica-course-completed p-2 mb-2 text-align-center font-weight-bold');
        }

        // Grade tables
        $kica_grades = NED::kg_get_grades_by_course($courseid, $userid, true)[$userid] ?? [];
        $average_data = NED::kg_get_kica_average($courseid, $userid, null, null, $ROUND, $kica_grades);

        foreach ($kicagroups as $kicagroupvalue => $kicagroup_title) {
            $table = NED::html_table('generaltable simple setup-grades kicagradingtable', null,
                [
                    $categoryicon . $kicagroup_title . "($kicagroupvalue%)",
                    $calc_natural ? NED::str('grade') : NED::str('rawgrade'),
                ]
            );
            foreach (NED::KICA_KEYS as $kica_key) {
                if ($calc_natural) {
                    $table->head[] = NED::str($kica_key)[0];
                } else {
                    $table->head[] = NED::str($kica_key)[0] . '<br>' . $kica->$kica_key . '%';
                }
            }

            $kica_activities = NED::ki_get_all_by_course($courseid, $kicagroupvalue, $userid);
            foreach ($kica_activities as $kicaitem) {
                // ki_get_all_by_course already check items for visibility for teacher and user
                $row = null;
                $kicagrade = $kica_grades[$kicaitem->id] ?? null;
                if (empty($kicagrade) || (!$grader && $kicagrade->flag)) {
                    continue;
                }

                $needsupdate = false;
                if ($user) {
                    $needsupdate = kica\helper::is_gradingform_needupdate($kicaitem, $userid) || $kicagrade->grade_mismatch();
                }

                $rowclass = '';
                if (!$isgradable = $kicagrade->is_gradable()) {
                    $rowclass = 'notgradable';
                    if ($kicagrade->area && $kicagrade->is_graded()) {
                        $rowclass = $needsupdate ? 'grademismatch' : '';
                    }
                } elseif (!$kicagrade->is_graded() && !$needsupdate) {
                    $rowclass = 'ungraded';
                } elseif ($needsupdate) {
                    $rowclass = 'grademismatch';
                }

                $activity_cell = NED::q_cm_student_link($kicaitem->cm, $user);
                if ($viewgradedisparity && NED::cm_is_test_or_assignment($kicaitem->cm)) {
                    $activity_cell .= $activityicons[$kicaitem->cm->modname];
                }

                // Prevents students to view grade mismatch errors.
                if (!$grader && $rowclass == 'grademismatch') {
                    $rowclass = 'ungraded';
                }

                $maskicon = '';
                if ($kicagrade->excluded) {
                    $maskicon = NED::fa('icon fa fa-first-order', '', NED::str('excluded'));
                } else if ($rowclass == 'ungraded') {
                    $maskicon = NED::img('i/duration', '');
                }

                $classes = [];
                foreach (NED::KICA_KEYS as $kica_key){
                    $classes[$kica_key] = [];
                    if ($kicagrade->has_grade_flag($kica_key)){
                        $classes[$kica_key][] = 'flag';
                    }
                    if ($calc_natural) {
                        $classes[$kica_key][] = 'dimmed_text';
                    }
                }

                if ($view){
                    $row = NED::row(null, $rowclass);
                    $row->cells[] = NED::cell($activity_cell, 'column-name');

                    $grade_url_view = null;
                    if ($kicaitem->cm->uservisible){
                        if (method_exists('\local_kicaquizgrading\helper', 'get_kica_grader_link_from_cm')) {
                            $grade_url_view = \local_kicaquizgrading\helper::get_kica_grader_link_from_cm($kicaitem->cm, $user->id);
                        }
                        if (empty($grade_url_view)){
                            if ($grader){
                                $grade_url_view = new \moodle_url($grade_url, ['mid' => $kicaitem->cm->id]);
                            } else {
                                $grade_url_view = $kicaitem->cm->get_url();
                            }
                        }
                    }

                    $row->cells[] =  NED::cell($kicagrade->get_format_finalgrade($ROUND, $grade_url_view, $maskicon));
                    foreach (NED::KICA_KEYS as $kica_key){
                        $row->cells[] = NED::cell($kicagrade->get_format_section($kica_key, $ROUND, null, $maskicon), $classes[$kica_key]);
                    }
                } elseif ($edit){
                    $row = NED::row(null, $rowclass);
                    $row->cells[] = NED::cell($activity_cell, 'column-name');
                    $row->cells[] = NED::cell($kicagrade->get_format_finalgrade());

                    foreach (NED::KICA_KEYS as $kica_key){
                        $max = $kicaitem->get_section($kica_key);
                        $row->cells[] = NED::cell(
                            $this->grade_input($kica_key, $kicaitem->id, $userid,
                                $kicagrade->get_section($kica_key), $max,
                                $kicaitem->incomplete, $isgradable) . ' / ' . $max,
                            $classes[$kica_key]
                        );
                    }
                }

                if ($row){
                    if ($gradeview != kica\VIEW_USER){
                        kica\add_row_action_menu($row, 'student', $kicaitem->cm, $userid);
                    }
                    $table->data[] = $row;
                }
            }

            $row = NED::row([NED::cell('', 'column-name') ], 'lastrow');

            if ($calc_natural) {
                if ($kicagroupvalue == 30 && is_null($average_data[$kicagroupvalue][NED::FINALGRADE])) {
                    $row->cells[] = NED::cell('(' . get_string('pending', 'local_kica') . ')');
                } else {
                    $row->cells[] = NED::cell(($average_data[$kicagroupvalue][NED::FINALGRADE] ?? null).' / 100');
                }
            } else {
                $row->cells[] = NED::cell('');
            }

            foreach (NED::KICA_KEYS as $kica_key){
                if ($calc_natural) {
                    $grade_max = $average_data['max'][$kicagroupvalue][$kica_key] ?? null;
                    $class = 'dimmed_text';
                } else {
                    $grade_max = $kica->$kica_key;
                    $class = '';
                }
                $avg = $average_data[$kicagroupvalue][$kica_key] ?? null;
                $row->cells[] = NED::cell(is_null($grade_max) ? '-' : ($avg.' / '.$grade_max), $class);
            }
            $table->data[] = $row;
            $o .= NED::render_table($table);
        }

        if ($edit) {
            $o .= $this->grade_formend();
        }

        if (!$return){
            echo $o;
        }
        return $o;
    }

    /**
     * Render user table with KICA grades for all $users
     *
     * @param      $courseid
     * @param      $users
     * @param bool $return
     *
     * @return string
     */
    public function print_users_grades($courseid, $users, $return=false){
        $ROUND = 2;
        $kica = NED::get_kica($courseid);
        $baseurl = new \moodle_url(NED::page()->url);
        $calc_natural = NED::is_kica_calculation_natural($kica);
        $kicaitems = NED::ki_get_all_by_course($courseid);
        $max_kicaitems_count = count($kicaitems);

        //region Short functions
        $get_user_url = function($userid) use ($baseurl){
            $url = clone $baseurl;
            $url->param(menu_bar::PAR_USER, $userid);
            return $url;
        };

        $get_group_link = function($user) use ($baseurl){
            if ($user->group->id ?? false){
                $url = clone $baseurl;
                $url->param(menu_bar::PAR_GROUP, $user->group->id);
                return NED::link($url, $user->group->name);
            } else {
                return '';
            }
        };

        $cell = function($text='', $class='', $sort=null, $attr=null){
            $sort = $sort ?? $text;
            $attr = $attr ?: [];
            $attr['data-sort-value'] = $sort;
            return NED::cell($text, $class, $attr);
        };

        $head_cell = function($name, $add_text=''){
            $text = NED::str($name);
            return NED::cell($text.$add_text, $name, ['title' => $text]);
        };

        $get_count_graded_activities = function($userid) use (&$kica, &$calc_natural, &$kicaitems){
            $all = 0;
            $graded = 0;
            foreach ($kicaitems as $itemid => $kicaitem){
                if (!$kicaitem->is_visible($userid)){
                    continue;
                }

                $all++;

                $kicagrade = kica\kica_grade::get_by_userid_itemid($userid, $itemid);
                if (empty($kicagrade->id) || !$kicagrade->is_graded()) continue;

                $graded++;
            }
            return [$all, $graded];
        };
        //endregion

        $average_data = ['grade' => []];
        $average_graded = 0;
        $table = new \html_table();
        $table->attributes['class'] = 'kica-table grade-user';
        $table->head = [
            $head_cell('student'),
            $head_cell('class'),
            $head_cell('graded', ' /'.$max_kicaitems_count),
            $head_cell('grade'),
        ];
        foreach (NED::KICA_KEYS as $kica_key) {
            $kk_str = NED::str($kica_key);
            $head_name = $kk_str[0];
            if (!$calc_natural){
                $head_name .= ' /' . $kica->$kica_key;
            }
            $table->head[] = NED::cell($head_name, 'kica-key-head ' . $kica_key, ['title' => $kk_str]);
            $average_data[$kica_key] = [];
        }

        NED::kg_get_grades_by_course($courseid, array_keys($users), false, true);

        foreach ($users as $user){
            $userid = $user->id;
            $row = NED::row();

            $username = NED::link($get_user_url($userid), NED::O()->user_picture($user,
                ['includefullname' => true, 'link' => false, 'courseid' => $courseid]));
            $row->cells[] = $cell($username, 'username', fullname($user));
            $row->cells[] = $cell($get_group_link($user), 'group', $user->group->name ?? '');
            list($kicaitems_count, $graded) = $get_count_graded_activities($userid);
            $average_graded += $graded;
            if ($kicaitems_count == $max_kicaitems_count){
                $gr_text = $graded;
            } else {
                $gr_text = $graded.'/'.$kicaitems_count;
            }
            $row->cells[] = $cell($gr_text, 'graded');

            $average_user_data = NED::kg_get_course_average($courseid, $userid);
            foreach (NED::FKICA_KEYS as $fkica_key){
                $grade_val = $average_user_data[$fkica_key] ?? null;
                if (is_null($grade_val)){
                    $sort_grade = -1;
                    $grade_val = '-';
                } else {
                    $sort_grade = $grade_val;
                    $average_data[$fkica_key][] = $grade_val;
                    $grade_val = round($grade_val, $ROUND);
                    if (!$calc_natural || $fkica_key == NED::FINALGRADE){
                        $grade_val .= '%';
                    }
                }
                $row->cells[] = $cell($grade_val, $fkica_key.'-grade', $sort_grade);
            }

            $table->data[] = $row;
        }

        // add average row
        $row = NED::row([], 'nosort average-row');
        $c = NED::cell(NED::str('totalaverage'), 'totalaverage');
        $c->colspan = 2;
        $row->cells[] = $c;

        $user_graded = round($average_graded / count($users), $ROUND);
        $average_graded = "{$user_graded}/{$max_kicaitems_count}";
        $row->cells[] = NED::cell($average_graded, 'graded');
        unset($average_data['graded']);

        foreach (NED::FKICA_KEYS as $fkica_key){
            $average_list = $average_data[$fkica_key] ?? [];
            $count = count($average_list);
            if ($count){
                $val = round(array_sum($average_list) / $count, $ROUND);
                if (!$calc_natural){
                    $max_val = $kica->$fkica_key ?? 100;
                    $val = $max_val ? round((($val / $max_val) * 100), $ROUND) : 0;
                }
                if (!$calc_natural || $fkica_key == NED::FINALGRADE){
                    $val .= '%';
                }
            } else {
                $val = '-';
            }
            $row->cells[] = NED::cell($val, $fkica_key);
        }
        $table->data[] = $row;

        $o = NED::render_table($table);
        NED::$C::js_call_amd('add_sorter', 'add_sort', ['table.kica-table']);
        if (!$return){
            echo $o;
        }
        return $o;
    }

    /**
     * Render activity table by its $itemid with KICA grades by all $users
     *
     * @param             $courseid
     * @param             $groupid
     * @param             $itemid
     * @param array       $users
     * @param             $filter
     * @param int         $mode
     * @param bool        $isgrader
     * @param array       $action_form - additional page selectors save here, they are rendered out of the function
     * @param bool        $return
     *
     * @return string
     */
    public function print_grade_activity($courseid, $groupid, $itemid, $users, $filter,
        $mode=kica\MODE_VIEW, $isgrader=false, &$action_form=[], $return=false){
        $thispageurl = new \moodle_url(NED::page()->url);
        $kica = NED::get_kica($courseid);
        $calc_weight = NED::is_kica_calculation_natural($kica);
        $o = '';
        $KMB = MB::get_menu_bar_by_courseid($courseid);
        $grade_url = clone $thispageurl;

        // Display first ungraded activity.
        if (empty($itemid)) {
            $itemms = $KMB->get_ungraded_activities();
            if (isset($itemms[0])) {
                $firstitem = reset($itemms[0]);
                if (isset($firstitem['item']->id)) {
                    $itemid = $firstitem['item']->id;
                }
            }
        }

        $cm = $kicaitem = null;
        if ($itemid) {
            $kicaitem = kica\kica_item::get_by_id($itemid);
            if ($kicaitem){
                $cm = $kicaitem->cm;
            }
        }

        $not_empty = !empty($cm->id) && !empty($kicaitem->id) && !empty($users);

        if ($cm){
            $action_form[] = NED::link($cm->get_url(),
                '', 'fa fa-external-link viewactivity',
                ['target' => '_blank', 'title' => NED::str('viewactivity')]);

            if (has_capability('moodle/course:manageactivities', NED::ctx(null, $cm->id))){
                $action_form[] = NED::link(["/course/modedit.php", ['update' => $cm->id]],
                    '', 'fa fa-cog activitysettings',
                    ['target' => '_blank', 'title' => NED::str('activitysettings')]);
            }
            $grade_url = $KMB->get_url(MB::PAGE_MM, ['getshow' => 1, 'mid' => $cm->id]);
        }

        if ($not_empty && $isgrader) {
            //$actionurl = clone $thispageurl;
            //$actionurl->param(kica\P_MODE, (($mode == kica\MODE_VIEW) ? kica\MODE_EDIT : kica\MODE_VIEW));
            //$action_form[] = $this->grade_action_button($actionurl, (($mode == kica\MODE_VIEW) ? 'editkicagrades' : 'cancel'));
        }

        if (is_siteadmin() && ($kicaitem->itemmodule == 'assign' || $kicaitem->itemmodule == 'quiz')) {
            $actionurl = new \moodle_url('/local/kica/recalculate.php');
            $actionurl->params(['courseid' => $courseid, 'itemid' => $itemid]);
            $action_form[] = $this->grade_action_button($actionurl, 'recalculatekicascores');
        }

        if (($formdata = data_submitted()) && confirm_sesskey()) {
            $submittedgrades = kica\helper::parse_grade($formdata);
            $redirect = false;
            foreach ($submittedgrades as $gradeitemid => $save_grades) {
                foreach ($save_grades as $userid => $save_grade) {
                    kica\helper::save_grade($gradeitemid, $userid, $save_grade);
                    $redirect = true;
                }
            }
            if ($redirect){
                NED::redirect($KMB->get_url());
            }
        }

        if ($not_empty) {
            NED::js_call_amd('grade');

            if ($mode == kica\MODE_EDIT) {
                $o .= $this->grade_formstart($thispageurl->out(), ['itemid' => $itemid, 'sesskey' => sesskey()]
                );
                $o .= $this->grade_controlbuttons();
            }

            $table = NED::html_table('generaltable simple setup-grades kicagradingtable', null,
                [
                    NED::cell(' ', 'cell column-rowspan rowspan level1 levelodd cell c0', ['rowspan' => count($users) + 1]),
                    ($calc_weight) ? NED::str('rawgrade') : NED::str('grade'),
                ]
            );

            if ($calc_weight) {
                $table->head[] = NED::str('kica_grade');
            }

            foreach (NED::KICA_KEYS as $kica_key) {
                if ($calc_weight) {
                    $table->head[] = NED::str($kica_key) . '<br>' . $kica->$kica_key . '%';
                } else {
                    $table->head[] = NED::str($kica_key);
                }
            }

            if ($users) {
                $controller = $kicaitem->get_grading_controller();
                foreach ($users as $user) {
                    if (!$kicaitem->is_visible($user)){
                        continue;
                    }

                    $row = null;
                    $kicagrade = kica\kica_grade::get_by_userid_itemid($user->id, $kicaitem->id);
                    $needsupdate = kica\helper::is_gradingform_needupdate($kicaitem, $user->id, $controller) || $kicagrade->grade_mismatch();

                    $rowclass = '';
                    $datasort = 4;
                    if (!$isgradable = $kicagrade->is_gradable()) {
                        $rowclass = 'notgradable';
                        $datasort = 3;
                        if ($kicagrade->area && $kicagrade->is_graded()) {
                            if ($needsupdate) {
                                $rowclass = 'grademismatch';
                                $datasort = 2;
                            } else {
                                $rowclass = '';
                                $datasort = 4;
                            }
                        }
                    } else if (!$kicagrade->is_graded() && !$kicagrade->grade_mismatch()) {
                        $rowclass = 'ungraded';
                        $datasort = 1;
                    } else if ($grademismatch = $kicagrade->grade_mismatch()) {
                        $rowclass = 'grademismatch';
                        $datasort = 2;
                    }

                    $cell = NED::O()->user_picture($user, ['visibletoscreenreaders' => false, 'includefullname' => true, 'link' => true]);

                    $flagged = false;
                    $classes = [];
                    foreach (NED::KICA_KEYS as $kica_key){
                        $classes[$kica_key] = [];
                        if ($kicagrade->has_grade_flag($kica_key)) {
                            $classes[$kica_key][] = 'flag';
                            $flagged = true;
                        }
                    }
                    if ($flagged) {
                        $datasort = 2;
                    }

                    if ($filter == 'flag' && !$flagged) {
                        continue;
                    }

                    $fullname = fullname($user);
                    if (($fullname[0] ?? '') == '['){
                        $datasort .= '000';
                    }
                    $datasort .= $fullname;

                    if ($mode == kica\MODE_VIEW) {
                        $grade_url->param(MB::PAR_SETUSER, $user->id);
                        if (method_exists('\local_kicaquizgrading\helper', 'get_kica_grader_link_from_cm')) {
                            if ($kicaquickgraderurl = \local_kicaquizgrading\helper::get_kica_grader_link_from_cm($cm, $user->id)) {
                                $grade_url = $kicaquickgraderurl;
                            }
                        }
                        $row = NED::row([NED::cell($cell, 'column-name')], $rowclass, ['data-sort' => $datasort]);
                        $row->cells[] = NED::cell($kicagrade->get_format_finalgrade(2, $grade_url));
                        if ($calc_weight) {
                            $row->cells[] = NED::cell($kicagrade->get_grade_percentage(true, 2, true));
                        }

                        foreach (NED::KICA_KEYS as $kica_key){
                            $row->cells[] = NED::cell($kicagrade->get_format_section($kica_key), $classes[$kica_key]);
                        }
                    } else if ($mode == kica\MODE_EDIT) {
                        $row = NED::row([NED::cell($cell, 'column-name')], $rowclass, ['data-sort' => $datasort]);
                        $row->cells[] = NED::cell($kicagrade->get_format_finalgrade());
                        if ($calc_weight) {
                            $row->cells[] = NED::cell($kicagrade->get_grade_percentage(true, 2, true));
                        }

                        foreach (NED::KICA_KEYS as $kica_key){
                            $max = $kicaitem->get_section($kica_key);
                            $row->cells[] = NED::cell(
                                $this->grade_input($kica_key, $kicaitem->id, $user->id,
                                    $kicagrade->get_section($kica_key), $max,
                                    $kicaitem->incomplete, $isgradable) . ' / ' . $max,
                                $classes[$kica_key]
                            );
                        }
                    }

                    if ($row){
                        kica\add_row_action_menu($row, 'activity', $cm, $user->id);
                        $table->data[] = $row;
                    }
                }
            }

            $o .= NED::render_table($table);

            if ($mode == kica\MODE_EDIT) {
                $o .= $this->grade_formend();
            }
        }

        if (!$return){
            echo $o;
        }
        return $o;
    }

    /**
     * Render all activity table with average KICA grades by all $users
     *
     * @param      $courseid
     * @param      $users
     * @param bool $return
     *
     * @return string
     */
    public function print_all_activity_grades($courseid, $users, $return=false){
        global $PAGE;
        $kica = NED::get_kica($courseid);
        $kicaitems = NED::ki_get_all_by_course($courseid);
        $user_count = count($users);
        $baseurl = new \moodle_url($PAGE->url);
        $kicagroups = NED::get_kica_groups($courseid);
        $kica_gradekeys = NED::FKICA_KEYS;
        $calc_natural = NED::is_kica_calculation_natural($kica);
        NED::kg_get_grades_by_course($courseid, array_keys($users), false, true);
        $o = '';

        //region Short functions
        $cell = function($text='', $class='', $sort=null, $attr=null){
            $sort = $sort ?? $text;
            $attr = $attr ?: [];
            $attr['data-sort-value'] = $sort;
            return NED::cell($text, $class, $attr);
        };

        $head_cell = function($name, $add_text=''){
            $text = NED::str($name);
            return NED::cell($text.$add_text, $name, ['title' => $text]);
        };

        $activity_cell = function($kicaitem) use ($baseurl, $cell){
            /** @var \local_kica\kica_item $kicaitem */
            $url = clone $baseurl;
            $url->param(kica\P_ITEM, $kicaitem->id);
            $text = NED::mod_link($kicaitem->cm, 20, true, 'icon').NED::link($url, $kicaitem->itemname);
            return $cell($text, 'activity', $kicaitem->itemname);
        };

        $average_data = [];
        $grade_cell = function($grade, $count, $type, $for_average=true) use (&$kicaitem, &$calc_natural, &$average_data, &$cell){
            if ($type === NED::FINALGRADE){
                $class = $type;
                $percent = !$calc_natural;
            } else {
                $class =  'kicagrade '.$type;
                $percent = false;
            }

            /** @var \local_kica\kica_item $kicaitem */
            if (!is_null($grade) && $count && ((!empty($kicaitem->id) && $kicaitem->has_section($type)) || !$for_average)){
                $grade = $calcgrade = $grade / $count;
                if ($for_average){
                    $maxgrade = $kicaitem->get_section($type);
                    $calcgrade = $percent ? $grade : ($grade / $maxgrade * 100);
                    if (isset($average_data[$type]) && !is_null($calcgrade)){
                        $average_data[$type][] = $calcgrade;
                    }
                }

                $kicagrade = round($grade, 2);
                if ($percent || !$for_average){
                    $kicagrade .= '%';
                } else {
                    $kicagrade .= ' / '.$maxgrade;
                }
            } else {
                $kicagrade = '-';
                $calcgrade = -1;
            }

            return $cell($kicagrade, $class, $calcgrade);
        };
        //endregion

        $visibility_data = NED::ki_get_visibility_data($kicaitems, $users);

        foreach ($kicagroups as $kgroup => $kica_group_name){
            $gr_out = '';
            $average_data = [];
            $table = NED::html_table('kica-table grade-activity', null, [
                $head_cell('activity'),
                $head_cell('graded', ' /'.$user_count),
                $head_cell('averagegrade'),
            ]);

            $average_data[NED::FINALGRADE] = [];
            $average_graded = 0;
            $checked_kicaitems = 0;
            foreach (NED::KICA_KEYS as $kica_key) {
                $kk_str = NED::str($kica_key);
                $table->head[] = NED::cell($kk_str[0], 'kica-key-head ' . $kica_key, ['title' => $kk_str]);
                $average_data[$kica_key] = [];
            }

            foreach ($kicaitems as $itemid => $kicaitem){
                if ($kicaitem->kicagroup != $kgroup) continue;

                $row = NED::row();
                $row->cells[] = $activity_cell($kicaitem);

                $grades_list = [];
                foreach ($kica_gradekeys as $kica_key){
                    $grades_list[$kica_key] = [];
                }
                $checked_users = 0;

                // calculate average by all users
                foreach ($users as $userid => $user){
                    if (!($visibility_data[$itemid][$userid] ?? false)) continue;

                    $checked_users++;
                    $kicagrade = kica\kica_grade::get_by_userid_itemid($userid, $itemid);
                    if (empty($kicagrade->id) || !$kicagrade->is_graded()) continue;

                    $grades_list[NED::FINALGRADE][] = $kicagrade->get_grade_percentage_or_finalgrade(!$calc_natural);
                    foreach (NED::KICA_KEYS as $kica_key){
                        $grades_list[$kica_key][] = $kicagrade->get_section($kica_key);
                    }
                }

                if (!$checked_users) continue;


                $checked_kicaitems++;
                $graded = count($grades_list[NED::FINALGRADE]);
                $average_graded += $graded/$checked_users;
                if ($checked_users == $user_count){
                    $gr_text = $graded;
                } else {
                    $gr_text = $graded.' / '.$checked_users;
                }
                $row->cells[] = $cell($gr_text, 'graded');
                foreach ($kica_gradekeys as $kica_key){
                    $row->cells[] = $grade_cell(array_sum($grades_list[$kica_key]), $graded, $kica_key);
                }

                $table->data[] = $row;
            }

            // add average row
            $row = NED::row([], 'nosort average-row');
            $row->cells[] = NED::cell(NED::str('totalaverage'), 'totalaverage');
            $average_graded = ($checked_kicaitems ? round(100 * $average_graded / $checked_kicaitems, 2) : 0) . '%';
            $row->cells[] = NED::cell($average_graded, 'graded');
            foreach ($average_data as $type => $average_list){
                $row->cells[] = $grade_cell(array_sum($average_list), count($average_list), $type, false);
            }
            $table->data[] = $row;
            // out group part
            $gr_out .= NED::div($kica_group_name, 'kicagroup-title');
            $gr_out .= NED::render_table($table);
            $o .= NED::div($gr_out, 'kicagroup kgroup-'.$kgroup);
        }

        NED::$C::js_call_amd('add_sorter', 'add_sort', ['table.kica-table']);

        if (!$return){
            echo $o;
        }
        return $o;
    }
}
