<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Fix KICA grades
 *
 * @package    local_kica
 * @copyright  2023 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kica\task;

use Exception;
use local_kica\shared_lib as NED;
use local_kica\kica_grade;

defined('MOODLE_INTERNAL') || die();

class fix_kica_grades extends \core\task\scheduled_task {

    public function get_name() {
        return get_string('fixkicagrades', 'local_kica');
    }

    public function execute() {
        global $DB;

        $time = time();
        $minuteago = $time - MINSECS;

        $sql = "SELECT ki.*, gi.itemtype, gi.grademax
                  FROM {local_kica_grade_items} ki
	              JOIN {grade_items} gi
	                ON ki.courseid = gi.courseid
                   AND ki.itemid = gi.id
	              JOIN {local_kica} k 
                    ON ki.courseid = k.courseid
                 WHERE k.enabled = 1 
                   AND gi.grademax > 0
                   AND gi.itemtype = 'mod'
	               AND (ki.knowledge + ki.inquiry + ki.communication + ki.application) 
	               AND ABS((ki.knowledge +  ki.inquiry + ki.communication + ki.application) - gi.grademax) > ?
                   AND ki.timemodified < ?
	               AND gi.timemodified < ?";

        if (!$items = $DB->get_records_sql($sql, [kica_grade::GRADE_TOLERANCE, $minuteago, $minuteago])) {
            return;
        }

        foreach ($items as $kicaitem) {

            if (empty($kicaitem->knowledge) && empty($kicaitem->inquiry) && empty($kicaitem->communication) && empty($kicaitem->application)) {
                return true;
            }

            $kicatotal = $kicaitem->knowledge + $kicaitem->inquiry + $kicaitem->communication + $kicaitem->application;

            $rate = $kicaitem->grademax / $kicatotal;

            mtrace('KICA item update: ID-' . $kicaitem->id . ' Grade max from '. $kicatotal . ' to ' . $kicaitem->grademax);

            try {
                $transaction = $DB->start_delegated_transaction();

                // Update KICA Item.
                $kicaupdate = new \stdClass();
                $kicaupdate->id = $kicaitem->id;
                $kicaupdate->knowledge = round(($kicaitem->knowledge * $rate), 5);
                $kicaupdate->inquiry = round(($kicaitem->inquiry * $rate), 5);
                $kicaupdate->communication = round(($kicaitem->communication * $rate), 5);
                $kicaupdate->application = round(($kicaitem->application * $rate), 5);
                $kicaupdate->timemodified = $time;

                $DB->update_record('local_kica_grade_items', $kicaupdate);

                // Update KICA grades.
                $sql = "SELECT kg.* FROM {local_kica_grade_grades} kg WHERE kg.courseid = ? AND kg.itemid = ?";

                if ($grades = $DB->get_records_sql($sql, [$kicaitem->courseid, $kicaitem->id])) {
                    foreach ($grades as $grade) {
                        $gradeupdate = new \stdClass();
                        $gradeupdate->id = $grade->id;
                        $gradeupdate->finalgrade = 0;
                        $update = false;

                        foreach (NED::KICA_KEYS as $key) {
                            if (!is_null($grade->{$key}) && $grade->{$key} > 0) {
                                $gradeupdate->{$key} = round(($grade->{$key} * $rate), 5);
                                $gradeupdate->finalgrade += $grade->{$key} * $rate;
                                $update = true;
                            }
                        }
                        if ($update) {
                            $gradeupdate->finalgrade = round($gradeupdate->finalgrade, 5);
                            $gradeupdate->timemodified = $time;
                            $DB->update_record('local_kica_grade_grades', $gradeupdate);
                        }
                    }
                }

                $transaction->allow_commit();
            } catch (Exception $e) {
                if (!empty($transaction)){
                    $transaction->rollback($e);
                }
            }
        }
    }
}