<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * KICA
 *
 * @package     local_kica
 * @category    string
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use local_kica as kica;
use local_kica\output\menu_bar as KMB;
use local_kica\shared_lib as NED;

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once($CFG->dirroot.'/local/kica/lib.php');

// Paging options.
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 20, PARAM_INT);

$KMB = new KMB(KMB::PAGE_KICA_ACTIVITIES, ['page' => $page, 'perpage' => $perpage]);
$KMB->show_error_for_not_teacher();
$courseid = $KMB->courseid;
$coursecontext = $KMB->context;

NED::require_capability('manage', $coursecontext);

$caneditgrade = has_capability('moodle/grade:edit', $coursecontext);
$kica = NED::get_kica($courseid);

// Page boilerplate stuff.
$KMB->set_page_params();
NED::$C::js_call_amd('add_sorter', 'add_sort', ['.kica-table']);
NED::js_call_amd('kica', 'unlock');

// Table columns.
$columns = [
    'itemname',
    'grademax',
    'knowledge',
    'inquiry',
    'communication',
    'application'
];
if ($caneditgrade) {
    $columns[] = 'actions';
}

foreach ($columns as $column) {
    if (in_array($column, NED::KICA_KEYS)) {
        $string[$column] = substr(NED::str($column), 0, 1);
    } else {
        $string[$column] = NED::str($column);
    }

    if (isset($kica->$column)) {
        $percentage = '<br>'.$kica->$column.'%';
    } else {
        $percentage = '';
    }

    $$column = $string[$column].$percentage;
}

// Grade Mismatch check.
list($data, $numberofuser) = $KMB->get_grade_mismatches();
$grademismatchactivities = [];
if ($data) {
    foreach ($data as $item) {
        $grademismatchactivities[] = $item['item']->itemid;
    }
}

echo $OUTPUT->header();
$KMB->render();
echo html_writer::start_div('kica-main');
$KMB->render_kica_selector_menu();

if ($caneditgrade) {
    $KMB->print_grading_errors();
}

if (!local_kica\helper::iskicaready($courseid) && local_kica\helper::has_kica_item($courseid)) {
    echo NED::notification('incompletekicactivitiesnotification', NED::NOTIFY_ERROR);
}

$kicagroups = NED::get_kica_groups($courseid);
foreach ($kicagroups as $kicagroupvalue => $kicatitle) {
    $table = NED::html_table('kica-table');
    $table->wrap = array();

    foreach ($columns as $column) {
        $table->head[$column] = $$column;
        $table->wrap[$column] = '';
    }
    // Prevent sort.
    if (isset($table->head['actions'])) {
        $table->head['actions'] = NED::cell($table->head['actions'], 'tablesorter-noSort');
    }

    $kicaitems = NED::ki_get_all_by_course($courseid, $kicagroupvalue);
    $counter = ($page * $perpage);

    foreach ($kicaitems as $kicaitem) {
        $row_class = [];
        if (!empty($grademismatchactivities) && in_array($kicaitem->itemid, $grademismatchactivities)) {
            $row_class[] = 'incomplete';
        }

        if ($kicaitem->incomplete) {
            $row_class[] = 'incomplete';
        }

        $cells = [];
        foreach ($columns as $column) {
            $cell_class = [$column];
            if ($column == 'itemname'){
                $cell_class[] = 'activity';
            }
            $cells[$column] = NED::cell(\local_kica\helper::kicaactivities_data($kicaitem, $column, $counter, $courseid), $cell_class);
        }

        $table->data[] = NED::row($cells, $row_class);
    }
    // render
    echo NED::div(NED::str('groupheader'.$kicagroupvalue, $kicatitle), 'group-header');
    echo NED::render_table($table);
    // Add record form.
    $formurl = new moodle_url('/local/kica/kicaactivities_add.php', array('courseid' => $courseid, 'kicagroup' => $kicagroupvalue));
    $submitbutton  = html_writer::tag('button', NED::str('add'), array(
        'class' => 'btn btn-primary add-record-btn',
        'type' => 'submit',
        'value' => 'submit',
    ));
    $form = html_writer::tag('form', $submitbutton, array(
        'action' => $formurl->out(false),
        'method' => 'post',
        'autocomplete' => 'off'
    ));
    if ($caneditgrade) {
        echo NED::div($form, 'add-record-btn-wrapper', ['id' => 'add-record-form']);
    }
}
echo html_writer::end_div();
echo $OUTPUT->footer();
