<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Observer
 *
 * @package    local_kicaquizgrading
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kicaquizgrading;

global $CFG;
require_once($CFG->dirroot . '/mod/quiz/locallib.php');

use \mod_quiz\event\edit_page_viewed;
use \mod_quiz\event\attempt_submitted;

/**
 * Event observers.
 */
abstract class observer {
    /**
     * @param edit_page_viewed $event
     * @return bool
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function edit_page_viewed(edit_page_viewed $event) {
        global $DB;

        if (!get_config('local_kicaquizgrading', 'enable')) {
            return false;
        }

        $data = $event->get_data();

        $contexid = $data['contextid'];
        $quizid = $data['other']['quizid'];

        $context = \context::instance_by_id($contexid);
        $cmid = $context->instanceid;

        if (!helper::quiz_has_kica_tag($cmid)) {
            return true;
        }

        if (!class_exists('\local_kica\kica_item')) {
            return true;
        }

        helper::update_kica_item($quizid);

        return true;
    }

    /**
     * @param \core\event\user_graded $event
     * @return bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function user_graded(\core\event\user_graded $event) {
        global $DB;

        if (!get_config('local_kicaquizgrading', 'enable')) {
            return false;
        }

        $data = $event->get_data();
        $itemid = $data['other']['itemid'];
        $userid = $event->relateduserid;

        if (!$gradeitem = $DB->get_record('grade_items', ['id' => $itemid, 'itemtype' => 'mod', 'itemmodule' => 'quiz'])) {
            return true;
        }

        $quiz = $DB->get_record('quiz', ['id' => $gradeitem->iteminstance]);

        if (!$cm = get_coursemodule_from_instance('quiz', $quiz->id)) {
            return true;
        }
        if (!helper::quiz_has_kica_tag($cm->id)) {
            return true;
        }

        if (!class_exists('\local_kica\kica_item')) {
            return true;
        }

        $attempts = quiz_get_user_attempts($quiz->id, $userid);
        $attempt = self::get_graded_attempt($quiz, $attempts);
        if (!($attempt->id ?? false)){
            return true;
        }

        $attemptobj = \quiz_attempt::create($attempt->id);
        helper::update_kica_gradebook($attemptobj);

        return true;
    }

    /**
     * @param $quiz
     * @param $attempts
     * @return false|mixed|null
     */
    public static function get_graded_attempt($quiz, $attempts) {
        switch ($quiz->grademethod) {

            case QUIZ_ATTEMPTFIRST:
                $firstattempt = reset($attempts);
                return $firstattempt;

            case QUIZ_ATTEMPTLAST:
                $lastattempt = end($attempts);
                return $lastattempt;

            case QUIZ_GRADEAVERAGE:
               return false;

            case QUIZ_GRADEHIGHEST:
            default:
                $max = null;
                $maxattempt = null;
                foreach ($attempts as $attempt) {
                    if ($attempt->sumgrades > $max) {
                        $max = $attempt->sumgrades;
                        $maxattempt = $attempt;
                    }
                }
                return $maxattempt;
        }
    }
}
